<?php
/**
 * Admin Page Renderer for Delcampe Plugin
 * File: class-admin-page.php
 * Version: 1.1.2.1
 *
 * Complete implementation with settings, category mapping, persistence, and saved category listing
 * Enhanced with branch import functionality for category trees
 * 
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Admin
 * @since 1.0.0
 * @version 1.1.2.1
 * 
 * Changelog:
 * 1.1.2.1 - Fixed translation loading issue by removing activation hook
 * 1.1.2.0 - Enhanced documentation and improved error handling
 * 1.1.1.0 - Fixed redirect issues after category save/import
 * 1.1.0.0 - Added branch import and improved navigation
 * 1.0.0.0 - Initial release with basic category management
 */

// Exit if accessed directly to prevent direct file access
if (!defined('ABSPATH')) exit;

// Include category API for fetching Delcampe categories
require_once plugin_dir_path(__FILE__) . '../includes/class-delcampe-category-api.php';

/**
 * Class Delcampe_Admin_Page
 * 
 * Handles rendering of all admin pages for the Delcampe integration plugin.
 * Manages settings registration, form processing, and UI generation.
 * Uses static methods as callbacks for WordPress admin pages.
 * 
 * @since 1.0.0
 * @version 1.1.2.1
 */
class Delcampe_Admin_Page {

    /**
     * Initialize the admin page functionality
     * 
     * Sets up WordPress hooks for settings registration and form handling.
     * Called during plugin initialization to prepare admin functionality.
     * Enhanced in version 1.1.2.0 with improved action handling.
     * 
     * @since 1.0.0
     * @version 1.1.2.0
     */
    public static function init() {
        // Register plugin settings with WordPress Settings API
        add_action('admin_init', array(__CLASS__, 'register_settings'));
        
        // Handle category save form submissions
        add_action('admin_post_save_delcampe_categories', array(__CLASS__, 'handle_category_save'));
        
        // Handle branch import form submissions
        add_action('admin_post_import_delcampe_branch', array(__CLASS__, 'handle_branch_import'));
        
        // Add AJAX handlers for category operations
        add_action('wp_ajax_delcampe_count_subcategories', array(__CLASS__, 'ajax_count_subcategories'));
        add_action('wp_ajax_delcampe_import_progress', array(__CLASS__, 'ajax_import_progress'));
        add_action('wp_ajax_delcampe_rebuild_category_flags', array(__CLASS__, 'ajax_rebuild_category_flags'));
        
        // Handle category deletion via AJAX
        add_action('wp_ajax_delcampe_delete_category', array(__CLASS__, 'ajax_delete_category'));
        add_action('wp_ajax_delcampe_bulk_delete_categories', array(__CLASS__, 'ajax_bulk_delete_categories'));
        add_action('wp_ajax_delcampe_refresh_categories', array(__CLASS__, 'ajax_refresh_categories'));
        add_action('wp_ajax_delcampe_check_category_leaf', array(__CLASS__, 'ajax_check_category_leaf'));
    }

    /**
     * Register plugin settings with WordPress
     * 
     * Uses WordPress Settings API to properly register and sanitize settings.
     * Creates settings sections and fields for the plugin configuration page.
     * Enhanced in version 1.1.2.0 with additional setting validations.
     * 
     * @since 1.0.0
     * @version 1.1.2.0
     */
    public static function register_settings() {
        // Register API key setting with sanitization callback
        register_setting(
            'delcampe_integration_settings', 
            'delcampe_api_key',
            array(
                'sanitize_callback' => array(__CLASS__, 'sanitize_api_key'),
                'default' => ''
            )
        );
        
        // Register logging enable/disable setting
        register_setting(
            'delcampe_integration_settings', 
            'delcampe_enable_logging',
            array(
                'sanitize_callback' => 'absint',
                'default' => 0
            )
        );

        // Add main settings section for API configuration
        add_settings_section(
            'delcampe_main_section',           // Section ID
            'API Configuration',               // Section title (hardcoded to avoid early translation)
            array(__CLASS__, 'render_section_description'), // Section description callback
            'delcampe_integration'            // Page slug
        );

        // Add API key field to settings section
        add_settings_field(
            'delcampe_api_key',                              // Field ID
            'Delcampe API Key',                              // Field label (hardcoded to avoid early translation)
            array(__CLASS__, 'render_api_key_field'),        // Render callback
            'delcampe_integration',                          // Page slug
            'delcampe_main_section'                          // Section ID
        );

        // Add logging toggle field to settings section
        add_settings_field(
            'delcampe_enable_logging',                       // Field ID
            'Enable Logging',                                // Field label (hardcoded to avoid early translation)
            array(__CLASS__, 'render_logging_field'),        // Render callback
            'delcampe_integration',                          // Page slug
            'delcampe_main_section'                          // Section ID
        );
    }
    
    /**
     * Render section description
     * 
     * Provides context for the API configuration section.
     * 
     * @since 1.1.2.0
     */
    public static function render_section_description() {
        echo '<p>' . __('Configure your Delcampe API settings below. All communication with Delcampe uses XML format.', 'wc-delcampe-integration') . '</p>';
    }
    
    /**
     * Sanitize API key input
     * 
     * Ensures API key is properly formatted and secure.
     * 
     * @since 1.1.2.0
     * @param string $input The API key input
     * @return string Sanitized API key
     */
    public static function sanitize_api_key($input) {
        // Remove any whitespace
        $input = trim($input);
        
        // Only allow alphanumeric characters, dashes, and underscores
        $input = preg_replace('/[^a-zA-Z0-9_-]/', '', $input);
        
        return $input;
    }

    /**
     * Render the saved categories page
     * 
     * Displays a table of all categories that have been selected
     * for mapping between WooCommerce and Delcampe.
     * Enhanced in version 1.1.2.0 with improved statistics and actions.
     * 
     * @since 1.0.0
     * @version 1.1.2.0
     */
    public static function render_saved_categories_page() {
        global $wpdb;
        
        // Get table name with proper prefix using constant
        $table = $wpdb->prefix . DELCAMPE_TABLE_SELECTED_CATEGORIES;
        
        // Query all saved categories with enhanced information
        $results = $wpdb->get_results(
            "SELECT category_id, category_name, parent_path, import_method, date_added 
             FROM $table 
             ORDER BY date_added DESC", 
            ARRAY_A
        );

        // Start page wrapper
        echo '<div class="wrap">';
        echo '<h1>' . __('Saved Delcampe Categories', 'wc-delcampe-integration') . '</h1>';

        // Add summary statistics
        $total_categories = count($results);
        $branch_imports = $wpdb->get_var("SELECT COUNT(DISTINCT parent_path) FROM $table WHERE import_method = 'branch'");
        $manual_imports = $total_categories - $wpdb->get_var("SELECT COUNT(*) FROM $table WHERE import_method = 'branch'");
        
        echo '<div class="category-stats" style="background: #fff; padding: 15px; margin-bottom: 20px; border-left: 4px solid #0073aa;">';
        echo '<p>' . sprintf(
            __('Total categories: <strong>%d</strong> | Branch imports: <strong>%d</strong> | Manual selections: <strong>%d</strong>', 'wc-delcampe-integration'),
            $total_categories,
            $branch_imports,
            $manual_imports
        ) . '</p>';
        // Rebuild flags controls
        $nonce = wp_create_nonce('delcampe_rebuild_flags');
        echo '<button id="rebuild-category-flags" class="button button-primary" data-nonce="' . esc_attr($nonce) . '">' . esc_html__('Rebuild Category Flags', 'wc-delcampe-integration') . '</button>';
        echo ' <span id="rebuild-flags-status" style="margin-left:10px;color:#666;"></span>';
        // Refresh saved categories
        $refresh_nonce = wp_create_nonce('delcampe_refresh_categories');
        echo '<div style="margin-top:10px;">';
        echo '<button id="refresh-saved-categories" class="button">' . esc_html__('Refresh Saved Categories', 'wc-delcampe-integration') . '</button>';
        echo ' <span id="refresh-cats-status" style="margin-left:10px;color:#666;">' . esc_html__('Updates names, paths, and remaps retired IDs.', 'wc-delcampe-integration') . '</span>';
        echo '<input type="hidden" id="refresh-cats-nonce" value="' . esc_attr($refresh_nonce) . '" />';
        echo '</div>';
        echo '</div>';

        // Check if any categories have been saved
        if (empty($results)) {
            echo '<p>' . __('No categories saved yet. Visit the Categories tab to select categories for synchronization.', 'wc-delcampe-integration') . '</p>';
        } else {
            // Add bulk actions
            echo '<div class="tablenav top">';
            echo '<div class="alignleft actions bulkactions">';
            echo '<label for="bulk-action-selector-top" class="screen-reader-text">' . __('Select bulk action', 'wc-delcampe-integration') . '</label>';
            echo '<select id="bulk-action-selector-top">';
            echo '<option value="-1">' . __('Bulk Actions', 'wc-delcampe-integration') . '</option>';
            echo '<option value="delete">' . __('Delete', 'wc-delcampe-integration') . '</option>';
            echo '</select>';
            echo '<input type="button" id="doaction" class="button action" value="' . __('Apply', 'wc-delcampe-integration') . '">';
            echo '</div>';
            echo '</div>';
            
            // Display categories in a WordPress-styled table
            echo '<table class="wp-list-table widefat fixed striped">';
            echo '<thead><tr>';
            echo '<td class="manage-column column-cb check-column"><input type="checkbox" id="cb-select-all-1"></td>';
            echo '<th>' . __('Category ID', 'wc-delcampe-integration') . '</th>';
            echo '<th>' . __('Category Name', 'wc-delcampe-integration') . '</th>';
            echo '<th>' . __('Category Path', 'wc-delcampe-integration') . '</th>';
            echo '<th>' . __('Import Method', 'wc-delcampe-integration') . '</th>';
            echo '<th>' . __('Date Added', 'wc-delcampe-integration') . '</th>';
            echo '<th>' . __('Actions', 'wc-delcampe-integration') . '</th>';
            echo '</tr></thead><tbody>';
            
            // Loop through and display each saved category
            foreach ($results as $row) {
                echo '<tr>';
                echo '<th scope="row" class="check-column"><input type="checkbox" name="categories[]" value="' . esc_attr($row['category_id']) . '"></th>';
                echo '<td>' . esc_html($row['category_id']) . '</td>';
                echo '<td>' . esc_html($row['category_name'] ?? 'N/A') . '</td>';
                echo '<td>' . esc_html($row['parent_path'] ?? 'N/A') . '</td>';
                echo '<td><span class="import-method-' . esc_attr($row['import_method']) . '">' . esc_html(ucfirst($row['import_method'] ?? 'manual')) . '</span></td>';
                echo '<td>' . esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($row['date_added']))) . '</td>';
                echo '<td>';
                echo '<a href="#" class="delete-category" data-id="' . esc_attr($row['category_id']) . '">' . __('Delete', 'wc-delcampe-integration') . '</a>';
                echo '</td>';
                echo '</tr>';
            }
            
            echo '</tbody></table>';
        }

        // Add JavaScript for handling deletions and rebuild
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            // Show inline progress while branch import is running
            $('#branch-import-form').on('submit', function() {
                var $btn = $('#import_branch');
                var $status = $('#branch-import-progress');
                // Change button label and disable
                if ($btn.length) {
                    try { $btn.val('<?php echo esc_js(__('Importing…', 'wc-delcampe-integration')); ?>'); } catch(e) {}
                    $btn.prop('disabled', true).addClass('disabled');
                }
                // Show spinner + message
                var spinner = '<span class="spinner is-active" style="float:none; visibility:visible; margin-right:6px;"></span>';
                $status.html(spinner + '<?php echo esc_js(__('Importing branch… this can take a while. Please keep this tab open.', 'wc-delcampe-integration')); ?>').show();
            });
            // Verify leaf/parent status for unknown nodes
            var checkNonce = '<?php echo wp_create_nonce('delcampe_check_leaf'); ?>';
            $('.leaf-status').each(function(){
                var $status = $(this);
                var id = $status.data('id');
                if (!id) return;
                if ($status.text().indexOf('Checking') === -1) return; // only unknown
                $.post(ajaxurl, {
                    action: 'delcampe_check_category_leaf',
                    nonce: checkNonce,
                    category_id: id
                }).done(function(resp){
                    var $cb = $('.category-checkbox[data-id="' + id + '"]');
                    if (resp && resp.success && resp.data) {
                        if (resp.data.leaf === true) {
                            $status.text('<?php echo esc_js(__('✓ Leaf category - can be selected', 'wc-delcampe-integration')); ?>').css('color', '#00a32a');
                            $cb.prop('disabled', false).css('opacity', 1);
                        } else if (resp.data.leaf === false) {
                            $status.text('<?php echo esc_js(__('⚠️ Has subcategories - cannot select', 'wc-delcampe-integration')); ?>').css('color', '#d63638');
                            $cb.prop('disabled', true).css('opacity', 0.5);
                        }
                    }
                });
            });
            // Rebuild flags in batches
            $('#rebuild-category-flags').on('click', function(e) {
                e.preventDefault();
                var $btn = $(this);
                var $status = $('#rebuild-flags-status');
                var nonce = $btn.data('nonce');
                var processed = 0;
                var batchSize = 10;
                $btn.prop('disabled', true).text('<?php echo esc_js(__('Rebuilding...', 'wc-delcampe-integration')); ?>');
                $status.text('<?php echo esc_js(__('Starting...', 'wc-delcampe-integration')); ?>');

                function runBatch(){
                    $.post(ajaxurl, {
                        action: 'delcampe_rebuild_category_flags',
                        nonce: nonce,
                        batch_size: batchSize
                    }).done(function(resp){
                        if (!resp || !resp.success) {
                            $status.text((resp && resp.data) ? resp.data : '<?php echo esc_js(__('Error during rebuild', 'wc-delcampe-integration')); ?>');
                            $btn.prop('disabled', false).text('<?php echo esc_js(__('Rebuild Category Flags', 'wc-delcampe-integration')); ?>');
                            return;
                        }
                        processed += parseInt(resp.data.processed || 0, 10);
                        var remaining = parseInt(resp.data.remaining || 0, 10);
                        $status.text(processed + ' <?php echo esc_js(__('updated,', 'wc-delcampe-integration')); ?> ' + remaining + ' <?php echo esc_js(__('remaining', 'wc-delcampe-integration')); ?>');
                        if (remaining > 0) {
                            setTimeout(runBatch, 250);
                        } else {
                            $status.text('<?php echo esc_js(__('Rebuild complete. Refreshing…', 'wc-delcampe-integration')); ?>');
                            setTimeout(function(){ location.reload(); }, 600);
                        }
                    }).fail(function(){
                        $status.text('<?php echo esc_js(__('Network error during rebuild', 'wc-delcampe-integration')); ?>');
                        $btn.prop('disabled', false).text('<?php echo esc_js(__('Rebuild Category Flags', 'wc-delcampe-integration')); ?>');
                    });
                }
                runBatch();
            });
            // Handle individual category deletion
            $('.delete-category').on('click', function(e) {
                e.preventDefault();
                
                if (!confirm('<?php _e('Are you sure you want to delete this category?', 'wc-delcampe-integration'); ?>')) {
                    return;
                }
                
                var categoryId = $(this).data('id');
                var row = $(this).closest('tr');
                
                $.post(ajaxurl, {
                    action: 'delcampe_delete_category',
                    category_id: categoryId,
                    _ajax_nonce: '<?php echo wp_create_nonce('delcampe_delete_category'); ?>'
                }, function(response) {
                    if (response.success) {
                        row.fadeOut(function() {
                            row.remove();
                        });
                    } else {
                        alert('<?php _e('Error deleting category', 'wc-delcampe-integration'); ?>');
                    }
                });
            });
            
            // Refresh saved categories in batches
            $('#refresh-saved-categories').on('click', function(e){
                e.preventDefault();
                var ids = [];
                $('input[name="categories[]"]').each(function(){ ids.push($(this).val()); });
                if (ids.length === 0) { alert('<?php echo esc_js(__('No categories to refresh', 'wc-delcampe-integration')); ?>'); return; }
                var $btn = $(this);
                var $status = $('#refresh-cats-status');
                var nonce = $('#refresh-cats-nonce').val();
                var batchSize = 20;
                var index = 0, updated=0, remapped=0, unresolved=0;
                $btn.prop('disabled', true).text('<?php echo esc_js(__('Refreshing...', 'wc-delcampe-integration')); ?>');
                function runBatch(){
                    var slice = ids.slice(index, index + batchSize);
                    if (slice.length === 0) {
                        $status.text(updated + ' <?php echo esc_js(__('updated,', 'wc-delcampe-integration')); ?> ' + remapped + ' <?php echo esc_js(__('remapped,', 'wc-delcampe-integration')); ?> ' + unresolved + ' <?php echo esc_js(__('unresolved', 'wc-delcampe-integration')); ?>');
                        setTimeout(function(){ location.reload(); }, 600);
                        return;
                    }
                    $.post(ajaxurl, {
                        action: 'delcampe_refresh_categories',
                        ids: slice,
                        _ajax_nonce: nonce
                    }).done(function(resp){
                        if (resp && resp.success) {
                            updated += parseInt(resp.data.updated || 0, 10);
                            remapped += parseInt(resp.data.remapped || 0, 10);
                            unresolved += parseInt(resp.data.unresolved || 0, 10);
                            $status.text('<?php echo esc_js(__('Progress:', 'wc-delcampe-integration')); ?> ' + (index + slice.length) + '/' + ids.length);
                            index += batchSize;
                            setTimeout(runBatch, 200);
                        } else {
                            $status.text((resp && resp.data) ? resp.data : '<?php echo esc_js(__('Refresh failed', 'wc-delcampe-integration')); ?>');
                            $btn.prop('disabled', false).text('<?php echo esc_js(__('Refresh Saved Categories', 'wc-delcampe-integration')); ?>');
                        }
                    }).fail(function(){
                        $status.text('<?php echo esc_js(__('Network error during refresh', 'wc-delcampe-integration')); ?>');
                        $btn.prop('disabled', false).text('<?php echo esc_js(__('Refresh Saved Categories', 'wc-delcampe-integration')); ?>');
                    });
                }
                runBatch();
            });

            // Handle bulk actions
            $('#doaction').on('click', function() {
                var action = $('#bulk-action-selector-top').val();
                if (action === 'delete') {
                    if (!confirm('<?php _e('Are you sure you want to delete the selected categories?', 'wc-delcampe-integration'); ?>')) {
                        return;
                    }
                    
                    var selected = [];
                    $('input[name="categories[]"]:checked').each(function() {
                        selected.push($(this).val());
                    });
                    
                    if (selected.length === 0) {
                        alert('<?php _e('No categories selected', 'wc-delcampe-integration'); ?>');
                        return;
                    }
                    
                    // Process deletions via AJAX
                    var $btn = $(this);
                    $btn.prop('disabled', true);
                    $.post(ajaxurl, {
                        action: 'delcampe_bulk_delete_categories',
                        ids: selected,
                        _ajax_nonce: '<?php echo wp_create_nonce('delcampe_bulk_delete'); ?>'
                    }, function(resp) {
                        if (resp && resp.success) {
                            // Remove deleted rows
                            selected.forEach(function(id){
                                $('input[name="categories[]"][value="' + id + '"]').closest('tr').remove();
                            });
                        } else {
                            alert((resp && resp.data) ? resp.data : '<?php _e('Bulk delete failed', 'wc-delcampe-integration'); ?>');
                        }
                        $btn.prop('disabled', false);
                    }).fail(function(){
                        alert('<?php _e('Network error during delete', 'wc-delcampe-integration'); ?>');
                        $btn.prop('disabled', false);
                    });
                }
            });
        });
        </script>
        <?php

        echo '</div>';
    }
    
    /**
     * AJAX handler for category deletion
     * 
     * Processes individual category deletion requests via AJAX.
     * 
     * @since 1.1.2.0
     */
    public static function ajax_delete_category() {
        // Verify nonce
        if (!check_ajax_referer('delcampe_delete_category', '_ajax_nonce', false)) {
            wp_send_json_error('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        // Get category ID
        $category_id = isset($_POST['category_id']) ? sanitize_text_field($_POST['category_id']) : '';
        
        if (empty($category_id)) {
            wp_send_json_error('No category ID provided');
        }
        
        // Delete from database
        global $wpdb;
        $table = $wpdb->prefix . DELCAMPE_TABLE_SELECTED_CATEGORIES;
        
        $result = $wpdb->delete(
            $table,
            array('category_id' => $category_id),
            array('%s')
        );
        
        if ($result !== false) {
            wp_send_json_success(array('deleted' => $category_id));
        } else {
            wp_send_json_error('Database error');
        }
    }

    /**
     * AJAX handler for bulk deletion of categories
     *
     * Deletes multiple selected categories by ID in a single request.
     *
     * @since 1.1.2.2
     */
    public static function ajax_bulk_delete_categories() {
        // Verify nonce
        if (!check_ajax_referer('delcampe_bulk_delete', '_ajax_nonce', false)) {
            wp_send_json_error(__('Security check failed', 'wc-delcampe-integration'));
        }

        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions', 'wc-delcampe-integration'));
        }

        // Validate IDs
        $ids = isset($_POST['ids']) && is_array($_POST['ids']) ? array_map('sanitize_text_field', $_POST['ids']) : array();
        $ids = array_filter($ids, function($v){ return $v !== '' && is_numeric($v); });
        if (empty($ids)) {
            wp_send_json_error(__('No categories selected', 'wc-delcampe-integration'));
        }

        global $wpdb;
        $table = $wpdb->prefix . DELCAMPE_TABLE_SELECTED_CATEGORIES;

        // Build placeholders for prepared IN clause
        $placeholders = implode(',', array_fill(0, count($ids), '%s'));
        $sql = $wpdb->prepare("DELETE FROM {$table} WHERE category_id IN ($placeholders)", $ids);
        $result = $wpdb->query($sql);

        if ($result === false) {
            wp_send_json_error(__('Database error during delete', 'wc-delcampe-integration'));
        }

        wp_send_json_success(array('deleted' => count($ids)));
    }

    /**
     * AJAX: Refresh saved categories (names, paths, and remap retired IDs)
     * Processes a batch of category IDs and updates rows in-place.
     */
    public static function ajax_refresh_categories() {
        if (!check_ajax_referer('delcampe_refresh_categories', '_ajax_nonce', false)) {
            wp_send_json_error(__('Security check failed', 'wc-delcampe-integration'));
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions', 'wc-delcampe-integration'));
        }

        $ids = isset($_POST['ids']) && is_array($_POST['ids']) ? array_map('sanitize_text_field', $_POST['ids']) : array();
        $ids = array_filter($ids, function($v){ return $v !== '' && is_numeric($v); });
        if (empty($ids)) {
            wp_send_json_error(__('No category IDs provided', 'wc-delcampe-integration'));
        }

        global $wpdb;
        $table = $wpdb->prefix . DELCAMPE_TABLE_SELECTED_CATEGORIES;
        $updated = 0; $remapped = 0; $unresolved = 0;

        foreach ($ids as $cid) {
            // Try to fetch by ID
            $xml = Delcampe_Category_API::get_category_by_id($cid, 'E');
            if (!is_wp_error($xml) && ($xml->category || isset($xml->name) || isset($xml->label))) {
                // Normalize node
                $node = isset($xml->category) ? $xml->category : $xml;
                $name = '';
                if (isset($node->name) && (string)$node->name !== '') { $name = (string)$node->name; }
                elseif (isset($node->label) && (string)$node->label !== '') { $name = (string)$node->label; }
                // Build full path via API helper
                $names = Delcampe_Category_API::get_category_path($cid, 'E');
                if (is_wp_error($names) || !is_array($names) || empty($names)) {
                    // Fallback: use name alone
                    $path_str = __('Root', 'wc-delcampe-integration') . ($name !== '' ? ' > ' . $name : '');
                } else {
                    $path_str = __('Root', 'wc-delcampe-integration') . ' > ' . implode(' > ', $names);
                }
                // Check children flag
                $hc = Delcampe_Category_API::has_children($cid, 'E');
                $has_children_val = (!is_wp_error($hc) && $hc) ? 1 : 0;
                $wpdb->update($table, array(
                    'category_name' => $name,
                    'parent_path'   => $path_str,
                    'has_children'  => $has_children_val,
                ), array('category_id' => $cid), array('%s','%s','%d'), array('%s'));
                $updated++;
                continue;
            }

            // If fetch failed: attempt to remap by walking names in stored parent_path
            $row = $wpdb->get_row($wpdb->prepare("SELECT parent_path FROM {$table} WHERE category_id = %s", $cid), ARRAY_A);
            $names_seq = array();
            if ($row && !empty($row['parent_path'])) {
                $parts = array_map('trim', explode('>', $row['parent_path']));
                // Expect parts like [Root, A, B, C]
                foreach ($parts as $p) {
                    if ($p !== '' && strtolower($p) !== 'root') { $names_seq[] = $p; }
                }
            }
            // Try to resolve by walking down via names
            $resolved_id = self::resolve_id_by_names($names_seq);
            if ($resolved_id) {
                // Update to new ID
                $new_names = Delcampe_Category_API::get_category_path($resolved_id, 'E');
                $path_str = __('Root', 'wc-delcampe-integration') . (is_wp_error($new_names) || empty($new_names) ? '' : ' > ' . implode(' > ', $new_names));
                $leaf_children = Delcampe_Category_API::has_children($resolved_id, 'E');
                $has_children_val = (!is_wp_error($leaf_children) && $leaf_children) ? 1 : 0;
                $leaf_name = (!empty($new_names)) ? end($new_names) : (isset($names_seq[count($names_seq)-1]) ? $names_seq[count($names_seq)-1] : '');
                $wpdb->update($table, array(
                    'category_id'   => $resolved_id,
                    'category_name' => $leaf_name,
                    'parent_path'   => $path_str,
                    'has_children'  => $has_children_val,
                ), array('category_id' => $cid), array('%s','%s','%s','%d'), array('%s'));
                $remapped++;
            } else {
                $unresolved++;
            }
        }

        wp_send_json_success(array('updated' => $updated, 'remapped' => $remapped, 'unresolved' => $unresolved));
    }

    /**
     * AJAX: Check if a category is a leaf (no subcategories)
     */
    public static function ajax_check_category_leaf() {
        if (!check_ajax_referer('delcampe_check_leaf', 'nonce', false)) {
            wp_send_json_error(__('Security check failed', 'wc-delcampe-integration'));
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions', 'wc-delcampe-integration'));
        }
        $cid = isset($_POST['category_id']) ? sanitize_text_field($_POST['category_id']) : '';
        if ($cid === '') {
            wp_send_json_error(__('No category ID provided', 'wc-delcampe-integration'));
        }
        $hc = Delcampe_Category_API::has_children($cid, 'E');
        if (is_wp_error($hc)) {
            // On error, conservatively mark as parent (not selectable) to avoid bad saves
            wp_send_json_success(array('leaf' => false));
        }
        wp_send_json_success(array('leaf' => ($hc ? false : true)));
    }

    /**
     * Helper: resolve category ID by walking a path of names from root.
     */
    private static function resolve_id_by_names($names_seq) {
        if (empty($names_seq)) return 0;
        // Fetch root categories
        $root = Delcampe_Category_API::get_categories(null, 'E');
        if (is_wp_error($root)) return 0;
        $list = null;
        if (isset($root->Notification_Data->body->category)) $list = $root->Notification_Data->body->category;
        elseif (isset($root->category)) $list = $root->category;
        elseif (isset($root->categories->category)) $list = $root->categories->category;
        elseif (isset($root->body->category)) $list = $root->body->category;
        if ($list === null) return 0;
        $parent_id = 0; $found = 0;
        foreach ($names_seq as $name) {
            $name_lc = mb_strtolower(trim($name));
            $match_id = 0;
            foreach ($list as $cat) {
                $label = isset($cat->name) ? (string)$cat->name : (isset($cat->label) ? (string)$cat->label : '');
                if ($label !== '' && mb_strtolower($label) === $name_lc) {
                    $match_id = (int)$cat->id; break;
                }
            }
            if ($match_id === 0) { return 0; }
            $found = $match_id; $parent_id = $match_id;
            // Load next level children for next iteration
            $children = Delcampe_Category_API::get_categories($parent_id, 'E');
            if (is_wp_error($children)) return 0;
            if (isset($children->Notification_Data->body->category)) $list = $children->Notification_Data->body->category;
            elseif (isset($children->category)) $list = $children->category;
            elseif (isset($children->categories->category)) $list = $children->categories->category;
            elseif (isset($children->body->category)) $list = $children->body->category;
            else $list = null;
        }
        return (int)$found;
    }

    /**
     * Handle category save form submission
     * 
     * Processes the form when users select individual categories to save.
     * Enhanced in version 1.1.2.0 with improved duplicate checking and validation.
     * 
     * @since 1.0.0
     * @version 1.1.2.0
     */
    public static function handle_category_save() {
        global $wpdb;
        
        // Verify user permissions
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.', 'wc-delcampe-integration'));
        }
        
        // Get table name with proper prefix
        $table_name = $wpdb->prefix . DELCAMPE_TABLE_SELECTED_CATEGORIES;

        // Check if categories were selected
        if (!empty($_POST['selected_cats']) && is_array($_POST['selected_cats'])) {
            // Get additional form data
            $category_names = isset($_POST['category_names']) ? $_POST['category_names'] : array();
            $parent_path = isset($_POST['parent_path']) ? sanitize_text_field($_POST['parent_path']) : '';
            
            // Sanitize and convert category IDs to integers
            $ids = array_map('intval', $_POST['selected_cats']);
            
            $saved_count = 0;
            $duplicate_count = 0;
            $skipped_parent_count = 0;
            
            // Insert each selected category into the database
            // Load category flags (has_children) cache
            $flags_option_key = 'delcampe_category_meta';
            $flags = get_option($flags_option_key, array());
            // Category manager for determining leaf/parent at save time
            if (!class_exists('Delcampe_Category_Manager')) {
                require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-delcampe-category-manager.php';
            }
            $manager = Delcampe_Category_Manager::get_instance();

            foreach ($ids as $cat_id) {
                $cat_name = isset($category_names[$cat_id]) ? sanitize_text_field($category_names[$cat_id]) : '';
                // If name is missing or appears to be just the ID, resolve real name via API
                if ($cat_name === '' || $cat_name === (string)$cat_id) {
                    $xmln = Delcampe_Category_API::get_category_by_id($cat_id, 'E');
                    if (!is_wp_error($xmln)) {
                        if (isset($xmln->category)) $node = $xmln->category;
                        elseif (isset($xmln->Notification_Data->body->category)) $node = $xmln->Notification_Data->body->category;
                        elseif (isset($xmln->body->category)) $node = $xmln->body->category;
                        else $node = $xmln;
                        if (isset($node->name) && (string)$node->name !== '') { $cat_name = (string)$node->name; }
                        elseif (isset($node->label) && (string)$node->label !== '') { $cat_name = (string)$node->label; }
                    }
                }
                
                // Check if category already exists
                $exists = $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(*) FROM $table_name WHERE category_id = %d",
                    $cat_id
                ));
                
                if (!$exists) {
                    // Determine has_children (API) – tolerate failures and rate limits
                    $hc = $manager->check_category_has_children($cat_id);
                    // If the category has children, skip storing it as a usable category
                    if (!is_wp_error($hc) && (bool)$hc === true) {
                        $skipped_parent_count++;
                        continue;
                    }

                    // Compute full path for this category for accurate storage
                    $full_path = $parent_path;
                    try {
                        $names_for_cat = Delcampe_Category_API::get_category_path($cat_id, 'E');
                        if (!is_wp_error($names_for_cat) && is_array($names_for_cat) && count($names_for_cat) > 0) {
                            $full_path = __('Root', 'wc-delcampe-integration') . ' > ' . implode(' > ', $names_for_cat);
                        }
                    } catch (Exception $e) {
                        // ignore and use posted parent_path
                    }
                    // Fallback: ensure the leaf name is included if we only have parent path
                    if ($full_path === $parent_path && $cat_name !== '') {
                        if ($full_path !== '') {
                            $full_path .= ' > ' . $cat_name;
                        } else {
                            $full_path = $cat_name;
                        }
                    }

                    $has_children_val = (is_wp_error($hc) ? null : 0);
                    $result = $wpdb->insert($table_name, [
                        'category_id' => $cat_id,
                        'category_name' => $cat_name,
                        'parent_path' => $full_path,
                        'import_method' => 'manual',
                        'has_children' => $has_children_val,
                        'date_added' => current_time('mysql')
                    ]);
                    
                    if ($result !== false) {
                        $saved_count++;
                        // Log successful save
                        error_log('[Delcampe Admin v1.1.2.1] Category saved: ID=' . $cat_id . ', Name=' . $cat_name);

                        // Determine and cache has_children flag for this saved category
                        try {
                            if (!isset($flags[$cat_id]) && !is_wp_error($hc)) {
                                    $flags[$cat_id] = array(
                                        'has_children' => (bool)$hc,
                                        'checked_at'   => time(),
                                    );
                                // Also persist to DB if insert omitted due to earlier error
                                if ($has_children_val === null) {
                                    $wpdb->update(
                                        $table_name,
                                        array('has_children' => ((bool)$hc ? 1 : 0)),
                                        array('category_id' => $cat_id),
                                        array('%d'),
                                        array('%d')
                                    );
                                }
                            }
                        } catch (Exception $e) {
                            // ignore caching errors
                        }
                    }
                } else {
                    $duplicate_count++;
                }
            }

            // Persist updated flags
            update_option($flags_option_key, $flags, false);
            
            // Build redirect URL with appropriate message
            $redirect_url = admin_url('admin.php?page=delcampe-settings&tab=categories');
            
            if ($saved_count > 0) {
                $redirect_url = add_query_arg('saved', $saved_count, $redirect_url);
            }

            if ($duplicate_count > 0) {
                $redirect_url = add_query_arg('duplicates', $duplicate_count, $redirect_url);
            }

            if ($skipped_parent_count > 0) {
                $redirect_url = add_query_arg('skipped_parents', $skipped_parent_count, $redirect_url);
            }
        } else {
            // No categories selected
            $redirect_url = admin_url('admin.php?page=delcampe-settings&tab=categories&error=none_selected');
        }

        // Redirect back to settings page categories tab
        wp_redirect($redirect_url);
        exit;
    }

    /**
     * Handle branch import form submission
     * 
     * Processes the import of an entire category branch including all subcategories.
     * Uses recursive fetching to get all descendants of the selected category.
     * Enhanced in version 1.1.2.0 with better error handling and progress tracking.
     * 
     * @since 1.0.0
     * @version 1.1.2.0
     */
    public static function handle_branch_import() {
        global $wpdb;
        
        // Verify nonce for security
        if (!isset($_POST['delcampe_branch_import_nonce']) || 
            !wp_verify_nonce($_POST['delcampe_branch_import_nonce'], 'delcampe_branch_import')) {
            wp_die(__('Security check failed', 'wc-delcampe-integration'));
        }
        
        // Verify user permissions
        if (!current_user_can('manage_options')) {
            wp_die(__('You do not have sufficient permissions to access this page.', 'wc-delcampe-integration'));
        }
        
        // Get form data
        $parent_id = isset($_POST['branch_parent_id']) ? sanitize_text_field($_POST['branch_parent_id']) : '';
        $parent_name = isset($_POST['branch_parent_name']) ? sanitize_text_field($_POST['branch_parent_name']) : '';
        $parent_path = isset($_POST['branch_parent_path']) ? sanitize_text_field($_POST['branch_parent_path']) : '';
        
        if (empty($parent_id) && $parent_id !== '0') {
            wp_redirect(admin_url('admin.php?page=delcampe-settings&tab=categories&error=no_parent'));
            exit;
        }
        
        // Log branch import start
        error_log('[Delcampe Admin v1.1.2.1] Starting branch import for parent ID: ' . $parent_id);
        
        // Set time limit for long operations
        @set_time_limit(300); // 5 minutes
        
        // Resolve accurate parent name and path server-side to avoid UI placeholder issues
        $resolved_parent_name = $parent_name;
        $resolved_parent_path = $parent_path;
        $xml = Delcampe_Category_API::get_category_by_id($parent_id, 'E');
        if (!is_wp_error($xml)) {
            if (isset($xml->category)) $node = $xml->category;
            elseif (isset($xml->Notification_Data->body->category)) $node = $xml->Notification_Data->body->category;
            elseif (isset($xml->body->category)) $node = $xml->body->category;
            else $node = $xml;
            $nm = '';
            if (isset($node->name) && (string)$node->name !== '') { $nm = (string)$node->name; }
            elseif (isset($node->label) && (string)$node->label !== '') { $nm = (string)$node->label; }
            if ($nm !== '') {
                $resolved_parent_name = $nm;
            }
            $names = Delcampe_Category_API::get_category_path($parent_id, 'E');
            if (!is_wp_error($names) && is_array($names) && !empty($names)) {
                $resolved_parent_path = __('Root', 'wc-delcampe-integration') . ' > ' . implode(' > ', $names);
            }
        }

        // Start the recursive import with resolved name/path
        $imported = self::import_category_branch($parent_id, $resolved_parent_name, $resolved_parent_path, 0);
        
        // Log completion
        error_log('[Delcampe Admin v1.1.2.1] Branch import completed. Imported ' . $imported . ' categories');
        
        // Redirect with success message
        wp_redirect(admin_url('admin.php?page=delcampe-settings&tab=categories&imported=' . $imported));
        exit;
    }

    /**
     * Recursively import a category branch
     * 
     * Fetches and stores a category and all its descendants.
     * Enhanced in version 1.1.2.0 with depth limiting and better error handling.
     * 
     * @since 1.0.0
     * @version 1.1.2.0
     * @param string $parent_id The parent category ID
     * @param string $parent_name The parent category name
     * @param string $parent_path The full path to this category
     * @param int $depth Current depth in the tree
     * @return int Number of categories imported
     */
    private static function import_category_branch($parent_id, $parent_name, $parent_path, $depth = 0) {
        global $wpdb;
        $table_name = $wpdb->prefix . DELCAMPE_TABLE_SELECTED_CATEGORIES;
        $imported_count = 0;
        
        // Safety check - limit depth to prevent infinite recursion
        if ($depth > 10) {
            error_log('[Delcampe Admin v1.1.2.1] Maximum depth reached, stopping recursion at depth: ' . $depth);
            return $imported_count;
        }
        
        // If this is not the root level, store the parent category itself
        if (!empty($parent_id) && $parent_id !== '0') {
            // Check if already exists
            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM $table_name WHERE category_id = %s",
                $parent_id
            ));
            
                if (!$exists) {
                    $result = $wpdb->insert($table_name, [
                        'category_id' => $parent_id,
                        'category_name' => $parent_name,
                        'parent_path' => $parent_path,
                        'import_method' => 'branch',
                        'depth' => $depth,
                        'date_added' => current_time('mysql')
                    ]);
                    
                    if ($result !== false) {
                        $imported_count++;
                        // Cache parent as non-leaf because it has children by definition
                        $flags = get_option('delcampe_category_meta', array());
                        $flags[$parent_id] = array(
                            'has_children' => true,
                            'checked_at'   => time(),
                        );
                        update_option('delcampe_category_meta', $flags, false);
                    }
                }
        }
        
        // Fetch subcategories from API
        $subcategories = Delcampe_Category_API::get_categories($parent_id);
        
        // Handle API errors
        if (is_wp_error($subcategories)) {
            error_log('[Delcampe Admin v1.1.2.1] Error fetching subcategories: ' . $subcategories->get_error_message());
            return $imported_count;
        }
        
        // Check different possible XML structures
        $category_list = null;
        if (isset($subcategories->Notification_Data->body->category)) {
            $category_list = $subcategories->Notification_Data->body->category;
        } elseif (isset($subcategories->category)) {
            $category_list = $subcategories->category;
        } elseif (isset($subcategories->categories->category)) {
            $category_list = $subcategories->categories->category;
        } elseif (isset($subcategories->body->category)) {
            $category_list = $subcategories->body->category;
        }
        
        if ($category_list !== null) {
                $child_count = 0;
                foreach ($category_list as $cat) {
                    $cat_id = (string) $cat->id;
                    $cat_name = isset($cat->name) ? (string)$cat->name : (isset($cat->label) ? (string)$cat->label : '');
                    
                    // Skip empty entries
                    if (empty($cat_id) || empty($cat_name)) {
                        continue;
                    }
                    $child_count++;
                    
                    // Build the full path
                    $cat_path = empty($parent_path) ? $cat_name : $parent_path . ' > ' . $cat_name;

                    // Cache the has_children flag from the API payload if available
                    $has_children = isset($cat->has_children) ? ((string)$cat->has_children === 'true') : null;
                    if ($has_children !== null) {
                        $flags = get_option('delcampe_category_meta', array());
                        $flags[$cat_id] = array(
                            'has_children' => (bool)$has_children,
                            'checked_at'   => time(),
                        );
                        update_option('delcampe_category_meta', $flags, false);
                    }
                
                // Recursively import this subcategory and its children
                $imported_count += self::import_category_branch($cat_id, $cat_name, $cat_path, $depth + 1);
                
                // Add small delay to avoid overwhelming the API
                if ($imported_count % 10 === 0) {
                    usleep(100000); // 0.1 second delay every 10 categories
                }
            }
            // If we discovered children, mark current parent as non-leaf
            if ($child_count > 0 && !empty($parent_id) && $parent_id !== '0') {
                $wpdb->update(
                    $table_name,
                    array('has_children' => 1),
                    array('category_id' => $parent_id),
                    array('%d'),
                    array('%s')
                );
            }
        }
        
        return $imported_count;
    }

    /**
     * AJAX handler to count subcategories
     * 
     * Returns the number of subcategories for a given parent category.
     * SIMPLIFIED VERSION - Only counts immediate children, not recursive
     * Enhanced in version 1.1.2.0 to prevent excessive API calls.
     * 
     * @since 1.0.0
     * @version 1.1.2.0
     */
    public static function ajax_count_subcategories() {
        $parent_id = isset($_POST['parent_id']) ? sanitize_text_field($_POST['parent_id']) : '';
        
        if (empty($parent_id) && $parent_id !== '0') {
            wp_send_json_error(__('No parent ID provided', 'wc-delcampe-integration'));
        }
        
        // For now, return a simple message instead of counting recursively
        // This prevents the excessive API calls
        wp_send_json_success([
            'count' => 'many', 
            'message' => __('This branch contains subcategories', 'wc-delcampe-integration'),
            'version' => '1.1.2.1'
        ]);
    }

    /**
     * Render API key settings field
     * 
     * Outputs the HTML for the API key input field.
     * Retrieves and displays the current value if set.
     * Enhanced in version 1.1.2.0 with improved help text and validation.
     * 
     * @since 1.0.0
     * @version 1.1.2.0
     */
    public static function render_api_key_field() {
        // Get current API key value, escape for display
        $value = esc_attr(get_option('delcampe_api_key', ''));
        
        // Output text input field with improved styling
        echo "<input type='text' name='delcampe_api_key' value='{$value}' class='regular-text' placeholder='" . esc_attr__('Enter your API key', 'wc-delcampe-integration') . "' />";
        
        // Add help text
        echo '<p class="description">' . __('Enter your Delcampe API key. This is required for authentication with the Delcampe API.', 'wc-delcampe-integration') . '</p>';
        echo '<p class="description">' . sprintf(
            __('All API communication uses %s format (v%s).', 'wc-delcampe-integration'),
            '<strong>' . DELCAMPE_API_FORMAT . '</strong>',
            '1.1.2.1'
        ) . '</p>';
        
        // Add test connection button
        echo '<p><button type="button" class="button" id="test-api-connection">' . __('Test Connection', 'wc-delcampe-integration') . '</button></p>';
        echo '<div id="test-connection-result"></div>';
        
        // Add JavaScript for testing connection
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            $('#test-api-connection').on('click', function() {
                var button = $(this);
                var resultDiv = $('#test-connection-result');
                var apiKey = $('input[name="delcampe_api_key"]').val();
                
                if (!apiKey) {
                    resultDiv.html('<p style="color: red;"><?php _e('Please enter an API key first.', 'wc-delcampe-integration'); ?></p>');
                    return;
                }
                
                button.prop('disabled', true);
                resultDiv.html('<p><?php _e('Testing connection...', 'wc-delcampe-integration'); ?></p>');
                
                // This would need to be implemented with proper AJAX handler
                setTimeout(function() {
                    button.prop('disabled', false);
                    resultDiv.html('<p style="color: green;"><?php _e('Connection test feature coming soon.', 'wc-delcampe-integration'); ?></p>');
                }, 1000);
            });
        });
        </script>
        <?php
    }

    /**
     * AJAX: Rebuild has_children flags in batches for saved categories
     */
    public static function ajax_rebuild_category_flags() {
        if (!check_ajax_referer('delcampe_rebuild_flags', 'nonce', false)) {
            wp_send_json_error(__('Security check failed', 'wc-delcampe-integration'));
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Insufficient permissions', 'wc-delcampe-integration'));
        }

        global $wpdb;
        $table = $wpdb->prefix . DELCAMPE_TABLE_SELECTED_CATEGORIES;

        // Ensure column exists (attempt dbDelta, then ALTER TABLE as fallback)
        $col = $wpdb->get_var( $wpdb->prepare("SHOW COLUMNS FROM {$table} LIKE %s", 'has_children') );
        if (!$col) {
            if (method_exists(__CLASS__, 'create_category_table')) {
                self::create_category_table();
            }
            $col = $wpdb->get_var( $wpdb->prepare("SHOW COLUMNS FROM {$table} LIKE %s", 'has_children') );
            if (!$col) {
                // Fallback: direct ALTER TABLE to add column + index
                $wpdb->query("ALTER TABLE {$table} ADD COLUMN has_children TINYINT(1) NULL");
                // Index (ignore errors if exists later)
                $wpdb->query("CREATE INDEX idx_has_children ON {$table} (has_children)");
                // Recheck
                $col = $wpdb->get_var( $wpdb->prepare("SHOW COLUMNS FROM {$table} LIKE %s", 'has_children') );
                if (!$col) {
                    wp_send_json_error(__('Schema migration failed: cannot add has_children column.', 'wc-delcampe-integration'));
                }
            }
        }

        $batch_size = isset($_POST['batch_size']) ? max(1, min(200, intval($_POST['batch_size']))) : 30;

        // Fast heuristic pass: mark parents that clearly appear in other rows' paths
        // This requires no API calls and can label many parent categories immediately.
        $wpdb->query(
            "UPDATE {$table} t
             SET has_children = 1
             WHERE has_children IS NULL
             AND EXISTS (
               SELECT 1 FROM {$table} t2
               WHERE t2.category_id <> t.category_id
               AND t2.parent_path LIKE CONCAT('% > ', t.category_name, ' > %')
             )"
        );
        $fast_processed = intval($wpdb->rows_affected);

        $rows = $wpdb->get_results( $wpdb->prepare("SELECT category_id FROM {$table} WHERE has_children IS NULL LIMIT %d", $batch_size), ARRAY_A );
        if (!empty($wpdb->last_error)) {
            wp_send_json_error( sprintf(__('Database error: %s', 'wc-delcampe-integration'), $wpdb->last_error) );
        }

        if (empty($rows)) {
            $remaining0 = intval( $wpdb->get_var("SELECT COUNT(*) FROM {$table} WHERE has_children IS NULL") );
            wp_send_json_success(array('processed' => $fast_processed, 'remaining' => $remaining0));
        }

        // Use lightweight API check for remaining unknowns
        if (!class_exists('Delcampe_Category_API')) {
            require_once plugin_dir_path(__FILE__) . '../includes/class-delcampe-category-api.php';
        }

        $processed = $fast_processed;
        foreach ($rows as $row) {
            $cid = $row['category_id'];
            $hc = Delcampe_Category_API::has_children($cid, 'E');
            if (!is_wp_error($hc)) {
                $wpdb->update(
                    $table,
                    array('has_children' => ($hc ? 1 : 0)),
                    array('category_id' => $cid),
                    array('%d'),
                    array('%s')
                );
                $processed++;
            }
        }

        $remaining = intval( $wpdb->get_var("SELECT COUNT(*) FROM {$table} WHERE has_children IS NULL") );
        wp_send_json_success(array('processed' => $processed, 'remaining' => $remaining));
    }

    /**
     * Render logging toggle field
     * 
     * Outputs the HTML for the logging enable/disable checkbox.
     * Allows users to control whether debug logging is active.
     * Enhanced in version 1.1.2.0 with location information and log viewer link.
     * 
     * @since 1.0.0
     * @version 1.1.2.0
     */
    public static function render_logging_field() {
        // Get current logging preference
        $value = get_option('delcampe_enable_logging', false);
        
        // Output checkbox field with label
        echo '<label for="delcampe_enable_logging">';
        echo "<input type='checkbox' id='delcampe_enable_logging' name='delcampe_enable_logging' value='1'" . checked(1, $value, false) . " />";
        echo ' ' . __('Enable debug logging', 'wc-delcampe-integration');
        echo '</label>';
        
        // Add help text
        echo '<p class="description">' . __('Enable debug logging to troubleshoot sync issues.', 'wc-delcampe-integration') . '</p>';
        echo '<p class="description">' . sprintf(
            __('Logs are stored in: %s', 'wc-delcampe-integration'),
            '<code>' . esc_html(DELCAMPE_LOG_FILE) . '</code>'
        ) . '</p>';
        
        // Add link to view logs if logging is enabled
        if ($value) {
            $log_url = admin_url('admin.php?page=delcampe-settings&tab=developer');
            echo '<p class="description">' . sprintf(
                __('View logs in the %s tab.', 'wc-delcampe-integration'),
                '<a href="' . esc_url($log_url) . '">' . __('Developer', 'wc-delcampe-integration') . '</a>'
            ) . '</p>';
        }
        
        echo '<p class="description">' . sprintf(
            __('Log entries include version information (v%s) for easier debugging.', 'wc-delcampe-integration'),
            '1.1.2.1'
        ) . '</p>';
    }

    /**
     * Render main settings page
     * 
     * Displays the plugin's main settings page with API configuration options.
     * Uses WordPress Settings API for proper form handling and security.
     * Enhanced in version 1.1.2.0 with version display and better organization.
     * 
     * @since 1.0.0
     * @version 1.1.2.0
     */
    public static function render_settings_page() {
        echo '<div class="wrap">';
        echo '<h1>' . __('Delcampe Integration Settings', 'wc-delcampe-integration') . '</h1>';
        echo '<p class="description">' . sprintf(
            __('Version %s - Configure your WooCommerce Delcampe Integration', 'wc-delcampe-integration'),
            DELCAMPE_PLUGIN_VERSION
        ) . '</p>';

        // Display any settings errors/updates
        settings_errors('delcampe_integration_settings');

        // WordPress settings form
        // action="options.php" is required for Settings API
        echo '<form method="post" action="options.php">';
        
        // Output security fields for settings page
        settings_fields('delcampe_integration_settings');
        
        // Output setting sections and fields
        do_settings_sections('delcampe_integration');
        
        // Output save button
        submit_button(__('Save Settings', 'wc-delcampe-integration'));
        
        echo '</form>';
        
        // Add additional information panel
        echo '<div class="card" style="margin-top: 20px;">';
        echo '<h2>' . __('Quick Start Guide', 'wc-delcampe-integration') . '</h2>';
        echo '<ol>';
        echo '<li>' . __('Enter your Delcampe API key above and save', 'wc-delcampe-integration') . '</li>';
        echo '<li>' . __('Enable logging if you want to track sync operations', 'wc-delcampe-integration') . '</li>';
        echo '<li>' . __('Visit the Categories tab to select Delcampe categories', 'wc-delcampe-integration') . '</li>';
        echo '<li>' . __('Use Test Sync to verify your configuration', 'wc-delcampe-integration') . '</li>';
        echo '</ol>';
        echo '</div>';
        
        echo '</div>';
    }

    /**
     * Render category mapping page
     * 
     * Enhanced page that allows users to navigate Delcampe category hierarchy
     * and select categories OR import entire branches for mapping.
     * Version 1.1.2.0 includes improved navigation, error handling, and UI.
     * 
     * @since 1.0.0
     * @version 1.1.2.0
     */
    public static function render_category_mapping_page() {
        echo '<div class="wrap">';
        echo '<h1>' . __('Category Mapping', 'wc-delcampe-integration') . '</h1>';

        // Display success/error messages
        if (isset($_GET['saved'])) {
            $count = intval($_GET['saved']);
            echo '<div class="notice notice-success is-dismissible"><p>' . 
                sprintf(_n('%d category saved successfully.', '%d categories saved successfully.', $count, 'wc-delcampe-integration'), $count) . 
                '</p></div>';
        }
        if (isset($_GET['duplicates'])) {
            $count = intval($_GET['duplicates']);
            echo '<div class="notice notice-warning is-dismissible"><p>' . 
                sprintf(_n('%d category was already saved.', '%d categories were already saved.', $count, 'wc-delcampe-integration'), $count) . 
                '</p></div>';
        }
        if (isset($_GET['imported'])) {
            $count = intval($_GET['imported']);
            echo '<div class="notice notice-success is-dismissible"><p>' . 
                sprintf(_n('Successfully imported %d category.', 'Successfully imported %d categories.', $count, 'wc-delcampe-integration'), $count) . 
                '</p></div>';
        }
        if (isset($_GET['skipped_parents'])) {
            $count = intval($_GET['skipped_parents']);
            echo '<div class="notice notice-warning is-dismissible"><p>' . 
                sprintf(_n('%d parent category was skipped (has subcategories).', '%d parent categories were skipped (have subcategories).', $count, 'wc-delcampe-integration'), $count) . 
                '</p></div>';
        }
        if (isset($_GET['error'])) {
            $error_type = sanitize_text_field($_GET['error']);
            $error_messages = array(
                'no_parent' => __('No parent category specified for import.', 'wc-delcampe-integration'),
                'none_selected' => __('No categories were selected.', 'wc-delcampe-integration'),
                'api_error' => __('API error occurred. Please check your settings.', 'wc-delcampe-integration')
            );
            $error_msg = isset($error_messages[$error_type]) ? $error_messages[$error_type] : __('An error occurred. Please try again.', 'wc-delcampe-integration');
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html($error_msg) . '</p></div>';
        }

        // Check if API is configured
        $api_key = get_option('delcampe_api_key', '');
        if (empty($api_key)) {
            echo '<div class="notice notice-warning"><p>' . 
                sprintf(
                    __('Please %s before selecting categories.', 'wc-delcampe-integration'),
                    '<a href="' . admin_url('admin.php?page=delcampe-settings') . '">' . __('configure your API key', 'wc-delcampe-integration') . '</a>'
                ) . 
                '</p></div>';
            echo '</div>';
            return;
        }

        /**
         * Category navigation logic
         * Path is stored as comma-separated category IDs
         * Allows drilling down through category hierarchy
         */
        $path = isset($_POST['delcampe_category_path']) 
            ? explode(',', sanitize_text_field($_POST['delcampe_category_path'])) 
            : [];
        
        // Clean up empty values from path
        $path = array_filter($path, function($val) {
            return $val !== '';
        });
        
        // Get current category ID (last in path)
        $current_id = end($path);
        if ($current_id === false) {
            $current_id = null; // Root level
        }
        
        // Calculate user's depth level (0 = root, 1 = first level, etc.)
        $user_level = count($path);
        
        // Build parent path names using API (Root > ...)
        $parent_path_names = array();
        $breadcrumb_names = array();
        if (!empty($path)) {
            // Resolve names for the current path with API
            $resolved_names = array();
            if ($current_id) {
                $resolved_names = Delcampe_Category_API::get_category_path($current_id, 'E'); // array of names from root to current
                if (is_wp_error($resolved_names)) {
                    $resolved_names = array();
                }
            }
            // Always start breadcrumb with Root for display
            $parent_path_names[] = __('Root', 'wc-delcampe-integration');
            // Align IDs in $path with names in $resolved_names by index; if empty, fall back per-ID lookup
            if (!empty($resolved_names)) {
                foreach ($path as $idx => $pid) {
                    $breadcrumb_names[$pid] = isset($resolved_names[$idx]) ? $resolved_names[$idx] : (string)$pid;
                }
            } else {
                foreach ($path as $pid) {
                    $xml = Delcampe_Category_API::get_category_by_id($pid, 'E');
                    if (!is_wp_error($xml)) {
                        if (isset($xml->category)) $node = $xml->category;
                        elseif (isset($xml->Notification_Data->body->category)) $node = $xml->Notification_Data->body->category;
                        elseif (isset($xml->body->category)) $node = $xml->body->category;
                        else $node = $xml;
                        $nm = '';
                        if (isset($node->name) && (string)$node->name !== '') { $nm = (string)$node->name; }
                        elseif (isset($node->label) && (string)$node->label !== '') { $nm = (string)$node->label; }
                        $breadcrumb_names[$pid] = $nm !== '' ? $nm : (string)$pid;
                    } else {
                        $breadcrumb_names[$pid] = (string)$pid;
                    }
                }
            }
        }
        
        // Display breadcrumb navigation
        if (!empty($path) || $user_level > 0) {
            echo '<div class="category-breadcrumb" style="margin-bottom: 20px; padding: 10px; background: #f0f0f0;">';
            echo '<strong>' . __('Current Path:', 'wc-delcampe-integration') . '</strong> ';
            echo __('Root', 'wc-delcampe-integration');
            
            $breadcrumb_path = array();
            foreach ($path as $p) {
                $breadcrumb_path[] = $p;
                echo ' › <a href="#" class="breadcrumb-link" data-path="' . esc_attr(implode(',', array_slice($breadcrumb_path, 0, -1))) . '">';
                $label = isset($breadcrumb_names[$p]) ? $breadcrumb_names[$p] : (string)$p;
                echo esc_html($label);
                echo '</a>';
            }
            echo '</div>';
        }
        
        // v1.10.35.2: Only load categories when explicitly requested
        $categories = null;
        $load_categories = isset($_POST['load_categories']) && $_POST['load_categories'] === 'true';
        
        if ($load_categories) {
            // Fetch categories from Delcampe API
            $categories = Delcampe_Category_API::get_categories($current_id);

            // Handle API errors
            if (is_wp_error($categories)) {
                echo '<div class="notice notice-error"><p>' . 
                    __('Error fetching categories:', 'wc-delcampe-integration') . ' ' . 
                    esc_html($categories->get_error_message()) . 
                    '</p></div>';
                echo '<p>' . __('Please check your API key in settings and ensure logging is enabled for more details.', 'wc-delcampe-integration') . '</p>';
                echo '</div>';
                return;
            }
        } else {
            // Show button to load categories
            echo '<div class="notice notice-info" style="margin-bottom: 20px;">';
            echo '<p>' . __('Categories are not automatically loaded to reduce API calls.', 'wc-delcampe-integration') . '</p>';
            echo '<form method="post" style="margin: 10px 0;">';
            echo '<input type="hidden" name="load_categories" value="true" />';
            echo '<input type="hidden" name="delcampe_category_path" value="' . esc_attr($path_str) . '" />';
            wp_nonce_field('delcampe_load_categories', 'delcampe_load_nonce');
            echo '<button type="submit" class="button button-primary">' . __('Load Categories from Delcampe', 'wc-delcampe-integration') . '</button>';
            echo '</form>';
            echo '</div>';
        }
        
        // Debug output (only show if logging is enabled and categories were loaded)
        if ($categories && get_option('delcampe_enable_logging', false)) {
            delcampe_debug('[Category Mapping v1.1.2.1] XML Structure: ' . print_r($categories, true));
        }
        
        // Only show navigation and forms if categories were loaded
        if ($categories) {
            // Navigation form - SEPARATE from the selection form
            echo '<form method="post" style="margin-bottom: 20px;" id="navigation-form">';
            echo '<input type="hidden" name="load_categories" value="true" />';
            
            // Back button navigation (if not at root)
            if (!empty($path)) {
                $back_path = $path;
                array_pop($back_path);  // Remove last element
                echo '<button class="button" name="delcampe_category_path" value="' . esc_attr(implode(',', $back_path)) . '">⬅ ' . __('Back', 'wc-delcampe-integration') . '</button>';
            }
            
            echo '</form>';
        }
        
        // Branch import form (available at any level, but only if categories are loaded)
        if ($categories && ($current_id !== null || $user_level > 0)) {
            echo '<div class="branch-import-section" style="background: #f1f1f1; padding: 15px; margin-bottom: 20px; border-radius: 5px;">';
            echo '<h3>' . __('Import Entire Branch', 'wc-delcampe-integration') . '</h3>';
            
            $current_name = __('Current Category', 'wc-delcampe-integration');
            if (!empty($path)) {
                // Get the name of current category from the response if possible
                $current_name = __('Selected Category', 'wc-delcampe-integration');
            }
            
            echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" id="branch-import-form">';
            echo '<input type="hidden" name="action" value="import_delcampe_branch" />';
            wp_nonce_field('delcampe_branch_import', 'delcampe_branch_import_nonce');
            echo '<input type="hidden" name="branch_parent_id" value="' . esc_attr($current_id ?: '0') . '" />';
            echo '<input type="hidden" name="branch_parent_name" value="' . esc_attr($current_name) . '" />';
            // Build accurate branch parent path using breadcrumb IDs → names
            $branch_path_str = __('Root', 'wc-delcampe-integration');
            if (!empty($path)) {
                $names_seq = array();
                foreach ($path as $pid) {
                    $label = isset($breadcrumb_names[$pid]) ? $breadcrumb_names[$pid] : '';
                    if ($label === '') {
                        $xml = Delcampe_Category_API::get_category_by_id($pid, 'E');
                        if (!is_wp_error($xml)) {
                            if (isset($xml->category)) $node = $xml->category;
                            elseif (isset($xml->Notification_Data->body->category)) $node = $xml->Notification_Data->body->category;
                            elseif (isset($xml->body->category)) $node = $xml->body->category;
                            else $node = $xml;
                            if (isset($node->name) && (string)$node->name !== '') { $label = (string)$node->name; }
                            elseif (isset($node->label) && (string)$node->label !== '') { $label = (string)$node->label; }
                        }
                    }
                    $names_seq[] = $label !== '' ? $label : (string)$pid;
                }
                if (!empty($names_seq)) {
                    $branch_path_str .= ' > ' . implode(' > ', $names_seq);
                }
            }
            echo '<input type="hidden" name="branch_parent_path" value="' . esc_attr($branch_path_str) . '" />';
            
            echo '<p>' . __('Import this category and all its subcategories with one click.', 'wc-delcampe-integration') . '</p>';
            echo '<p class="subcategory-count" data-parent="' . esc_attr($current_id ?: '0') . '">' . __('This will import all subcategories recursively.', 'wc-delcampe-integration') . '</p>';
            
            submit_button(__('Import Entire Branch', 'wc-delcampe-integration'), 'primary', 'import_branch', false);
            echo ' <span id="branch-import-progress" style="margin-left:10px; color:#666; display:none;"></span>';
            echo '</form>';
            echo '</div>';
        }
        
        // Individual category selection section (only if categories are loaded)
        if ($categories) {
            echo '<h3>' . __('Select Individual Categories', 'wc-delcampe-integration') . '</h3>';
            
            // Check different possible XML structures
            $categories_found = false;
            $category_list = null;
        
        // Try different XML paths where categories might be
        if (isset($categories->Notification_Data->body->category)) {
            $category_list = $categories->Notification_Data->body->category;
            $categories_found = true;
        } elseif (isset($categories->category)) {
            $category_list = $categories->category;
            $categories_found = true;
        } elseif (isset($categories->categories->category)) {
            $category_list = $categories->categories->category;
            $categories_found = true;
        } elseif (isset($categories->body->category)) {
            $category_list = $categories->body->category;
            $categories_found = true;
        }
        
        // Treat empty body as a valid "no children" state for leaf nodes
        if (!$categories_found) {
            $categories_found = true;
            $category_list = array();
        }

        // Display categories if found
        if ($categories_found && $category_list !== null) {
            // Check if it's an empty result
            if (count($category_list) == 0) {
                echo '<p><em>' . __('No subcategories found at this level. This might be a leaf category.', 'wc-delcampe-integration') . '</em></p>';
                // Allow selecting the current category if it is a leaf
                if ($current_id) {
                    // Resolve names by walking the breadcrumb IDs
                    $names_from_ids = array();
                    foreach ($path as $pid) {
                        $xml = Delcampe_Category_API::get_category_by_id($pid, 'E');
                        if (!is_wp_error($xml)) {
                            if (isset($xml->category)) $node = $xml->category;
                            elseif (isset($xml->Notification_Data->body->category)) $node = $xml->Notification_Data->body->category;
                            elseif (isset($xml->body->category)) $node = $xml->body->category;
                            else $node = $xml;
                            $nm = '';
                            if (isset($node->name) && (string)$node->name !== '') { $nm = (string)$node->name; }
                            elseif (isset($node->label) && (string)$node->label !== '') { $nm = (string)$node->label; }
                            $names_from_ids[] = $nm !== '' ? $nm : (string)$pid;
                        } else {
                            $names_from_ids[] = (string)$pid;
                        }
                    }
                    $leaf_name = !empty($names_from_ids) ? end($names_from_ids) : (string)$current_id;
                    $leaf_parent_path = __('Root', 'wc-delcampe-integration');
                    if (!empty($names_from_ids)) {
                        $leaf_parent_path .= ' > ' . implode(' > ', $names_from_ids);
                    }

                    echo '<div class="card" style="padding: 12px; max-width: 800px;">';
                    echo '<h4>' . __('Select This Category', 'wc-delcampe-integration') . '</h4>';
                    echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '">';
                    echo '<input type="hidden" name="action" value="save_delcampe_categories" />';
                    echo '<input type="hidden" name="delcampe_category_path" value="' . esc_attr(implode(',', $path)) . '" />';
                    echo '<input type="hidden" name="parent_path" value="' . esc_attr($leaf_parent_path) . '" />';
                    echo '<label>'; 
                    echo '<input type="checkbox" name="selected_cats[]" value="' . esc_attr($current_id) . '" checked /> ';
                    echo esc_html($leaf_name) . ' (ID: ' . esc_html($current_id) . ')';
                    echo '</label>';
                    echo '<input type="hidden" name="category_names[' . esc_attr($current_id) . ']" value="' . esc_attr($leaf_name) . '" />';
                    echo '<div style="margin-top: 12px;">';
                    echo '<button type="submit" class="button button-primary">' . esc_html__('Save Selected Category', 'wc-delcampe-integration') . '</button>';
                    echo '</div>';
                    echo '</form>';
                    echo '</div>';
                }
            } else {
                echo '<ul style="list-style: none; padding-left: 0;">';
                
                foreach ($category_list as $cat) {
                    $id = (string) $cat->id;
                    $label = isset($cat->name) ? (string)$cat->name : (isset($cat->label) ? (string)$cat->label : '');
                    // Determine children flag: true/false if present; null if unknown (will verify via Ajax)
                    $has_children = isset($cat->has_children) ? ((string) $cat->has_children === 'true') : null;
                    
                    // Skip empty entries
                    if (empty($id) || empty($label)) {
                        continue;
                    }
                    
                    // Style differently based on whether it has children
                    $bg_color = $has_children ? '#fffbf0' : '#f0f8ff';
                    $border_color = $has_children ? '#ffcc00' : '#0073aa';
                    
                    echo '<li style="margin: 10px 0; padding: 10px; background: ' . $bg_color . '; border: 1px solid ' . $border_color . ';">';
                    
                    // Navigation form - SEPARATE for each category
                    $next_path = array_merge($path, [$id]);
                    echo '<div style="float: right;">';
                    // Always allow drilling down; if no subs, next view shows leaf message
                    echo '<form method="post" style="display: inline;">';
                    echo '<input type="hidden" name="load_categories" value="true" />';
                    echo '<button type="submit" name="delcampe_category_path" value="' . esc_attr(implode(',', $next_path)) . '" class="button-link">';
                    echo __('View Subcategories', 'wc-delcampe-integration') . ' →';
                    echo '</button>';
                    echo '</form>';
                    echo '</div>';
                    
                    // Category info with checkbox (disabled for parent categories)
                    echo '<label style="font-weight: bold; ' . (($has_children === true) ? 'color: #666;' : '') . '">';
                    
                    if ($has_children === true) {
                        // Confirmed parent category - disable selection
                        echo '<input type="checkbox" disabled style="opacity: 0.5;" /> ';
                        echo esc_html($label) . ' (ID: ' . esc_html($id) . ')';
                        echo ' <span class="leaf-status" data-id="' . esc_attr($id) . '" style="color: #d63638; font-weight: normal;">⚠️ ' . __('Has subcategories - cannot select', 'wc-delcampe-integration') . '</span>';
                    } elseif ($has_children === false) {
                        // Confirmed leaf
                        echo '<input type="checkbox" class="category-checkbox" data-id="' . esc_attr($id) . '" data-name="' . esc_attr($label) . '" /> ';
                        echo esc_html($label) . ' (ID: ' . esc_html($id) . ')';
                        echo ' <span class="leaf-status" data-id="' . esc_attr($id) . '" style="color: #00a32a; font-weight: normal;">✓ ' . __('Leaf category - can be selected', 'wc-delcampe-integration') . '</span>';
                    } else {
                        // Unknown: disable until verified via Ajax
                        echo '<input type="checkbox" class="category-checkbox" data-id="' . esc_attr($id) . '" data-name="' . esc_attr($label) . '" disabled style="opacity:0.5;" /> ';
                        echo esc_html($label) . ' (ID: ' . esc_html($id) . ')';
                        echo ' <span class="leaf-status" data-id="' . esc_attr($id) . '" style="color: #666; font-weight: normal;">' . __('Checking…', 'wc-delcampe-integration') . '</span>';
                    }
                    
                    echo '</label>';
                    
                    echo '</li>';
                }
                
                echo '</ul>';
                
                // Save form - SEPARATE from navigation
                echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '" id="save-categories-form">';
                echo '<input type="hidden" name="action" value="save_delcampe_categories" />';
                echo '<input type="hidden" name="delcampe_category_path" value="' . esc_attr(implode(',', $path)) . '" />';
                // Persist accurate parent path for manual saves using breadcrumb IDs
                $manual_parent_path = __('Root', 'wc-delcampe-integration');
                $names_from_ids = array();
                if (!empty($path)) {
                    foreach ($path as $pid) {
                        $xml = Delcampe_Category_API::get_category_by_id($pid, 'E');
                        if (!is_wp_error($xml)) {
                            $node = isset($xml->category) ? $xml->category : $xml;
                            $nm = '';
                            if (isset($node->name) && (string)$node->name !== '') { $nm = (string)$node->name; }
                            elseif (isset($node->label) && (string)$node->label !== '') { $nm = (string)$node->label; }
                            $names_from_ids[] = $nm !== '' ? $nm : (string)$pid;
                        } else {
                            $names_from_ids[] = (string)$pid;
                        }
                    }
                    if (!empty($names_from_ids)) {
                        $manual_parent_path .= ' > ' . implode(' > ', $names_from_ids);
                    }
                }
                echo '<input type="hidden" name="parent_path" value="' . esc_attr($manual_parent_path) . '" />';
                
                // Hidden container for selected categories (populated by JavaScript)
                echo '<div id="selected-categories-container"></div>';
                
                // Save button for individual selections
                echo '<div style="margin-top: 20px;">';
                submit_button(__('Save Selected Categories', 'wc-delcampe-integration'), 'secondary', 'save_selected', false);
                echo '</div>';
                
                echo '</form>';
            }
        } else {
            echo '<p>' . __('No categories found at this level. The XML structure might be different than expected.', 'wc-delcampe-integration') . '</p>';
            // Debug: show raw XML structure
            if (get_option('delcampe_enable_logging', false)) {
                echo '<details>';
                echo '<summary>' . __('Debug: Raw XML Response', 'wc-delcampe-integration') . '</summary>';
                echo '<pre>' . esc_html(print_r($categories, true)) . '</pre>';
                echo '</details>';
            }
        }
        } // End if ($categories)
        
        // Enhanced JavaScript for handling selections and navigation
        ?>
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            // Handle checkbox selections
            $('.category-checkbox').on('change', function() {
                updateSelectedCategories();
            });
            
            // Handle breadcrumb navigation
            $('.breadcrumb-link').on('click', function(e) {
                e.preventDefault();
                var path = $(this).data('path');
                
                // Create and submit navigation form
                var form = $('<form>', {
                    'method': 'post',
                    'style': 'display: none;'
                });
                
                form.append($('<input>', {
                    'type': 'hidden',
                    'name': 'load_categories',
                    'value': 'true'
                }));
                
                form.append($('<input>', {
                    'type': 'hidden',
                    'name': 'delcampe_category_path',
                    'value': path
                }));
                
                $('body').append(form);
                form.submit();
            });
            
            function updateSelectedCategories() {
                $('#selected-categories-container').empty();
                
                $('.category-checkbox:checked').each(function() {
                    var id = $(this).data('id');
                    var name = $(this).data('name');
                    
                    $('#selected-categories-container').append(
                        '<input type="hidden" name="selected_cats[]" value="' + id + '" />' +
                        '<input type="hidden" name="category_names[' + id + ']" value="' + name + '" />'
                    );
                });
                
                // Update save button state
                var count = $('.category-checkbox:checked').length;
                var button = $('#save_selected');
                
                if (count > 0) {
                    button.val('<?php echo esc_js(sprintf(__('Save %d Selected Categories', 'wc-delcampe-integration'), '')) ?>' + count + ' <?php echo esc_js(__('Selected Categories', 'wc-delcampe-integration')) ?>');
                    button.prop('disabled', false);
                } else {
                    button.val('<?php echo esc_js(__('Save Selected Categories', 'wc-delcampe-integration')) ?>');
                    button.prop('disabled', true);
                }
            }
            
            // Initialize button state
            updateSelectedCategories();
        });
        </script>
        <?php

        echo '</div>';
    }

    /**
     * Render sync status page
     * 
     * Displays the synchronization log file contents.
     * Helps users monitor and troubleshoot sync operations.
     * Enhanced in version 1.1.2.0 with better log formatting and controls.
     * 
     * @since 1.0.0
     * @version 1.1.2.0
     */
    public static function render_sync_status_page() {
        echo '<div class="wrap">';
        echo '<h1>' . __('Sync Status / Logs', 'wc-delcampe-integration') . '</h1>';
        echo '<p class="description">' . sprintf(__('Version %s', 'wc-delcampe-integration'), '1.1.2.1') . '</p>';

        // Path to log file using constant
        $log_file = DELCAMPE_LOG_FILE;
        
        // Check if log file exists
        if ( file_exists( $log_file ) ) {
            echo '<h2>' . __('Recent Log Entries', 'wc-delcampe-integration') . '</h2>';
            
            // Add log controls
            echo '<div style="margin-bottom: 10px;">';
            echo '<button class="button" id="refresh-log">' . __('Refresh', 'wc-delcampe-integration') . '</button> ';
            echo '<button class="button" id="clear-log">' . __('Clear Log', 'wc-delcampe-integration') . '</button> ';
            echo '<button class="button" id="download-log">' . __('Download Log', 'wc-delcampe-integration') . '</button>';
            echo '</div>';
            
            // Display log contents in a scrollable box
            echo '<pre style="background:#fff; padding:10px; border:1px solid #ccc; max-height:500px; overflow:auto" id="log-content">';
            
            // Read last 1000 lines of log file (to avoid memory issues with large logs)
            $lines = file($log_file);
            $last_lines = array_slice($lines, -1000);
            echo esc_html(implode('', $last_lines));
            
            echo '</pre>';
            
            // Add log file info
            echo '<p>' . sprintf(__('Log file location: %s', 'wc-delcampe-integration'), '<code>' . esc_html($log_file) . '</code>') . '</p>';
            echo '<p>' . sprintf(__('File size: %s', 'wc-delcampe-integration'), size_format(filesize($log_file))) . '</p>';
            echo '<p>' . sprintf(__('Last modified: %s', 'wc-delcampe-integration'), date_i18n(get_option('date_format') . ' ' . get_option('time_format'), filemtime($log_file))) . '</p>';
            
            // Add JavaScript for log controls
            ?>
            <script type="text/javascript">
            jQuery(document).ready(function($) {
                $('#refresh-log').on('click', function() {
                    location.reload();
                });
                
                $('#clear-log').on('click', function() {
                    if (confirm('<?php _e('Are you sure you want to clear the log file?', 'wc-delcampe-integration'); ?>')) {
                        // This would need to be implemented with AJAX
                        alert('<?php _e('Clear log feature coming soon.', 'wc-delcampe-integration'); ?>');
                    }
                });
                
                $('#download-log').on('click', function() {
                    // Create download link
                    var content = $('#log-content').text();
                    var blob = new Blob([content], {type: 'text/plain'});
                    var url = window.URL.createObjectURL(blob);
                    var a = document.createElement('a');
                    a.href = url;
                    a.download = 'delcampe-sync-log-' + new Date().toISOString().slice(0,10) + '.txt';
                    document.body.appendChild(a);
                    a.click();
                    document.body.removeChild(a);
                    window.URL.revokeObjectURL(url);
                });
            });
            </script>
            <?php
        } else {
            // No log file yet
            echo '<p>' . __('No log file found. It will be created automatically when logging is enabled and sync operations occur.', 'wc-delcampe-integration') . '</p>';
            echo '<p>' . sprintf(__('Expected location: %s', 'wc-delcampe-integration'), '<code>' . esc_html($log_file) . '</code>') . '</p>';
            echo '<p>' . __('Make sure logging is enabled in the settings.', 'wc-delcampe-integration') . '</p>';
            
            // Add quick link to enable logging
            $settings_url = admin_url('admin.php?page=delcampe-settings');
            echo '<p><a href="' . esc_url($settings_url) . '" class="button">' . __('Go to Settings', 'wc-delcampe-integration') . '</a></p>';
        }

        echo '</div>';
    }

    /**
     * Create category database table
     * 
     * Creates the custom table for storing selected Delcampe categories.
     * Enhanced in version 1.1.2.0 with additional indexes for performance.
     * 
     * @since 1.0.0
     * @version 1.1.2.1
     */
    public static function create_category_table() {
        global $wpdb;
        
        // Table name with WordPress prefix
        $table_name = $wpdb->prefix . DELCAMPE_TABLE_SELECTED_CATEGORIES;

        // Get charset for table creation
        $charset_collate = $wpdb->get_charset_collate();

        // SQL for table creation with enhanced fields
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            category_id VARCHAR(50) NOT NULL,
            category_name VARCHAR(255) DEFAULT NULL,
            parent_path TEXT DEFAULT NULL,
            import_method VARCHAR(20) DEFAULT 'manual',
            depth INT DEFAULT 0,
            has_children TINYINT(1) NULL,
            date_added DATETIME DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY unique_category (category_id),
            KEY idx_import_method (import_method),
            KEY idx_has_children (has_children),
            KEY idx_date_added (date_added)
        ) $charset_collate;";

        // Use dbDelta for safe table creation/updates
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);
        
        // Log table creation
        error_log('[Delcampe Admin v1.1.2.1] Category table created/updated: ' . $table_name);
    }
}

// Initialize the admin page functionality after init to avoid early hook registration (v1.10.5.12)
add_action( 'init', array( 'Delcampe_Admin_Page', 'init' ) );

// NOTE: Activation hook removed in v1.1.2.1 to fix translation loading issue
// Table creation is now handled in the main plugin file's activation hook
