<?php
/**
 * Delcampe Business Log Admin Interface
 * 
 * Provides a production-friendly interface to view business events
 * and sync activity between WooCommerce and Delcampe.
 * 
 * @package Delcampe_Sync
 * @since 1.10.12.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Business_Log_Admin {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Logger instance
     */
    private $logger;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Get logger instance
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-business-logger.php';
        $this->logger = Delcampe_Business_Logger::get_instance();
        
        // Add admin menu
        add_action('admin_menu', array($this, 'add_menu_page'), 15);
        
        // Add AJAX handlers
        add_action('wp_ajax_delcampe_get_business_events', array($this, 'ajax_get_events'));
        add_action('wp_ajax_delcampe_get_event_stats', array($this, 'ajax_get_stats'));
    }
    
    /**
     * Add menu page
     */
    public function add_menu_page() {
        add_submenu_page(
            'delcampe',
            __('Business Events', 'wc-delcampe-integration'),
            __('Business Events', 'wc-delcampe-integration'),
            'manage_woocommerce',
            'delcampe-business-events',
            array($this, 'render_page')
        );
    }
    
    /**
     * Render the admin page
     */
    public function render_page() {
        // Get statistics
        $stats = $this->logger->get_statistics(7);
        
        // Get filters from request
        $filter_type = isset($_GET['event_type']) ? sanitize_text_field($_GET['event_type']) : '';
        $filter_date_from = isset($_GET['date_from']) ? sanitize_text_field($_GET['date_from']) : date('Y-m-d', strtotime('-7 days'));
        $filter_date_to = isset($_GET['date_to']) ? sanitize_text_field($_GET['date_to']) : date('Y-m-d');
        
        // Pagination
        $current_page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
        $per_page = 50;
        $offset = ($current_page - 1) * $per_page;
        
        // Get events
        $filters = array();
        if ($filter_type) {
            $filters['event_type'] = $filter_type;
        }
        if ($filter_date_from) {
            $filters['date_from'] = $filter_date_from . ' 00:00:00';
        }
        if ($filter_date_to) {
            $filters['date_to'] = $filter_date_to . ' 23:59:59';
        }
        
        $events = $this->logger->get_recent_events($per_page, $offset, $filters);
        
        ?>
        <div class="wrap">
            <h1><?php _e('Business Events Log', 'wc-delcampe-integration'); ?></h1>
            
            <!-- Statistics Dashboard -->
            <div class="delcampe-stats-dashboard">
                <h2><?php _e('Last 7 Days Summary', 'wc-delcampe-integration'); ?></h2>
                <div class="stats-grid">
                    <div class="stat-box">
                        <span class="stat-label"><?php _e('Delcampe Orders', 'wc-delcampe-integration'); ?></span>
                        <span class="stat-value"><?php echo esc_html($stats['totals']['orders_from_delcampe']); ?></span>
                    </div>
                    <div class="stat-box">
                        <span class="stat-label"><?php _e('Website Orders', 'wc-delcampe-integration'); ?></span>
                        <span class="stat-value"><?php echo esc_html($stats['totals']['orders_from_site']); ?></span>
                    </div>
                    <div class="stat-box">
                        <span class="stat-label"><?php _e('Stock Updates', 'wc-delcampe-integration'); ?></span>
                        <span class="stat-value"><?php echo esc_html($stats['totals']['stock_updates']); ?></span>
                    </div>
                    <div class="stat-box <?php echo $stats['totals']['sync_errors'] > 0 ? 'has-errors' : ''; ?>">
                        <span class="stat-label"><?php _e('Sync Errors', 'wc-delcampe-integration'); ?></span>
                        <span class="stat-value"><?php echo esc_html($stats['totals']['sync_errors']); ?></span>
                    </div>
                </div>
            </div>
            
            <!-- Filters -->
            <div class="tablenav top">
                <form method="get">
                    <input type="hidden" name="page" value="delcampe-business-events">
                    
                    <select name="event_type">
                        <option value=""><?php _e('All Events', 'wc-delcampe-integration'); ?></option>
                        <option value="order_from_delcampe" <?php selected($filter_type, 'order_from_delcampe'); ?>><?php _e('Orders from Delcampe', 'wc-delcampe-integration'); ?></option>
                        <option value="order_from_site" <?php selected($filter_type, 'order_from_site'); ?>><?php _e('Website Orders', 'wc-delcampe-integration'); ?></option>
                        <option value="stock_updated" <?php selected($filter_type, 'stock_updated'); ?>><?php _e('Stock Updates', 'wc-delcampe-integration'); ?></option>
                        <option value="stock_synced_to_delcampe" <?php selected($filter_type, 'stock_synced_to_delcampe'); ?>><?php _e('Stock Synced to Delcampe', 'wc-delcampe-integration'); ?></option>
                        <option value="listing_created" <?php selected($filter_type, 'listing_created'); ?>><?php _e('Listings Created', 'wc-delcampe-integration'); ?></option>
                        <option value="listing_updated" <?php selected($filter_type, 'listing_updated'); ?>><?php _e('Listings Updated', 'wc-delcampe-integration'); ?></option>
                        <option value="listing_ended" <?php selected($filter_type, 'listing_ended'); ?>><?php _e('Listings Ended', 'wc-delcampe-integration'); ?></option>
                        <option value="sync_error" <?php selected($filter_type, 'sync_error'); ?>><?php _e('Sync Errors', 'wc-delcampe-integration'); ?></option>
                    </select>
                    
                    <input type="date" name="date_from" value="<?php echo esc_attr($filter_date_from); ?>" placeholder="<?php esc_attr_e('From Date', 'wc-delcampe-integration'); ?>">
                    <input type="date" name="date_to" value="<?php echo esc_attr($filter_date_to); ?>" placeholder="<?php esc_attr_e('To Date', 'wc-delcampe-integration'); ?>">
                    
                    <input type="submit" class="button" value="<?php esc_attr_e('Filter', 'wc-delcampe-integration'); ?>">
                    
                    <?php if ($filter_type || $filter_date_from || $filter_date_to) : ?>
                        <a href="<?php echo admin_url('admin.php?page=delcampe-business-events'); ?>" class="button"><?php _e('Clear Filters', 'wc-delcampe-integration'); ?></a>
                    <?php endif; ?>
                </form>
            </div>
            
            <!-- Events Table -->
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php _e('Time', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('Event', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('Source', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('Product/SKU', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('Quantity Change', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('Stock Level', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('Order', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('Status', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('Details', 'wc-delcampe-integration'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if (empty($events)) : ?>
                        <tr>
                            <td colspan="9"><?php _e('No events found for the selected criteria.', 'wc-delcampe-integration'); ?></td>
                        </tr>
                    <?php else : ?>
                        <?php foreach ($events as $event) : ?>
                            <?php
                            $details = $event->details ? json_decode($event->details, true) : array();
                            $event_class = '';
                            if ($event->status === 'error') {
                                $event_class = 'event-error';
                            } elseif (strpos($event->event_type, 'order') !== false) {
                                $event_class = 'event-order';
                            } elseif (strpos($event->event_type, 'stock') !== false) {
                                $event_class = 'event-stock';
                            }
                            ?>
                            <tr class="<?php echo esc_attr($event_class); ?>">
                                <td><?php echo esc_html(date('Y-m-d H:i:s', strtotime($event->created_at))); ?></td>
                                <td><?php echo esc_html($this->get_event_label($event->event_type)); ?></td>
                                <td>
                                    <span class="source-badge source-<?php echo esc_attr($event->event_source); ?>">
                                        <?php echo esc_html(ucfirst($event->event_source)); ?>
                                    </span>
                                </td>
                                <td>
                                    <?php if ($event->product_id) : ?>
                                        <a href="<?php echo get_edit_post_link($event->product_id); ?>">
                                            <?php echo esc_html($event->product_name ?: 'Product #' . $event->product_id); ?>
                                        </a>
                                        <?php if ($event->sku) : ?>
                                            <br><small>SKU: <?php echo esc_html($event->sku); ?></small>
                                        <?php endif; ?>
                                    <?php else : ?>
                                        -
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($event->quantity_change !== null) : ?>
                                        <span class="quantity-change <?php echo $event->quantity_change < 0 ? 'negative' : 'positive'; ?>">
                                            <?php echo $event->quantity_change > 0 ? '+' : ''; ?><?php echo esc_html($event->quantity_change); ?>
                                        </span>
                                    <?php else : ?>
                                        -
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($event->quantity_after !== null) : ?>
                                        <?php echo esc_html($event->quantity_after); ?>
                                    <?php elseif ($event->quantity_before !== null && $event->quantity_change !== null) : ?>
                                        <?php echo esc_html($event->quantity_before + $event->quantity_change); ?>
                                    <?php else : ?>
                                        -
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <?php if ($event->order_id) : ?>
                                        <a href="<?php echo admin_url('post.php?post=' . $event->order_id . '&action=edit'); ?>">
                                            #<?php echo esc_html($event->order_id); ?>
                                        </a>
                                    <?php else : ?>
                                        -
                                    <?php endif; ?>
                                </td>
                                <td>
                                    <span class="status-badge status-<?php echo esc_attr($event->status); ?>">
                                        <?php echo esc_html(ucfirst($event->status)); ?>
                                    </span>
                                </td>
                                <td>
                                    <?php if (!empty($details)) : ?>
                                        <button type="button" class="button-link" onclick="toggleDetails(this)">
                                            <?php _e('View', 'wc-delcampe-integration'); ?>
                                        </button>
                                        <div class="event-details" style="display:none;">
                                            <pre><?php echo esc_html(json_encode($details, JSON_PRETTY_PRINT)); ?></pre>
                                        </div>
                                    <?php else : ?>
                                        -
                                    <?php endif; ?>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php endif; ?>
                </tbody>
            </table>
            
            <!-- Pagination -->
            <?php if (count($events) >= $per_page) : ?>
                <div class="tablenav bottom">
                    <div class="tablenav-pages">
                        <?php
                        $base_url = add_query_arg(array(
                            'page' => 'delcampe-business-events',
                            'event_type' => $filter_type,
                            'date_from' => $filter_date_from,
                            'date_to' => $filter_date_to
                        ), admin_url('admin.php'));
                        
                        if ($current_page > 1) {
                            echo '<a class="prev-page button" href="' . add_query_arg('paged', $current_page - 1, $base_url) . '">&lsaquo;</a>';
                        }
                        
                        echo '<span class="paging-input">' . sprintf(__('Page %d', 'wc-delcampe-integration'), $current_page) . '</span>';
                        
                        if (count($events) == $per_page) {
                            echo '<a class="next-page button" href="' . add_query_arg('paged', $current_page + 1, $base_url) . '">&rsaquo;</a>';
                        }
                        ?>
                    </div>
                </div>
            <?php endif; ?>
        </div>
        
        <style>
            .delcampe-stats-dashboard {
                background: #fff;
                border: 1px solid #ccd0d4;
                border-radius: 4px;
                padding: 20px;
                margin: 20px 0;
            }
            
            .stats-grid {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
                gap: 20px;
                margin-top: 15px;
            }
            
            .stat-box {
                text-align: center;
                padding: 15px;
                border: 1px solid #e1e1e1;
                border-radius: 4px;
                background: #f9f9f9;
            }
            
            .stat-box.has-errors {
                background: #fff5f5;
                border-color: #ffb3b3;
            }
            
            .stat-label {
                display: block;
                font-size: 12px;
                color: #666;
                margin-bottom: 5px;
            }
            
            .stat-value {
                display: block;
                font-size: 24px;
                font-weight: bold;
                color: #333;
            }
            
            .source-badge {
                display: inline-block;
                padding: 2px 8px;
                border-radius: 3px;
                font-size: 11px;
                font-weight: bold;
                text-transform: uppercase;
            }
            
            .source-delcampe { background: #e8f4ff; color: #0073aa; }
            .source-website { background: #f0f8ea; color: #46783c; }
            .source-admin { background: #f5f5f5; color: #666; }
            .source-system { background: #f0f0f7; color: #5850ec; }
            .source-ajax { background: #fff4e6; color: #ff8c00; }
            
            .quantity-change {
                font-weight: bold;
            }
            
            .quantity-change.negative { color: #d63638; }
            .quantity-change.positive { color: #00a32a; }
            
            .status-badge {
                display: inline-block;
                padding: 2px 8px;
                border-radius: 3px;
                font-size: 11px;
            }
            
            .status-success { background: #d7f7c2; color: #006312; }
            .status-error { background: #ffd6d6; color: #8a0020; }
            .status-pending { background: #fff4d6; color: #996800; }
            
            .event-error { background: #fff5f5; }
            .event-order { background: #f7f9ff; }
            .event-stock { background: #fffef7; }
            
            .event-details {
                margin-top: 10px;
                padding: 10px;
                background: #f5f5f5;
                border-radius: 3px;
            }
            
            .event-details pre {
                margin: 0;
                font-size: 11px;
                max-height: 200px;
                overflow: auto;
            }
        </style>
        
        <script>
            function toggleDetails(button) {
                var details = button.nextElementSibling;
                if (details.style.display === 'none') {
                    details.style.display = 'block';
                    button.textContent = '<?php _e('Hide', 'wc-delcampe-integration'); ?>';
                } else {
                    details.style.display = 'none';
                    button.textContent = '<?php _e('View', 'wc-delcampe-integration'); ?>';
                }
            }
        </script>
        <?php
    }
    
    /**
     * Get human-readable event label
     */
    private function get_event_label($event_type) {
        $labels = array(
            'order_from_delcampe' => __('Order from Delcampe', 'wc-delcampe-integration'),
            'order_from_site' => __('Website Order', 'wc-delcampe-integration'),
            'stock_updated' => __('Stock Updated', 'wc-delcampe-integration'),
            'stock_synced_to_delcampe' => __('Stock Synced to Delcampe', 'wc-delcampe-integration'),
            'listing_created' => __('Listing Created', 'wc-delcampe-integration'),
            'listing_updated' => __('Listing Updated', 'wc-delcampe-integration'),
            'listing_ended' => __('Listing Ended', 'wc-delcampe-integration'),
            'listing_quantity_updated' => __('Listing Quantity Updated', 'wc-delcampe-integration'),
            'price_updated' => __('Price Updated', 'wc-delcampe-integration'),
            'sync_error' => __('Sync Error', 'wc-delcampe-integration'),
            'order_status_changed' => __('Order Status Changed', 'wc-delcampe-integration')
        );
        
        return isset($labels[$event_type]) ? $labels[$event_type] : ucfirst(str_replace('_', ' ', $event_type));
    }
    
    /**
     * AJAX handler to get events
     */
    public function ajax_get_events() {
        check_ajax_referer('delcampe_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $filters = array();
        if (isset($_POST['event_type'])) {
            $filters['event_type'] = sanitize_text_field($_POST['event_type']);
        }
        if (isset($_POST['date_from'])) {
            $filters['date_from'] = sanitize_text_field($_POST['date_from']);
        }
        if (isset($_POST['date_to'])) {
            $filters['date_to'] = sanitize_text_field($_POST['date_to']);
        }
        
        $limit = isset($_POST['limit']) ? intval($_POST['limit']) : 50;
        $offset = isset($_POST['offset']) ? intval($_POST['offset']) : 0;
        
        $events = $this->logger->get_recent_events($limit, $offset, $filters);
        
        wp_send_json_success($events);
    }
    
    /**
     * AJAX handler to get statistics
     */
    public function ajax_get_stats() {
        check_ajax_referer('delcampe_nonce', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $days = isset($_POST['days']) ? intval($_POST['days']) : 7;
        $stats = $this->logger->get_statistics($days);
        
        wp_send_json_success($stats);
    }
}

// Initialize the admin interface
add_action('init', function() {
    if (is_admin()) {
        Delcampe_Business_Log_Admin::get_instance();
    }
}, 25);