<?php
/**
 * Delcampe Category Admin Interface
 * Version: 1.1.0.0
 *
 * Manages the admin UI for category mapping and selection.
 *
 * NOTE: All responses from Delcampe are in XML format. Our backend converts the XML
 * response into a PHP array and then outputs JSON for ease of processing in the admin UI.
 *
 * @package Delcampe_Integration
 * @since 1.0.0
 * @version 1.1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Delcampe_Category_Admin
 * 
 * Handles the category administration interface for mapping Delcampe categories
 * to WooCommerce product categories. Uses AJAX for dynamic category loading.
 * 
 * @version 1.1.0.0
 */
class Delcampe_Category_Admin {

	/**
	 * Singleton instance.
	 *
	 * @var Delcampe_Category_Admin
	 */
	private static $instance = null;

	/**
	 * Option key for storing category mapping.
	 *
	 * @var string
	 */
	private $option_key = 'delcampe_category_mapping';

	/**
	 * Returns the singleton instance.
	 * 
	 * Ensures only one instance of the category admin exists
	 * to prevent duplicate menu registrations and maintain consistency.
	 *
	 * @return Delcampe_Category_Admin
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Private constructor.
	 * 
	 * Sets up WordPress hooks for menu registration, AJAX handling,
	 * and form processing. Uses singleton pattern.
	 */
	private function __construct() {
		// Add menu page.
		add_action( 'admin_menu', array( $this, 'add_menu_page' ) );
		// AJAX handler for dynamic category loading.
		add_action( 'wp_ajax_delcampe_load_subcategories', array( $this, 'handle_ajax_requests' ) );
		// Handle saving category mapping.
		add_action( 'admin_post_delcampe_save_category_mapping', array( $this, 'save_category_mapping' ) );
	}

	/**
	 * Adds the category management submenu page.
	 * 
	 * Creates a submenu item under the main Delcampe Integration menu
	 * for managing category mappings between platforms.
	 * Enhanced in version 1.1.0.0 with improved menu structure.
	 *
	 * @return void
	 */
	public function add_menu_page() {
		add_submenu_page(
			'delcampe_integration',                                  // Parent slug.
			'Category Management',                                   // Page title (hardcoded to avoid early translation).
			'Categories',                                            // Menu title (hardcoded to avoid early translation).
			'manage_options',                                        // Capability.
			'delcampe-categories',                                   // Menu slug.
			array( $this, 'render_category_page' )                   // Callback.
		);
	}

	/**
	 * Renders the category mapping interface.
	 * 
	 * Displays a dynamic tree view of Delcampe categories with checkboxes
	 * for selecting which categories to map. Uses AJAX for loading subcategories
	 * on demand to improve performance and avoid overwhelming the user.
	 * Enhanced in version 1.1.0.0 with better UI and error handling.
	 *
	 * @return void
	 */
	public function render_category_page() {
		// Retrieve existing mapping.
		$mapping = get_option( $this->option_key, array() );
		?>
		<div class="wrap">
			<h1>
				<?php esc_html_e( 'Delcampe Category Management', 'wc-delcampe-integration' ); ?>
				<span style="font-size: 0.8em; color: #666;">(v1.1.0.0)</span>
			</h1>
			<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
				<?php wp_nonce_field( 'delcampe_category_mapping', 'delcampe_category_mapping_nonce' ); ?>
				<input type="hidden" name="action" value="delcampe_save_category_mapping">
				
				<!-- Information about XML format added in v1.1.0.0 -->
				<p class="description">
					<?php esc_html_e( 'All category data is retrieved from Delcampe API in XML format and converted for display.', 'wc-delcampe-integration' ); ?>
				</p>
				
				<!-- Checkbox to force a fresh reload -->
				<p>
					<label>
						<input type="checkbox" name="reload_categories" value="1">
						<?php esc_html_e( 'Reload categories from scratch (clear cache)', 'wc-delcampe-integration' ); ?>
					</label>
				</p>
				<div id="delcampe-category-container">
					<!-- Dynamic category tree will be loaded here via AJAX -->
					<p><?php esc_html_e( 'Loading categories...', 'wc-delcampe-integration' ); ?></p>
				</div>
				<p>
					<input type="submit" class="button-primary" value="<?php esc_attr_e( 'Save Category Mapping', 'wc-delcampe-integration' ); ?>">
				</p>
			</form>
		</div>
		<script type="text/javascript">
		jQuery(document).ready(function($) {
			// Function to load categories via AJAX - Enhanced in v1.1.0.0
			function loadCategories(parentId, container) {
				$.ajax({
					url: ajaxurl,
					method: 'POST',
					dataType: 'json',
					data: {
						action: 'delcampe_load_subcategories',
						parent: parentId,
						nonce: '<?php echo wp_create_nonce( "delcampe_category_nonce" ); ?>'
					},
					beforeSend: function() {
						container.html('<p><?php esc_html_e( "Loading...", "delcampe-integration" ); ?></p>');
					},
					success: function(response) {
						// Log the entire response to inspect the structure - Enhanced logging in v1.1.0.0
						console.log("AJAX Response (v1.1.0.0):", response);
						if(response.success) {
							// Use response.data.data as the category array,
							// and response.data.mapping as the saved mapping.
							var categories = response.data.data;
							var mapping = response.data.mapping;
							var html = '<ul>';
							$.each(categories, function(index, category) {
								var catId = category.id ? category.id : 'undefined';
								var catName = category.name ? category.name : 'undefined';
								html += '<li data-id="'+ catId +'">';
								html += '<span class="category-name">' + catName + '</span>';
								html += ' <input type="checkbox" name="category_mapping[]" value="'+ catId +'" ' + (mapping && mapping.indexOf(catId) !== -1 ? 'checked' : '') + '>';
								html += '<div class="child-categories"></div>';
								html += '</li>';
							});
							html += '</ul>';
							container.html(html);
						} else {
							// Enhanced error message in v1.1.0.0
							var errorMsg = response.data ? response.data : '<?php esc_html_e( "Unknown error", "delcampe-integration" ); ?>';
							container.html('<p style="color: red;"><?php esc_html_e( "Failed to load categories:", "delcampe-integration" ); ?> ' + errorMsg + '</p>');
						}
					},
					error: function(jqXHR, textStatus, errorThrown) {
						// Enhanced error handling in v1.1.0.0
						container.html('<p style="color: red;"><?php esc_html_e( "AJAX error occurred:", "delcampe-integration" ); ?> ' + textStatus + ' - ' + errorThrown + '</p>');
					}
				});
			}

			// Load base categories initially.
			loadCategories('', $('#delcampe-category-container'));

			// When a category name is clicked, load its subcategories.
			$('#delcampe-category-container').on('click', '.category-name', function(){
				var li = $(this).closest('li');
				var categoryId = li.data('id');
				var childContainer = li.find('.child-categories').first();
				if(childContainer.children().length === 0){
					loadCategories(categoryId, childContainer);
				} else {
					childContainer.toggle();
				}
			});
		});
		</script>
		<style>
			/* Enhanced styles in v1.1.0.0 */
			#delcampe-category-container ul {
				list-style: none;
				margin-left: 20px;
			}
			#delcampe-category-container li {
				margin: 5px 0;
				padding: 5px;
				border-left: 2px solid transparent;
				transition: border-color 0.2s;
			}
			#delcampe-category-container li:hover {
				border-left-color: #0073aa;
			}
			.category-name {
				cursor: pointer;
				font-weight: bold;
				color: #0073aa;
			}
			.category-name:hover {
				text-decoration: underline;
			}
		</style>
		<?php
	}

	/**
	 * Handles AJAX requests for loading subcategories.
	 * 
	 * Processes AJAX requests to load subcategories dynamically.
	 * Converts XML responses from Delcampe API to JSON for JavaScript processing.
	 * Enhanced in version 1.1.0.0 with better error reporting.
	 *
	 * @return void Outputs JSON response.
	 */
	public function handle_ajax_requests() {
		// Security check with nonce verification
		check_ajax_referer( 'delcampe_category_nonce', 'nonce' );
		
		// Get parent category ID from request
		$parent = isset( $_POST['parent'] ) ? sanitize_text_field( wp_unslash( $_POST['parent'] ) ) : '';

		// Log AJAX request in v1.1.0.0
		delcampe_log( '[Delcampe Category Admin v1.1.0.0] AJAX request for parent: ' . $parent );

		// Use category manager to fetch categories
		$category_manager = Delcampe_Category_Manager::get_instance();
		if ( empty( $parent ) ) {
			$categories = $category_manager->fetch_base_categories();
		} else {
			$categories = $category_manager->fetch_subcategories( $parent );
		}

		// Handle errors from API
		if ( is_wp_error( $categories ) ) {
			delcampe_log( '[Delcampe Category Admin v1.1.0.0] Error fetching categories: ' . $categories->get_error_message() );
			wp_send_json_error( $categories->get_error_message() );
		}

		// Convert category data to standardized format
		// v1.10.24.3 - Fixed to properly extract 'id' field from normalized data
		$data = array();
		foreach ( $categories as $cat ) {
			// The normalized data from fetch_subcategories has 'id' not 'id_category'
			$data[] = array(
				'id'   => isset( $cat['id'] ) ? $cat['id'] : '',
				'name' => isset( $cat['name'] ) ? $cat['name'] : '',
			);
		}

		// Get existing mapping for pre-checking checkboxes
		$mapping = get_option( $this->option_key, array() );
		
		// Send JSON response with categories and mapping
		wp_send_json_success( array(
			'data'    => $data,
			'mapping' => $mapping,
			'version' => '1.1.0.0', // Added in v1.1.0.0 for debugging
		) );
	}

	/**
	 * Saves the user category mapping selections.
	 * 
	 * Processes form submission to save selected categories.
	 * Can also clear the category cache if requested by user.
	 * Enhanced in version 1.1.0.0 with better validation and logging.
	 *
	 * @return void Redirects back with status.
	 */
	public function save_category_mapping() {
		// Verify nonce for security
		if ( ! isset( $_POST['delcampe_category_mapping_nonce'] ) || ! wp_verify_nonce( $_POST['delcampe_category_mapping_nonce'], 'delcampe_category_mapping' ) ) {
			wp_die( __( 'Security check failed', 'wc-delcampe-integration' ) );
		}

		// Check user permissions
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( __( 'You do not have permission to perform this action.', 'wc-delcampe-integration' ) );
		}

		// If "Reload categories" checkbox is checked, delete the transient.
		if ( isset( $_POST['reload_categories'] ) && '1' === $_POST['reload_categories'] ) {
			delete_transient( 'delcampe_categories' );
			delcampe_log( '[Delcampe Category Admin v1.1.0.0] Transient "delcampe_categories" deleted per user request.' );
		}

		// Get and sanitize the category mapping data
		$mapping = isset( $_POST['category_mapping'] ) ? array_map( 'sanitize_text_field', $_POST['category_mapping'] ) : array();
		
		// Log the save operation in v1.1.0.0
		delcampe_log( '[Delcampe Category Admin v1.1.0.0] Saving category mapping with ' . count( $mapping ) . ' categories' );
		
		// Update the option in database
		update_option( $this->option_key, $mapping );
		
		// Redirect back with success message
		wp_redirect( add_query_arg( 'message', 'success', admin_url( 'admin.php?page=delcampe-categories' ) ) );
		exit;
	}
}

// Initialize the category admin singleton on plugins_loaded to ensure translations are ready
add_action( 'plugins_loaded', function() {
	Delcampe_Category_Admin::get_instance();
}, 5 );
