<?php
/**
 * Delcampe Shipping Admin Interface
 * 
 * Provides the admin interface for managing shipping models and their mappings.
 * Handles the UI for configuring shipping between WooCommerce and Delcampe.
 * 
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Admin
 * @since 1.3.0.0
 * @version 1.3.0.0
 * 
 * This class provides:
 * - Shipping models display and management interface
 * - Mapping configuration between WooCommerce and Delcampe shipping
 * - Profile-specific shipping settings
 * - Testing tools for shipping calculations
 * 
 * All shipping configurations are stored per profile for flexibility.
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class Delcampe_Shipping_Admin
 * 
 * Manages the admin interface for shipping configuration
 * 
 * @version 1.3.0.0
 */
class Delcampe_Shipping_Admin {
    
    /**
     * Singleton instance
     * @var Delcampe_Shipping_Admin
     */
    private static $instance = null;
    
    /**
     * Shipping models manager instance
     * @var Delcampe_Shipping_Models
     */
    private $shipping_models;
    
    /**
     * Get singleton instance
     * 
     * @return Delcampe_Shipping_Admin
     * @since 1.3.0.0
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     * 
     * @since 1.3.0.0
     */
    private function __construct() {
        // Get shipping models manager
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-shipping-models.php';
        $this->shipping_models = Delcampe_Shipping_Models::get_instance();
        
        // Add admin scripts
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // Add AJAX handler for test calculation
        add_action('wp_ajax_delcampe_test_shipping_calculation', array($this, 'ajax_test_shipping_calculation'));
    }
    
    /**
     * Render the shipping tab content
     * 
     * Called from the main tabbed interface to display shipping configuration
     * 
     * @since 1.3.0.0
     */
    public function render_shipping_tab() {
        ?>
        <div class="wrap">
            <!-- Available Shipping Models Section -->
            <div class="delcampe-section">
                <h3><?php _e('Available Delcampe Shipping Models', 'wc-delcampe-integration'); ?></h3>
                
                <p class="description">
                    <?php _e('These are the shipping models configured in your Delcampe account. You can map them to WooCommerce shipping methods in each profile.', 'wc-delcampe-integration'); ?>
                    <br><strong><?php _e('Note: Shipping costs and zones must be configured directly on the Delcampe website.', 'wc-delcampe-integration'); ?></strong>
                </p>
                
                <div id="shipping-models-container">
                    <?php $this->render_shipping_models_list(); ?>
                </div>
                
                <p style="margin-top: 15px;">
                    <button type="button" class="button" id="refresh-shipping-models">
                        <span class="dashicons dashicons-update" style="vertical-align: middle;"></span>
                        <?php _e('Refresh Shipping Models', 'wc-delcampe-integration'); ?>
                    </button>
                    <span id="refresh-feedback" style="margin-left: 10px; display: none;"></span>
                </p>
            </div>
            
            <!-- Shipping Conditions Section -->
            <div class="delcampe-section">
                <h3><?php _e('Your Shipping Conditions', 'wc-delcampe-integration'); ?></h3>
                
                <p class="description">
                    <?php _e('These are your textual shipping conditions from Delcampe that buyers will see.', 'wc-delcampe-integration'); ?>
                </p>
                
                <div id="shipping-conditions-container" style="background: #f9f9f9; padding: 15px; border: 1px solid #ddd; border-radius: 4px;">
                    <?php 
                    $conditions = $this->shipping_models->get_shipping_conditions();
                    if (is_wp_error($conditions)) {
                        echo '<em>' . __('Unable to fetch shipping conditions.', 'wc-delcampe-integration') . '</em>';
                    } elseif (empty($conditions) || $conditions === 'false') {
                        echo '<em>' . __('No shipping conditions defined. You can add them on the Delcampe website.', 'wc-delcampe-integration') . '</em>';
                    } else {
                        echo '<div style="white-space: pre-wrap; font-family: monospace;">' . esc_html($conditions) . '</div>';
                    }
                    ?>
                </div>
                
                <p style="margin-top: 10px;">
                    <a href="https://www.delcampe.net/en_GB/collectables/store-settings/shipping-conditions" target="_blank" class="button button-small">
                        <?php _e('Edit Shipping Conditions', 'wc-delcampe-integration'); ?>
                        <span class="dashicons dashicons-external" style="vertical-align: middle; font-size: 14px;"></span>
                    </a>
                </p>
            </div>
            
            <!-- WooCommerce Shipping Methods Section -->
            <div class="delcampe-section">
                <h3><?php _e('WooCommerce Shipping Configuration', 'wc-delcampe-integration'); ?></h3>
                
                <p class="description">
                    <?php _e('Your WooCommerce shipping zones and methods. Configure shipping mappings in each Delcampe profile.', 'wc-delcampe-integration'); ?>
                </p>
                
                <?php $this->render_woocommerce_shipping_methods(); ?>
            </div>
            
            <!-- Shipping Configuration Guide -->
            <div class="delcampe-section">
                <h3><?php _e('Shipping Configuration Guide', 'wc-delcampe-integration'); ?></h3>
                
                <h4><?php _e('How Shipping Works', 'wc-delcampe-integration'); ?></h4>
                <ol>
                    <li><?php _e('Configure shipping models in your Delcampe account', 'wc-delcampe-integration'); ?></li>
                    <li><?php _e('Create or edit a Delcampe profile in this plugin', 'wc-delcampe-integration'); ?></li>
                    <li><?php _e('Select the appropriate shipping model for that profile', 'wc-delcampe-integration'); ?></li>
                    <li><?php _e('Products using that profile will use the selected shipping model', 'wc-delcampe-integration'); ?></li>
                </ol>
                
                <h4><?php _e('Shipping Model Types', 'wc-delcampe-integration'); ?></h4>
                <ul>
                    <li><strong><?php _e('Fixed Price:', 'wc-delcampe-integration'); ?></strong> <?php _e('Single shipping cost regardless of quantity', 'wc-delcampe-integration'); ?></li>
                    <li><strong><?php _e('Per Item:', 'wc-delcampe-integration'); ?></strong> <?php _e('Shipping cost multiplied by quantity', 'wc-delcampe-integration'); ?></li>
                    <li><strong><?php _e('Weight Based:', 'wc-delcampe-integration'); ?></strong> <?php _e('Cost calculated based on total weight', 'wc-delcampe-integration'); ?></li>
                    <li><strong><?php _e('Zone Based:', 'wc-delcampe-integration'); ?></strong> <?php _e('Different costs for different countries/zones', 'wc-delcampe-integration'); ?></li>
                </ul>
                
                <h4><?php _e('Free Shipping', 'wc-delcampe-integration'); ?></h4>
                <p><?php _e('To offer free shipping, create a shipping model in Delcampe with 0.00 cost and assign it to the appropriate profile.', 'wc-delcampe-integration'); ?></p>
            </div>
            
            <!-- Test Shipping Calculation -->
            <div class="delcampe-section">
                <h3><?php _e('Test Shipping Calculation', 'wc-delcampe-integration'); ?></h3>
                
                <p class="description">
                    <?php _e('Test how shipping costs will be calculated for a specific product and destination.', 'wc-delcampe-integration'); ?>
                </p>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="test-product-id"><?php _e('Product', 'wc-delcampe-integration'); ?></label>
                        </th>
                        <td>
                            <select id="test-product-id" style="width: 300px;">
                                <option value=""><?php _e('Select a product...', 'wc-delcampe-integration'); ?></option>
                                <?php
                                // Get products with Delcampe profiles
                                $args = array(
                                    'post_type' => 'product',
                                    'posts_per_page' => -1,
                                    'meta_query' => array(
                                        array(
                                            'key' => '_delcampe_profile_id',
                                            'compare' => 'EXISTS'
                                        )
                                    )
                                );
                                $products = get_posts($args);
                                
                                foreach ($products as $product_post) {
                                    $product = wc_get_product($product_post->ID);
                                    if ($product) {
                                        echo '<option value="' . esc_attr($product->get_id()) . '">' . 
                                             esc_html($product->get_name()) . ' (#' . $product->get_id() . ')' .
                                             '</option>';
                                    }
                                }
                                ?>
                            </select>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="test-country"><?php _e('Destination Country', 'wc-delcampe-integration'); ?></label>
                        </th>
                        <td>
                            <select id="test-country">
                                <option value=""><?php _e('Default', 'wc-delcampe-integration'); ?></option>
                                <?php
                                $countries = WC()->countries->get_countries();
                                foreach ($countries as $code => $name) {
                                    echo '<option value="' . esc_attr($code) . '">' . esc_html($name) . '</option>';
                                }
                                ?>
                            </select>
                        </td>
                    </tr>
                </table>
                
                <p>
                    <button type="button" class="button button-primary" id="test-shipping-calc">
                        <?php _e('Calculate Shipping', 'wc-delcampe-integration'); ?>
                    </button>
                </p>
                
                <div id="shipping-test-result" style="margin-top: 20px;"></div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Get profiles using a specific shipping model
     * 
     * @param int $model_id The shipping model ID
     * @return array Array of profile names using this model
     * @since 1.3.0.0
     */
    private function get_profiles_using_model($model_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_profiles';
        $profiles = $wpdb->get_results($wpdb->prepare(
            "SELECT profile_name FROM $table_name WHERE shipping_model_id = %d",
            $model_id
        ), ARRAY_A);
        
        return wp_list_pluck($profiles, 'profile_name');
    }
    
    /**
     * Render shipping models list
     * 
     * @since 1.3.0.0
     */
    private function render_shipping_models_list() {
        $models = $this->shipping_models->get_shipping_models();
        
        if (is_wp_error($models)) {
            echo '<div class="notice notice-error inline">';
            echo '<p>' . sprintf(
                __('Error fetching shipping models: %s', 'wc-delcampe-integration'),
                esc_html($models->get_error_message())
            ) . '</p>';
            echo '</div>';
            return;
        }
        
        if (empty($models)) {
            echo '<p>' . __('No shipping models found. Please configure shipping models in your Delcampe account.', 'wc-delcampe-integration') . '</p>';
            return;
        }
        
        ?>
        <table class="wp-list-table widefat fixed striped">
            <thead>
                <tr>
                    <th><?php _e('Shipping Model', 'wc-delcampe-integration'); ?></th>
                    <th><?php _e('Type', 'wc-delcampe-integration'); ?></th>
                    <th><?php _e('Usage', 'wc-delcampe-integration'); ?></th>
                    <th><?php _e('Actions', 'wc-delcampe-integration'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php 
                $model_count = 0;
                foreach ($models as $model) : 
                    $model_count++;
                    // Determine icon based on type
                    $icon = 'cart';
                    $icon_color = '#666';
                    if (stripos($model['type'], 'free') !== false) {
                        $icon = 'yes-alt';
                        $icon_color = '#46b450';
                    } elseif (stripos($model['type'], 'weight') !== false) {
                        $icon = 'performance';
                        $icon_color = '#0073aa';
                    }
                ?>
                    <tr>
                        <td>
                            <strong style="font-size: 14px;">
                                <span class="dashicons dashicons-<?php echo $icon; ?>" style="color: <?php echo $icon_color; ?>; vertical-align: middle;"></span>
                                <?php echo esc_html($model['name']); ?>
                            </strong>
                            <br>
                            <small style="color: #999;">
                                <?php printf(__('Model #%d', 'wc-delcampe-integration'), $model_count); ?>
                                <?php if ($model['id'] > 1000000) : ?>
                                    • <?php _e('Custom model', 'wc-delcampe-integration'); ?>
                                <?php endif; ?>
                            </small>
                        </td>
                        <td>
                            <?php 
                            $type_display = ucfirst(str_replace('_', ' ', $model['type']));
                            echo esc_html($type_display);
                            ?>
                        </td>
                        <td>
                            <?php
                            // Check if this model is used in any profiles
                            $profiles_using = $this->get_profiles_using_model($model['id']);
                            if (!empty($profiles_using)) {
                                echo '<span style="color: #46b450;">' . sprintf(_n('Used in %d profile', 'Used in %d profiles', count($profiles_using), 'wc-delcampe-integration'), count($profiles_using)) . '</span>';
                            } else {
                                echo '<span style="color: #999;">' . __('Not currently used', 'wc-delcampe-integration') . '</span>';
                            }
                            ?>
                        </td>
                        <td>
                            <a href="https://www.delcampe.net/en_GB/collectables/store-settings/shipping-models" target="_blank" class="button button-small">
                                <?php _e('Configure', 'wc-delcampe-integration'); ?>
                                <span class="dashicons dashicons-external" style="vertical-align: middle; font-size: 14px;"></span>
                            </a>
                            <?php if ($model['id'] != 1) : // Don't allow deletion of default free shipping ?>
                            <button type="button" class="button button-small button-link-delete" style="color: #a00;" onclick="if(confirm('<?php esc_attr_e('This will only remove the model from this plugin cache. To permanently delete, use Delcampe website.', 'wc-delcampe-integration'); ?>')) { /* Add delete handler */ }">
                                <?php _e('Remove', 'wc-delcampe-integration'); ?>
                            </button>
                            <?php endif; ?>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <?php
    }
    
    /**
     * Render WooCommerce shipping methods
     * 
     * @since 1.3.0.0
     */
    private function render_woocommerce_shipping_methods() {
        $shipping_methods = $this->shipping_models->get_woocommerce_shipping_methods();
        
        if (empty($shipping_methods)) {
            echo '<p>' . __('No shipping methods configured in WooCommerce.', 'wc-delcampe-integration') . '</p>';
            return;
        }
        
        // Group by zone
        $zones = array();
        foreach ($shipping_methods as $method) {
            $zone_id = $method['zone_id'];
            if (!isset($zones[$zone_id])) {
                $zones[$zone_id] = array(
                    'name' => $method['zone_name'],
                    'methods' => array()
                );
            }
            $zones[$zone_id]['methods'][] = $method;
        }
        
        ?>
        <div class="shipping-zones-list">
            <?php foreach ($zones as $zone_id => $zone) : ?>
                <div class="shipping-zone" style="margin-bottom: 20px;">
                    <h4><?php echo esc_html($zone['name']); ?></h4>
                    <table class="wp-list-table widefat">
                        <thead>
                            <tr>
                                <th><?php _e('Method', 'wc-delcampe-integration'); ?></th>
                                <th><?php _e('Type', 'wc-delcampe-integration'); ?></th>
                                <th><?php _e('Status', 'wc-delcampe-integration'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($zone['methods'] as $method) : ?>
                                <tr>
                                    <td><?php echo esc_html($method['method_title']); ?></td>
                                    <td><?php echo esc_html($method['method_type']); ?></td>
                                    <td>
                                        <?php if ($method['enabled']) : ?>
                                            <span class="dashicons dashicons-yes" style="color: #46b450;"></span>
                                            <?php _e('Enabled', 'wc-delcampe-integration'); ?>
                                        <?php else : ?>
                                            <span class="dashicons dashicons-no" style="color: #dc3232;"></span>
                                            <?php _e('Disabled', 'wc-delcampe-integration'); ?>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            <?php endforeach; ?>
        </div>
        <?php
    }
    
    /**
     * AJAX handler for test shipping calculation
     * 
     * @since 1.3.0.0
     */
    public function ajax_test_shipping_calculation() {
        // Check nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'delcampe_admin_nonce')) {
            wp_send_json_error(array(
                'message' => __('Security check failed', 'wc-delcampe-integration')
            ));
        }
        
        $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
        $country = isset($_POST['country']) ? sanitize_text_field($_POST['country']) : '';
        
        if (!$product_id) {
            wp_send_json_error(array(
                'message' => __('Please select a product', 'wc-delcampe-integration')
            ));
        }
        
        // Get product
        $product = wc_get_product($product_id);
        if (!$product) {
            wp_send_json_error(array(
                'message' => __('Product not found', 'wc-delcampe-integration')
            ));
        }
        
        // Get product's profile
        $profile_id = get_post_meta($product_id, '_delcampe_profile_id', true);
        if (!$profile_id) {
            wp_send_json_error(array(
                'message' => __('This product does not have a Delcampe profile assigned', 'wc-delcampe-integration')
            ));
        }
        
        // Get profile
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/profiles/class-delcampe-profiles-model.php';
        $profiles_model = Delcampe_Profiles_Model::get_instance();
        $profile = $profiles_model->get_profile($profile_id);
        
        if (!$profile) {
            wp_send_json_error(array(
                'message' => __('Profile not found', 'wc-delcampe-integration')
            ));
        }
        
        // Get shipping model ID
        $shipping_model_id = isset($profile['shipping_model_id']) ? $profile['shipping_model_id'] : 0;
        
        if (!$shipping_model_id) {
            wp_send_json_error(array(
                'message' => __('No shipping model configured for this profile', 'wc-delcampe-integration')
            ));
        }
        
        // Calculate shipping
        $shipping_cost = $this->shipping_models->calculate_shipping_cost($product, $shipping_model_id, $country);
        
        if (is_wp_error($shipping_cost)) {
            wp_send_json_error(array(
                'message' => $shipping_cost->get_error_message()
            ));
        }
        
        // Get shipping model details
        $model = $this->shipping_models->get_shipping_model($shipping_model_id);
        
        // Build result HTML
        ob_start();
        ?>
        <div class="notice notice-success inline">
            <h4><?php _e('Shipping Calculation Result', 'wc-delcampe-integration'); ?></h4>
            <table class="form-table">
                <tr>
                    <th><?php _e('Product:', 'wc-delcampe-integration'); ?></th>
                    <td><?php echo esc_html($product->get_name()); ?></td>
                </tr>
                <tr>
                    <th><?php _e('Profile:', 'wc-delcampe-integration'); ?></th>
                    <td><?php echo esc_html($profile['profile_name']); ?></td>
                </tr>
                <tr>
                    <th><?php _e('Shipping Model:', 'wc-delcampe-integration'); ?></th>
                    <td>
                        <?php echo esc_html($model['name']); ?>
                        <?php if (!empty($model['description'])) : ?>
                            <br><small><?php echo esc_html($model['description']); ?></small>
                        <?php endif; ?>
                    </td>
                </tr>
                <tr>
                    <th><?php _e('Product Weight:', 'wc-delcampe-integration'); ?></th>
                    <td>
                        <?php 
                        $weight = $product->get_weight();
                        if ($weight) {
                            echo esc_html($weight . ' ' . get_option('woocommerce_weight_unit'));
                        } else {
                            echo __('Not set', 'wc-delcampe-integration');
                        }
                        ?>
                    </td>
                </tr>
                <tr>
                    <th><?php _e('Destination:', 'wc-delcampe-integration'); ?></th>
                    <td>
                        <?php 
                        if ($country) {
                            $countries = WC()->countries->get_countries();
                            echo esc_html($countries[$country] ?? $country);
                        } else {
                            echo __('Default', 'wc-delcampe-integration');
                        }
                        ?>
                    </td>
                </tr>
                <tr>
                    <th><?php _e('Calculated Shipping Cost:', 'wc-delcampe-integration'); ?></th>
                    <td><strong><?php echo wc_price($shipping_cost); ?></strong></td>
                </tr>
            </table>
        </div>
        <?php
        $html = ob_get_clean();
        
        wp_send_json_success(array(
            'html' => $html
        ));
    }
    
    /**
     * Enqueue admin scripts and styles
     * 
     * @param string $hook Current admin page hook
     * @since 1.3.0.0
     */
    public function enqueue_admin_scripts($hook) {
        // Only load on our settings page
        if ($hook !== 'delcampe_page_delcampe-settings') {
            return;
        }
        
        // Add CSS for spinning animation
        wp_add_inline_style('wp-admin', '
            .dashicons.spinning {
                animation: spin 1s linear infinite;
            }
            @keyframes spin {
                0% { transform: rotate(0deg); }
                100% { transform: rotate(360deg); }
            }
            #refresh-feedback {
                display: inline-block;
                padding: 5px 10px;
                background: #f0f0f0;
                border-radius: 3px;
            }
            #refresh-feedback.success {
                background: #d4edda;
                color: #155724;
            }
            #refresh-feedback.error {
                background: #f8d7da;
                color: #721c24;
            }
        ');
        
        // Add shipping-specific JavaScript
        wp_add_inline_script('jquery', '
            jQuery(document).ready(function($) {
                // Refresh shipping models
                $("#refresh-shipping-models").on("click", function() {
                    var $btn = $(this);
                    var $feedback = $("#refresh-feedback");
                    var originalHtml = $btn.html();
                    
                    $btn.prop("disabled", true).html(\'<span class="dashicons dashicons-update spinning"></span> ' . esc_js(__('Refreshing...', 'wc-delcampe-integration')) . '\');
                    $feedback.hide();
                    
                    $.post(ajaxurl, {
                        action: "delcampe_refresh_shipping_models",
                        nonce: "' . wp_create_nonce('delcampe_admin_nonce') . '"
                    }, function(response) {
                        if (response.success) {
                            $feedback.removeClass("error").addClass("success").html(
                                \'<span class="dashicons dashicons-yes" style="color: #46b450;"></span> \' + 
                                response.data.message
                            ).fadeIn();
                            
                            // Reload after showing success message
                            setTimeout(function() {
                                location.reload();
                            }, 1500);
                        } else {
                            $feedback.removeClass("success").addClass("error").html(
                                \'<span class="dashicons dashicons-warning" style="color: #dc3232;"></span> \' + 
                                response.data.message
                            ).fadeIn();
                            $btn.prop("disabled", false).html(originalHtml);
                        }
                    }).fail(function() {
                        $feedback.removeClass("success").addClass("error").html(
                            \'<span class="dashicons dashicons-warning" style="color: #dc3232;"></span> ' . 
                            esc_js(__('Network error. Please try again.', 'wc-delcampe-integration')) . '\'
                        ).fadeIn();
                        $btn.prop("disabled", false).html(originalHtml);
                    });
                });
                
                // Test shipping calculation
                $("#test-shipping-calc").on("click", function() {
                    var productId = $("#test-product-id").val();
                    var country = $("#test-country").val();
                    
                    if (!productId) {
                        alert("' . esc_js(__('Please select a product', 'wc-delcampe-integration')) . '");
                        return;
                    }
                    
                    var $btn = $(this);
                    var originalText = $btn.text();
                    
                    $btn.prop("disabled", true).text("' . esc_js(__('Calculating...', 'wc-delcampe-integration')) . '");
                    $("#shipping-test-result").html("");
                    
                    $.post(ajaxurl, {
                        action: "delcampe_test_shipping_calculation",
                        product_id: productId,
                        country: country,
                        nonce: "' . wp_create_nonce('delcampe_admin_nonce') . '"
                    }, function(response) {
                        $btn.prop("disabled", false).text(originalText);
                        
                        if (response.success) {
                            $("#shipping-test-result").html(response.data.html);
                        } else {
                            $("#shipping-test-result").html(
                                "<div class=\"notice notice-error inline\"><p>" + response.data.message + "</p></div>"
                            );
                        }
                    });
                });
            });
        ');
        
        // Add shipping-specific CSS
        wp_add_inline_style('wp-admin', '
            .shipping-zones-list {
                margin-top: 15px;
            }
            .shipping-zone {
                background: #f9f9f9;
                padding: 15px;
                border: 1px solid #ddd;
                border-radius: 3px;
            }
            .shipping-zone h4 {
                margin-top: 0;
                margin-bottom: 10px;
            }
            .shipping-zone table {
                margin-bottom: 0;
            }
            details summary {
                cursor: pointer;
                color: #0073aa;
            }
            details summary:hover {
                color: #00a0d2;
            }
        ');
    }
}

// Initialize the shipping admin
Delcampe_Shipping_Admin::get_instance();
