#!/usr/bin/php
<?php
/**
 * Clean up duplicate webhook registrations
 * 
 * This utility removes duplicate webhooks, keeping only one registration
 * per webhook type. It uses the same URL normalization as the Webhook Guardian
 * to properly identify duplicates.
 * 
 * @since 1.10.35.8
 */

require_once(dirname(dirname(dirname(dirname(dirname(__FILE__))))) . '/wp-load.php');

echo "\n=== WEBHOOK DUPLICATE CLEANUP ===\n";
echo "Version: 1.10.35.8\n";
echo "Time: " . current_time('Y-m-d H:i:s') . "\n\n";

// Get auth token
$auth = Delcampe_Auth::get_instance();
$token = $auth->get_auth_token();

if (is_wp_error($token)) {
    die("❌ Authentication failed: " . $token->get_error_message() . "\n");
}

/**
 * Normalize URL for comparison (same logic as Webhook Guardian)
 */
function normalize_webhook_url($url) {
    $parts = wp_parse_url($url);
    if (!$parts || empty($parts['host'])) {
        return $url;
    }
    
    $scheme = isset($parts['scheme']) ? strtolower($parts['scheme']) : 'https';
    $host = strtolower($parts['host']);
    $port = isset($parts['port']) ? (int)$parts['port'] : null;
    $path = isset($parts['path']) ? rtrim($parts['path'], '/') : '';
    $query_params = array();
    
    if (!empty($parts['query'])) {
        parse_str($parts['query'], $query_params);
        ksort($query_params); // Sort for stable equality
    }
    
    // Strip default port
    $port_str = '';
    if ($port && !(($scheme === 'https' && $port === 443) || ($scheme === 'http' && $port === 80))) {
        $port_str = ':' . $port;
    }
    
    $query = http_build_query($query_params, '', '&', PHP_QUERY_RFC3986);
    return $scheme . '://' . $host . $port_str . $path . ($query ? '?' . $query : '');
}

// 1. List current webhook registrations
echo "=== CURRENT WEBHOOK REGISTRATIONS ===\n\n";

$url = DELCAMPE_API_URL . '/notification/settings?token=' . urlencode($token);
$response = wp_remote_get($url, array(
    'timeout' => 20,
    'headers' => array('Accept' => 'application/xml')
));

if (is_wp_error($response)) {
    die("❌ Failed to list webhooks: " . $response->get_error_message() . "\n");
}

$xml = simplexml_load_string(wp_remote_retrieve_body($response));
if (!$xml || !isset($xml->Notification_Data->body->notification_settings)) {
    die("❌ Failed to parse webhook response\n");
}

$notifications = array();
foreach ($xml->Notification_Data->body->notification_settings as $setting) {
    $notifications[] = array(
        'id' => (string)$setting->id_notification,
        'type' => (string)$setting->type,
        'channel' => (string)$setting->channel,
        'destination' => (string)$setting->destination,
        'active' => isset($setting->active) ? ((string)$setting->active === '1') : true
    );
    
    echo sprintf("ID %s: %s → %s %s\n", 
        $setting->id_notification,
        $setting->type,
        $setting->destination,
        (isset($setting->active) && (string)$setting->active === '0') ? '[INACTIVE]' : ''
    );
}

if (empty($notifications)) {
    echo "No webhooks registered.\n";
    exit(0);
}

echo "\nTotal: " . count($notifications) . " webhook(s)\n\n";

// 2. Group by type and normalized destination
echo "=== ANALYZING DUPLICATES ===\n\n";

$groups = array();
foreach ($notifications as $notification) {
    // Only process Curl webhooks (actual webhooks, not email notifications)
    if (strpos($notification['channel'], 'Curl') !== 0) {
        continue;
    }
    
    $normalized_dest = normalize_webhook_url($notification['destination']);
    $key = $notification['type'] . '|' . $normalized_dest;
    
    if (!isset($groups[$key])) {
        $groups[$key] = array();
    }
    $groups[$key][] = $notification;
}

$duplicates_found = false;
$to_delete = array();

foreach ($groups as $key => $group) {
    if (count($group) > 1) {
        $duplicates_found = true;
        list($type, $dest) = explode('|', $key, 2);
        echo "Found " . count($group) . " duplicates for: $type\n";
        echo "  Destination: $dest\n";
        
        // Sort by active status and ID (keep first active, or first ID if none active)
        usort($group, function($a, $b) {
            if ($a['active'] === $b['active']) {
                return (int)$a['id'] <=> (int)$b['id'];
            }
            return $a['active'] ? -1 : 1;
        });
        
        // Keep first, delete rest
        $keeper = array_shift($group);
        echo "  ✅ Keeping: ID " . $keeper['id'] . ($keeper['active'] ? ' [ACTIVE]' : ' [INACTIVE]') . "\n";
        
        foreach ($group as $duplicate) {
            echo "  ❌ Will delete: ID " . $duplicate['id'] . ($duplicate['active'] ? ' [ACTIVE]' : ' [INACTIVE]') . "\n";
            $to_delete[] = $duplicate;
        }
        echo "\n";
    }
}

if (!$duplicates_found) {
    echo "✅ No duplicate webhooks found!\n\n";
    exit(0);
}

// 3. Ask for confirmation
echo "=== CONFIRMATION ===\n";
echo "Found " . count($to_delete) . " duplicate webhook(s) to remove.\n";
echo "Do you want to proceed with cleanup? (yes/no): ";

$handle = fopen("php://stdin", "r");
$line = fgets($handle);
$answer = trim(strtolower($line));
fclose($handle);

if ($answer !== 'yes' && $answer !== 'y') {
    echo "\n❌ Cleanup cancelled.\n";
    exit(0);
}

// 4. Delete duplicates
echo "\n=== REMOVING DUPLICATES ===\n\n";

$success_count = 0;
$fail_count = 0;

foreach ($to_delete as $webhook) {
    echo "Deleting webhook ID " . $webhook['id'] . " (" . $webhook['type'] . ")... ";
    
    $delete_url = DELCAMPE_API_URL . '/notification/' . $webhook['id'] . '?token=' . urlencode($token);
    $response = wp_remote_request($delete_url, array(
        'method' => 'DELETE',
        'timeout' => 20,
        'headers' => array('Accept' => 'application/xml')
    ));
    
    if (is_wp_error($response)) {
        echo "❌ Error: " . $response->get_error_message() . "\n";
        $fail_count++;
    } else {
        $code = wp_remote_retrieve_response_code($response);
        if ($code >= 200 && $code < 300) {
            echo "✅ Deleted\n";
            $success_count++;
            delcampe_log('[Webhook Cleanup] Deleted duplicate webhook ID ' . $webhook['id'] . ' (' . $webhook['type'] . ')');
        } else {
            echo "❌ Failed (HTTP $code)\n";
            $fail_count++;
        }
    }
}

// 5. Summary
echo "\n=== CLEANUP COMPLETE ===\n";
echo "✅ Deleted: $success_count webhook(s)\n";
if ($fail_count > 0) {
    echo "❌ Failed: $fail_count webhook(s)\n";
}

// 6. Verify final state
echo "\n=== FINAL WEBHOOK STATE ===\n";

$response = wp_remote_get($url, array(
    'timeout' => 20,
    'headers' => array('Accept' => 'application/xml')
));

if (!is_wp_error($response)) {
    $xml = simplexml_load_string(wp_remote_retrieve_body($response));
    if ($xml && isset($xml->Notification_Data->body->notification_settings)) {
        $count = 0;
        $types = array();
        foreach ($xml->Notification_Data->body->notification_settings as $setting) {
            if (strpos((string)$setting->channel, 'Curl') === 0) {
                $count++;
                $type = str_replace('Curl_', '', (string)$setting->type);
                if (!in_array($type, $types)) {
                    $types[] = $type;
                }
                echo "  ✅ " . $setting->type . " → " . $setting->destination . "\n";
            }
        }
        echo "\nTotal: $count webhook(s) for " . count($types) . " unique type(s)\n";
    }
}

echo "\n✅ Cleanup complete!\n";
echo "\nNote: Automatic webhook registration has been disabled.\n";
echo "Webhooks will only be created when manually requested.\n\n";