#!/usr/bin/php
<?php
/**
 * Clean up webhook URL variations
 * 
 * WordPress REST API endpoints can be accessed with or without /index.php/
 * This causes duplicate webhooks. This script removes all variations and
 * keeps only the canonical URL format.
 * 
 * @since 1.10.35.8
 */

require_once(dirname(dirname(dirname(dirname(dirname(__FILE__))))) . '/wp-load.php');

echo "\n=== WEBHOOK URL VARIATION CLEANUP ===\n";
echo "Version: 1.10.35.8\n";
echo "Time: " . current_time('Y-m-d H:i:s') . "\n\n";

// Get auth token
$auth = Delcampe_Auth::get_instance();
$token = $auth->get_auth_token();

if (is_wp_error($token)) {
    die("❌ Authentication failed: " . $token->get_error_message() . "\n");
}

// 1. Determine canonical webhook URL
$webhook_token = get_option('delcampe_webhook_token');
if (!$webhook_token) {
    $webhook_token = wp_generate_password(32, false, false);
    update_option('delcampe_webhook_token', $webhook_token);
}

// Canonical URL (without index.php)
$canonical_url = home_url('/wp-json/delcampe/v1/webhook');
$canonical_url = str_replace('http://', 'https://', $canonical_url);
$canonical_url = add_query_arg('token', $webhook_token, $canonical_url);

echo "Canonical webhook URL: $canonical_url\n\n";

// 2. List current webhook registrations
echo "=== CURRENT WEBHOOK REGISTRATIONS ===\n\n";

$url = DELCAMPE_API_URL . '/notification/settings?token=' . urlencode($token);
$response = wp_remote_get($url, array(
    'timeout' => 20,
    'headers' => array('Accept' => 'application/xml')
));

if (is_wp_error($response)) {
    die("❌ Failed to list webhooks: " . $response->get_error_message() . "\n");
}

$xml = simplexml_load_string(wp_remote_retrieve_body($response));
if (!$xml || !isset($xml->Notification_Data->body->notification_settings)) {
    die("❌ Failed to parse webhook response\n");
}

$notifications = array();
$variations = array();
$canonical_exists = array();

foreach ($xml->Notification_Data->body->notification_settings as $setting) {
    $id = (string)$setting->id_notification;
    $type = (string)$setting->type;
    $channel = (string)$setting->channel;
    $destination = (string)$setting->destination;
    $active = isset($setting->active) ? ((string)$setting->active === '1') : true;
    
    // Only process Curl webhooks
    if (strpos($channel, 'Curl') !== 0) {
        continue;
    }
    
    $notifications[] = array(
        'id' => $id,
        'type' => $type,
        'channel' => $channel,
        'destination' => $destination,
        'active' => $active
    );
    
    // Check if it's a variation of our webhook
    if (strpos($destination, 'dev.fdstamps.com') !== false && 
        strpos($destination, '/wp-json/delcampe/v1/webhook') !== false &&
        strpos($destination, $webhook_token) !== false) {
        
        if ($destination === $canonical_url) {
            $canonical_exists[$type] = $id;
            echo "✅ CANONICAL: $type (ID: $id)\n";
        } else {
            $variations[] = array('id' => $id, 'type' => $type, 'destination' => $destination);
            echo "❌ VARIATION: $type (ID: $id)\n";
            echo "   URL: $destination\n";
        }
    }
}

echo "\nFound " . count($canonical_exists) . " canonical webhook(s)\n";
echo "Found " . count($variations) . " variation(s) to remove\n\n";

if (empty($variations)) {
    echo "✅ No URL variations found!\n\n";
    exit(0);
}

// 3. Remove all variations
echo "=== REMOVING VARIATIONS ===\n\n";

foreach ($variations as $webhook) {
    echo "Deleting variation: " . $webhook['type'] . " (ID: " . $webhook['id'] . ")... ";
    
    $delete_url = DELCAMPE_API_URL . '/notification/' . $webhook['id'] . '?token=' . urlencode($token);
    $response = wp_remote_request($delete_url, array(
        'method' => 'DELETE',
        'timeout' => 20,
        'headers' => array('Accept' => 'application/xml')
    ));
    
    if (is_wp_error($response)) {
        echo "❌ Error: " . $response->get_error_message() . "\n";
    } else {
        $code = wp_remote_retrieve_response_code($response);
        if ($code >= 200 && $code < 300) {
            echo "✅ Deleted\n";
            delcampe_log('[Webhook Cleanup] Deleted URL variation webhook ID ' . $webhook['id'] . ' (' . $webhook['type'] . ')');
        } else {
            echo "❌ Failed (HTTP $code)\n";
        }
    }
}

// 4. Register missing canonical webhooks
echo "\n=== ENSURING CANONICAL WEBHOOKS ===\n\n";

$essential_types = array(
    'Curl_Seller_Item_Close_Sold',
    'Curl_Seller_Payment_Received'
);

foreach ($essential_types as $webhook_type) {
    if (!isset($canonical_exists[$webhook_type])) {
        echo "Registering missing webhook: $webhook_type... ";
        
        $register_url = DELCAMPE_API_URL . '/notification/settings?token=' . urlencode($token);
        $body = array(
            'notificationType' => $webhook_type,
            'destination' => $canonical_url
        );
        
        $response = wp_remote_post($register_url, array(
            'timeout' => 20,
            'headers' => array(
                'Content-Type' => 'application/x-www-form-urlencoded',
                'Accept' => 'application/xml'
            ),
            'body' => http_build_query($body)
        ));
        
        if (is_wp_error($response)) {
            echo "❌ Error: " . $response->get_error_message() . "\n";
        } else {
            $code = wp_remote_retrieve_response_code($response);
            if ($code >= 200 && $code < 300) {
                echo "✅ Registered\n";
                delcampe_log('[Webhook Cleanup] Registered canonical webhook: ' . $webhook_type);
            } else {
                echo "❌ Failed (HTTP $code)\n";
            }
        }
    } else {
        echo "✅ Already registered: $webhook_type (ID: " . $canonical_exists[$webhook_type] . ")\n";
    }
}

// 5. Verify final state
echo "\n=== FINAL WEBHOOK STATE ===\n\n";

$response = wp_remote_get($url, array(
    'timeout' => 20,
    'headers' => array('Accept' => 'application/xml')
));

if (!is_wp_error($response)) {
    $xml = simplexml_load_string(wp_remote_retrieve_body($response));
    if ($xml && isset($xml->Notification_Data->body->notification_settings)) {
        foreach ($xml->Notification_Data->body->notification_settings as $setting) {
            if (strpos((string)$setting->channel, 'Curl') === 0) {
                $dest = (string)$setting->destination;
                $is_canonical = ($dest === $canonical_url);
                echo sprintf("%s %s → %s\n",
                    $is_canonical ? '✅' : '⚠️',
                    $setting->type,
                    $setting->destination
                );
            }
        }
    }
}

echo "\n✅ Cleanup complete!\n";
echo "\nCanonical webhook URL: $canonical_url\n";
echo "All webhooks now use the canonical URL format.\n";
echo "\nNote: Automatic webhook registration has been disabled.\n";
echo "Webhooks will only be created when manually requested.\n\n";