<?php
/**
 * Diagnostic page for Product T0010
 * This page helps investigate why product T0010 isn't showing on Delcampe
 */

// Load WordPress if accessing directly
if (!defined('ABSPATH')) {
    require_once(dirname(dirname(dirname(dirname(dirname(__FILE__))))) . '/wp-load.php');
}

// Check permissions
if (!current_user_can('manage_options')) {
    wp_die('You do not have permission to access this page.');
}

global $wpdb;
$listings_table = $wpdb->prefix . 'delcampe_listings';

?>
<!DOCTYPE html>
<html>
<head>
    <title>Delcampe Diagnostic - Product T0010</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .section { margin-bottom: 30px; background: #f9f9f9; padding: 15px; border: 1px solid #ddd; }
        .error { color: #dc3232; font-weight: bold; }
        .success { color: #46b450; font-weight: bold; }
        .warning { color: #f0ad4e; font-weight: bold; }
        table { border-collapse: collapse; width: 100%; margin-top: 10px; }
        th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
        th { background-color: #f2f2f2; }
        pre { background: #f1f1f1; padding: 10px; overflow-x: auto; }
        .button { 
            display: inline-block;
            padding: 10px 20px;
            background: #0073aa;
            color: white;
            text-decoration: none;
            border-radius: 3px;
            margin-top: 10px;
        }
        .button:hover { background: #005177; }
    </style>
</head>
<body>
    <h1>Delcampe Diagnostic - Product T0010</h1>
    
    <div class="section">
        <h2>1. Search for Product T0010</h2>
        <?php
        // Search by title, SKU, or ID
        $search_query = "
            SELECT p.ID, p.post_title, p.post_status, p.post_type,
                   pm_sku.meta_value as sku
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm_sku ON p.ID = pm_sku.post_id AND pm_sku.meta_key = '_sku'
            WHERE (p.post_title LIKE '%T0010%' 
                   OR pm_sku.meta_value LIKE '%T0010%'
                   OR p.ID = 'T0010')
            AND p.post_type IN ('product', 'product_variation')
            ORDER BY p.ID DESC
        ";
        
        $products = $wpdb->get_results($search_query);
        
        if ($products) {
            echo "<p class='success'>Found " . count($products) . " products matching 'T0010':</p>";
            echo "<table>";
            echo "<tr><th>ID</th><th>Title</th><th>SKU</th><th>Status</th><th>Type</th></tr>";
            
            foreach ($products as $product) {
                echo "<tr>";
                echo "<td>{$product->ID}</td>";
                echo "<td>{$product->post_title}</td>";
                echo "<td>" . ($product->sku ?: '-') . "</td>";
                echo "<td>{$product->post_status}</td>";
                echo "<td>{$product->post_type}</td>";
                echo "</tr>";
            }
            echo "</table>";
            
            // Store for further investigation
            $target_product_id = $products[0]->ID;
        } else {
            echo "<p class='error'>No products found matching 'T0010'</p>";
            $target_product_id = null;
        }
        ?>
    </div>
    
    <?php if ($target_product_id): ?>
    <div class="section">
        <h2>2. Delcampe Listing Status for Product #<?php echo $target_product_id; ?></h2>
        <?php
        $listing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $listings_table WHERE product_id = %d",
            $target_product_id
        ));
        
        if ($listing) {
            echo "<p class='success'>Listing found in database:</p>";
            echo "<table>";
            echo "<tr><th>Field</th><th>Value</th></tr>";
            foreach ($listing as $key => $value) {
                if ($key === 'last_errors' && !empty($value)) {
                    $value = "<span class='error'>" . htmlspecialchars($value) . "</span>";
                } elseif (empty($value)) {
                    $value = "<em>empty</em>";
                } else {
                    $value = htmlspecialchars($value);
                }
                echo "<tr><td><strong>$key</strong></td><td>$value</td></tr>";
            }
            echo "</table>";
            
            // Check if actually published
            if ($listing->status === 'published' && empty($listing->delcampe_id)) {
                echo "<p class='warning'>⚠️ Listing marked as 'published' but has no Delcampe ID!</p>";
            }
        } else {
            echo "<p class='error'>No listing found in database for this product!</p>";
            echo "<p>This means the product has never been synced to Delcampe.</p>";
        }
        ?>
    </div>
    
    <div class="section">
        <h2>3. Product Meta Data</h2>
        <?php
        $meta_keys = array(
            '_delcampe_sync_status' => 'Sync Status',
            '_delcampe_listing_id' => 'Delcampe Listing ID',
            '_delcampe_last_error' => 'Last Error',
            '_delcampe_last_sync' => 'Last Sync Time',
            '_stock_status' => 'Stock Status',
            '_stock' => 'Stock Quantity',
            '_price' => 'Price',
            '_regular_price' => 'Regular Price',
            '_sku' => 'SKU',
            '_manage_stock' => 'Manage Stock'
        );
        
        echo "<table>";
        echo "<tr><th>Meta Key</th><th>Value</th></tr>";
        
        foreach ($meta_keys as $key => $label) {
            $value = get_post_meta($target_product_id, $key, true);
            if (empty($value) && $value !== '0') {
                $value = "<em>not set</em>";
            } else {
                $value = htmlspecialchars($value);
            }
            echo "<tr><td><strong>$label</strong> ($key)</td><td>$value</td></tr>";
        }
        echo "</table>";
        ?>
    </div>
    
    <div class="section">
        <h2>4. Sync Queue Status</h2>
        <?php
        $sync_queue = get_option('delcampe_sync_queue', array());
        $in_queue = false;
        
        foreach ($sync_queue as $item) {
            if ($item['product_id'] == $target_product_id) {
                $in_queue = true;
                echo "<p class='warning'>Product is in sync queue:</p>";
                echo "<pre>" . print_r($item, true) . "</pre>";
                break;
            }
        }
        
        if (!$in_queue) {
            echo "<p>Product is not in the sync queue.</p>";
        }
        
        echo "<p>Total items in queue: " . count($sync_queue) . "</p>";
        ?>
    </div>
    
    <div class="section">
        <h2>5. Actions</h2>
        <?php if (!$listing): ?>
            <p>Since this product has no listing, you can:</p>
            <ol>
                <li>Go to the product edit page and click "Publish to Delcampe"</li>
                <li>Use the bulk action on the products list to publish to Delcampe</li>
                <li>Add the product to the sync queue manually</li>
            </ol>
            <a href="<?php echo admin_url('post.php?post=' . $target_product_id . '&action=edit'); ?>" class="button">
                Edit Product
            </a>
        <?php elseif ($listing->status !== 'published'): ?>
            <p>The listing exists but is not published. Current status: <strong><?php echo $listing->status; ?></strong></p>
            <a href="<?php echo admin_url('admin.php?page=delcampe-listings'); ?>" class="button">
                View Listings
            </a>
        <?php elseif (empty($listing->delcampe_id)): ?>
            <p>The listing is marked as published but has no Delcampe ID. This indicates a sync failure.</p>
            <p>Try republishing the product or check the error logs.</p>
        <?php else: ?>
            <p>The listing appears to be properly synced. Delcampe ID: <strong><?php echo $listing->delcampe_id; ?></strong></p>
            <?php if ($listing->view_item_url): ?>
                <a href="<?php echo esc_url($listing->view_item_url); ?>" target="_blank" class="button">
                    View on Delcampe
                </a>
            <?php endif; ?>
        <?php endif; ?>
    </div>
    
    <?php endif; // if target_product_id ?>
    
    <div class="section">
        <h2>6. Recent Sync Logs</h2>
        <?php
        $upload_dir = wp_upload_dir();
        $log_file = $upload_dir['basedir'] . '/delcampe-debug.log';
        
        if (file_exists($log_file)) {
            $log_contents = file_get_contents($log_file);
            $lines = explode("\n", $log_contents);
            $recent_lines = array_slice($lines, -50); // Last 50 lines
            
            // Filter for T0010 related entries
            $filtered_lines = array_filter($recent_lines, function($line) {
                return stripos($line, 'T0010') !== false || stripos($line, 'product ' . ($GLOBALS['target_product_id'] ?? '')) !== false;
            });
            
            if ($filtered_lines) {
                echo "<p>Recent log entries related to this product:</p>";
                echo "<pre>" . implode("\n", $filtered_lines) . "</pre>";
            } else {
                echo "<p>No recent log entries found for this product.</p>";
            }
        } else {
            echo "<p>Debug log file not found.</p>";
        }
        ?>
    </div>
    
    <hr>
    <p><a href="<?php echo admin_url('admin.php?page=delcampe-settings&tab=sync'); ?>">Back to Delcampe Settings</a></p>
</body>
</html>