#!/usr/bin/php
<?php
/**
 * Disable automatic webhook registration
 * Webhooks should only be created once, manually by the user
 * 
 * This script:
 * 1. Removes all scheduled webhook cron events
 * 2. Disables auto-registration setting
 * 3. Logs all changes for audit trail
 * 
 * @since 1.10.35.8
 */

require_once(dirname(dirname(dirname(dirname(dirname(__FILE__))))) . '/wp-load.php');

echo "\n=== DISABLING AUTOMATIC WEBHOOK REGISTRATION ===\n";
echo "Version: 1.10.35.8\n";
echo "Time: " . current_time('Y-m-d H:i:s') . "\n\n";

// 1. Remove all webhook-related cron events
$webhook_hooks = array(
    'delcampe_ensure_webhooks',      // Twice daily webhook check
    'delcampe_daily_webhook_check',   // Daily guardian check
    'process_delcampe_webhook'        // Any lingering webhook processing
);

$removed_count = 0;
foreach ($webhook_hooks as $hook) {
    // Remove all scheduled events for this hook
    $crons = _get_cron_array();
    $updated = false;
    
    foreach ($crons as $timestamp => $hooks) {
        if (isset($hooks[$hook])) {
            unset($crons[$timestamp][$hook]);
            if (empty($crons[$timestamp])) {
                unset($crons[$timestamp]);
            }
            $updated = true;
            $removed_count++;
            echo "✅ Removed scheduled event: $hook (was scheduled for " . date('Y-m-d H:i:s', $timestamp) . ")\n";
            delcampe_log('[Webhook Cleanup] Removed scheduled event: ' . $hook);
        }
    }
    
    if ($updated) {
        _set_cron_array($crons);
    }
}

if ($removed_count === 0) {
    echo "No scheduled webhook events found to remove.\n";
}

echo "\n";

// 2. Disable auto-registration setting
$old_setting = get_option('delcampe_auto_register_webhooks', 'yes');
update_option('delcampe_auto_register_webhooks', 'no');
echo "✅ Auto-registration setting changed from '$old_setting' to 'no'\n";
delcampe_log('[Webhook Cleanup] Auto-registration disabled (was: ' . $old_setting . ')');

// 3. Show current webhook status
echo "\n=== CURRENT WEBHOOK STATUS ===\n";

// Check for any remaining webhook crons
$crons = _get_cron_array();
$remaining = array();
foreach ($crons as $timestamp => $hooks) {
    foreach ($hooks as $hook => $data) {
        if (strpos($hook, 'webhook') !== false) {
            $remaining[] = $hook . ' @ ' . date('Y-m-d H:i:s', $timestamp);
        }
    }
}

if (empty($remaining)) {
    echo "✅ No webhook-related cron events are scheduled\n";
} else {
    echo "⚠️ Found remaining webhook events:\n";
    foreach ($remaining as $event) {
        echo "  - $event\n";
    }
}

// 4. Check current webhook registrations with Delcampe
echo "\n=== EXISTING WEBHOOK REGISTRATIONS ===\n";

$auth = Delcampe_Auth::get_instance();
$token = $auth->get_auth_token();

if (!is_wp_error($token)) {
    $url = DELCAMPE_API_URL . '/notification/settings?token=' . urlencode($token);
    $response = wp_remote_get($url, array(
        'timeout' => 20,
        'headers' => array('Accept' => 'application/xml')
    ));
    
    if (!is_wp_error($response)) {
        $xml = simplexml_load_string(wp_remote_retrieve_body($response));
        if ($xml && isset($xml->Notification_Data->body->notification_settings)) {
            $count = 0;
            foreach ($xml->Notification_Data->body->notification_settings as $setting) {
                $count++;
                echo "  - " . (string)$setting->type . " → " . (string)$setting->destination . "\n";
            }
            echo "\nTotal: $count webhook(s) registered with Delcampe\n";
        }
    }
}

// 5. Instructions for manual webhook management
echo "\n=== WEBHOOK MANAGEMENT ===\n";
echo "Automatic webhook registration has been DISABLED.\n";
echo "Webhooks will NOT be created or re-created automatically.\n\n";
echo "To manually manage webhooks:\n";
echo "1. Register webhooks: php testing/check-webhook-registration.php register\n";
echo "2. Check status: php testing/check-webhook-registration.php\n";
echo "3. Remove all: php testing/check-webhook-registration.php unregister\n";
echo "\nOr use the admin interface: Delcampe → Settings → Webhooks\n";

echo "\n✅ Automatic webhook registration disabled successfully!\n";
echo "Webhooks will now only be created when manually requested by the user.\n\n";