/**
 * Delcampe Listings Admin Scripts
 * Version: 1.6.1.0
 * 
 * Fixed Import Existing Listings button functionality
 * - Simplified debugging approach after removing conflicting inline JavaScript
 * - Enhanced error handling and AJAX response processing
 */

(function($) {
    'use strict';

    // Initialize when document is ready
    $(document).ready(function() {
        
        // Debug: Log initialization
        console.log('[Delcampe v1.6.1.0] Listings Admin JS loaded');
        console.log('[Delcampe v1.6.1.0] Import button found:', $('#import-existing-listings').length);
        console.log('[Delcampe v1.6.1.0] AJAX URL:', window.delcampe_listings ? delcampe_listings.ajax_url : 'NOT AVAILABLE');
        
        // Handle import existing listings button (v1.6.1.0 - Enhanced debugging and error handling)
        $('#import-existing-listings').on('click', function(e) {
            console.log('[Delcampe v1.6.1.0] Import button clicked');
            e.preventDefault();
            
            var $button = $(this);
            var $results = $('#import-results');
            
            // Check if required objects exist
            if (!window.delcampe_listings || !window.delcampe_listings.ajax_url) {
                console.error('[Delcampe v1.6.1.0] delcampe_listings object not available');
                alert('Configuration error. Please refresh the page and try again.');
                return;
            }
            
            console.log('[Delcampe v1.6.1.0] Starting AJAX request to:', delcampe_listings.ajax_url);
            console.log('[Delcampe v1.6.1.0] Using admin nonce:', delcampe_listings.admin_nonce);
            
            // Disable button and show loading state
            $button.prop('disabled', true).text(delcampe_listings.i18n.importing);
            
            // Send AJAX request to import existing listings
            $.ajax({
                url: delcampe_listings.ajax_url,
                type: 'POST',
                data: {
                    action: 'delcampe_import_existing_listings',
                    nonce: delcampe_listings.admin_nonce
                },
                dataType: 'json', // Explicitly expect JSON response
                beforeSend: function() {
                    console.log('[Delcampe v1.6.1.0] Sending AJAX request...');
                },
                success: function(response) {
                    console.log('[Delcampe v1.6.1.0] AJAX success response:', response);
                    
                    if (response.success) {
                        // Show success message
                        $results.removeClass('notice-error').addClass('notice-success').show();
                        $results.find('p').text(response.data.message);
                        
                        // Log details to console for debugging
                        if (response.data.results && response.data.results.details) {
                            console.log('[Delcampe v1.6.1.0] Import details:', response.data.results.details);
                        }
                        
                        // Reload page after 2 seconds to show updated listings
                        setTimeout(function() {
                            window.location.reload();
                        }, 2000);
                    } else {
                        // Show error message
                        $results.removeClass('notice-success').addClass('notice-error').show();
                        $results.find('p').text(response.data.message || delcampe_listings.i18n.import_error);
                        console.error('[Delcampe v1.6.1.0] Import failed:', response.data.message);
                    }
                },
                error: function(xhr, status, error) {
                    console.error('[Delcampe v1.6.1.0] AJAX error:', status, error);
                    console.error('[Delcampe v1.6.1.0] XHR response:', xhr.responseText);
                    
                    // Show error message
                    $results.removeClass('notice-success').addClass('notice-error').show();
                    
                    // Try to parse error response
                    var errorMessage = delcampe_listings.i18n.import_error;
                    try {
                        var errorResponse = JSON.parse(xhr.responseText);
                        if (errorResponse.data && errorResponse.data.message) {
                            errorMessage = errorResponse.data.message;
                        }
                    } catch (e) {
                        // If not JSON, include the raw error
                        errorMessage += ' (' + error + ')';
                    }
                    
                    $results.find('p').text(errorMessage);
                },
                complete: function() {
                    console.log('[Delcampe v1.6.1.0] AJAX request complete');
                    // Re-enable button with original text
                    $button.prop('disabled', false).text($button.data('original-text') || 'Import Existing Listings');
                }
            });
        });
        
        // Store original button text for restoration
        $('#import-existing-listings').each(function() {
            $(this).data('original-text', $(this).text());
        });
        
        // Handle delete confirmation
        $('.row-actions .delete a, .submitdelete').on('click', function(e) {
            if (!confirm(delcampe_listings.i18n.confirm_delete)) {
                e.preventDefault();
                return false;
            }
        });
        
        // Handle end listing confirmation
        $('.row-actions .end a').on('click', function(e) {
            if (!confirm(delcampe_listings.i18n.confirm_end)) {
                e.preventDefault();
                return false;
            }
        });
        
        // Handle bulk actions
        $('#doaction, #doaction2').on('click', function(e) {
            var action = $(this).siblings('select').val();
            
            if (action === 'delete') {
                if (!confirm(delcampe_listings.i18n.confirm_delete)) {
                    e.preventDefault();
                    return false;
                }
            } else if (action === 'end') {
                if (!confirm(delcampe_listings.i18n.confirm_end)) {
                    e.preventDefault();
                    return false;
                }
            }
        });
        
        // Handle listing preview
        $('.preview-listing').on('click', function(e) {
            e.preventDefault();
            var url = $(this).attr('href');
            tb_show('', url);
        });
        
        // Handle variations toggle
        $('.show-variations').on('click', function(e) {
            e.preventDefault();
            var listingId = $(this).data('listing-id');
            $('#variations-' + listingId).slideToggle();
        });
        
        // Handle status filter change
        $('.subsubsub a').on('click', function() {
            var $link = $(this);
            var status = $link.data('status');
            
            // Update URL without reloading
            if (history.pushState) {
                var newUrl = $link.attr('href');
                history.pushState({status: status}, '', newUrl);
            }
        });
        
        // Handle AJAX actions
        $('.ajax-action').on('click', function(e) {
            e.preventDefault();
            
            var $button = $(this);
            var action = $button.data('action');
            var listingId = $button.data('listing-id');
            
            // Show spinner
            $button.prop('disabled', true);
            $button.after('<span class="spinner is-active delcampe-spinner"></span>');
            
            // Send AJAX request
            $.post(delcampe_listings.ajax_url, {
                action: 'delcampe_listing_action',
                listing_action: action,
                listing_id: listingId,
                nonce: delcampe_listings.nonce
            })
            .done(function(response) {
                if (response.success) {
                    // Reload the page to show updated status
                    location.reload();
                } else {
                    alert(response.data.message || 'An error occurred');
                }
            })
            .fail(function() {
                alert('An error occurred. Please try again.');
            })
            .always(function() {
                // Remove spinner
                $button.prop('disabled', false);
                $button.siblings('.delcampe-spinner').remove();
            });
        });
        
        // Handle profile selection in prepare page
        $('#profile_id').on('change', function() {
            var profileId = $(this).val();
            if (profileId) {
                // Load profile details via AJAX
                $.post(delcampe_listings.ajax_url, {
                    action: 'delcampe_get_profile_details',
                    profile_id: profileId,
                    nonce: delcampe_listings.nonce
                })
                .done(function(response) {
                    if (response.success) {
                        // Display profile details
                        $('#profile-details').html(response.data.html);
                    }
                });
            } else {
                $('#profile-details').empty();
            }
        });
        
        // Handle search form
        $('#listings-search-input').on('keypress', function(e) {
            if (e.which === 13) {
                e.preventDefault();
                $('#search-submit').click();
            }
        });
        
        // Handle inline edit
        $('.inline-edit').on('click', function(e) {
            e.preventDefault();
            
            var $row = $(this).closest('tr');
            var listingId = $row.find('input[name="listing[]"]').val();
            
            // Toggle inline edit form
            var $editRow = $('#inline-edit-' + listingId);
            if ($editRow.length) {
                $editRow.toggle();
            } else {
                // Load inline edit form via AJAX
                loadInlineEditForm(listingId, $row);
            }
        });
        
        // Load inline edit form
        function loadInlineEditForm(listingId, $row) {
            $.post(delcampe_listings.ajax_url, {
                action: 'delcampe_inline_edit_form',
                listing_id: listingId,
                nonce: delcampe_listings.nonce
            })
            .done(function(response) {
                if (response.success) {
                    $row.after(response.data.html);
                    
                    // Handle inline save
                    $('#inline-edit-' + listingId + ' .save').on('click', function() {
                        saveInlineEdit(listingId);
                    });
                    
                    // Handle inline cancel
                    $('#inline-edit-' + listingId + ' .cancel').on('click', function() {
                        $('#inline-edit-' + listingId).remove();
                    });
                }
            });
        }
        
        // Save inline edit
        function saveInlineEdit(listingId) {
            var $form = $('#inline-edit-' + listingId);
            var data = {
                action: 'delcampe_save_inline_edit',
                listing_id: listingId,
                nonce: delcampe_listings.nonce,
                listing_title: $form.find('input[name="listing_title"]').val(),
                quantity: $form.find('input[name="quantity"]').val(),
                price: $form.find('input[name="price"]').val()
            };
            
            $.post(delcampe_listings.ajax_url, data)
            .done(function(response) {
                if (response.success) {
                    location.reload();
                } else {
                    alert(response.data.message || 'Error saving changes');
                }
            });
        }
        
        // Handle column sorting
        $('.sorted, .sortable').on('click', 'a', function(e) {
            // Let WordPress handle the sorting
        });
        
        // Handle screen options
        $('#screen-options-apply').on('click', function() {
            var perPage = $('#listings_per_page').val();
            
            // Update screen option via AJAX
            $.post(delcampe_listings.ajax_url, {
                action: 'delcampe_update_screen_option',
                option: 'delcampe_listings_per_page',
                value: perPage,
                nonce: delcampe_listings.nonce
            });
        });
        
        // Initialize tooltips
        if ($.fn.tipTip) {
            $('.tips').tipTip({
                'attribute': 'data-tip',
                'fadeIn': 50,
                'fadeOut': 50,
                'delay': 200
            });
        }
        
        // Handle mass edit
        $('#bulk-edit').on('click', function(e) {
            e.preventDefault();
            
            var selectedListings = [];
            $('input[name="listing[]"]:checked').each(function() {
                selectedListings.push($(this).val());
            });
            
            if (selectedListings.length === 0) {
                alert('Please select at least one listing.');
                return;
            }
            
            // Show mass edit dialog
            showMassEditDialog(selectedListings);
        });
        
        // Show mass edit dialog
        function showMassEditDialog(listingIds) {
            // Implementation for mass edit dialog
            // This would typically open a modal or inline form
        }
        
        // Auto-refresh for changed listings
        if ($('.listings-status.changed').length > 0) {
            // Set up auto-refresh every 30 seconds
            setInterval(function() {
                checkForUpdates();
            }, 30000);
        }
        
        // Check for updates
        function checkForUpdates() {
            $.post(delcampe_listings.ajax_url, {
                action: 'delcampe_check_listing_updates',
                nonce: delcampe_listings.nonce
            })
            .done(function(response) {
                if (response.success && response.data.has_updates) {
                    // Show notification
                    showUpdateNotification();
                }
            });
        }
        
        // Show update notification
        function showUpdateNotification() {
            var $notice = $('<div class="notice notice-info is-dismissible">' +
                '<p>Listings have been updated. <a href="#" class="refresh-page">Refresh page</a></p>' +
                '</div>');
            
            $('.wrap h1').after($notice);
            
            $notice.find('.refresh-page').on('click', function(e) {
                e.preventDefault();
                location.reload();
            });
        }
        
        // Log final initialization status
        console.log('[Delcampe v1.6.1.0] Listings admin JS initialization complete');
        
    });
    
})(jQuery);
