#!/usr/bin/php
<?php
/**
 * Reset webhooks - Remove ALL and register only canonical ones
 * 
 * @since 1.10.35.8
 */

require_once(dirname(dirname(dirname(dirname(dirname(__FILE__))))) . '/wp-load.php');

echo "\n=== WEBHOOK RESET - CLEAN SLATE ===\n";
echo "Version: 1.10.35.8\n";
echo "Time: " . current_time('Y-m-d H:i:s') . "\n\n";

// Get auth token
$auth = Delcampe_Auth::get_instance();
$token = $auth->get_auth_token();

if (is_wp_error($token)) {
    die("❌ Authentication failed: " . $token->get_error_message() . "\n");
}

// 1. Get canonical webhook URL
$webhook_token = get_option('delcampe_webhook_token');
if (!$webhook_token) {
    $webhook_token = wp_generate_password(32, false, false);
    update_option('delcampe_webhook_token', $webhook_token);
}

$canonical_url = home_url('/wp-json/delcampe/v1/webhook');
$canonical_url = str_replace('http://', 'https://', $canonical_url);
$canonical_url = add_query_arg('token', $webhook_token, $canonical_url);

echo "Canonical webhook URL: $canonical_url\n\n";

// 2. List and remove ALL webhooks
echo "=== STEP 1: REMOVING ALL WEBHOOKS ===\n\n";

$url = DELCAMPE_API_URL . '/notification/settings?token=' . urlencode($token);
$response = wp_remote_get($url, array(
    'timeout' => 20,
    'headers' => array('Accept' => 'application/xml')
));

if (is_wp_error($response)) {
    die("❌ Failed to list webhooks: " . $response->get_error_message() . "\n");
}

$xml = simplexml_load_string(wp_remote_retrieve_body($response));
if ($xml && isset($xml->Notification_Data->body->notification_settings)) {
    $removed_count = 0;
    foreach ($xml->Notification_Data->body->notification_settings as $setting) {
        // Only remove Curl webhooks for our site
        if (strpos((string)$setting->channel, 'Curl') === 0 &&
            (strpos((string)$setting->destination, 'dev.fdstamps.com') !== false ||
             strpos((string)$setting->destination, 'fdstamps.com') !== false)) {
            
            $id = (string)$setting->id_notification;
            echo "Removing: " . $setting->type . " (ID: $id)... ";
            
            $delete_url = DELCAMPE_API_URL . '/notification/' . $id . '?token=' . urlencode($token);
            $del_response = wp_remote_request($delete_url, array(
                'method' => 'DELETE',
                'timeout' => 20,
                'headers' => array('Accept' => 'application/xml')
            ));
            
            if (!is_wp_error($del_response)) {
                $code = wp_remote_retrieve_response_code($del_response);
                if ($code >= 200 && $code < 300) {
                    echo "✅\n";
                    $removed_count++;
                } else {
                    echo "❌ (HTTP $code)\n";
                }
            } else {
                echo "❌ (" . $del_response->get_error_message() . ")\n";
            }
        }
    }
    echo "\nRemoved $removed_count webhook(s)\n";
}

// 3. Wait a moment for API to process deletions
echo "\nWaiting 2 seconds for API to process deletions...\n";
sleep(2);

// 4. Register only the essential webhooks with canonical URL
echo "\n=== STEP 2: REGISTERING ESSENTIAL WEBHOOKS ===\n\n";

$essential_webhooks = array(
    'Curl_Seller_Item_Close_Sold' => 'Item sold notification',
    'Curl_Seller_Payment_Received' => 'Payment received notification'
);

$registered_count = 0;
foreach ($essential_webhooks as $type => $description) {
    echo "Registering: $description... ";
    
    $register_url = DELCAMPE_API_URL . '/notification/settings?token=' . urlencode($token);
    $body = array(
        'notificationType' => $type,
        'destination' => $canonical_url
    );
    
    $response = wp_remote_post($register_url, array(
        'timeout' => 20,
        'headers' => array(
            'Content-Type' => 'application/x-www-form-urlencoded',
            'Accept' => 'application/xml'
        ),
        'body' => http_build_query($body)
    ));
    
    if (!is_wp_error($response)) {
        $code = wp_remote_retrieve_response_code($response);
        if ($code >= 200 && $code < 300) {
            echo "✅\n";
            $registered_count++;
            delcampe_log('[Webhook Reset] Registered: ' . $type);
        } else {
            echo "❌ (HTTP $code)\n";
            $body = wp_remote_retrieve_body($response);
            if ($body) {
                echo "  Response: " . substr($body, 0, 200) . "\n";
            }
        }
    } else {
        echo "❌ (" . $response->get_error_message() . ")\n";
    }
}

echo "\nRegistered $registered_count webhook(s)\n";

// 5. Verify final state
echo "\n=== STEP 3: VERIFYING FINAL STATE ===\n\n";

sleep(1);
$response = wp_remote_get($url, array(
    'timeout' => 20,
    'headers' => array('Accept' => 'application/xml')
));

if (!is_wp_error($response)) {
    $xml = simplexml_load_string(wp_remote_retrieve_body($response));
    if ($xml && isset($xml->Notification_Data->body->notification_settings)) {
        $count = 0;
        foreach ($xml->Notification_Data->body->notification_settings as $setting) {
            if (strpos((string)$setting->channel, 'Curl') === 0 &&
                strpos((string)$setting->destination, 'fdstamps.com') !== false) {
                $count++;
                echo "✅ " . $setting->type . " → " . $setting->destination . "\n";
            }
        }
        echo "\nTotal: $count webhook(s) registered\n";
    }
}

// 6. Disable auto-registration to prevent future duplicates
update_option('delcampe_auto_register_webhooks', 'no');

echo "\n✅ WEBHOOK RESET COMPLETE!\n";
echo "\nSettings:\n";
echo "- Canonical URL: $canonical_url\n";
echo "- Auto-registration: DISABLED\n";
echo "- Webhooks will only be created when manually requested\n";
echo "\nTo check webhook status in the future:\n";
echo "  php testing/check-webhook-registration.php\n\n";