/**
 * Bulk Publish JavaScript
 * 
 * Handles the bulk publish interface interactions
 * 
 * @version 1.9.0.0
 */
(function($) {
    'use strict';
    
    var DelcampeBulkPublish = {
        
        currentMethod: 'category',
        selectedProducts: [],
        profileId: 0,
        
        init: function() {
            this.bindEvents();
        },
        
        bindEvents: function() {
            // Tab switching
            $('.method-tabs a').on('click', this.switchTab);
            
            // Preview button
            $('#preview-products').on('click', this.previewProducts.bind(this));
            
            // Execute publish button
            $('#execute-publish').on('click', this.executePublish.bind(this));
            
            // Cancel button
            $('#cancel-publish').on('click', this.cancelPublish.bind(this));
            
            // Profile selection
            $('#delcampe-profile').on('change', this.profileChanged.bind(this));
            
            // Select all checkboxes
            $('.select-all-categories').on('change', this.selectAllCategories);
            
            // Parent category checkbox
            $('.category-item > input[type="checkbox"]').on('change', this.categoryCheckChanged);
            
            // Select all products checkbox
            $(document).on('change', '#select-all-products', this.selectAllProducts.bind(this));
        },
        
        switchTab: function(e) {
            e.preventDefault();
            
            var $tab = $(this);
            var target = $tab.attr('href');
            var method = target.replace('#method-', '');
            
            // Update tabs
            $('.method-tabs a').removeClass('active');
            $tab.addClass('active');
            
            // Update panels
            $('.method-panel').removeClass('active');
            $(target).addClass('active');
            
            // Store current method
            DelcampeBulkPublish.currentMethod = method;
        },
        
        profileChanged: function() {
            this.profileId = $('#delcampe-profile').val();
            $('#preview-products').prop('disabled', !this.profileId);
        },
        
        categoryCheckChanged: function() {
            var $checkbox = $(this);
            var $subcategories = $checkbox.siblings('.subcategories').find('input[type="checkbox"]');
            
            // Check/uncheck all subcategories
            $subcategories.prop('checked', $checkbox.prop('checked'));
        },
        
        previewProducts: function() {
            var self = this;
            var $button = $('#preview-products');
            var data = {
                action: 'delcampe_bulk_publish_preview',
                nonce: delcampeBulkPublish.nonce,
                method: this.currentMethod,
                profile_id: this.profileId,
                max_batch_size: $('#max-batch-size').val() || 100
            };
            
            // Always exclude already listed products by default for all methods
            data.exclude_listed = 1;
            
            // Collect method-specific data
            switch (this.currentMethod) {
                case 'category':
                    data.categories = [];
                    $('input[name="categories[]"]:checked').each(function() {
                        data.categories.push($(this).val());
                    });
                    break;
                    
                case 'tags':
                    data.tags = [];
                    $('input[name="tags[]"]:checked').each(function() {
                        data.tags.push($(this).val());
                    });
                    break;
                    
                case 'filter':
                    data.filter_category = $('select[name="filter_category"]').val();
                    data.product_status = $('select[name="product_status"]').val();
                    data.stock_status = $('select[name="stock_status"]').val();
                    data.sku_filter = $('input[name="sku_filter"]').val();
                    data.price_min = $('input[name="price_min"]').val();
                    data.price_max = $('input[name="price_max"]').val();
                    // For filter mode, respect the checkbox setting (overrides default)
                    data.exclude_listed = $('input[name="exclude_listed"]').prop('checked') ? 1 : 0;
                    break;
                    
                case 'all':
                    if (!$('input[name="confirm_all"]').prop('checked')) {
                        alert(delcampeBulkPublish.strings.confirmAll);
                        return;
                    }
                    // Uses the default exclude_listed = 1 set above
                    break;
            }
            
            // Show loading
            $button.prop('disabled', true).text('Loading...');
            
            $.post(delcampeBulkPublish.ajaxUrl, data)
                .done(function(response) {
                    if (response.success) {
                        self.displayPreview(response.data);
                    } else {
                        alert(response.data || delcampeBulkPublish.strings.error);
                    }
                })
                .fail(function() {
                    alert(delcampeBulkPublish.strings.error);
                })
                .always(function() {
                    $button.prop('disabled', false).text(delcampeBulkPublish.strings.previewProducts || 'Preview Products');
                });
        },
        
        displayPreview: function(data) {
            var self = this;
            
            // Store data
            this.selectedProducts = data.products.map(function(p) { return p.id; });
            
            // Update summary
            var summaryText = sprintf(
                'Found %d products to publish with profile #%d',
                data.total,
                data.profile_id
            );
            $('.summary-text').text(summaryText);
            
            // Clear existing products
            var $tbody = $('#preview-products-list').empty();
            
            // Add products to table
            if (data.products.length > 0) {
                $.each(data.products, function(i, product) {
                    var row = '<tr data-product-id="' + product.id + '">' +
                        '<td class="check-column" style="text-align: center; width: 30px;">' +
                        '<input type="checkbox" class="product-select" value="' + product.id + '" checked="checked" />' +
                        '</td>' +
                        '<td style="width: 60px;"><img src="' + product.image + '" style="width: 40px; height: auto; display: block;" /></td>' +
                        '<td>' + product.title + '</td>' +
                        '<td>' + (product.sku || '—') + '</td>' +
                        '<td>' + self.formatPrice(product.price) + '</td>' +
                        '<td>' + (product.stock || '∞') + '</td>' +
                        '<td>' + product.categories + '</td>' +
                        '</tr>';
                    $tbody.append(row);
                });
                
                // Show note if more products than displayed
                if (data.total > data.products.length) {
                    $tbody.append('<tr><td colspan="7" style="text-align: center; font-style: italic;">... and ' + 
                        (data.total - data.products.length) + ' more products</td></tr>');
                }
                
                $('#execute-publish').prop('disabled', false);
            } else {
                $tbody.append('<tr><td colspan="7">' + delcampeBulkPublish.strings.noProducts + '</td></tr>');
                $('#execute-publish').prop('disabled', true);
            }
            
            // Make sure the select all checkbox is checked by default
            $('#select-all-products').prop('checked', true);
            
            // Show review step
            $('.step-review').show();
            
            // Scroll to review
            $('html, body').animate({
                scrollTop: $('.step-review').offset().top - 100
            }, 500);
        },
        
        executePublish: function() {
            var self = this;
            
            // Get selected products
            var selectedProducts = [];
            $('.product-select:checked').each(function() {
                selectedProducts.push(parseInt($(this).val()));
            });
            
            if (selectedProducts.length === 0) {
                alert('Please select at least one product to publish.');
                return;
            }
            
            // Update selected products
            this.selectedProducts = selectedProducts;
            
            if (!confirm(sprintf(delcampeBulkPublish.strings.confirmPublish, this.selectedProducts.length))) {
                return;
            }
            
            // Hide buttons, show progress
            $('.publish-actions').hide();
            $('.publish-progress').show();
            
            // Start publishing
            this.publishBatch(0);
        },
        
        publishBatch: function(offset) {
            var self = this;
            
            // Update progress text to show we're starting
            if (offset === 0) {
                $('.progress-text').text('Preparing to publish ' + this.selectedProducts.length + ' products...');
            }
            
            var data = {
                action: 'delcampe_bulk_publish_execute',
                nonce: delcampeBulkPublish.nonce,
                product_ids: this.selectedProducts,
                profile_id: this.profileId,
                offset: offset,
                // Always use batch queue so jobs appear in Batch Queue UI and process reliably
                use_batch_queue: 1,
                // Honor the user-chosen max items per batch (defaults to 100 in UI)
                batch_size: parseInt($('#max-batch-size').val() || 100, 10)
            };
            
            $.post(delcampeBulkPublish.ajaxUrl, data)
                .done(function(response) {
                    if (response.success) {
                        // Check if batch queue was used (for large operations)
                        if (response.data.use_batch_queue) {
                            // Batch queue was created - show completion
                            $('.progress-fill').css('width', '100%');
                            $('.progress-text').html('<strong>Publishing complete!</strong> ' + 
                                self.selectedProducts.length + ' products have been queued for sync to Delcampe.');
                            
                            // Show success message with link to batch monitor
                            setTimeout(function() {
                                if (confirm('Batch created successfully! ' + self.selectedProducts.length + 
                                    ' products are being processed in the background. Would you like to go to the Batch Queue to track progress?')) {
                                    window.location.href = 'admin.php?page=delcampe-settings&tab=inventory';
                                } else {
                                    location.reload();
                                }
                            }, 1500);
                        } else {
                            // Normal progressive processing
                            // Update progress
                            var percent = Math.round((response.data.processed / response.data.total) * 100);
                            $('.progress-fill').css('width', percent + '%');
                            $('.progress-text').text(sprintf(
                                delcampeBulkPublish.strings.published,
                                response.data.processed,
                                response.data.total
                            ));
                            
                            // Continue with next batch
                            if (response.data.has_more) {
                                setTimeout(function() {
                                    self.publishBatch(response.data.next_offset);
                                }, 1000); // Small delay between batches
                            } else {
                                // Complete!
                                self.publishComplete(response.data);
                            }
                        }
                    } else {
                        alert(response.data || delcampeBulkPublish.strings.error);
                        self.cancelPublish();
                    }
                })
                .fail(function() {
                    alert(delcampeBulkPublish.strings.error);
                    self.cancelPublish();
                });
        },
        
        publishComplete: function(data) {
            $('.progress-text').html('<strong>Publishing complete!</strong> ' + data.processed + ' products have been queued for sync to Delcampe.');
            
            // Show success message
            setTimeout(function() {
                if (confirm('Publishing complete! Would you like to go to the Sync Management page to monitor progress?')) {
                    window.location.href = 'admin.php?page=delcampe-settings&tab=sync';
                } else {
                    location.reload();
                }
            }, 2000);
        },
        
        cancelPublish: function() {
            $('.step-review').hide();
            $('.publish-actions').show();
            $('.publish-progress').hide();
            $('.progress-fill').css('width', '0%');
        },
        
        selectAllProducts: function(e) {
            var isChecked = $(e.target).prop('checked');
            $('.product-select').prop('checked', isChecked);
            console.log('Select all triggered: ' + isChecked);
        },
        
        formatPrice: function(price) {
            if (!price) return '—';
            return '$' + parseFloat(price).toFixed(2);
        }
    };
    
    // Helper function for sprintf-like formatting
    function sprintf(format) {
        var args = Array.prototype.slice.call(arguments, 1);
        return format.replace(/%[sd]/g, function(match) {
            var replacement = args.shift();
            return replacement !== undefined ? replacement : match;
        });
    }
    
    // Initialize when ready
    $(document).ready(function() {
        DelcampeBulkPublish.init();
    });
    
})(jQuery);
