/**
 * Delcampe Listings Importer JavaScript
 * 
 * @package WC_Delcampe_Integration
 * @since 1.9.0.6
 */

(function($) {
    'use strict';
    
    var DelcampeImporter = {
        listings: [],
        matched: {},
        selectedListings: [],
        
        init: function() {
            this.bindEvents();
        },
        
        bindEvents: function() {
            $('#fetch-listings-btn').on('click', this.fetchListings.bind(this));
            $('#import-selected-btn').on('click', this.importSelected.bind(this));
            $('#cancel-import-btn').on('click', this.cancelImport.bind(this));
            $('#import-more-btn').on('click', this.resetImporter.bind(this));
            $('#select-all-listings').on('change', this.toggleSelectAll.bind(this));
            $(document).on('change', '.listing-checkbox', this.updateSelectedCount.bind(this));
        },
        
        fetchListings: function() {
            var self = this;
            var $button = $('#fetch-listings-btn');
            var $spinner = $button.siblings('.spinner');
            
            $button.prop('disabled', true);
            $spinner.addClass('is-active');
            
            $('#import-progress').show();
            $('.progress-text').text(delcampe_importer.strings.fetching);
            
            var data = {
                action: 'delcampe_fetch_listings',
                nonce: delcampe_importer.nonce,
                include_active: $('#import-active').is(':checked'),
                include_ended: $('#import-ended').is(':checked'),
                include_sold: $('#import-sold').is(':checked')
            };
            
            $.post(delcampe_importer.ajax_url, data)
                .done(function(response) {
                    if (response.success) {
                        self.listings = response.data.listings;
                        $('.listings-count').text('Found ' + response.data.total + ' listings');
                        
                        // Now match products
                        self.matchProducts();
                    } else {
                        alert(delcampe_importer.strings.error + ' ' + response.data.message);
                        $('#import-progress').hide();
                    }
                })
                .fail(function() {
                    alert(delcampe_importer.strings.error + ' Network error');
                    $('#import-progress').hide();
                })
                .always(function() {
                    $button.prop('disabled', false);
                    $spinner.removeClass('is-active');
                });
        },
        
        matchProducts: function() {
            var self = this;
            $('.progress-text').text(delcampe_importer.strings.matching);
            
            var data = {
                action: 'delcampe_match_products',
                nonce: delcampe_importer.nonce,
                listings: this.listings,
                method: $('#matching-method').val()
            };
            
            $.post(delcampe_importer.ajax_url, data)
                .done(function(response) {
                    if (response.success) {
                        self.matched = response.data.matched;
                        self.displayListings();
                    } else {
                        alert(delcampe_importer.strings.error + ' ' + response.data.message);
                    }
                    $('#import-progress').hide();
                })
                .fail(function() {
                    alert(delcampe_importer.strings.error + ' Network error');
                    $('#import-progress').hide();
                });
        },
        
        displayListings: function() {
            var self = this;
            var $tbody = $('#listings-table-body');
            $tbody.empty();
            
            $.each(this.listings, function(index, listing) {
                var productId = self.matched[listing.id] || 0;
                var matchedClass = productId > 0 ? 'matched' : 'no-match';
                var matchedText = productId > 0 ? 'Product #' + productId : 'No match';
                
                var row = '<tr class="' + matchedClass + '">' +
                    '<th scope="row" class="check-column">' +
                        '<input type="checkbox" class="listing-checkbox" value="' + listing.id + '" data-index="' + index + '">' +
                    '</th>' +
                    '<td>' + listing.id + '</td>' +
                    '<td>' + listing.title + '</td>' +
                    '<td>' + listing.status + '</td>' +
                    '<td>' + listing.price + ' ' + listing.currency + '</td>' +
                    '<td>' + matchedText + '</td>' +
                '</tr>';
                
                $tbody.append(row);
            });
            
            $('#listings-preview').show();
        },
        
        toggleSelectAll: function(e) {
            $('.listing-checkbox').prop('checked', $(e.target).is(':checked'));
            this.updateSelectedCount();
        },
        
        updateSelectedCount: function() {
            var count = $('.listing-checkbox:checked').length;
            $('#import-selected-btn').prop('disabled', count === 0);
            
            if (count > 0) {
                $('#import-selected-btn').text('Import ' + count + ' Selected Listings');
            } else {
                $('#import-selected-btn').text('Import Selected Listings');
            }
        },
        
        importSelected: function() {
            if (!confirm(delcampe_importer.strings.confirm_import)) {
                return;
            }
            
            var self = this;
            this.selectedListings = [];
            
            $('.listing-checkbox:checked').each(function() {
                self.selectedListings.push($(this).data('index'));
            });
            
            if (this.selectedListings.length === 0) {
                return;
            }
            
            $('#listings-preview').hide();
            $('#import-progress').show();
            $('.progress-bar-inner').css('width', '0%');
            
            this.importNextListing(0);
        },
        
        importNextListing: function(index) {
            var self = this;
            
            if (index >= this.selectedListings.length) {
                this.showResults();
                return;
            }
            
            var listingIndex = this.selectedListings[index];
            var listing = this.listings[listingIndex];
            var progress = ((index + 1) / this.selectedListings.length) * 100;
            
            $('.progress-bar-inner').css('width', progress + '%');
            $('.progress-text').text(
                delcampe_importer.strings.importing.replace('%d', index + 1).replace('%d', this.selectedListings.length)
            );
            
            var data = {
                action: 'delcampe_import_listing',
                nonce: delcampe_importer.nonce,
                listing: listing,
                product_id: this.matched[listing.id] || 0,
                profile_id: $('#default-profile').val(),
                create_products: $('#create-products').is(':checked')
            };
            
            $.post(delcampe_importer.ajax_url, data)
                .done(function(response) {
                    if (!response.success) {
                        console.error('Failed to import listing ' + listing.id + ': ' + response.data.message);
                    }
                })
                .always(function() {
                    // Continue with next listing
                    self.importNextListing(index + 1);
                });
        },
        
        showResults: function() {
            $('#import-progress').hide();
            $('#import-results').show();
            
            var matched = 0;
            var self = this;
            
            $.each(this.selectedListings, function(i, index) {
                var listing = self.listings[index];
                if (self.matched[listing.id]) {
                    matched++;
                }
            });
            
            var summary = '<strong>' + delcampe_importer.strings.complete + '</strong><br>' +
                'Imported: ' + this.selectedListings.length + ' listings<br>' +
                'Matched with products: ' + matched + '<br>' +
                'Unmatched: ' + (this.selectedListings.length - matched);
            
            $('.results-summary').html(summary);
        },
        
        cancelImport: function() {
            $('#listings-preview').hide();
            this.listings = [];
            this.matched = {};
        },
        
        resetImporter: function() {
            $('#import-results').hide();
            $('#listings-preview').hide();
            this.listings = [];
            this.matched = {};
            this.selectedListings = [];
            $('#select-all-listings').prop('checked', false);
        }
    };
    
    // Initialize when document is ready
    $(document).ready(function() {
        DelcampeImporter.init();
    });
    
})(jQuery);