<?php
/**
 * Delcampe Authentication Handler
 * 
 * Manages API authentication for Delcampe integration
 * Handles token retrieval, storage, and validation
 * 
 * @package     WooCommerce_Delcampe_Integration
 * @subpackage  Authentication
 * @since       1.0.7.7
 * @version     1.8.0.0
 * 
 * @author      Frank Kahle
 * @copyright   2024 Frank Kahle
 * @license     Proprietary
 * 
 * Changelog:
 * 1.8.0.0 - Production release with comprehensive documentation
 *         - Enhanced security with input validation
 *         - Improved error handling and logging
 *         - Added detailed inline comments
 *         - Standardized code formatting
 *         - Enhanced token validation logic
 *         - Added constants usage for configuration
 * 1.2.2.1 - Fixed XML parsing to handle nested token in Delcampe_Notification structure
 * 1.2.2.0 - Updated to match actual Delcampe API authentication
 *         - Changed from GET to POST method
 *         - Updated endpoint to /seller
 *         - Fixed token parsing from response
 * 1.0.9.0 - Enhanced error handling and logging capabilities
 * 1.0.8.0 - Added comprehensive documentation
 * 1.0.7.7 - Initial implementation
 */

// Exit if accessed directly to prevent direct file access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class Delcampe_Auth
 * 
 * Singleton class responsible for handling all authentication operations
 * with the Delcampe API. Manages token lifecycle including retrieval,
 * caching, validation, and refresh.
 * 
 * The Delcampe API uses token-based authentication:
 * 1. Send API key to /seller endpoint via POST
 * 2. Receive XML response with authentication token
 * 3. Use token for subsequent API calls
 * 4. Token expires after 30 minutes
 * 
 * @since   1.0.7.7
 * @version 1.8.0.0
 */
class Delcampe_Auth {

    /**
     * Singleton instance of the authentication handler
     * Ensures only one authentication manager exists
     * 
     * @var Delcampe_Auth|null
     * @since 1.0.7.7
     */
    private static $instance = null;

    /**
     * Base URL for Delcampe API endpoints
     * Uses constant for centralized configuration
     * 
     * @var string
     * @since 1.0.7.7
     */
    private $api_base_url;

    /**
     * WordPress transient key for storing the auth token
     * Uses constant for consistency across plugin
     * 
     * @var string
     * @since 1.0.7.7
     */
    private $token_transient_key;

    /**
     * Token expiration time in seconds
     * Delcampe tokens expire after 30 minutes as per API documentation
     * 
     * @var int
     * @since 1.0.7.7
     */
    private $token_expiry;

    /**
     * Get singleton instance
     * 
     * Ensures only one instance of the authentication handler exists
     * throughout the plugin lifecycle. This prevents multiple token
     * requests and maintains consistency.
     * 
     * @since  1.0.7.7
     * @version 1.8.0.0
     * @return Delcampe_Auth The singleton instance
     */
    public static function get_instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Private constructor to enforce singleton pattern
     * 
     * Initializes properties using plugin constants for consistency.
     * Sets up the authentication handler with proper configuration.
     * 
     * @since  1.0.7.7
     * @version 1.8.0.0
     */
    private function __construct() {
        // Initialize properties from constants
        $this->api_base_url = DELCAMPE_API_BASE_URL;
        $this->token_transient_key = DELCAMPE_TOKEN_TRANSIENT;
        $this->token_expiry = DELCAMPE_TOKEN_EXPIRY;
        
        // Log initialization
        $this->log( 'Authentication handler initialized', 'info' );
    }

    /**
     * Get authentication token
     * 
     * Retrieves the authentication token from cache or requests a new one
     * if the cached token is expired or doesn't exist. This method is the
     * main entry point for getting a valid token.
     * 
     * Flow:
     * 1. Check for cached token in WordPress transients
     * 2. If valid cached token exists, return it
     * 3. If no valid token, request new one from API
     * 4. Cache the new token for future use
     * 
     * @since  1.0.7.7
     * @version 1.8.0.0
     * @return string|WP_Error Authentication token or error object
     */
    public function get_auth_token() {
        // v1.10.35.6: Enhanced error handling and token recovery
        // Check if we have a cached token
        $cached_token = get_transient( $this->token_transient_key );
        
        if ( $cached_token !== false && ! empty( $cached_token ) ) {
            $this->log( 'Using cached authentication token', 'debug' );
            return $cached_token;
        }

        // No valid cached token, request a new one
        $this->log( 'Requesting new authentication token from Delcampe API', 'info' );
        
        // Try up to 3 times with exponential backoff
        $max_attempts = 3;
        $attempt = 0;
        
        while ( $attempt < $max_attempts ) {
            $attempt++;
            $token = $this->request_new_token();
            
            if ( ! is_wp_error( $token ) ) {
                return $token;
            }
            
            // If rate limited or temporary failure, wait and retry
            $error_code = $token->get_error_code();
            if ( in_array( $error_code, array( 'rate_limited', 'request_failed' ) ) && $attempt < $max_attempts ) {
                $wait_time = pow( 2, $attempt - 1 ); // 1, 2, 4 seconds
                $this->log( sprintf( 'Auth attempt %d failed, waiting %d seconds before retry', $attempt, $wait_time ), 'warning' );
                sleep( $wait_time );
            } else {
                // Permanent error, don't retry
                break;
            }
        }
        
        return $token;
    }

    /**
     * Request a new authentication token from Delcampe API
     * 
     * Makes a POST request to the /seller endpoint with the API key
     * to retrieve a new authentication token. Handles all aspects of
     * the HTTP request including error handling and response parsing.
     * 
     * @since  1.0.7.7
     * @version 1.8.0.0
     * @return string|WP_Error Authentication token or error object
     */
    private function request_new_token() {
        // Get API key from settings
        $api_key = get_option( DELCAMPE_API_KEY_OPTION, '' );
        
        // v1.10.35.6: Validate API key with clearer error messages
        if ( empty( $api_key ) ) {
            $error_msg = 'API key not configured. Please go to Delcampe > Settings to configure your API key.';
            $this->log( $error_msg, 'error' );
            
            // Check if we're in the admin and show admin notice
            if ( is_admin() && current_user_can( 'manage_options' ) ) {
                add_action( 'admin_notices', function() {
                    echo '<div class="notice notice-error is-dismissible">';
                    echo '<p><strong>Delcampe Sync:</strong> ' . esc_html__( 'API key is not configured. Please configure it in the settings.', 'wc-delcampe-integration' ) . '</p>';
                    echo '<p><a href="' . admin_url( 'admin.php?page=delcampe-settings' ) . '" class="button button-primary">' . esc_html__( 'Configure Now', 'wc-delcampe-integration' ) . '</a></p>';
                    echo '</div>';
                });
            }
            
            return new WP_Error( 
                'no_api_key', 
                __( 'Delcampe API key is not configured. Please set it in the plugin settings.', 'wc-delcampe-integration' ) 
            );
        }

        // Validate API key format
        if ( ! $this->validate_api_key( $api_key ) ) {
            return new WP_Error( 
                'invalid_api_key', 
                __( 'Invalid API key format. Please check your API key.', 'wc-delcampe-integration' ) 
            );
        }

        // Prepare the POST data
        $post_data = array(
            'apikey' => sanitize_text_field( $api_key )
        );

        // v1.10.35.5: Check rate limit before making request
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-delcampe-rate-limiter.php';
        $rate_limiter = Delcampe_Rate_Limiter::get_instance();
        
        $can_request = $rate_limiter->can_make_request();
        if ( is_wp_error( $can_request ) ) {
            $this->log( 'Rate limited: ' . $can_request->get_error_message(), 'warning' );
            return $can_request;
        }
        
        // Prepare the request arguments
        $request_url = $this->api_base_url . '/seller';
        $args = array(
            'method'      => 'POST',
            'timeout'     => 30,
            'redirection' => 5,
            'httpversion' => '1.1',
            'blocking'    => true,
            'headers'     => array(
                'Content-Type' => 'application/x-www-form-urlencoded',
                'User-Agent'   => DELCAMPE_USER_AGENT
            ),
            'body'        => $post_data,
            'cookies'     => array()
        );

        $this->log( 'Sending authentication request to: ' . $request_url, 'debug' );

        // Make the API request
        $response = wp_remote_post( $request_url, $args );
        
        // Record the request for rate limiting
        $response_code = is_wp_error( $response ) ? 0 : wp_remote_retrieve_response_code( $response );
        $rate_limiter->record_request( '/seller', $response_code );

        // Check for request errors
        if ( is_wp_error( $response ) ) {
            $error_msg = 'Authentication request failed: ' . $response->get_error_message();
            $this->log( $error_msg, 'error' );
            return new WP_Error( 
                'request_failed', 
                sprintf( 
                    __( 'Failed to connect to Delcampe API: %s', 'wc-delcampe-integration' ), 
                    $response->get_error_message() 
                )
            );
        }

        // Get response code and body
        $response_code = wp_remote_retrieve_response_code( $response );
        $response_body = wp_remote_retrieve_body( $response );

        $this->log( 'Authentication response code: ' . $response_code, 'debug' );
        
        // Log response body only in debug mode for security
        if ( delcampe_is_debug_mode() ) {
            $this->log( 'Authentication response body: ' . $response_body, 'debug' );
        }

        // Check response code
        if ( $response_code !== 200 ) {
            $error_msg = sprintf( 'Authentication failed with response code: %d', $response_code );
            $this->log( $error_msg, 'error' );
            
            // Provide user-friendly error messages based on response code
            $user_message = __( 'Authentication failed.', 'wc-delcampe-integration' );
            if ( $response_code === 401 ) {
                $user_message = __( 'Invalid API key. Please check your credentials.', 'wc-delcampe-integration' );
            } elseif ( $response_code === 429 ) {
                // v1.10.35.5: Handle rate limiting
                $user_message = __( 'API rate limit exceeded. Please wait before trying again.', 'wc-delcampe-integration' );
                // Clear token cache to force renewal later
                delete_transient( $this->token_transient_key );
            } elseif ( $response_code === 500 ) {
                $user_message = __( 'Delcampe API server error. Please try again later.', 'wc-delcampe-integration' );
            } elseif ( $response_code === 503 ) {
                $user_message = __( 'Delcampe API is temporarily unavailable. Please try again later.', 'wc-delcampe-integration' );
            }
            
            return new WP_Error( 'auth_failed', $user_message );
        }

        // Parse the XML response to extract token
        $token = $this->parse_token_response( $response_body );

        if ( is_wp_error( $token ) ) {
            return $token;
        }

        // Cache the token for future use
        set_transient( $this->token_transient_key, $token, $this->token_expiry );
        $this->log( 'Authentication successful, token cached for 30 minutes', 'info' );

        return $token;
    }

    /**
     * Parse authentication token from API response
     * 
     * Extracts the authentication token from the XML response body.
     * Handles multiple possible XML structures for backward compatibility.
     * 
     * Expected format variations:
     * 1. <Delcampe_Notification><Notification_Data><body><token>...</token></body>...
     * 2. <response><token>...</token></response>
     * 3. <body><token>...</token></body>
     * 
     * @since  1.0.7.7
     * @version 1.8.0.0
     * @param  string $response_body The XML response body from API
     * @return string|WP_Error Extracted token or error object
     */
    private function parse_token_response( $response_body ) {
        // Validate response body
        if ( empty( $response_body ) ) {
            $error_msg = 'Empty response from authentication API';
            $this->log( $error_msg, 'error' );
            return new WP_Error( 
                'empty_response', 
                __( 'Empty response received from Delcampe API.', 'wc-delcampe-integration' ) 
            );
        }

        // Suppress XML errors and use internal error handling
        libxml_use_internal_errors( true );

        try {
            // Attempt to parse the XML response
            $xml = simplexml_load_string( $response_body );
            
            if ( $xml === false ) {
                // Collect XML parsing errors
                $errors = libxml_get_errors();
                $error_messages = array();
                foreach ( $errors as $error ) {
                    $error_messages[] = trim( $error->message );
                }
                libxml_clear_errors();
                
                $error_msg = 'XML Parse Error: ' . implode( '; ', $error_messages );
                $this->log( $error_msg, 'error' );
                
                return new WP_Error( 
                    'xml_parse_error', 
                    __( 'Failed to parse API response. The response may not be valid XML.', 'wc-delcampe-integration' ) 
                );
            }

            // Check for error elements in response
            if ( isset( $xml->error ) || isset( $xml->e ) ) {
                $error_msg = isset( $xml->error ) ? (string) $xml->error : (string) $xml->e;
                $this->log( 'API returned error: ' . $error_msg, 'error' );
                return new WP_Error( 'api_error', $error_msg );
            }

            // Extract token from various possible locations
            $token = null;
            
            // According to API docs, token should be in <body><token>...</token></body>
            if ( isset( $xml->token ) ) {
                $token = (string) $xml->token;
                $this->log( 'Found token directly in body element', 'debug' );
            }
            // Try nested structures for backward compatibility
            elseif ( isset( $xml->Notification_Data->body->token ) ) {
                $token = (string) $xml->Notification_Data->body->token;
                $this->log( 'Found token in Delcampe_Notification structure', 'debug' );
            }
            // Alternative structures
            elseif ( isset( $xml->body->token ) ) {
                $token = (string) $xml->body->token;
                $this->log( 'Found token in nested body element', 'debug' );
            }
            elseif ( isset( $xml->request ) && isset( $xml->request->authentication_token ) ) {
                $token = (string) $xml->request->authentication_token;
                $this->log( 'Found token in request element', 'debug' );
            }
            
            // Validate extracted token
            if ( empty( $token ) ) {
                $error_msg = 'Token is empty or not found in API response';
                $this->log( $error_msg . ' - Response structure: ' . substr( $response_body, 0, 200 ) . '...', 'error' );
                return new WP_Error( 
                    'empty_token', 
                    __( 'Authentication token not found in API response. The API response format may have changed.', 'wc-delcampe-integration' ) 
                );
            }
            
            // Additional token validation
            $token = trim( $token );
            if ( strlen( $token ) < 10 ) {
                $this->log( 'Token appears to be too short: ' . strlen( $token ) . ' characters', 'warning' );
            }
            
            $this->log( 'Successfully extracted authentication token', 'debug' );
            return $token;

        } catch ( Exception $e ) {
            $error_msg = 'Exception during token parsing: ' . $e->getMessage();
            $this->log( $error_msg, 'error' );
            return new WP_Error( 
                'parse_exception', 
                __( 'An error occurred while processing the authentication response.', 'wc-delcampe-integration' ) 
            );
        } finally {
            // Always restore error handling
            libxml_use_internal_errors( false );
        }
    }

    /**
     * Clear cached authentication token
     * 
     * Removes the cached token from WordPress transients.
     * Useful when token needs to be refreshed manually or
     * when API key settings have changed.
     * 
     * @since  1.0.7.7
     * @version 1.8.0.0
     * @return bool True on success, false on failure
     */
    public function clear_cached_token() {
        $result = delete_transient( $this->token_transient_key );
        
        if ( $result ) {
            $this->log( 'Cached authentication token cleared successfully', 'info' );
        } else {
            $this->log( 'Failed to clear cached authentication token (may not exist)', 'debug' );
        }
        
        return $result;
    }

    /**
     * Validate API key format
     * 
     * Performs validation on the API key format to ensure it meets
     * expected criteria before sending to the API.
     * 
     * @since  1.0.8.0
     * @version 1.8.0.0
     * @param  string $api_key The API key to validate
     * @return bool True if valid, false otherwise
     */
    public function validate_api_key( $api_key ) {
        // SECURITY FIX v1.10.21.0: Enhanced API key validation
        
        // Basic validation - ensure it's not empty
        if ( empty( $api_key ) ) {
            $this->log( 'API key validation failed: empty key', 'warning' );
            return false;
        }
        
        // Type check - must be string
        if ( ! is_string( $api_key ) ) {
            $this->log( 'API key validation failed: not a string', 'warning' );
            return false;
        }
        
        // Check minimum length (Delcampe API keys are typically 32+ characters)
        if ( strlen( $api_key ) < 32 ) {
            $this->log( 'API key validation failed: too short (' . strlen( $api_key ) . ' characters)', 'warning' );
            return false;
        }
        
        // Check maximum length to prevent buffer overflow attacks
        if ( strlen( $api_key ) > 256 ) {
            $this->log( 'API key validation failed: too long (' . strlen( $api_key ) . ' characters)', 'warning' );
            return false;
        }
        
        // Strict character validation - only alphanumeric, hyphens, and underscores
        if ( ! preg_match( '/^[a-zA-Z0-9\-_]+$/', $api_key ) ) {
            $this->log( 'API key validation failed: contains invalid characters', 'warning' );
            return false;
        }
        
        // Check for suspicious patterns
        $suspicious_patterns = array(
            '/^(test|demo|example|sample|dummy)/i',
            '/^[0]+$/',  // All zeros
            '/^[1]+$/',  // All ones
            '/^[a]+$/i', // All same letter
            '/^1234/',   // Sequential start
            '/^abcd/i'   // Sequential letters
        );
        
        foreach ( $suspicious_patterns as $pattern ) {
            if ( preg_match( $pattern, $api_key ) ) {
                $this->log( 'API key validation failed: suspicious pattern detected', 'warning' );
                return false;
            }
        }
        
        // Ensure reasonable entropy (not all same character repeated)
        $unique_chars = count( array_unique( str_split( $api_key ) ) );
        if ( $unique_chars < 10 ) {
            $this->log( 'API key validation failed: insufficient entropy', 'warning' );
            return false;
        }
        
        $this->log( 'API key validation passed', 'debug' );
        return true;
    }

    /**
     * Test authentication with current settings
     * 
     * Attempts to authenticate with the current API key settings.
     * Useful for verifying configuration is correct and API is accessible.
     * 
     * @since  1.0.8.0
     * @version 1.8.0.0
     * @return array Test result with 'success' boolean and 'message' string
     */
    public function test_authentication() {
        $this->log( 'Testing authentication with current settings', 'info' );
        
        // Clear any cached token to force fresh authentication
        $this->clear_cached_token();
        
        // Attempt to get a new token
        $token = $this->get_auth_token();
        
        if ( is_wp_error( $token ) ) {
            return array(
                'success' => false,
                'message' => $token->get_error_message(),
                'code'    => $token->get_error_code()
            );
        }
        
        return array(
            'success' => true,
            'message' => __( 'Authentication successful! API connection is working.', 'wc-delcampe-integration' ),
            'token'   => substr( $token, 0, 10 ) . '...' // Show partial token for security
        );
    }

    /**
     * Get authentication status information
     * 
     * Returns detailed information about the current authentication status
     * including token validity and expiration details.
     * 
     * @since  1.0.9.0
     * @version 1.8.0.0
     * @return array Status information array
     */
    public function get_auth_status() {
        $cached_token = get_transient( $this->token_transient_key );
        $token_expiry = get_option( '_transient_timeout_' . $this->token_transient_key );
        
        // No token cached
        if ( $cached_token === false ) {
            return array(
                'authenticated' => false,
                'has_token'     => false,
                'token_valid'   => false,
                'expires_at'    => null,
                'expires_in'    => null,
                'status_text'   => __( 'Not authenticated', 'wc-delcampe-integration' )
            );
        }
        
        // Calculate expiration details
        $current_time = time();
        $expires_in = $token_expiry - $current_time;
        $is_valid = $expires_in > 0;
        
        // Format status text
        if ( $is_valid ) {
            $minutes_left = round( $expires_in / 60 );
            $status_text = sprintf( 
                __( 'Authenticated (expires in %d minutes)', 'wc-delcampe-integration' ), 
                $minutes_left 
            );
        } else {
            $status_text = __( 'Token expired', 'wc-delcampe-integration' );
        }
        
        return array(
            'authenticated' => true,
            'has_token'     => true,
            'token_valid'   => $is_valid,
            'expires_at'    => $token_expiry,
            'expires_in'    => max( 0, $expires_in ),
            'status_text'   => $status_text
        );
    }

    /**
     * Log messages for debugging
     * 
     * Logs messages to WordPress debug log if logging is enabled
     * in plugin settings. Error level messages are always logged.
     * 
     * @since  1.0.8.0
     * @version 1.8.0.0
     * @param  string $message The message to log
     * @param  string $level Log level (debug, info, warning, error)
     * @return void
     */
    private function log( $message, $level = 'info' ) {
        // Check if logging is enabled (errors are always logged)
        $logging_enabled = get_option( DELCAMPE_LOGGING_OPTION, false );
        
        if ( ! $logging_enabled && $level !== 'error' ) {
            return;
        }

        // Format the log message with timestamp and context
        $formatted_message = sprintf(
            '[%s] [Delcampe Auth v%s] [%s] %s',
            current_time( 'Y-m-d H:i:s' ),
            DELCAMPE_PLUGIN_VERSION,
            strtoupper( $level ),
            $message
        );

        // Only log to custom log file if enabled, not to WordPress debug log
        if ( defined( 'DELCAMPE_LOG_FILE' ) ) {
            $log_dir = dirname( DELCAMPE_LOG_FILE );
            if ( ! file_exists( $log_dir ) ) {
                wp_mkdir_p( $log_dir );
            }
            @file_put_contents( 
                DELCAMPE_LOG_FILE, 
                $formatted_message . PHP_EOL, 
                FILE_APPEND | LOCK_EX 
            );
        }
    }
}
