<?php
/**
 * Delcampe Batch Queue Manager
 * 
 * Handles batch processing with proper queue management, position tracking,
 * retry logic, and recovery mechanisms.
 * 
 * @package WC_Delcampe_Integration
 * @since 1.10.18.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Batch_Queue {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Database table name
     */
    private $table_name;
    
    /**
     * Maximum retry attempts
     */
    const MAX_RETRIES = 5;
    
    /**
     * Processing timeout in seconds
     */
    const PROCESSING_TIMEOUT = 600; // 10 minutes
    
    /**
     * Batch size limits
     */
    const MIN_BATCH_SIZE = 1;
    const MAX_BATCH_SIZE = 100;
    const DEFAULT_BATCH_SIZE = 30;  // Increased from 10 for faster processing
    
    /**
     * Queue status constants
     */
    const STATUS_PENDING = 'pending';
    const STATUS_QUEUED = 'queued';
    const STATUS_PROCESSING = 'processing';
    const STATUS_COMPLETED = 'completed';
    const STATUS_FAILED = 'failed';
    const STATUS_RETRY = 'retry';
    const STATUS_CANCELLED = 'cancelled';
    
    /**
     * Constructor
     */
    private function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'delcampe_batch_queue';
        
        // Hook for creating/updating table structure
        add_action('init', array($this, 'maybe_create_table'));
        
        // Schedule background processing
        add_action('delcampe_process_batch_queue', array($this, 'process_queue'));
        add_action('delcampe_check_stuck_batches', array($this, 'recover_stuck_batches'));
        
        // Ensure cron events are scheduled AFTER init when custom schedules are registered
        // This prevents "could_not_set" errors due to missing schedule intervals
        add_action('init', array($this, 'schedule_cron_events'), 20);
    }
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Create database table
     */
    public function maybe_create_table() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS {$this->table_name} (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            batch_id varchar(64) NOT NULL,
            product_ids text NOT NULL,
            profile_id bigint(20) UNSIGNED NOT NULL,
            action varchar(20) NOT NULL DEFAULT 'create',
            status varchar(20) NOT NULL DEFAULT 'pending',
            queue_position int(11) DEFAULT NULL,
            batch_size int(11) NOT NULL DEFAULT 10,
            processed_count int(11) NOT NULL DEFAULT 0,
            total_count int(11) NOT NULL DEFAULT 0,
            retry_count int(11) NOT NULL DEFAULT 0,
            max_retries int(11) NOT NULL DEFAULT 5,
            last_error text DEFAULT NULL,
            error_details longtext DEFAULT NULL,
            processing_started datetime DEFAULT NULL,
            processing_completed datetime DEFAULT NULL,
            processing_pid int(11) DEFAULT NULL,
            next_retry_at datetime DEFAULT NULL,
            created_by bigint(20) UNSIGNED DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            metadata longtext DEFAULT NULL,
            PRIMARY KEY (id),
            UNIQUE KEY batch_id (batch_id),
            KEY status (status),
            KEY queue_position (queue_position),
            KEY next_retry_at (next_retry_at),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // Add version option for future migrations
        add_option('delcampe_batch_queue_db_version', '1.0');
    }
    
    /**
     * Schedule cron events for queue processing
     */
    public function schedule_cron_events() {
        // Get available schedules to verify our custom intervals exist
        $schedules = wp_get_schedules();
        
        // v1.10.35.17: Disabled automatic batch processing - should be manual or triggered by user action
        // Process queue every minute - DISABLED
        /*
        if (!wp_next_scheduled('delcampe_process_batch_queue')) {
            // Only schedule if the interval exists
            if (isset($schedules['delcampe_batch_interval'])) {
                wp_schedule_event(time(), 'delcampe_batch_interval', 'delcampe_process_batch_queue');
            } else {
                // Fallback to hourly if our custom schedule doesn't exist
                wp_schedule_event(time(), 'hourly', 'delcampe_process_batch_queue');
                error_log('[Delcampe Batch Queue] Warning: delcampe_batch_interval not found, using hourly');
            }
        }
        */
        
        // Check for stuck batches every 5 minutes - DISABLED (not needed with manual processing)
        /*
        if (!wp_next_scheduled('delcampe_check_stuck_batches')) {
            // Only schedule if the interval exists
            if (isset($schedules['delcampe_stuck_check_interval'])) {
                wp_schedule_event(time(), 'delcampe_stuck_check_interval', 'delcampe_check_stuck_batches');
            } else {
                // Fallback to twicedaily if our custom schedule doesn't exist
                wp_schedule_event(time(), 'twicedaily', 'delcampe_check_stuck_batches');
                error_log('[Delcampe Batch Queue] Warning: delcampe_stuck_check_interval not found, using twicedaily');
            }
        }
        */
    }
    
    /**
     * Create a new batch
     */
    public function create_batch($product_ids, $profile_id, $action = 'create', $batch_size = null) {
        global $wpdb;
        
        if (empty($product_ids) || !is_array($product_ids)) {
            return new WP_Error('invalid_products', 'Invalid product IDs');
        }
        
        $batch_size = $this->validate_batch_size($batch_size);
        $batch_id = $this->generate_batch_id();
        
        $data = array(
            'batch_id' => $batch_id,
            'product_ids' => json_encode($product_ids),
            'profile_id' => $profile_id,
            'action' => $action,
            'status' => self::STATUS_PENDING,
            'batch_size' => $batch_size,
            'total_count' => count($product_ids),
            'created_by' => get_current_user_id()
        );
        
        $result = $wpdb->insert($this->table_name, $data);
        
        if ($result === false) {
            return new WP_Error('db_error', 'Failed to create batch');
        }
        
        // Check if we should start processing immediately
        $this->maybe_start_processing($batch_id);
        
        return array(
            'batch_id' => $batch_id,
            'queue_position' => $this->get_queue_position($batch_id),
            'status' => $data['status']
        );
    }
    
    /**
     * Add batch to queue with position tracking
     */
    public function enqueue_batch($batch_id) {
        global $wpdb;
        
        // Get the next queue position
        $max_position = $wpdb->get_var("
            SELECT MAX(queue_position) 
            FROM {$this->table_name} 
            WHERE status IN ('queued', 'processing')
        ");
        
        $next_position = ($max_position === null) ? 1 : $max_position + 1;
        
        $result = $wpdb->update(
            $this->table_name,
            array(
                'status' => self::STATUS_QUEUED,
                'queue_position' => $next_position
            ),
            array('batch_id' => $batch_id)
        );
        
        if ($result === false) {
            return new WP_Error('db_error', 'Failed to enqueue batch');
        }
        
        $this->log_batch_event($batch_id, 'Batch added to queue at position ' . $next_position);
        
        return $next_position;
    }
    
    /**
     * Get next batch to process with improved atomic locking
     * 
     * v1.10.23.0 - Enhanced atomic locking to prevent duplicate batch claiming
     */
    public function get_next_batch() {
        global $wpdb;
        
        // Use transaction with proper isolation level for atomic operation
        $wpdb->query('SET TRANSACTION ISOLATION LEVEL READ COMMITTED');
        $wpdb->query('START TRANSACTION');
        
        try {
            // First, check if any batch is currently processing (with row lock)
            $processing = $wpdb->get_var("
                SELECT COUNT(*) 
                FROM {$this->table_name} 
                WHERE status = 'processing' 
                AND processing_started > DATE_SUB(NOW(), INTERVAL " . self::PROCESSING_TIMEOUT . " SECOND)
                FOR UPDATE
            ");
            
            if ($processing > 0) {
                $wpdb->query('COMMIT');
                return null; // Another batch is processing
            }
            
            // Get and immediately lock the next queued batch or retry batch
            // Prefer FOR UPDATE SKIP LOCKED; gracefully fall back if unsupported
            $sql = "
                SELECT * FROM {$this->table_name}
                WHERE (
                    status = 'queued'
                    OR (status = 'retry' AND next_retry_at <= NOW())
                )
                ORDER BY 
                    CASE WHEN status = 'retry' THEN 0 ELSE 1 END,
                    queue_position ASC,
                    created_at ASC
                LIMIT 1
                FOR UPDATE SKIP LOCKED
            ";
            $batch = $wpdb->get_row($sql);
            if (!$batch && !empty($wpdb->last_error)) {
                // Retry without SKIP LOCKED for older MySQL/MariaDB
                $fallback_sql = str_replace(' FOR UPDATE SKIP LOCKED', ' FOR UPDATE', $sql);
                $batch = $wpdb->get_row($fallback_sql);
            }
            
            if (!$batch) {
                $wpdb->query('COMMIT');
                return null;
            }
            
            // Immediately update status to processing within the same transaction
            // This ensures atomicity - no other process can claim this batch
            $update_result = $wpdb->update(
                $this->table_name,
                array(
                    'status' => self::STATUS_PROCESSING,
                    'processing_started' => current_time('mysql'),
                    'queue_position' => null
                ),
                array('id' => $batch->id),
                array('%s', '%s', '%d'),
                array('%d')
            );
            
            if ($update_result === false) {
                // If update fails, rollback and return null
                $wpdb->query('ROLLBACK');
                $this->log_batch_event($batch->batch_id, 'Failed to claim batch - rolling back');
                return null;
            }
            
            // Commit the transaction to release locks
            $wpdb->query('COMMIT');
            
            $this->log_batch_event($batch->batch_id, sprintf(
                'Batch processing started by PID %d',
                getmypid()
            ));
            
            return $batch;
            
        } catch (Exception $e) {
            // On any error, rollback the transaction
            $wpdb->query('ROLLBACK');
            $this->log_batch_event('system', 'Batch claiming error: ' . $e->getMessage());
            return null;
        }
    }
    
    /**
     * Process a batch
     */
    public function process_batch($batch) {
        $product_ids = json_decode($batch->product_ids, true);
        $processed = $batch->processed_count;
        $batch_size = $batch->batch_size;
        $errors = array();
        
        // Get sync handler instance
        $sync_handler = Delcampe_Sync_Handler::get_instance();
        
        // Process items in chunks
        $chunk = array_slice($product_ids, $processed, $batch_size);
        
        foreach ($chunk as $product_id) {
            // Add to sync queue
            $result = $sync_handler->add_to_queue($product_id, $batch->profile_id, $batch->action);

            if (is_wp_error($result)) {
                $errors[] = sprintf('Product %d: %s', $product_id, $result->get_error_message());
            } elseif ($result === false) {
                // Treat false (e.g., invalid product status) as an error so the batch surfaces issues properly
                $errors[] = sprintf('Product %d: not queued (unpublished or invalid)', $product_id);
            }
            
            $processed++;
            
            // Update progress
            $this->update_batch_progress($batch->batch_id, $processed);
            
            // Check memory usage
            if ($this->is_memory_critical()) {
                $this->log_batch_event($batch->batch_id, 'Pausing due to memory constraints');
                break;
            }
        }
        
        // Trigger sync queue processing
        do_action('delcampe_process_sync_queue');
        
        // Check if batch is complete
        if ($processed >= $batch->total_count) {
            $this->complete_batch($batch->batch_id, empty($errors), $errors);
        } else {
            // More items to process - requeue
            $this->requeue_batch_for_continuation($batch->batch_id, $processed);
        }
        
        return array(
            'processed' => $processed,
            'total' => $batch->total_count,
            'errors' => $errors
        );
    }
    
    /**
     * Complete batch processing
     */
    public function complete_batch($batch_id, $success = true, $errors = array()) {
        global $wpdb;
        
        $status = $success ? self::STATUS_COMPLETED : self::STATUS_FAILED;
        $error_message = !empty($errors) ? implode('; ', array_slice($errors, 0, 5)) : null;
        $error_details = !empty($errors) ? json_encode($errors) : null;
        
        $wpdb->update(
            $this->table_name,
            array(
                'status' => $status,
                'processing_completed' => current_time('mysql'),
                'last_error' => $error_message,
                'error_details' => $error_details,
                'queue_position' => null
            ),
            array('batch_id' => $batch_id)
        );
        
        $this->log_batch_event($batch_id, 'Batch ' . $status);
        
        // If batch completed successfully, update listing status from verified to published
        if ($status === self::STATUS_COMPLETED) {
            $batch = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$this->table_name} WHERE batch_id = %s",
                $batch_id
            ));
            
            // FIX: Check product_ids (plural) not product_id (singular)
            if ($batch && $batch->product_ids) {
                $product_ids = json_decode($batch->product_ids, true);
                
                if (is_array($product_ids)) {
                    $listings_table = $wpdb->prefix . 'delcampe_listings';
                    
                    // Process ALL products in the batch
                    foreach ($product_ids as $product_id) {
                        // Update listing status to published
                        $listing = $wpdb->get_row($wpdb->prepare(
                            "SELECT * FROM {$listings_table} WHERE product_id = %d AND status = 'verified'",
                            $product_id
                        ));
                        
                        if ($listing) {
                            // Check if we have a Delcampe ID from the personal reference
                            $delcampe_id = null;
                            if (!empty($listing->personal_reference)) {
                                // Try to get from product meta first
                                $delcampe_id = get_post_meta($product_id, '_delcampe_listing_id', true);
                            }
                            
                            $wpdb->update(
                                $listings_table,
                                array(
                                    'status' => 'published',
                                    'delcampe_id' => $delcampe_id ?: $listing->delcampe_id,
                                    'date_published' => current_time('mysql'),
                                    'date_updated' => current_time('mysql')
                                ),
                                array('id' => $listing->id),
                                array('%s', '%s', '%s', '%s'),
                                array('%d')
                            );
                            
                            // Update product meta
                            update_post_meta($product_id, '_delcampe_sync_status', 'active');
                            
                            delcampe_log('[Batch Queue] Updated listing ' . $listing->id . ' for product ' . $product_id . ' from verified to published after batch completion');
                        }
                    }
                }
            }
        }
        
        // Update queue positions
        $this->reindex_queue_positions();
        
        // Trigger next batch processing
        $this->maybe_process_next_batch();
    }
    
    /**
     * Schedule batch for retry with exponential backoff
     */
    public function schedule_retry($batch_id, $error_message = null) {
        global $wpdb;
        
        $batch = $wpdb->get_row($wpdb->prepare(
            "SELECT retry_count, max_retries FROM {$this->table_name} WHERE batch_id = %s",
            $batch_id
        ));
        
        if (!$batch) {
            return false;
        }
        
        $retry_count = $batch->retry_count + 1;
        
        // Check if max retries exceeded
        if ($retry_count > $batch->max_retries) {
            $this->complete_batch($batch_id, false, array('Max retries exceeded'));
            return false;
        }
        
        // Calculate exponential backoff delay (in seconds)
        $delays = array(60, 300, 900, 3600, 7200); // 1min, 5min, 15min, 1hr, 2hrs
        $delay_index = min($retry_count - 1, count($delays) - 1);
        $delay = $delays[$delay_index];
        
        $next_retry = date('Y-m-d H:i:s', time() + $delay);
        
        $wpdb->update(
            $this->table_name,
            array(
                'status' => self::STATUS_RETRY,
                'retry_count' => $retry_count,
                'next_retry_at' => $next_retry,
                'last_error' => $error_message,
                'queue_position' => null
            ),
            array('batch_id' => $batch_id)
        );
        
        $this->log_batch_event($batch_id, sprintf(
            'Scheduled retry #%d at %s',
            $retry_count,
            $next_retry
        ));
        
        return true;
    }
    
    /**
     * Recover stuck batches
     */
    public function recover_stuck_batches() {
        global $wpdb;
        
        $timeout = self::PROCESSING_TIMEOUT;
        
        // Find stuck batches
        $stuck_batches = $wpdb->get_results("
            SELECT batch_id 
            FROM {$this->table_name}
            WHERE status = 'processing'
            AND processing_started < DATE_SUB(NOW(), INTERVAL {$timeout} SECOND)
        ");
        
        foreach ($stuck_batches as $batch) {
            $this->log_batch_event($batch->batch_id, 'Recovering stuck batch');
            
            // Schedule for retry
            $this->schedule_retry($batch->batch_id, 'Batch processing timeout');
        }
        
        return count($stuck_batches);
    }
    
    /**
     * Process queue (background job)
     */
    public function process_queue() {
        // v1.10.35.6: Optimized lock timing to reduce overhead
        global $wpdb;
        
        // Check if processing is enabled
        if (get_transient('delcampe_batch_processing_paused')) {
            return;
        }
        
        // First do a quick check if lock exists to avoid database queries
        $process_lock_key = 'delcampe_batch_processing_lock';
        $existing_lock = get_transient( $process_lock_key );
        
        if ( $existing_lock && isset( $existing_lock['time'] ) ) {
            $lock_age = time() - $existing_lock['time'];
            if ( $lock_age < 300 ) { // Lock valid for 5 minutes
                // Lock is still valid, skip logging to reduce noise
                return;
            }
        }
        
        // Acquire global batch processing lock to prevent multiple workers
        $process_lock_timeout = 300; // 5 minutes
        $process_id = wp_generate_uuid4();
        
        // Try to acquire process lock atomically
        $option_name = '_transient_' . $process_lock_key;
        $result = $wpdb->query( $wpdb->prepare(
            "INSERT IGNORE INTO {$wpdb->options} (option_name, option_value, autoload) 
             VALUES (%s, %s, 'no')",
            $option_name,
            serialize( array( 'process_id' => $process_id, 'time' => time() ) )
        ) );
        
        if ( $result !== 1 ) {
            // Lock exists, check if it's stale
            $existing_lock = get_transient( $process_lock_key );
            if ( $existing_lock && isset( $existing_lock['time'] ) ) {
                $lock_age = time() - $existing_lock['time'];
                if ( $lock_age < $process_lock_timeout ) {
                    // Lock is still valid, another process is working
                    // v1.10.35.6: Only log if significant time has passed to reduce log noise
                    if ( $lock_age > 30 ) {
                        delcampe_log('[Batch Queue] Another batch processor is already running (lock age: ' . $lock_age . 's)');
                    }
                    return;
                }
                // Lock is stale, clear it
                delete_transient( $process_lock_key );
                delcampe_log('[Batch Queue] Cleared stale batch processing lock (age: ' . $lock_age . 's)');
            }
            return;
        }
        
        // Set lock expiration
        set_transient( $process_lock_key, array( 'process_id' => $process_id, 'time' => time() ), $process_lock_timeout );
        delcampe_log('[Batch Queue] Acquired batch processing lock (process: ' . $process_id . ')');
        
        // Check for stuck master sync lock (v1.10.26.0)
        $master_sync_lock = get_transient('delcampe_master_sync_lock');
        if ($master_sync_lock) {
            // Check if lock is older than 10 minutes (600 seconds)
            $lock_age = time() - $master_sync_lock;
            if ($lock_age > 600) {
                // Clear stuck lock
                delete_transient('delcampe_master_sync_lock');
                delcampe_log('[Batch Queue] Cleared stuck master sync lock (age: ' . $lock_age . ' seconds)');
            } else {
                // Lock is still valid, wait
                delcampe_log('[Batch Queue] Master sync lock active, skipping batch processing');
                delete_transient( $process_lock_key ); // Release our lock
                return;
            }
        }
        
        $batch = $this->get_next_batch();
        
        if (!$batch) {
            return; // No batches to process
        }
        
        try {
            $result = $this->process_batch($batch);

            // Only schedule a retry if there are errors AND the batch still has remaining work.
            // If the batch reached total_count, process_batch() already marked it completed/failed.
            if (!empty($result['errors'])) {
                $updated = $this->get_batch_status($batch->batch_id);
                if ($updated && ($updated->processed_count < $updated->total_count) && $updated->status !== self::STATUS_COMPLETED) {
                    $this->schedule_retry($batch->batch_id, implode('; ', $result['errors']));
                }
            }
        } catch (Exception $e) {
            $this->log_batch_event($batch->batch_id, 'Processing error: ' . $e->getMessage());
            $this->schedule_retry($batch->batch_id, $e->getMessage());
        } finally {
            // v1.10.35.3: Always release the processing lock when done
            delete_transient( $process_lock_key );
            delcampe_log('[Batch Queue] Released batch processing lock (process: ' . $process_id . ')');
        }
    }
    
    /**
     * Get batch status and queue information
     */
    public function get_batch_status($batch_id) {
        global $wpdb;
        
        $batch = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->table_name} WHERE batch_id = %s",
            $batch_id
        ));
        
        if (!$batch) {
            return null;
        }
        
        // Add queue position info if queued
        if ($batch->status === self::STATUS_QUEUED) {
            $batch->queue_info = $this->get_queue_info();
        }
        
        return $batch;
    }
    
    /**
     * Get queue statistics
     */
    public function get_queue_stats() {
        global $wpdb;
        
        $stats = $wpdb->get_results("
            SELECT status, COUNT(*) as count
            FROM {$this->table_name}
            WHERE created_at > DATE_SUB(NOW(), INTERVAL 24 HOUR)
            GROUP BY status
        ");
        
        $result = array();
        foreach ($stats as $stat) {
            $result[$stat->status] = $stat->count;
        }
        
        // Add processing metrics
        $result['avg_processing_time'] = $wpdb->get_var("
            SELECT AVG(TIMESTAMPDIFF(SECOND, processing_started, processing_completed))
            FROM {$this->table_name}
            WHERE status = 'completed'
            AND processing_completed > DATE_SUB(NOW(), INTERVAL 7 DAY)
        ");
        
        $result['queue_length'] = $wpdb->get_var("
            SELECT COUNT(*) FROM {$this->table_name}
            WHERE status IN ('queued', 'retry')
        ");
        
        return $result;
    }
    
    /**
     * Cancel a batch
     */
    public function cancel_batch($batch_id) {
        global $wpdb;
        
        $result = $wpdb->update(
            $this->table_name,
            array(
                'status' => self::STATUS_CANCELLED,
                'queue_position' => null
            ),
            array('batch_id' => $batch_id)
        );
        
        if ($result) {
            $this->log_batch_event($batch_id, 'Batch cancelled by user');
            $this->reindex_queue_positions();
        }
        
        return $result !== false;
    }
    
    /**
     * Helper: Generate unique batch ID
     */
    private function generate_batch_id() {
        return 'batch_' . time() . '_' . wp_generate_password(8, false);
    }
    
    /**
     * Helper: Validate and sanitize batch size
     */
    private function validate_batch_size($batch_size) {
        if ($batch_size === null) {
            $batch_size = get_option('delcampe_batch_size', self::DEFAULT_BATCH_SIZE);
        }
        
        return max(self::MIN_BATCH_SIZE, min(self::MAX_BATCH_SIZE, intval($batch_size)));
    }
    
    /**
     * Helper: Check if memory usage is critical
     */
    private function is_memory_critical() {
        $memory_limit = $this->get_memory_limit();
        $memory_usage = memory_get_usage(true);
        
        return ($memory_usage / $memory_limit) > 0.8;
    }
    
    /**
     * Helper: Get PHP memory limit in bytes
     */
    private function get_memory_limit() {
        $memory_limit = ini_get('memory_limit');
        
        if (preg_match('/^(\d+)(.)$/', $memory_limit, $matches)) {
            if ($matches[2] == 'M') {
                return $matches[1] * 1024 * 1024;
            } else if ($matches[2] == 'K') {
                return $matches[1] * 1024;
            }
        }
        
        return 128 * 1024 * 1024; // Default 128MB
    }
    
    /**
     * Helper: Log batch event
     */
    private function log_batch_event($batch_id, $message) {
        if (function_exists('delcampe_log')) {
            delcampe_log('[Batch Queue] ' . $batch_id . ': ' . $message);
        }
    }
    
    /**
     * Helper: Update batch progress
     */
    private function update_batch_progress($batch_id, $processed_count) {
        global $wpdb;
        
        $wpdb->update(
            $this->table_name,
            array('processed_count' => $processed_count),
            array('batch_id' => $batch_id)
        );
    }
    
    /**
     * Helper: Requeue batch for continuation
     */
    private function requeue_batch_for_continuation($batch_id, $processed_count) {
        global $wpdb;
        
        $wpdb->update(
            $this->table_name,
            array(
                'status' => self::STATUS_QUEUED,
                'processed_count' => $processed_count,
                'queue_position' => 1 // Priority for continuation
            ),
            array('batch_id' => $batch_id)
        );
    }
    
    /**
     * Helper: Check and start processing if queue is idle
     */
    private function maybe_start_processing($batch_id) {
        global $wpdb;
        
        // First enqueue the batch
        $queue_position = $this->enqueue_batch($batch_id);
        
        // Check if anything is currently processing
        $processing_count = $wpdb->get_var("
            SELECT COUNT(*) FROM {$this->table_name}
            WHERE status = 'processing'
            AND processing_started > DATE_SUB(NOW(), INTERVAL " . self::PROCESSING_TIMEOUT . " SECOND)
        ");
        
        if ($processing_count == 0) {
            // No batches processing - trigger processing immediately
            delcampe_log('[Batch Queue] No active processing, triggering immediate batch processing');
            
            // Schedule immediate processing via cron (only once)
            if (!wp_next_scheduled('delcampe_process_batch_queue')) {
                wp_schedule_single_event(time(), 'delcampe_process_batch_queue');
            }
            
            // Also try direct processing if not in AJAX context
            if (!defined('DOING_AJAX') || !DOING_AJAX) {
                // Process immediately
                $this->process_queue();
            }
        } else {
            delcampe_log('[Batch Queue] Batch queued at position ' . $queue_position . ', waiting for current processing to complete');
        }
    }
    
    /**
     * Helper: Trigger processing of next batch in queue
     */
    private function maybe_process_next_batch() {
        // Schedule immediate processing check
        wp_schedule_single_event(time() + 5, 'delcampe_process_batch_queue');
    }
    
    /**
     * Helper: Reindex queue positions after changes
     */
    private function reindex_queue_positions() {
        global $wpdb;
        
        $queued_batches = $wpdb->get_results("
            SELECT id FROM {$this->table_name}
            WHERE status = 'queued'
            ORDER BY queue_position ASC, created_at ASC
        ");
        
        $position = 1;
        foreach ($queued_batches as $batch) {
            $wpdb->update(
                $this->table_name,
                array('queue_position' => $position++),
                array('id' => $batch->id)
            );
        }
    }
    
    /**
     * Helper: Get queue position for a batch
     */
    private function get_queue_position($batch_id) {
        global $wpdb;
        
        return $wpdb->get_var($wpdb->prepare(
            "SELECT queue_position FROM {$this->table_name} WHERE batch_id = %s",
            $batch_id
        ));
    }
    
    /**
     * Helper: Get general queue information
     */
    private function get_queue_info() {
        global $wpdb;
        
        return array(
            'total_queued' => $wpdb->get_var("
                SELECT COUNT(*) FROM {$this->table_name}
                WHERE status IN ('queued', 'retry')
            "),
            'currently_processing' => $wpdb->get_var("
                SELECT COUNT(*) FROM {$this->table_name}
                WHERE status = 'processing'
            "),
            'estimated_wait_time' => $this->estimate_wait_time()
        );
    }
    
    /**
     * Helper: Estimate wait time based on average processing time
     */
    private function estimate_wait_time() {
        global $wpdb;
        
        $avg_time = $wpdb->get_var("
            SELECT AVG(TIMESTAMPDIFF(SECOND, processing_started, processing_completed))
            FROM {$this->table_name}
            WHERE status = 'completed'
            AND processing_completed > DATE_SUB(NOW(), INTERVAL 1 DAY)
        ");
        
        $queued_count = $wpdb->get_var("
            SELECT COUNT(*) FROM {$this->table_name}
            WHERE status = 'queued'
        ");
        
        if (!$avg_time || !$queued_count) {
            return 0;
        }
        
        return round($avg_time * $queued_count);
    }
}
