<?php
/**
 * Delcampe Bulk Operations
 * 
 * Enhanced bulk operations for managing Delcampe listings
 * 
 * @package WC_Delcampe_Integration
 * @version 1.10.10.0
 * @since   1.10.10.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Delcampe_Bulk_Operations class
 *
 * @since 1.10.10.0
 */
class Delcampe_Bulk_Operations {
    
    /**
     * Singleton instance
     * @var Delcampe_Bulk_Operations
     */
    private static $instance = null;
    
    /**
     * Auth instance
     * @var Delcampe_Auth
     */
    private $auth;
    
    /**
     * Logger instance
     * @var Delcampe_Sync_Logger
     */
    private $logger;
    
    /**
     * Listings model
     * @var Delcampe_Listings_Model
     */
    private $listings_model;
    
    /**
     * Get singleton instance
     *
     * @return Delcampe_Bulk_Operations
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->init();
    }
    
    /**
     * Initialize bulk operations
     */
    private function init() {
        // Initialize dependencies
        $this->auth = Delcampe_Auth::get_instance();
        $this->logger = Delcampe_Sync_Logger::get_instance();
        $this->listings_model = new Delcampe_Listings_Model();
        
        // Add AJAX handlers
        add_action('wp_ajax_delcampe_bulk_update', array($this, 'ajax_bulk_update'));
        add_action('wp_ajax_delcampe_bulk_close', array($this, 'ajax_bulk_close'));
        add_action('wp_ajax_delcampe_bulk_delete', array($this, 'ajax_bulk_delete'));
        add_action('wp_ajax_delcampe_bulk_relist', array($this, 'ajax_bulk_relist'));
        add_action('wp_ajax_delcampe_bulk_sync_inventory', array($this, 'ajax_bulk_sync_inventory'));
        add_action('wp_ajax_delcampe_bulk_sync_prices', array($this, 'ajax_bulk_sync_prices'));
        add_action('wp_ajax_delcampe_bulk_operation_status', array($this, 'ajax_get_operation_status'));
        
        // Add bulk actions to products list
        add_filter('bulk_actions-edit-product', array($this, 'add_product_bulk_actions'));
        add_filter('handle_bulk_actions-edit-product', array($this, 'handle_product_bulk_actions'), 10, 3);
        
        // Add bulk actions to listings page
        add_filter('delcampe_listings_bulk_actions', array($this, 'add_listings_bulk_actions'));
        
        // Add admin notices
        add_action('admin_notices', array($this, 'show_bulk_action_notices'));
    }
    
    /**
     * Add bulk actions to product list
     *
     * @param array $actions Current actions
     * @return array Modified actions
     */
    public function add_product_bulk_actions($actions) {
        $actions['delcampe_publish'] = __('Publish to Delcampe', 'wc-delcampe-integration');
        $actions['delcampe_update'] = __('Update on Delcampe', 'wc-delcampe-integration');
        $actions['delcampe_close'] = __('Close on Delcampe', 'wc-delcampe-integration');
        $actions['delcampe_sync_inventory'] = __('Sync Inventory with Delcampe', 'wc-delcampe-integration');
        $actions['delcampe_sync_prices'] = __('Sync Prices with Delcampe', 'wc-delcampe-integration');
        return $actions;
    }
    
    /**
     * Handle product bulk actions
     *
     * @param string $redirect_to Redirect URL
     * @param string $action Action name
     * @param array $post_ids Post IDs
     * @return string Redirect URL
     */
    public function handle_product_bulk_actions($redirect_to, $action, $post_ids) {
        if (strpos($action, 'delcampe_') !== 0) {
            return $redirect_to;
        }
        
        // Create bulk operation
        $operation_id = $this->create_bulk_operation($action, $post_ids);
        
        // Process in background
        $this->schedule_bulk_operation($operation_id);
        
        // Add query arg for notice
        $redirect_to = add_query_arg(array(
            'delcampe_bulk_action' => $action,
            'delcampe_operation_id' => $operation_id,
            'delcampe_count' => count($post_ids)
        ), $redirect_to);
        
        return $redirect_to;
    }
    
    /**
     * Create bulk operation record
     *
     * @param string $action Action name
     * @param array $items Item IDs
     * @return int Operation ID
     */
    private function create_bulk_operation($action, $items) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_bulk_operations';
        
        // Create table if not exists
        $this->create_bulk_operations_table();
        
        // Insert operation
        $wpdb->insert(
            $table_name,
            array(
                'action' => $action,
                'items' => json_encode($items),
                'total_items' => count($items),
                'processed_items' => 0,
                'status' => 'pending',
                'user_id' => get_current_user_id(),
                'created_at' => current_time('mysql')
            ),
            array('%s', '%s', '%d', '%d', '%s', '%d', '%s')
        );
        
        return $wpdb->insert_id;
    }
    
    /**
     * Schedule bulk operation for processing
     *
     * @param int $operation_id Operation ID
     */
    private function schedule_bulk_operation($operation_id) {
        // Schedule immediate processing
        wp_schedule_single_event(time(), 'delcampe_process_bulk_operation', array($operation_id));
    }
    
    /**
     * Process bulk operation
     *
     * @param int $operation_id Operation ID
     */
    public function process_bulk_operation($operation_id) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_bulk_operations';
        
        // Get operation
        $operation = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table_name} WHERE id = %d",
            $operation_id
        ));
        
        if (!$operation) {
            return;
        }
        
        // Update status to processing
        $wpdb->update(
            $table_name,
            array('status' => 'processing', 'started_at' => current_time('mysql')),
            array('id' => $operation_id),
            array('%s', '%s'),
            array('%d')
        );
        
        $items = json_decode($operation->items, true);
        $processed = 0;
        $errors = array();
        
        foreach ($items as $item_id) {
            try {
                switch ($operation->action) {
                    case 'delcampe_publish':
                        $this->publish_single_item($item_id);
                        break;
                        
                    case 'delcampe_update':
                        $this->update_single_item($item_id);
                        break;
                        
                    case 'delcampe_close':
                        $this->close_single_item($item_id);
                        break;
                        
                    case 'delcampe_delete':
                        $this->delete_single_item($item_id);
                        break;
                        
                    case 'delcampe_relist':
                        $this->relist_single_item($item_id);
                        break;
                        
                    case 'delcampe_sync_inventory':
                        $this->sync_inventory_single($item_id);
                        break;
                        
                    case 'delcampe_sync_prices':
                        $this->sync_price_single($item_id);
                        break;
                }
                
                $processed++;
                
            } catch (Exception $e) {
                $errors[$item_id] = $e->getMessage();
            }
            
            // Update progress
            $wpdb->update(
                $table_name,
                array('processed_items' => $processed),
                array('id' => $operation_id),
                array('%d'),
                array('%d')
            );
        }
        
        // Update final status
        $final_status = empty($errors) ? 'completed' : 'completed_with_errors';
        $wpdb->update(
            $table_name,
            array(
                'status' => $final_status,
                'errors' => json_encode($errors),
                'completed_at' => current_time('mysql')
            ),
            array('id' => $operation_id),
            array('%s', '%s', '%s'),
            array('%d')
        );
        
        // Log completion
        $this->logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'BULK_OPERATION_COMPLETED',
            'operation_id' => $operation_id,
            'action' => $operation->action,
            'total' => $operation->total_items,
            'processed' => $processed,
            'errors' => count($errors)
        ));
    }
    
    /**
     * Publish single item to Delcampe
     *
     * @param int $product_id Product ID
     * @throws Exception
     */
    private function publish_single_item($product_id) {
        // Check if already published
        $delcampe_id = get_post_meta($product_id, '_delcampe_listing_id', true);
        if ($delcampe_id) {
            throw new Exception('Already published to Delcampe');
        }
        
        // Get profile
        $profile_id = get_post_meta($product_id, '_delcampe_profile_id', true);
        if (!$profile_id) {
            throw new Exception('No Delcampe profile assigned');
        }
        
        // Queue for publishing
        global $wpdb;
        $sync_table = $wpdb->prefix . 'delcampe_sync_queue';
        
        $wpdb->insert(
            $sync_table,
            array(
                'product_id' => $product_id,
                'action' => 'create',
                'status' => 'pending',
                'created_at' => current_time('mysql')
            ),
            array('%d', '%s', '%s', '%s')
        );
    }
    
    /**
     * Update single item on Delcampe
     *
     * @param int $product_id Product ID
     * @throws Exception
     */
    private function update_single_item($product_id) {
        $delcampe_id = get_post_meta($product_id, '_delcampe_listing_id', true);
        if (!$delcampe_id) {
            throw new Exception('Product not published to Delcampe');
        }
        
        // Queue for update
        global $wpdb;
        $sync_table = $wpdb->prefix . 'delcampe_sync_queue';
        
        $wpdb->insert(
            $sync_table,
            array(
                'product_id' => $product_id,
                'action' => 'update',
                'status' => 'pending',
                'created_at' => current_time('mysql')
            ),
            array('%d', '%s', '%s', '%s')
        );
    }
    
    /**
     * Close single item on Delcampe
     *
     * @param int $product_id Product ID
     * @throws Exception
     */
    private function close_single_item($product_id) {
        $delcampe_id = get_post_meta($product_id, '_delcampe_listing_id', true);
        if (!$delcampe_id) {
            throw new Exception('Product not published to Delcampe');
        }
        
        // Get auth token
        $token = $this->auth->get_auth_token();
        if (!$token) {
            throw new Exception('Failed to get auth token');
        }
        
        // Close listing via API
        $api_url = 'https://rest.delcampe.net/item/' . $delcampe_id . '/close';
        
        $response = wp_remote_post($api_url, array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $token
            ),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            throw new Exception($response->get_error_message());
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200 && $status_code !== 204) {
            throw new Exception('Failed to close listing');
        }
        
        // Update local status
        $this->listings_model->update_listing_status($product_id, 'closed');
        
        // Remove Delcampe ID from product
        delete_post_meta($product_id, '_delcampe_listing_id');
    }
    
    /**
     * Delete single item from Delcampe
     *
     * @param int $product_id Product ID
     * @throws Exception
     */
    private function delete_single_item($product_id) {
        $delcampe_id = get_post_meta($product_id, '_delcampe_listing_id', true);
        if (!$delcampe_id) {
            return; // Already not on Delcampe
        }
        
        // Get auth token
        $token = $this->auth->get_auth_token();
        if (!$token) {
            throw new Exception('Failed to get auth token');
        }
        
        // Delete listing via API
        $api_url = 'https://rest.delcampe.net/item/' . $delcampe_id;
        
        $response = wp_remote_request($api_url, array(
            'method' => 'DELETE',
            'headers' => array(
                'Authorization' => 'Bearer ' . $token
            ),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            throw new Exception($response->get_error_message());
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200 && $status_code !== 204) {
            throw new Exception('Failed to delete listing');
        }
        
        // Remove from local database
        $this->listings_model->delete_listing($product_id);
        
        // Remove Delcampe ID from product
        delete_post_meta($product_id, '_delcampe_listing_id');
    }
    
    /**
     * Relist single item on Delcampe
     *
     * @param int $product_id Product ID
     * @throws Exception
     */
    private function relist_single_item($product_id) {
        $delcampe_id = get_post_meta($product_id, '_delcampe_listing_id', true);
        if (!$delcampe_id) {
            throw new Exception('Product not previously listed on Delcampe');
        }
        
        // Get auth token
        $token = $this->auth->get_auth_token();
        if (!$token) {
            throw new Exception('Failed to get auth token');
        }
        
        // Relist via API
        $api_url = 'https://rest.delcampe.net/item/' . $delcampe_id . '/relist';
        
        $response = wp_remote_post($api_url, array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $token
            ),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            throw new Exception($response->get_error_message());
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200 && $status_code !== 201) {
            throw new Exception('Failed to relist item');
        }
        
        // Update local status
        $this->listings_model->update_listing_status($product_id, 'published');
    }
    
    /**
     * Sync inventory for single item
     *
     * @param int $product_id Product ID
     * @throws Exception
     */
    private function sync_inventory_single($product_id) {
        $delcampe_id = get_post_meta($product_id, '_delcampe_listing_id', true);
        if (!$delcampe_id) {
            throw new Exception('Product not published to Delcampe');
        }
        
        $product = wc_get_product($product_id);
        if (!$product) {
            throw new Exception('Product not found');
        }
        
        $stock_quantity = $product->get_stock_quantity();
        
        // Get auth token
        $token = $this->auth->get_auth_token();
        if (!$token) {
            throw new Exception('Failed to get auth token');
        }
        
        // Update stock via API
        $api_url = 'https://rest.delcampe.net/item/' . $delcampe_id;
        
        $response = wp_remote_request($api_url, array(
            'method' => 'PUT',
            'headers' => array(
                'Authorization' => 'Bearer ' . $token,
                'Content-Type' => 'application/json'
            ),
            'body' => json_encode(array(
                'quantity' => $stock_quantity
            )),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            throw new Exception($response->get_error_message());
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200 && $status_code !== 204) {
            throw new Exception('Failed to update inventory');
        }
    }
    
    /**
     * Sync price for single item
     *
     * @param int $product_id Product ID
     * @throws Exception
     */
    private function sync_price_single($product_id) {
        $delcampe_id = get_post_meta($product_id, '_delcampe_listing_id', true);
        if (!$delcampe_id) {
            throw new Exception('Product not published to Delcampe');
        }
        
        $product = wc_get_product($product_id);
        if (!$product) {
            throw new Exception('Product not found');
        }
        
        $price = $product->get_price();
        
        // Get auth token
        $token = $this->auth->get_auth_token();
        if (!$token) {
            throw new Exception('Failed to get auth token');
        }
        
        // Update price via API
        $api_url = 'https://rest.delcampe.net/item/' . $delcampe_id;
        
        $response = wp_remote_request($api_url, array(
            'method' => 'PUT',
            'headers' => array(
                'Authorization' => 'Bearer ' . $token,
                'Content-Type' => 'application/json'
            ),
            'body' => json_encode(array(
                'price' => floatval($price)
            )),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            throw new Exception($response->get_error_message());
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200 && $status_code !== 204) {
            throw new Exception('Failed to update price');
        }
    }
    
    /**
     * AJAX handler for bulk update
     */
    public function ajax_bulk_update() {
        // Check nonce
        if (!check_ajax_referer('delcampe_bulk_operations', 'nonce', false)) {
            wp_send_json_error('Invalid nonce');
        }
        
        // Check permissions
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $product_ids = array_map('intval', $_POST['product_ids'] ?? array());
        
        if (empty($product_ids)) {
            wp_send_json_error('No products selected');
        }
        
        $operation_id = $this->create_bulk_operation('delcampe_update', $product_ids);
        $this->schedule_bulk_operation($operation_id);
        
        wp_send_json_success(array(
            'operation_id' => $operation_id,
            'message' => sprintf(__('Updating %d products on Delcampe', 'wc-delcampe-integration'), count($product_ids))
        ));
    }
    
    /**
     * AJAX handler for bulk close
     */
    public function ajax_bulk_close() {
        // Check nonce
        if (!check_ajax_referer('delcampe_bulk_operations', 'nonce', false)) {
            wp_send_json_error('Invalid nonce');
        }
        
        // Check permissions
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        $product_ids = array_map('intval', $_POST['product_ids'] ?? array());
        
        if (empty($product_ids)) {
            wp_send_json_error('No products selected');
        }
        
        $operation_id = $this->create_bulk_operation('delcampe_close', $product_ids);
        $this->schedule_bulk_operation($operation_id);
        
        wp_send_json_success(array(
            'operation_id' => $operation_id,
            'message' => sprintf(__('Closing %d listings on Delcampe', 'wc-delcampe-integration'), count($product_ids))
        ));
    }
    
    /**
     * AJAX handler to get operation status
     */
    public function ajax_get_operation_status() {
        // Check nonce
        if (!check_ajax_referer('delcampe_bulk_operations', 'nonce', false)) {
            wp_send_json_error('Invalid nonce');
        }
        
        $operation_id = intval($_POST['operation_id'] ?? 0);
        
        if (!$operation_id) {
            wp_send_json_error('Invalid operation ID');
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'delcampe_bulk_operations';
        
        $operation = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table_name} WHERE id = %d",
            $operation_id
        ));
        
        if (!$operation) {
            wp_send_json_error('Operation not found');
        }
        
        $progress = $operation->total_items > 0 
            ? round(($operation->processed_items / $operation->total_items) * 100) 
            : 0;
        
        wp_send_json_success(array(
            'status' => $operation->status,
            'progress' => $progress,
            'processed' => $operation->processed_items,
            'total' => $operation->total_items,
            'errors' => json_decode($operation->errors, true)
        ));
    }
    
    /**
     * Show bulk action notices
     */
    public function show_bulk_action_notices() {
        if (!isset($_GET['delcampe_bulk_action'])) {
            return;
        }
        
        $action = sanitize_text_field($_GET['delcampe_bulk_action']);
        $count = intval($_GET['delcampe_count'] ?? 0);
        $operation_id = intval($_GET['delcampe_operation_id'] ?? 0);
        
        $message = '';
        switch ($action) {
            case 'delcampe_publish':
                $message = sprintf(__('%d products queued for publishing to Delcampe.', 'wc-delcampe-integration'), $count);
                break;
            case 'delcampe_update':
                $message = sprintf(__('%d products queued for update on Delcampe.', 'wc-delcampe-integration'), $count);
                break;
            case 'delcampe_close':
                $message = sprintf(__('%d listings queued for closing on Delcampe.', 'wc-delcampe-integration'), $count);
                break;
            case 'delcampe_sync_inventory':
                $message = sprintf(__('%d products queued for inventory sync with Delcampe.', 'wc-delcampe-integration'), $count);
                break;
            case 'delcampe_sync_prices':
                $message = sprintf(__('%d products queued for price sync with Delcampe.', 'wc-delcampe-integration'), $count);
                break;
        }
        
        if ($message) {
            ?>
            <div class="notice notice-success is-dismissible">
                <p><?php echo esc_html($message); ?></p>
                <?php if ($operation_id): ?>
                <p>
                    <a href="#" class="delcampe-check-operation-status" data-operation-id="<?php echo esc_attr($operation_id); ?>">
                        <?php esc_html_e('Check operation status', 'wc-delcampe-integration'); ?>
                    </a>
                </p>
                <?php endif; ?>
            </div>
            <?php
        }
    }
    
    /**
     * Add bulk actions to listings page
     *
     * @param array $actions Current actions
     * @return array Modified actions
     */
    public function add_listings_bulk_actions($actions) {
        $actions['update'] = __('Update Listings', 'wc-delcampe-integration');
        $actions['close'] = __('Close Listings', 'wc-delcampe-integration');
        $actions['delete'] = __('Delete Listings', 'wc-delcampe-integration');
        $actions['relist'] = __('Relist Items', 'wc-delcampe-integration');
        $actions['sync_inventory'] = __('Sync Inventory', 'wc-delcampe-integration');
        $actions['sync_prices'] = __('Sync Prices', 'wc-delcampe-integration');
        return $actions;
    }
    
    /**
     * Create bulk operations table
     */
    private function create_bulk_operations_table() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'delcampe_bulk_operations';
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS {$table_name} (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            action varchar(50) NOT NULL,
            items longtext NOT NULL,
            total_items int(11) NOT NULL,
            processed_items int(11) DEFAULT 0,
            status varchar(20) DEFAULT 'pending',
            errors longtext,
            user_id bigint(20) NOT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            started_at datetime,
            completed_at datetime,
            PRIMARY KEY (id),
            KEY status (status),
            KEY user_id (user_id)
        ) {$charset_collate};";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
}

// Initialize the class only after all plugins are loaded
add_action('plugins_loaded', function() {
    if (class_exists('Delcampe_Sync_Logger') && class_exists('Delcampe_Auth') && class_exists('Delcampe_Listings_Model')) {
        Delcampe_Bulk_Operations::get_instance();
    }
}, 20);

// Add cron processor after class is initialized
add_action('plugins_loaded', function() {
    if (class_exists('Delcampe_Bulk_Operations')) {
        add_action('delcampe_process_bulk_operation', array(Delcampe_Bulk_Operations::get_instance(), 'process_bulk_operation'));
    }
}, 25);