<?php
/**
 * Bulk Publish to Delcampe
 *
 * Handles bulk publishing of products to Delcampe with various selection methods
 *
 * @package WC_Delcampe_Integration
 * @version 1.9.0.1
 * @since   1.9.0.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Delcampe_Bulk_Publish class
 *
 * @since 1.9.0.0
 */
class Delcampe_Bulk_Publish {
    
    /**
     * Singleton instance
     * @var Delcampe_Bulk_Publish
     */
    private static $instance = null;
    
    /**
     * Get singleton instance
     *
     * @since 1.9.0.0
     * @return Delcampe_Bulk_Publish
     */
    public static function get_instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     *
     * @since 1.9.0.0
     */
    private function __construct() {
        // Add admin menu
        add_action('admin_menu', array($this, 'add_menu_page'), 15);
        
        // Register AJAX handlers
        add_action('wp_ajax_delcampe_bulk_publish_preview', array($this, 'ajax_preview_products'));
        add_action('wp_ajax_delcampe_bulk_publish_execute', array($this, 'ajax_execute_publish'));
        add_action('wp_ajax_delcampe_get_subcategories', array($this, 'ajax_get_subcategories'));
        
        // Enqueue scripts
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
    }
    
    /**
     * Add menu page
     *
     * @since 1.9.0.0
     */
    public function add_menu_page() {
        add_submenu_page(
            'delcampe',
            'Bulk Publish',                        // Page title (hardcoded to avoid early translation)
            'Bulk Publish',                        // Menu title (hardcoded to avoid early translation)
            'manage_woocommerce',
            'delcampe-bulk-publish',
            array($this, 'render_page')
        );
    }
    
    /**
     * Enqueue scripts and styles
     *
     * @since 1.9.0.0
     */
    public function enqueue_scripts($hook) {
        if ($hook !== 'delcampe_page_delcampe-bulk-publish') {
            return;
        }
        
        wp_enqueue_script(
            'delcampe-bulk-publish',
            plugin_dir_url(dirname(__FILE__)) . 'assets/js/bulk-publish.js',
            array('jquery', 'wp-util'),
            defined('DELCAMPE_PLUGIN_VERSION') ? DELCAMPE_PLUGIN_VERSION : time(),
            true
        );
        
        wp_localize_script('delcampe-bulk-publish', 'delcampeBulkPublish', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('delcampe_bulk_publish'),
            'strings' => array(
                'confirmPublish' => __('Are you sure you want to publish %d products to Delcampe?', 'wc-delcampe-integration'),
                'publishing' => __('Publishing...', 'wc-delcampe-integration'),
                'published' => __('Published %d of %d products', 'wc-delcampe-integration'),
                'error' => __('An error occurred. Please try again.', 'wc-delcampe-integration'),
                'noProducts' => __('No products found matching your criteria.', 'wc-delcampe-integration'),
                'selectProfile' => __('Please select a Delcampe profile.', 'wc-delcampe-integration')
            )
        ));
        
        wp_enqueue_style(
            'delcampe-bulk-publish',
            plugin_dir_url(dirname(__FILE__)) . 'assets/css/bulk-publish.css',
            array(),
            defined('DELCAMPE_PLUGIN_VERSION') ? DELCAMPE_PLUGIN_VERSION : time()
        );
    }
    
    /**
     * Render the bulk publish page
     *
     * @since 1.9.0.0
     */
    public function render_page() {
        // Get available profiles
        global $wpdb;
        $profiles_table = $wpdb->prefix . 'delcampe_profiles';
        $profiles = $wpdb->get_results("SELECT profile_id, profile_name FROM {$profiles_table} ORDER BY profile_name");
        
        // Get product categories
        $product_categories = get_terms(array(
            'taxonomy' => 'product_cat',
            'hide_empty' => true,
            'parent' => 0
        ));
        
        // Get product tags
        $product_tags = get_terms(array(
            'taxonomy' => 'product_tag',
            'hide_empty' => true
        ));
        
        ?>
        <div class="wrap delcampe-bulk-publish">
            <h1><?php _e('Bulk Publish to Delcampe', 'wc-delcampe-integration'); ?></h1>
            
            <div class="notice notice-info">
                <p><?php _e('Select products to publish to Delcampe. You can filter by category, tags, or other criteria.', 'wc-delcampe-integration'); ?></p>
            </div>
            
            <?php
            // Display API rate limit status
            require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-rate-limit.php';
            $rate_limit = Delcampe_Rate_Limit::get_instance();
            echo $rate_limit->get_rate_limit_display();
            ?>
            
            <div class="bulk-publish-container">
                <!-- Step 1: Select Profile -->
                <div class="publish-step step-profile">
                    <h2><?php _e('Step 1: Select Delcampe Profile', 'wc-delcampe-integration'); ?></h2>
                    <select id="delcampe-profile" class="regular-text">
                        <option value=""><?php _e('— Select Profile —', 'wc-delcampe-integration'); ?></option>
                        <?php foreach ($profiles as $profile): ?>
                            <option value="<?php echo esc_attr($profile->profile_id); ?>">
                                <?php echo esc_html($profile->profile_name); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <!-- Step 2: Select Products -->
                <div class="publish-step step-products">
                    <h2><?php _e('Step 2: Select Products', 'wc-delcampe-integration'); ?></h2>
                    
                    <div class="selection-methods">
                        <ul class="method-tabs">
                            <li><a href="#method-category" class="active"><?php _e('By Category', 'wc-delcampe-integration'); ?></a></li>
                            <li><a href="#method-tags"><?php _e('By Tags', 'wc-delcampe-integration'); ?></a></li>
                            <li><a href="#method-filter"><?php _e('By Filter', 'wc-delcampe-integration'); ?></a></li>
                            <li><a href="#method-all"><?php _e('All Products', 'wc-delcampe-integration'); ?></a></li>
                        </ul>
                        
                        <!-- By Category -->
                        <div id="method-category" class="method-panel active">
                            <h3><?php _e('Select Categories', 'wc-delcampe-integration'); ?></h3>
                            <div class="category-tree">
                                <?php foreach ($product_categories as $category): ?>
                                    <label class="category-item">
                                        <input type="checkbox" name="categories[]" value="<?php echo esc_attr($category->term_id); ?>">
                                        <span><?php echo esc_html($category->name); ?> (<?php echo $category->count; ?>)</span>
                                        <?php
                                        $subcategories = get_terms(array(
                                            'taxonomy' => 'product_cat',
                                            'hide_empty' => true,
                                            'parent' => $category->term_id
                                        ));
                                        if (!empty($subcategories)): ?>
                                            <div class="subcategories" style="margin-left: 20px;">
                                                <?php foreach ($subcategories as $subcategory): ?>
                                                    <label class="category-item">
                                                        <input type="checkbox" name="categories[]" value="<?php echo esc_attr($subcategory->term_id); ?>">
                                                        <span><?php echo esc_html($subcategory->name); ?> (<?php echo $subcategory->count; ?>)</span>
                                                    </label>
                                                <?php endforeach; ?>
                                            </div>
                                        <?php endif; ?>
                                    </label>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        
                        <!-- By Tags -->
                        <div id="method-tags" class="method-panel">
                            <h3><?php _e('Select Tags', 'wc-delcampe-integration'); ?></h3>
                            <div class="tag-list">
                                <?php foreach ($product_tags as $tag): ?>
                                    <label class="tag-item">
                                        <input type="checkbox" name="tags[]" value="<?php echo esc_attr($tag->term_id); ?>">
                                        <span><?php echo esc_html($tag->name); ?> (<?php echo $tag->count; ?>)</span>
                                    </label>
                                <?php endforeach; ?>
                            </div>
                        </div>
                        
                        <!-- By Filter -->
                        <div id="method-filter" class="method-panel">
                            <h3><?php _e('Filter Products', 'wc-delcampe-integration'); ?></h3>
                            <table class="form-table">
                                <tr>
                                    <th><?php _e('Category', 'wc-delcampe-integration'); ?></th>
                                    <td>
                                        <select name="filter_category" class="regular-text">
                                            <option value=""><?php _e('Any Category', 'wc-delcampe-integration'); ?></option>
                                            <?php
                                            $all_categories = get_terms(array(
                                                'taxonomy' => 'product_cat',
                                                'hide_empty' => true,
                                                'orderby' => 'name',
                                                'order' => 'ASC'
                                            ));
                                            foreach ($all_categories as $cat) {
                                                $indent = str_repeat('&nbsp;&nbsp;', $cat->parent ? 1 : 0);
                                                echo '<option value="' . esc_attr($cat->term_id) . '">' . 
                                                     $indent . esc_html($cat->name) . ' (' . $cat->count . ')' .
                                                     '</option>';
                                            }
                                            ?>
                                        </select>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php _e('Product Status', 'wc-delcampe-integration'); ?></th>
                                    <td>
                                        <select name="product_status">
                                            <option value=""><?php _e('Any Status', 'wc-delcampe-integration'); ?></option>
                                            <option value="publish"><?php _e('Published', 'wc-delcampe-integration'); ?></option>
                                            <option value="draft"><?php _e('Draft', 'wc-delcampe-integration'); ?></option>
                                        </select>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php _e('Stock Status', 'wc-delcampe-integration'); ?></th>
                                    <td>
                                        <select name="stock_status">
                                            <option value=""><?php _e('Any Stock Status', 'wc-delcampe-integration'); ?></option>
                                            <option value="instock"><?php _e('In Stock', 'wc-delcampe-integration'); ?></option>
                                            <option value="outofstock"><?php _e('Out of Stock', 'wc-delcampe-integration'); ?></option>
                                        </select>
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php _e('SKU Contains', 'wc-delcampe-integration'); ?></th>
                                    <td>
                                        <input type="text" name="sku_filter" placeholder="<?php esc_attr_e('e.g. STAMP-', 'wc-delcampe-integration'); ?>">
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php _e('Price Range', 'wc-delcampe-integration'); ?></th>
                                    <td>
                                        <input type="number" name="price_min" placeholder="<?php esc_attr_e('Min', 'wc-delcampe-integration'); ?>" step="0.01" style="width: 100px;">
                                        -
                                        <input type="number" name="price_max" placeholder="<?php esc_attr_e('Max', 'wc-delcampe-integration'); ?>" step="0.01" style="width: 100px;">
                                    </td>
                                </tr>
                                <tr>
                                    <th><?php _e('Exclude Already Listed', 'wc-delcampe-integration'); ?></th>
                                    <td>
                                        <label>
                                            <input type="checkbox" name="exclude_listed" value="1" checked>
                                            <?php _e('Skip products already listed on Delcampe', 'wc-delcampe-integration'); ?>
                                        </label>
                                    </td>
                                </tr>
                            </table>
                        </div>
                        
                        <!-- All Products -->
                        <div id="method-all" class="method-panel">
                            <h3><?php _e('Publish All Products', 'wc-delcampe-integration'); ?></h3>
                            <p><?php _e('This will queue all products (excluding those already listed) for publishing to Delcampe.', 'wc-delcampe-integration'); ?></p>
                            <label>
                                <input type="checkbox" name="confirm_all" value="1">
                                <?php _e('I understand this will publish ALL products', 'wc-delcampe-integration'); ?>
                            </label>
                        </div>
                    </div>
                    
                    <div class="preview-actions">
                        <div style="margin-bottom: 10px;">
                            <label for="max-batch-size">
                                <?php _e('Max items per batch:', 'wc-delcampe-integration'); ?>
                                <input type="number" id="max-batch-size" name="max_batch_size" value="100" min="10" max="500" style="width: 80px;">
                                <span class="description"><?php _e('(10-500 items)', 'wc-delcampe-integration'); ?></span>
                            </label>
                        </div>
                        <button type="button" id="preview-products" class="button button-secondary">
                            <?php _e('Preview Products', 'wc-delcampe-integration'); ?>
                        </button>
                    </div>
                </div>
                
                <!-- Step 3: Review and Publish -->
                <div class="publish-step step-review" style="display: none;">
                    <h2><?php _e('Step 3: Review and Publish', 'wc-delcampe-integration'); ?></h2>
                    
                    <div class="preview-summary">
                        <p class="summary-text"></p>
                    </div>
                    
                    <div class="preview-results">
                        <table class="wp-list-table widefat fixed striped">
                            <thead>
                                <tr>
                                    <th style="width: 30px;" class="check-column">
                                        <input type="checkbox" id="select-all-products" />
                                    </th>
                                    <th style="width: 50px;"><?php _e('Image', 'wc-delcampe-integration'); ?></th>
                                    <th><?php _e('Product', 'wc-delcampe-integration'); ?></th>
                                    <th><?php _e('SKU', 'wc-delcampe-integration'); ?></th>
                                    <th><?php _e('Price', 'wc-delcampe-integration'); ?></th>
                                    <th><?php _e('Stock', 'wc-delcampe-integration'); ?></th>
                                    <th><?php _e('Categories', 'wc-delcampe-integration'); ?></th>
                                </tr>
                            </thead>
                            <tbody id="preview-products-list">
                                <!-- Products will be loaded here -->
                            </tbody>
                        </table>
                    </div>
                    
                    <div class="publish-actions">
                        <button type="button" id="execute-publish" class="button button-primary" disabled>
                            <?php _e('Publish to Delcampe', 'wc-delcampe-integration'); ?>
                        </button>
                        <button type="button" id="cancel-publish" class="button">
                            <?php _e('Cancel', 'wc-delcampe-integration'); ?>
                        </button>
                    </div>
                    
                    <div class="publish-progress" style="display: none;">
                        <div class="progress-bar">
                            <div class="progress-fill" style="width: 0%;"></div>
                        </div>
                        <p class="progress-text"></p>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * AJAX handler for product preview
     *
     * @since 1.9.0.0
     */
    public function ajax_preview_products() {
        check_ajax_referer('delcampe_bulk_publish', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('Permission denied', 'wc-delcampe-integration'));
        }
        
        $method = sanitize_text_field($_POST['method'] ?? 'category');
        $profile_id = intval($_POST['profile_id'] ?? 0);
        
        if (!$profile_id) {
            wp_send_json_error(__('Please select a profile', 'wc-delcampe-integration'));
        }
        
        // Build query args
        $args = array(
            'post_type' => 'product',
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'fields' => 'ids'
        );
        
        // Add method-specific filters
        switch ($method) {
            case 'category':
                $categories = array_map('intval', $_POST['categories'] ?? array());
                if (!empty($categories)) {
                    $args['tax_query'] = array(
                        array(
                            'taxonomy' => 'product_cat',
                            'field' => 'term_id',
                            'terms' => $categories,
                            'include_children' => false
                        )
                    );
                }
                break;
                
            case 'tags':
                $tags = array_map('intval', $_POST['tags'] ?? array());
                if (!empty($tags)) {
                    $args['tax_query'] = array(
                        array(
                            'taxonomy' => 'product_tag',
                            'field' => 'term_id',
                            'terms' => $tags
                        )
                    );
                }
                break;
                
            case 'filter':
                // Category filter
                if (!empty($_POST['filter_category'])) {
                    $args['tax_query'] = array(
                        array(
                            'taxonomy' => 'product_cat',
                            'field' => 'term_id',
                            'terms' => intval($_POST['filter_category']),
                            'include_children' => true  // Include subcategories
                        )
                    );
                }
                
                // Status filter
                if (!empty($_POST['product_status'])) {
                    $args['post_status'] = sanitize_text_field($_POST['product_status']);
                }
                
                // Stock status filter
                if (!empty($_POST['stock_status'])) {
                    $args['meta_query'][] = array(
                        'key' => '_stock_status',
                        'value' => sanitize_text_field($_POST['stock_status'])
                    );
                }
                
                // SKU filter
                if (!empty($_POST['sku_filter'])) {
                    $args['meta_query'][] = array(
                        'key' => '_sku',
                        'value' => sanitize_text_field($_POST['sku_filter']),
                        'compare' => 'LIKE'
                    );
                }
                
                // Price filter
                if (!empty($_POST['price_min']) || !empty($_POST['price_max'])) {
                    $price_query = array(
                        'key' => '_price',
                        'type' => 'DECIMAL',
                        'compare' => 'BETWEEN',
                        'value' => array(
                            floatval($_POST['price_min'] ?? 0),
                            floatval($_POST['price_max'] ?? 999999)
                        )
                    );
                    $args['meta_query'][] = $price_query;
                }
                break;
                
            case 'all':
                // No additional filters
                break;
        }
        
        // Exclude already listed products if requested
        if (!empty($_POST['exclude_listed'])) {
            global $wpdb;
            $listings_table = $wpdb->prefix . 'delcampe_listings';
            
            // Get products already in listings table
            // Include 'verified' status as these are also published products
            $listed_products = $wpdb->get_col("SELECT DISTINCT product_id FROM {$listings_table} WHERE status IN ('published', 'active', 'processing', 'verified')");
            
            // ALSO get products with sync status metadata (pending, processing, or published)
            $synced_products = $wpdb->get_col("
                SELECT DISTINCT post_id 
                FROM {$wpdb->postmeta} 
                WHERE meta_key = '_delcampe_sync_status' 
                AND meta_value IN ('pending', 'processing', 'published', 'active', 'queued')
            ");
            
            // CHECK batch queue for products currently in batches to prevent duplicates
            $batch_table = $wpdb->prefix . 'delcampe_batch_queue';
            $batched_products = $wpdb->get_results("
                SELECT product_ids 
                FROM {$batch_table}
                WHERE status IN ('pending', 'queued', 'processing')
            ", ARRAY_A);
            
            // Extract all product IDs from batch JSONs
            $all_batched_ids = array();
            if (!empty($batched_products)) {
                foreach ($batched_products as $row) {
                    $ids = json_decode($row['product_ids'], true);
                    if (is_array($ids)) {
                        $all_batched_ids = array_merge($all_batched_ids, $ids);
                    }
                }
            }
            
            // Merge all three lists to exclude
            $exclude_products = array_unique(array_merge(
                $listed_products ?: array(),
                $synced_products ?: array(),
                $all_batched_ids ?: array()
            ));
            
            if (!empty($exclude_products)) {
                $args['post__not_in'] = $exclude_products;
            }
        }
        
        // Always exclude out-of-stock products regardless of UI filters
        if (!isset($args['meta_query'])) { $args['meta_query'] = array(); }
        $args['meta_query'][] = array(
            'key' => '_stock_status',
            'value' => 'instock',
        );

        // Get products
        $query = new WP_Query($args);
        $product_ids = $query->posts;
        
        // Get max batch size from request (default 100, max 500)
        $max_batch_size = min(500, max(10, intval($_POST['max_batch_size'] ?? 100)));
        
        // Prepare response data
        $products = array();
        foreach (array_slice($product_ids, 0, $max_batch_size) as $product_id) { // Limit preview to max batch size
            $product = wc_get_product($product_id);
            if (!$product) continue;
            
            // Skip any product that is not in stock at runtime (double-check)
            if (!method_exists($product, 'is_in_stock') || !$product->is_in_stock()) {
                continue;
            }

            $image = wp_get_attachment_image_src($product->get_image_id(), 'thumbnail');
            $categories = wp_get_post_terms($product_id, 'product_cat', array('fields' => 'names'));
            
            $products[] = array(
                'id' => $product_id,
                'title' => $product->get_name(),
                'sku' => $product->get_sku(),
                'price' => $product->get_price(),
                'stock' => $product->get_stock_quantity(),
                'stock_status' => $product->get_stock_status(),
                'image' => $image ? $image[0] : wc_placeholder_img_src(),
                'categories' => implode(', ', $categories)
            );
        }
        
        wp_send_json_success(array(
            'total' => count($product_ids),
            'products' => $products,
            'profile_id' => $profile_id
        ));
    }
    
    /**
     * Perform pre-flight validation on products
     * Checks eligibility WITHOUT modifying any state
     * 
     * @since 1.10.28.0
     * @param array $product_ids Array of product IDs to validate
     * @param int $profile_id Delcampe profile ID
     * @return array Results with 'eligible' and 'skipped' arrays
     */
    private function preflight_validation($product_ids, $profile_id) {
        global $wpdb;
        
        $results = array(
            'eligible' => array(),
            'skipped' => array(),
            'details' => array()
        );
        
        if (empty($product_ids) || !$profile_id) {
            return $results;
        }
        
        // Get all exclusion data in single atomic query to prevent race conditions
        $listings_table = $wpdb->prefix . 'delcampe_listings';
        $batch_table = $wpdb->prefix . 'delcampe_batch_queue';
        $sync_table = $wpdb->prefix . 'delcampe_sync_queue';
        
        // Get already listed products
        $listed_products = $wpdb->get_col("
            SELECT DISTINCT product_id 
            FROM {$listings_table} 
            WHERE status IN ('published', 'active', 'processing', 'verified')
        ");
        
        // Get products with sync status
        $synced_products = $wpdb->get_col("
            SELECT DISTINCT post_id 
            FROM {$wpdb->postmeta} 
            WHERE meta_key = '_delcampe_sync_status' 
            AND meta_value IN ('pending', 'processing', 'published', 'active', 'queued')
        ");
        
        // Get products in batch queue
        $batched_products_raw = $wpdb->get_col("
            SELECT product_ids 
            FROM {$batch_table}
            WHERE status IN ('pending', 'queued', 'processing', 'retry')
        ");
        
        $batched_products = array();
        foreach ($batched_products_raw as $json_ids) {
            $ids = json_decode($json_ids, true);
            if (is_array($ids)) {
                $batched_products = array_merge($batched_products, $ids);
            }
        }
        
        // Get products in sync queue  
        $sync_queue_products = $wpdb->get_col("
            SELECT DISTINCT product_id
            FROM {$sync_table}
            WHERE status IN ('pending', 'processing')
        ");
        
        // Combine all exclusions
        $all_excluded = array_unique(array_merge(
            $listed_products ?: array(),
            $synced_products ?: array(),
            $batched_products ?: array(),
            $sync_queue_products ?: array()
        ));
        
        // Check each product
        foreach ($product_ids as $product_id) {
            $product = wc_get_product($product_id);
            
            // Determine skip reason
            $skip_reason = null;
            
            if (!$product) {
                $skip_reason = 'Product not found';
            } elseif (in_array($product_id, $all_excluded)) {
                if (in_array($product_id, $listed_products)) {
                    $skip_reason = 'Already published on Delcampe';
                } elseif (in_array($product_id, $batched_products)) {
                    $skip_reason = 'Already in batch queue';
                } elseif (in_array($product_id, $sync_queue_products)) {
                    $skip_reason = 'Already in sync queue';  
                } else {
                    $skip_reason = 'Already being processed';
                }
            } elseif ($product->get_status() !== 'publish') {
                $skip_reason = 'Product not published';
            } elseif (!$product->is_in_stock()) {
                $skip_reason = 'Product out of stock';
            } elseif (!$product->get_sku()) {
                $skip_reason = 'Missing SKU';
            }
            
            if ($skip_reason) {
                $results['skipped'][$product_id] = $skip_reason;
                $results['details'][] = array(
                    'product_id' => $product_id,
                    'title' => $product ? $product->get_name() : 'Unknown',
                    'reason' => $skip_reason
                );
            } else {
                $results['eligible'][] = $product_id;
            }
        }
        
        return $results;
    }
    
    /**
     * AJAX handler for executing bulk publish
     *
     * @since 1.9.0.0
     * @version 1.10.28.0 - Added pre-flight validation
     */
    public function ajax_execute_publish() {
        check_ajax_referer('delcampe_bulk_publish', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('Permission denied', 'wc-delcampe-integration'));
        }
        
        $product_ids = array_map('intval', $_POST['product_ids'] ?? array());
        $profile_id = intval($_POST['profile_id'] ?? 0);
        $use_batch_queue = isset($_POST['use_batch_queue']) ? (bool)$_POST['use_batch_queue'] : true;
        $batch_size = intval($_POST['batch_size'] ?? 10);
        $offset = intval($_POST['offset'] ?? 0);
        
        if (empty($product_ids) || !$profile_id) {
            wp_send_json_error(__('Invalid request', 'wc-delcampe-integration'));
        }
        
        // Check API rate limit before proceeding
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-rate-limit.php';
        $rate_limit = Delcampe_Rate_Limit::get_instance();
        
        if ($rate_limit->should_pause_operations()) {
            wp_send_json_error(__('API rate limit reached. Please wait before continuing bulk operations.', 'wc-delcampe-integration'));
        }
        
        // Perform pre-flight validation FIRST
        $validation = $this->preflight_validation($product_ids, $profile_id);
        
        // If requested, return validation results for user confirmation
        if (isset($_POST['validate_only']) && $_POST['validate_only']) {
            wp_send_json_success(array(
                'validation' => $validation,
                'eligible_count' => count($validation['eligible']),
                'skipped_count' => count($validation['skipped'])
            ));
            return;
        }
        
        // If no eligible products, return error
        if (empty($validation['eligible'])) {
            wp_send_json_error(__('No eligible products to publish. All products were skipped.', 'wc-delcampe-integration'));
        }
        
        // Use only ELIGIBLE products for batch creation
        $eligible_product_ids = $validation['eligible'];
        
        // Always use batch queue when requested (ensures proper visibility + processing)
        if ($use_batch_queue) {
            // Load batch queue manager
            require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-batch-queue.php';
            $batch_queue = Delcampe_Batch_Queue::get_instance();
            
            // Get product integration for updating sync status
            $product_integration = Delcampe_Product_Integration::get_instance();
            
            global $wpdb;
            
            // Use transaction for atomic state changes
            $wpdb->query('START TRANSACTION');
            
            try {
                // Mark ONLY eligible products as pending
                foreach ($eligible_product_ids as $product_id) {
                    // Update profile
                    update_post_meta($product_id, '_delcampe_profile_id', $profile_id);
                    // Mark as pending
                    $product_integration->update_sync_status($product_id, 'pending');
                }
                
                // Create batch for eligible products only
                $batch_result = $batch_queue->create_batch($eligible_product_ids, $profile_id, 'create', $batch_size);
                
                if (is_wp_error($batch_result)) {
                    throw new Exception($batch_result->get_error_message());
                }
                
                // Commit transaction
                $wpdb->query('COMMIT');
                
            } catch (Exception $e) {
                // Rollback on any error
                $wpdb->query('ROLLBACK');
                
                // Clean up any partial state
                foreach ($eligible_product_ids as $product_id) {
                    delete_post_meta($product_id, '_delcampe_sync_status');
                    delete_post_meta($product_id, '_delcampe_profile_id');
                }
                
                wp_send_json_error($e->getMessage());
            }
            
            // Proactively nudge batch and sync processors
            if (function_exists('as_enqueue_async_action')) {
                // Kick batch processor
                as_enqueue_async_action('delcampe_process_batch_queue', array(), 'delcampe');
                // Kick sync queue processor as well (batch will also trigger)
                as_enqueue_async_action('delcampe_process_sync_queue', array(), 'delcampe');
            } else {
                // Fallback immediate triggers
                do_action('delcampe_process_batch_queue');
                do_action('delcampe_process_sync_queue');
            }

            // Return batch information with validation summary
            wp_send_json_success(array(
                'batch_id' => $batch_result['batch_id'],
                'queue_position' => $batch_result['queue_position'],
                'status' => $batch_result['status'],
                'message' => sprintf(
                    __('Batch created successfully. %d products queued (position: %d). %d products skipped.', 'wc-delcampe-integration'),
                    count($eligible_product_ids),
                    $batch_result['queue_position'],
                    count($validation['skipped'])
                ),
                'use_batch_queue' => true,
                'eligible_count' => count($eligible_product_ids),
                'skipped_count' => count($validation['skipped']),
                'skipped_details' => $validation['details']
            ));
            return;
        }
        
        // Get sync handler for immediate processing
        $sync_handler = Delcampe_Sync_Handler::get_instance();
        $product_integration = Delcampe_Product_Integration::get_instance();
        
        // Process batch
        $batch = array_slice($product_ids, $offset, $batch_size);
        $processed = 0;
        $errors = array();
        
        foreach ($batch as $product_id) {
            $product = wc_get_product($product_id);
            if (!$product) {
                $errors[] = array('product_id' => $product_id, 'reason' => 'invalid_product');
                continue;
            }
            // Update product with profile
            update_post_meta($product_id, '_delcampe_profile_id', $profile_id);
            
            // Add to sync queue
            $result = $sync_handler->add_to_queue($product_id, $profile_id, 'create');
            
            if ($result) {
                $product_integration->update_sync_status($product_id, 'pending');
                $processed++;
            } else {
                $errors[] = array('product_id' => $product_id, 'reason' => 'queue_failed');
            }
        }
        
        $total_processed = $offset + $processed;
        $has_more = $total_processed < count($product_ids);
        
        // Kick off processing immediately when possible to avoid relying solely on WP‑Cron
        if ($processed > 0) {
            if (function_exists('as_enqueue_async_action')) {
                as_enqueue_async_action('delcampe_process_sync_queue', array(), 'delcampe');
            } else {
                // Synchronous trigger as a fallback; lock prevents overlap
                do_action('delcampe_process_sync_queue');
            }
        }

        wp_send_json_success(array(
            'processed' => $total_processed,
            'total' => count($product_ids),
            'has_more' => $has_more,
            'next_offset' => $total_processed,
            'errors' => $errors
        ));
    }
}

// Initialize the class
add_action('init', array('Delcampe_Bulk_Publish', 'get_instance'));
