<?php
/**
 * Delcampe Category API Wrapper
 * Version: 1.1.2.1
 *
 * Retrieves Delcampe categories using the authenticated token.
 * Handles XML parsing and error management for category-related API calls.
 * 
 * @package WooCommerce_Delcampe_Integration
 * @subpackage API
 * @since 1.0.0
 * @version 1.1.2.1
 * 
 * Changelog:
 * 1.1.2.1 - Fixed method name from get_token() to get_auth_token()
 * 1.1.2.0 - Enhanced documentation and improved error handling
 * 1.1.1.0 - Fixed compatibility issues with constants
 * 1.1.0.0 - Added search functionality placeholder and better logging
 * 1.0.0.0 - Initial release with basic category retrieval
 */

// Exit if accessed directly to prevent direct file access
if (!defined('ABSPATH')) exit;

/**
 * Class Delcampe_Category_API
 * 
 * Static wrapper class for making category-related API calls to Delcampe.
 * Handles authentication, request formatting, and XML response parsing.
 * All responses from Delcampe API are in XML format.
 * 
 * @since 1.0.0
 * @version 1.1.2.1
 */
class Delcampe_Category_API {

    /**
     * Get categories from Delcampe API
     * 
     * Retrieves category listings from Delcampe with optional filtering.
     * Can fetch root categories or subcategories of a specific parent.
     * Supports language selection and filtering to show only leaf categories.
     * 
     * Enhanced in version 1.1.2.0 with better error handling and logging.
     * Fixed in version 1.1.2.1 to use correct auth method name.
     * 
     * @since 1.0.0
     * @version 1.1.2.1
     * @param int|null $id_parent Parent category ID (null for root categories)
     * @param string $lang Language code for category names (E=English, F=French, N=Dutch)
     * @param int $show_only_leaves 1 to show only leaf categories, 0 for all
     * @return SimpleXMLElement|WP_Error XML object with categories or error
     */
    public static function get_categories($id_parent = null, $lang = 'E', $show_only_leaves = 0) {
        // Get authentication token using singleton auth manager
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token(); // Fixed method name from get_token() to get_auth_token()
        
        // Handle authentication errors properly
        if (is_wp_error($token)) {
            delcampe_error('[Delcampe Category API v1.1.2.1] Authentication error: ' . $token->get_error_message());
            return $token;
        }
        
        // Validate token exists
        if (empty($token)) {
            delcampe_error('[Delcampe Category API v1.1.2.1] Failed to get authentication token');
            return new WP_Error('no_token', __('No valid Delcampe token available', 'wc-delcampe-integration'));
        }

        /**
         * Build query parameters for API request
         * Required parameters:
         * - token: Authentication token
         * - lang: Language code for category names
         * - showOnlyLeaves: Filter for leaf categories only
         */
        $params = array(
            'token' => $token,
            'lang'  => $lang,
        );

        // Add parent ID if specified (for subcategory requests)
        if (!empty($id_parent)) {
            // Delcampe /category endpoint expects idParent for subcategories
            $params['idParent'] = $id_parent;
            delcampe_debug('[Delcampe Category API v1.1.2.1] Fetching subcategories for parent ID: ' . $id_parent);
        } else {
            delcampe_debug('[Delcampe Category API v1.1.2.1] Fetching root categories');
        }

        // Construct full API URL with query parameters using constant
        $url = DELCAMPE_API_BASE_URL . '/category?' . http_build_query($params);
        
        // Log the request URL (without token for security)
        $safe_url = str_replace($token, '***TOKEN***', $url);
        delcampe_debug('[Delcampe Category API v1.1.2.1] Request URL: ' . $safe_url);

        /**
         * Make HTTP GET request to Delcampe API
         * Using WordPress HTTP API for compatibility
         */
        $response = wp_remote_get($url, [
            'timeout' => 20,  // 20 second timeout for API response
            'headers' => [
                'User-Agent' => DELCAMPE_USER_AGENT, // Use constant for user agent
                'Accept' => 'application/xml' // Explicitly request XML
            ]
        ]);

        // Check for WordPress HTTP errors (network issues, timeouts, etc.)
        if (is_wp_error($response)) {
            delcampe_error('[Delcampe Category API v1.1.2.1] HTTP Error: ' . $response->get_error_message());
            return new WP_Error(
                'http_request_failed',
                sprintf(
                    __('Failed to connect to Delcampe API: %s', 'wc-delcampe-integration'),
                    $response->get_error_message()
                )
            );
        }

        // Get HTTP response code
        $response_code = wp_remote_retrieve_response_code($response);
        delcampe_debug('[Delcampe Category API v1.1.2.1] Response code: ' . $response_code);
        
        // Get response headers for debugging
        $headers = wp_remote_retrieve_headers($response);
        if (isset($headers['content-type'])) {
            delcampe_debug('[Delcampe Category API v1.1.2.1] Content-Type: ' . $headers['content-type']);
        }
        
        // Check for non-200 response codes
        if ($response_code !== 200) {
            // Try to get error message from response body
            $body = wp_remote_retrieve_body($response);
            $error_msg = '';
            
            // Try to parse error from XML
            if (!empty($body)) {
                $xml = @simplexml_load_string($body);
                if ($xml && isset($xml->error)) {
                    $error_msg = (string)$xml->error;
                } elseif ($xml && isset($xml->message)) {
                    $error_msg = (string)$xml->message;
                }
            }
            
            return new WP_Error(
                'http_error',
                sprintf(
                    __('API returned HTTP %d: %s', 'wc-delcampe-integration'),
                    $response_code,
                    !empty($error_msg) ? $error_msg : __('Unknown error', 'wc-delcampe-integration')
                )
            );
        }

        // Extract response body containing XML
        $body = wp_remote_retrieve_body($response);
        
        // Check for empty response
        if (empty($body)) {
            delcampe_error('[Delcampe Category API v1.1.2.1] Empty response body received');
            return new WP_Error('empty_response', __('Empty response from Delcampe API', 'wc-delcampe-integration'));
        }
        
        // Log response size for debugging
        delcampe_debug('[Delcampe Category API v1.1.2.1] Response size: ' . strlen($body) . ' bytes');
        
        // Log first 500 characters of response for debugging (safely)
        $preview = substr($body, 0, 500);
        delcampe_debug('[Delcampe Category API v1.1.2.1] Response preview: ' . $preview);

        /**
         * Parse XML response
         * Delcampe returns categories in XML format
         * Using SimpleXML for parsing as it's PHP's standard XML parser
         */
        if (!function_exists('simplexml_load_string')) {
            // SimpleXML extension not available (rare but possible)
            delcampe_error('[Delcampe Category API v1.1.2.1] SimpleXML extension not available');
            return new WP_Error('no_xml_parser', __('SimpleXML PHP extension not available', 'wc-delcampe-integration'));
        }
        
        // Suppress errors during XML parsing and handle them properly
        libxml_use_internal_errors(true);
        
        // Attempt to parse XML
        $xml = simplexml_load_string($body);
        
        // Check if parsing failed
        if ($xml === false) {
            // Get XML parsing errors for debugging
            $errors = libxml_get_errors();
            $error_messages = array();
            
            foreach ($errors as $error) {
                $error_messages[] = trim($error->message);
            }
            
            delcampe_error('[Delcampe Category API v1.1.2.1] XML Parse errors: ' . implode(', ', $error_messages));
            
            // Clear error buffer
            libxml_clear_errors();
            
            return new WP_Error(
                'bad_xml',
                sprintf(
                    __('Could not parse category XML response: %s', 'wc-delcampe-integration'),
                    implode(', ', $error_messages)
                )
            );
        }
        
        // Clear any XML errors even if parsing succeeded
        libxml_clear_errors();
        
        // Log successful parsing with category count
        $category_count = 0;
        if (isset($xml->category)) {
            $category_count = count($xml->category);
        }
        delcampe_debug('[Delcampe Category API v1.1.2.1] XML parsed successfully. Categories found: ' . $category_count);
        
        // Return parsed XML object
        return $xml;
    }

    /**
     * Get category by ID
     * 
     * Retrieves a specific category by its ID.
     * Useful for getting detailed information about a single category.
     * Enhanced in version 1.1.2.0 with improved error handling.
     * Fixed in version 1.1.2.1 to use correct auth method name.
     * 
     * @since 1.0.0
     * @version 1.1.2.1
     * @param int $category_id The category ID to retrieve
     * @param string $lang Language code for category name
     * @return SimpleXMLElement|WP_Error Category data or error
     */
    public static function get_category_by_id($category_id, $lang = 'E') {
        // Validate category ID
        if (empty($category_id) || !is_numeric($category_id)) {
            return new WP_Error('invalid_id', __('Valid category ID is required', 'wc-delcampe-integration'));
        }

        // Get authentication token
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token(); // Fixed method name from get_token() to get_auth_token()
        
        // Handle authentication errors
        if (is_wp_error($token)) {
            return $token;
        }
        
        if (empty($token)) {
            return new WP_Error('no_token', __('No valid Delcampe token available', 'wc-delcampe-integration'));
        }

        // Build API URL for specific category
        $url = DELCAMPE_API_BASE_URL . '/category/' . $category_id . '?' . http_build_query([
            'token' => $token,
            'lang' => $lang
        ]);

        // Log request
        delcampe_debug('[Delcampe Category API v1.1.2.1] Fetching category ID: ' . $category_id);

        // Make API request
        $response = wp_remote_get($url, [
            'timeout' => 20,
            'headers' => [
                'User-Agent' => DELCAMPE_USER_AGENT,
                'Accept' => 'application/xml'
            ]
        ]);

        // Handle errors same as get_categories
        if (is_wp_error($response)) {
            return new WP_Error(
                'http_request_failed',
                sprintf(
                    __('Failed to retrieve category %d: %s', 'wc-delcampe-integration'),
                    $category_id,
                    $response->get_error_message()
                )
            );
        }

        // Check response code
        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code !== 200) {
            return new WP_Error(
                'http_error',
                sprintf(
                    __('API returned HTTP %d for category %d', 'wc-delcampe-integration'),
                    $response_code,
                    $category_id
                )
            );
        }

        $body = wp_remote_retrieve_body($response);
        
        if (empty($body)) {
            return new WP_Error('empty_response', __('Empty response from API', 'wc-delcampe-integration'));
        }
        
        // Parse XML response
        libxml_use_internal_errors(true);
        $xml = simplexml_load_string($body);
        
        if ($xml === false) {
            $errors = libxml_get_errors();
            libxml_clear_errors();
            
            return new WP_Error('bad_xml', __('Could not parse category XML response', 'wc-delcampe-integration'));
        }
        
        libxml_clear_errors();
        return $xml;
    }

    /**
     * Search categories by name
     * 
     * Searches for categories matching a given name pattern.
     * Useful for finding categories when mapping from WooCommerce.
     * Enhanced in version 1.1.2.0 for future API search implementation.
     * 
     * @since 1.0.0
     * @version 1.1.2.0
     * @param string $search_term The term to search for
     * @param string $lang Language code for search
     * @return SimpleXMLElement|WP_Error Search results or error
     */
    public static function search_categories($search_term, $lang = 'E') {
        // Validate search term
        $term = trim((string) $search_term);
        if ($term === '') {
            return new WP_Error('invalid_search', __('Search term cannot be empty', 'wc-delcampe-integration'));
        }

        $term_lc = mb_strtolower($term);

        // Cached search results by language + term
        $cache_key = 'delcampe_cat_search_' . md5($lang . '|' . $term_lc);
        $cached = get_transient($cache_key);
        if ($cached !== false) {
            delcampe_debug('[Delcampe Category API v1.1.2.1] Returning cached search for term: ' . $term);
            return $cached;
        }

        // Breadth-first traversal of the category tree with sane limits
        $max_results = 50;           // Limit results for UX
        $max_nodes   = 2000;         // Safety limit on total nodes visited

        $results = array();

        // Queue entries: [parent_id, path_names[]]
        $queue = array();
        $queue[] = array(null, array()); // start at root

        $visited = 0;

        while (!empty($queue) && count($results) < $max_results && $visited < $max_nodes) {
            list($parent_id, $path) = array_shift($queue);

            $xml = self::get_categories($parent_id, $lang, 0);
            if (is_wp_error($xml)) {
                // If a subtree fails, continue with others
                continue;
            }

            if (isset($xml->category)) {
                foreach ($xml->category as $cat) {
                    $visited++;
                    if ($visited > $max_nodes) break;

                    $id   = null;
                    if (isset($cat->id_category)) {
                        $id = (int) $cat->id_category;
                    } elseif (isset($cat->id)) {
                        $id = (int) $cat->id;
                    }
                    $name = isset($cat->name) ? (string) $cat->name : '';

                    // Build path string without extra API calls
                    $new_path   = $path;
                    $new_path[] = $name;
                    $path_str   = implode(' > ', $new_path);

                    if ($id && $name !== '') {
                        // Match if search term appears in name or path (case-insensitive)
                        $name_lc = mb_strtolower($name);
                        $path_lc = mb_strtolower($path_str);
                        if (strpos($name_lc, $term_lc) !== false || strpos($path_lc, $term_lc) !== false) {
                            $results[] = array(
                                'id' => $id,
                                'name' => $name,
                                'path' => $path_str,
                            );
                            if (count($results) >= $max_results) {
                                break;
                            }
                        }

                        // Enqueue this node to explore its children
                        $queue[] = array($id, $new_path);
                    }
                }
            }
        }

        // Cache the results for 1 hour to reduce API load
        set_transient($cache_key, $results, HOUR_IN_SECONDS);

        delcampe_debug('[Delcampe Category API v1.1.2.1] Search found ' . count($results) . ' results for term: ' . $term);

        return $results;
    }
    
    /**
     * Get category breadcrumb path
     * 
     * Retrieves the full path from root to a specific category.
     * Useful for displaying category hierarchy in the UI.
     * 
     * @since 1.1.2.0
     * @param int $category_id The category ID to get path for
     * @param string $lang Language code
     * @return array|WP_Error Array of category names from root to leaf, or error
     */
    public static function get_category_path($category_id, $lang = 'E') {
        $path = array();
        $current_id = $category_id;
        
        // Prevent infinite loops with a reasonable limit
        $max_depth = 20;
        $depth = 0;
        
        while ($current_id && $depth < $max_depth) {
            $xml = self::get_category_by_id($current_id, $lang);

            if (is_wp_error($xml)) {
                return $xml;
            }

            // Find category node across possible wrapper shapes
            if (isset($xml->category)) {
                $node = $xml->category;
            } elseif (isset($xml->Notification_Data->body->category)) {
                $node = $xml->Notification_Data->body->category;
            } elseif (isset($xml->body->category)) {
                $node = $xml->body->category;
            } else {
                $node = $xml;
            }

            // Extract name (name or label)
            $name = null;
            if (isset($node->name) && (string)$node->name !== '') {
                $name = (string)$node->name;
            } elseif (isset($node->label) && (string)$node->label !== '') {
                $name = (string)$node->label;
            }
            if ($name !== null) {
                array_unshift($path, $name);
            }

            // Determine parent id field
            $parent_id = 0;
            if (isset($node->id_parent)) {
                $parent_id = (int)$node->id_parent;
            } elseif (isset($node->parentId)) {
                $parent_id = (int)$node->parentId;
            }

            if ($parent_id > 0) {
                $current_id = $parent_id;
            } else {
                break; // reached root
            }

            $depth++;
        }
        
        if ($depth >= $max_depth) {
            delcampe_warning('[Delcampe Category API v1.1.2.1] Max depth reached when building category path');
        }
        
        return $path;
    }
    
    /**
     * Validate API response
     * 
     * Helper method to validate common API response issues.
     * Used internally to standardize error checking.
     * 
     * @since 1.1.2.0
     * @param mixed $response Response to validate
     * @param string $context Context for error messages
     * @return bool|WP_Error True if valid, WP_Error otherwise
     */
    private static function validate_response($response, $context = '') {
        if (is_wp_error($response)) {
            return $response;
        }
        
        if (empty($response)) {
            return new WP_Error(
                'invalid_response',
                sprintf(
                    __('Invalid response received%s', 'wc-delcampe-integration'),
                    !empty($context) ? ' for ' . $context : ''
                )
            );
        }
        
        return true;
    }
    
    /**
     * Check if category has children
     * 
     * Determines if a category has subcategories.
     * Useful for UI display and navigation.
     * 
     * @since 1.1.2.0
     * @param int $category_id Category ID to check
     * @param string $lang Language code
     * @return bool|WP_Error True if has children, false if not, WP_Error on failure
     */
    public static function has_children($category_id, $lang = 'E') {
        $children = self::get_categories($category_id, $lang);

        if (is_wp_error($children)) {
            return $children;
        }

        // Support multiple XML response shapes
        $list = null;
        if (isset($children->Notification_Data->body->category)) {
            $list = $children->Notification_Data->body->category;
        } elseif (isset($children->category)) {
            $list = $children->category;
        } elseif (isset($children->categories->category)) {
            $list = $children->categories->category;
        } elseif (isset($children->body->category)) {
            $list = $children->body->category;
        }

        return ($list !== null && count($list) > 0);
    }
}
