<?php
/**
 * Optimized Delcampe Category Manager
 * 
 * Version: 1.10.35.2
 * 
 * Prevents unnecessary category API calls when there are no active listings
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Category_Manager_Optimized extends Delcampe_Category_Manager {
    
    /**
     * Override fetch_base_categories to check for active listings first
     */
    public function fetch_base_categories() {
        // Check if we have any active listings first
        global $wpdb;
        $listings_table = $wpdb->prefix . 'delcampe_listings';
        
        $active_count = $wpdb->get_var("
            SELECT COUNT(*) 
            FROM {$listings_table}
            WHERE status IN ('active', 'published', 'verified')
            AND delcampe_id IS NOT NULL
        ");
        
        // If no active listings and not explicitly creating a new profile, 
        // return cached or empty categories
        if ($active_count == 0 && !$this->is_profile_creation_context()) {
            // Check if we have cached categories
            $cached = get_transient(self::CATEGORY_TRANSIENT_KEY);
            if ($cached !== false) {
                return $cached;
            }
            
            // Return empty array to prevent API calls
            delcampe_log('[Category Manager] No active listings - skipping category fetch');
            return array();
        }
        
        // Otherwise, proceed with normal fetch
        return parent::fetch_base_categories();
    }
    
    /**
     * Check if we're in a profile creation/edit context
     */
    private function is_profile_creation_context() {
        // Check if we're on the profiles admin page
        if (!is_admin()) {
            return false;
        }
        
        // Check if we're creating or editing a profile
        $page = isset($_GET['page']) ? $_GET['page'] : '';
        $action = isset($_GET['action']) ? $_GET['action'] : '';
        
        return ($page === 'delcampe-profiles' && in_array($action, ['new', 'edit']));
    }
    
    /**
     * Override fetch_subcategories with same optimization
     */
    public function fetch_subcategories($parent_id) {
        // Check for active listings first
        global $wpdb;
        $listings_table = $wpdb->prefix . 'delcampe_listings';
        
        $active_count = $wpdb->get_var("
            SELECT COUNT(*) 
            FROM {$listings_table}
            WHERE status IN ('active', 'published', 'verified')
            AND delcampe_id IS NOT NULL
        ");
        
        if ($active_count == 0 && !$this->is_profile_creation_context()) {
            // Check cache first
            $cache_key = 'delcampe_subcategories_' . $parent_id;
            $cached = get_transient($cache_key);
            if ($cached !== false) {
                return $cached;
            }
            
            // Return empty array to prevent API calls
            delcampe_log('[Category Manager] No active listings - skipping subcategory fetch');
            return array();
        }
        
        // Otherwise, proceed with normal fetch
        return parent::fetch_subcategories($parent_id);
    }
}