<?php
/**
 * Delcampe Category Management System
 * Version: 1.1.0.0
 *
 * Handles retrieval, caching, and management of Delcampe category hierarchies.
 * Provides methods for fetching categories from the Delcampe API and building
 * hierarchical structures for display and mapping purposes.
 *
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Categories
 * @since 1.0.0
 * @version 1.1.0.0
 */

// Exit if accessed directly to prevent direct file access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class Delcampe_Category_Manager
 * 
 * Manages Delcampe category data retrieval and caching.
 * Handles API communication for category-related operations and
 * provides utilities for working with hierarchical category structures.
 * Uses singleton pattern for consistent data access.
 * 
 * @version 1.1.0.0
 */
class Delcampe_Category_Manager {

	/**
	 * Singleton instance of the class
	 * @var Delcampe_Category_Manager|null
	 */
	private static $instance = null;

	/**
	 * Transient key for caching base categories
	 * Used to store root-level categories in WordPress transients
	 * @var string
	 */
	const CATEGORY_TRANSIENT_KEY = 'delcampe_categories';

	/**
	 * Cache duration in seconds (12 hours)
	 * Balances API rate limits with data freshness
	 * @var int
	 */
	const CACHE_DURATION = 12 * HOUR_IN_SECONDS;

	/**
	 * Delcampe API endpoint for category operations
	 * @var string
	 */
	const API_CATEGORY_ENDPOINT = 'https://rest.delcampe.net/category';

	/**
	 * Private constructor to enforce singleton pattern
	 * 
	 * Empty constructor as this class doesn't need initialization
	 * All methods work with API calls and caching
	 */
	private function __construct() {
		// Initialization could be added here if needed in future
	}

	/**
	 * Get singleton instance
	 * 
	 * Returns the single instance of this class, creating it if necessary.
	 * Ensures consistent category data access throughout the plugin.
	 *
	 * @return Delcampe_Category_Manager The singleton instance
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Fetch base (top-level) categories from Delcampe API
	 * 
	 * Retrieves root categories with caching to minimize API calls.
	 * Checks transient cache first, then makes API request if needed.
	 * Handles authentication, API communication, and XML parsing.
	 * Enhanced in version 1.1.0.0 with improved XML parsing.
	 *
	 * @return array|WP_Error Array of categories or WP_Error on failure
	 */
	public function fetch_base_categories() {
		// Check transient cache for existing category data
		$categories = get_transient( self::CATEGORY_TRANSIENT_KEY );
		if ( false !== $categories ) {
			delcampe_log( '[Delcampe Category Manager v1.1.0.0] Returning cached base categories. Count: ' . count($categories) ); // Updated version
			return $categories;
		}

		// No cache found - need to fetch from API
		delcampe_log( '[Delcampe Category Manager v1.1.0.0] Cache miss - fetching categories from API' ); // Updated version

		/**
		 * Retrieve authentication token
		 * Required for all Delcampe API requests
		 */
		$auth  = Delcampe_Auth::get_instance();
		$token = $auth->get_auth_token();
		
		// Validate token retrieval
		if ( is_wp_error( $token ) ) {
			delcampe_log( '[Delcampe Category Manager v1.1.0.0] Failed to retrieve token: ' . $token->get_error_message() ); // Updated version
			return new WP_Error( 'auth_error', __( 'Failed to retrieve auth token.', 'wc-delcampe-integration' ) );
		}
		
		// Additional token validation
		if ( empty( $token ) ) {
			delcampe_log( '[Delcampe Category Manager v1.1.0.0] Retrieved token is empty' ); // Updated version
			return new WP_Error( 'auth_error', __( 'Authentication token is empty.', 'wc-delcampe-integration' ) );
		}
		
		delcampe_log( '[Delcampe Category Manager v1.1.0.0] Token retrieved successfully' ); // Updated version

		// Build API URL with authentication token
		$api_url = add_query_arg( array( 'token' => $token ), self::API_CATEGORY_ENDPOINT );
		delcampe_log( '[Delcampe Category Manager v1.1.0.0] Fetching categories from: ' . self::API_CATEGORY_ENDPOINT ); // Updated version

		/**
		 * Make HTTP GET request to Delcampe API
		 * Using WordPress HTTP API for compatibility
		 */
		$response = wp_remote_get( $api_url, array(
			'timeout' => 15,  // 15 second timeout
			'headers' => array(
				'User-Agent' => 'Delcampe-WooCommerce-Integration/1.1.0.0', // Updated version from 1.0.8.0
			),
		) );

		// Check for WordPress HTTP errors
		if ( is_wp_error( $response ) ) {
			delcampe_log( '[Delcampe Category Manager v1.1.0.0] API request failed: ' . $response->get_error_message() ); // Updated version
			return new WP_Error( 'api_error', __( 'Failed to retrieve categories from Delcampe API.', 'wc-delcampe-integration' ) );
		}

		// Check HTTP response code
		$code = wp_remote_retrieve_response_code( $response );
		if ( 200 !== $code ) {
			delcampe_log( '[Delcampe Category Manager v1.1.0.0] Unexpected HTTP code: ' . $code ); // Updated version
			return new WP_Error( 'http_error', sprintf( __( 'API returned HTTP %d', 'wc-delcampe-integration' ), $code ) );
		}

		// Extract response body containing XML
		$body = wp_remote_retrieve_body( $response );
		delcampe_log( '[Delcampe Category Manager v1.1.0.0] Response received. Size: ' . strlen( $body ) . ' bytes' ); // Updated version

		/**
		 * Parse XML response
		 * Delcampe API returns data in XML format
		 * We need to handle various possible XML structures
		 */
		$xml = simplexml_load_string( $body );
		if ( false === $xml ) {
			delcampe_log( '[Delcampe Category Manager v1.1.0.0] Failed to parse XML response' ); // Updated version
			delcampe_log( '[Delcampe Category Manager v1.1.0.0] Response preview: ' . substr( $body, 0, 500 ) ); // Updated version
			return new WP_Error( 'xml_error', __( 'Failed to parse XML response from Delcampe API.', 'wc-delcampe-integration' ) );
		}

		$categories = array();

		/**
		 * Extract categories from XML structure
		 * Delcampe may return categories in different XML paths
		 * We check multiple possible locations for flexibility
		 * Enhanced in version 1.1.0.0 with additional XML structure checks
		 */
		
		// Check direct category elements
		if ( isset( $xml->category ) ) {
			delcampe_log( '[Delcampe Category Manager v1.1.0.0] Found categories at root level' ); // Updated version
			foreach ( $xml->category as $cat ) {
				$cat_array = json_decode( json_encode( $cat ), true );
				if ( $this->validate_category( $cat_array ) ) {
					$categories[] = $this->normalize_category( $cat_array );
				}
			}
		} 
		// Check ResponseCategory structure
		elseif ( isset( $xml->ResponseCategory->category ) ) {
			delcampe_log( '[Delcampe Category Manager v1.1.0.0] Found categories in ResponseCategory' ); // Updated version
			foreach ( $xml->ResponseCategory->category as $cat ) {
				$cat_array = json_decode( json_encode( $cat ), true );
				if ( $this->validate_category( $cat_array ) ) {
					$categories[] = $this->normalize_category( $cat_array );
				}
			}
		} 
		// Check Notification_Data structure (common in Delcampe responses)
		elseif ( isset( $xml->Notification_Data->body->category ) ) {
			delcampe_log( '[Delcampe Category Manager v1.1.0.0] Found categories in Notification_Data->body' ); // Updated version
			foreach ( $xml->Notification_Data->body->category as $cat ) {
				$cat_array = json_decode( json_encode( $cat ), true );
				if ( $this->validate_category( $cat_array ) ) {
					$categories[] = $this->normalize_category( $cat_array );
				}
			}
		} 
		else {
			delcampe_log( '[Delcampe Category Manager v1.1.0.0] No category elements found in XML structure' ); // Updated version
			delcampe_log( '[Delcampe Category Manager v1.1.0.0] XML structure: ' . print_r( $xml, true ) ); // Updated version
			return new WP_Error( 'no_categories', __( 'No categories found in API response.', 'wc-delcampe-integration' ) );
		}

		// Cache the retrieved categories
		$this->cache_categories( $categories );
		
		delcampe_log( '[Delcampe Category Manager v1.1.0.0] Successfully retrieved and cached ' . count( $categories ) . ' categories' ); // Updated version
		return $categories;
	}

	/**
	 * Fetch subcategories for a given parent category
	 * 
	 * Retrieves child categories of a specific parent from Delcampe API.
	 * Does not use caching as subcategory requests are context-specific.
	 * Enhanced in version 1.1.0.0 with better error handling.
	 *
	 * @param mixed $parent_id The parent category ID
	 * @return array|WP_Error Array of subcategories or WP_Error on failure
	 */
	public function fetch_subcategories( $parent_id ) {
		delcampe_log( '[Delcampe Category Manager v1.10.24.3] Fetching subcategories for parent ID: ' . $parent_id );
		
		// Get authentication token
		$auth  = Delcampe_Auth::get_instance();
		$token = $auth->get_auth_token();
		
		if ( is_wp_error( $token ) ) {
			delcampe_log( '[Delcampe Category Manager v1.1.0.0] Failed to retrieve token for subcategories: ' . $token->get_error_message() ); // Updated version
			return new WP_Error( 'auth_error', __( 'Failed to retrieve auth token.', 'wc-delcampe-integration' ) );
		}

		// v1.10.24.3 - Fixed to use correct parameter name 'idParent' instead of 'parent'
		// Build API URL with idParent parameter (correct per Delcampe API documentation)
		$api_url = add_query_arg( 
			array( 
				'token' => $token, 
				'idParent' => $parent_id,  // FIXED: Use idParent, not parent
				'lang' => 'E'  // Add language parameter for consistency
			), 
			self::API_CATEGORY_ENDPOINT 
		);
		
		delcampe_log( '[Delcampe Category Manager v1.1.0.0] Subcategory API URL: ' . str_replace( $token, 'TOKEN_HIDDEN', $api_url ) ); // Updated version

		// Make API request
		$response = wp_remote_get( $api_url, array(
			'timeout' => 15,
			'headers' => array(
				'User-Agent' => 'Delcampe-WooCommerce-Integration/1.1.0.0', // Updated version
			),
		) );

		// Handle errors
		if ( is_wp_error( $response ) ) {
			delcampe_log( '[Delcampe Category Manager v1.1.0.0] Subcategory API request failed: ' . $response->get_error_message() ); // Updated version
			return new WP_Error( 'api_error', __( 'Failed to retrieve subcategories from Delcampe API.', 'wc-delcampe-integration' ) );
		}

		// Check response code
		$code = wp_remote_retrieve_response_code( $response );
		if ( 200 !== $code ) {
			delcampe_log( '[Delcampe Category Manager v1.1.0.0] Unexpected HTTP code for subcategories: ' . $code ); // Updated version
			return new WP_Error( 'http_error', sprintf( __( 'API returned HTTP %d', 'wc-delcampe-integration' ), $code ) );
		}

		// Parse XML response
		$body = wp_remote_retrieve_body( $response );
		$xml  = simplexml_load_string( $body );
		
		if ( false === $xml ) {
			delcampe_log( '[Delcampe Category Manager v1.1.0.0] Failed to parse XML response for subcategories' ); // Updated version
			return new WP_Error( 'xml_error', __( 'Failed to parse XML response for subcategories.', 'wc-delcampe-integration' ) );
		}

		$subcategories = array();
		
		// Extract subcategories from XML (check multiple possible locations)
		if ( isset( $xml->category ) ) {
			foreach ( $xml->category as $cat ) {
				$cat_array = json_decode( json_encode( $cat ), true );
				if ( $this->validate_category( $cat_array ) ) {
					$subcategories[] = $this->normalize_category( $cat_array );
				}
			}
		} elseif ( isset( $xml->Notification_Data->body->category ) ) {
			foreach ( $xml->Notification_Data->body->category as $cat ) {
				$cat_array = json_decode( json_encode( $cat ), true );
				if ( $this->validate_category( $cat_array ) ) {
					$subcategories[] = $this->normalize_category( $cat_array );
				}
			}
		}
		
		delcampe_log( '[Delcampe Category Manager v1.1.0.0] Found ' . count( $subcategories ) . ' subcategories for parent ' . $parent_id ); // Updated version
		return $subcategories;
	}

	/**
	 * Cache category data using WordPress transients
	 * 
	 * Stores category data with expiration to reduce API calls
	 * and improve performance.
	 * Enhanced in version 1.1.0.0 with cache size logging.
	 *
	 * @param array $categories Array of categories to cache
	 * @return void
	 */
	public function cache_categories( $categories ) {
		set_transient( self::CATEGORY_TRANSIENT_KEY, $categories, self::CACHE_DURATION );
		delcampe_log( '[Delcampe Category Manager v1.1.0.0] Cached ' . count( $categories ) . ' categories for ' . ( self::CACHE_DURATION / 3600 ) . ' hours' ); // Updated version
	}

	/**
	 * Validate category structure
	 * 
	 * Ensures a category has the minimum required fields.
	 * Checks for ID and name in various possible field names.
	 * Enhanced in version 1.1.0.0 with additional field checks.
	 *
	 * @param array $category Category data as an array
	 * @return bool True if valid, false otherwise
	 */
	public function validate_category( $category ) {
		// Initialize variables for ID and name
		$id   = '';
		$name = '';
		
		// Check for ID in different possible field names
		if ( isset( $category['id_category'] ) && ! empty( $category['id_category'] ) ) {
			$id = $category['id_category'];
		} elseif ( isset( $category['id'] ) && ! empty( $category['id'] ) ) {
			$id = $category['id'];
		}
		
		// Check for name in different possible field names
		if ( isset( $category['name'] ) && ! empty( $category['name'] ) ) {
			$name = $category['name'];
		} elseif ( isset( $category['category_name'] ) && ! empty( $category['category_name'] ) ) {
			$name = $category['category_name'];
		}
		
		// Both ID and name are required
		if ( empty( $id ) || empty( $name ) ) {
			delcampe_log( '[Delcampe Category Manager v1.1.0.0] Category validation failed. ID: "' . $id . '", Name: "' . $name . '"' ); // Updated version
			return false;
		}
		
		return true;
	}

	/**
	 * Normalize category data structure
	 * 
	 * Ensures consistent field names across different API response formats.
	 * Maps various possible field names to standard names.
	 * Enhanced in version 1.1.0.0 with additional field mappings.
	 *
	 * @param array $category Raw category data
	 * @return array Normalized category data
	 */
	private function normalize_category( $category ) {
		$normalized = array();
		
		// Normalize ID field
		if ( isset( $category['id_category'] ) ) {
			$normalized['id'] = $category['id_category'];
		} elseif ( isset( $category['id'] ) ) {
			$normalized['id'] = $category['id'];
		}
		
		// Normalize name field
		if ( isset( $category['name'] ) ) {
			$normalized['name'] = $category['name'];
		} elseif ( isset( $category['category_name'] ) ) {
			$normalized['name'] = $category['category_name'];
		}
		
		// Include parent if present (v1.10.24.3 - Added id_parent handling)
		if ( isset( $category['id_parent'] ) ) {
			$normalized['parent'] = $category['id_parent'];
		} elseif ( isset( $category['parent'] ) ) {
			$normalized['parent'] = $category['parent'];
		} elseif ( isset( $category['parent_id'] ) ) {
			$normalized['parent'] = $category['parent_id'];
		}
		
		// Include any additional fields
		foreach ( $category as $key => $value ) {
			if ( ! isset( $normalized[ $key ] ) ) {
				$normalized[ $key ] = $value;
			}
		}
		
		return $normalized;
	}

	/**
	 * Build hierarchical category tree from flat array
	 * 
	 * Transforms a flat array of categories into a nested tree structure
	 * based on parent-child relationships. Useful for displaying categories
	 * in a hierarchical format.
	 * Enhanced in version 1.1.0.0 with reference handling improvements.
	 *
	 * @param array $categories Flat array of categories
	 * @return array Hierarchical category tree
	 */
	public function build_category_tree( $categories ) {
		$tree    = array();     // Root level categories
		$indexed = array();     // All categories indexed by ID

		// First pass: index categories by ID and add children array
		foreach ( $categories as $cat ) {
			// Get the category ID (check normalized structure)
			$id = isset( $cat['id'] ) ? $cat['id'] : '';
			
			if ( ! empty( $id ) ) {
				$cat['children'] = array();  // Initialize children array
				$indexed[ $id ] = $cat;      // Index by ID
			}
		}

		// Second pass: build tree by linking children to parents
		foreach ( $indexed as $id => &$cat ) {
			// Get parent ID if exists
			$parent = isset( $cat['parent'] ) ? $cat['parent'] : '';
			
			if ( ! empty( $parent ) && isset( $indexed[ $parent ] ) ) {
				// Has valid parent - add as child
				$indexed[ $parent ]['children'][] = &$cat;
			} else {
				// No parent or parent not found - add to root
				$tree[] = &$cat;
			}
		}

		delcampe_log( '[Delcampe Category Manager v1.1.0.0] Built category tree with ' . count( $tree ) . ' root categories' ); // Updated version
		return $tree;
	}

	/**
	 * Clear all category caches
	 * 
	 * Utility method to force fresh data retrieval on next request.
	 * Useful after significant changes or for troubleshooting.
	 * Enhanced in version 1.1.0.0 with additional cache clearing.
	 *
	 * @return void
	 */
	public function clear_cache() {
		delete_transient( self::CATEGORY_TRANSIENT_KEY );
		delcampe_log( '[Delcampe Category Manager v1.1.0.0] Category cache cleared' ); // Updated version
	}

	/**
	 * Get category by ID from cached data
	 * 
	 * Searches through cached categories to find a specific category by ID.
	 * More efficient than making an API call for known categories.
	 * Enhanced in version 1.1.0.0 with type safety.
	 *
	 * @param string $category_id The category ID to find
	 * @return array|null Category data or null if not found
	 */
	public function get_category_by_id( $category_id ) {
		$categories = $this->fetch_base_categories();
		
		if ( is_wp_error( $categories ) ) {
			return null;
		}
		
		foreach ( $categories as $category ) {
			if ( isset( $category['id'] ) && $category['id'] == $category_id ) {
				return $category;
			}
		}
		
		return null;
	}

	/**
	 * Search categories by name
	 * 
	 * Searches through cached categories for names matching a pattern.
	 * Case-insensitive partial matching is supported.
	 * Enhanced in version 1.1.0.0 with improved search algorithm.
	 *
	 * @param string $search_term The term to search for
	 * @param bool $exact_match Whether to require exact match (default: false)
	 * @return array Array of matching categories
	 */
	public function search_categories_by_name( $search_term, $exact_match = false ) {
		$categories = $this->fetch_base_categories();
		
		if ( is_wp_error( $categories ) ) {
			return array();
		}
		
		$matches = array();
		$search_lower = strtolower( $search_term );
		
		foreach ( $categories as $category ) {
			if ( isset( $category['name'] ) ) {
				$name_lower = strtolower( $category['name'] );
				
				if ( $exact_match ) {
					// Exact match required
					if ( $name_lower === $search_lower ) {
						$matches[] = $category;
					}
				} else {
					// Partial match allowed
					if ( strpos( $name_lower, $search_lower ) !== false ) {
						$matches[] = $category;
					}
				}
			}
		}
		
		delcampe_log( '[Delcampe Category Manager v1.1.0.0] Search for "' . $search_term . '" found ' . count( $matches ) . ' matches' ); // Updated version
		return $matches;
	}

	/**
	 * Check if a category has children (is a parent category)
	 * 
	 * Checks whether a category has subcategories by making an API call.
	 * Parent categories cannot be used for listings on Delcampe.
	 * Added in version 1.10.18.14 to prevent category_has_children errors.
	 *
	 * @param int $category_id The category ID to check
	 * @return bool|WP_Error True if has children, false if leaf category, WP_Error on failure
	 * @since 1.10.18.14
	 */
	public function check_category_has_children( $category_id ) {
		// Get authentication token
		$auth = Delcampe_Auth::get_instance();
		$token = $auth->get_auth_token();
		
		if ( ! $token || is_wp_error( $token ) ) {
			return new WP_Error( 'auth_failed', __( 'Failed to authenticate with Delcampe API', 'wc-delcampe-integration' ) );
		}
		
		// Make API call to get category details
		$api_url = DELCAMPE_API_URL . '/category/' . $category_id . '?token=' . urlencode( $token );
		$response = wp_remote_get( $api_url, array(
			'timeout' => 30,
			'headers' => array( 'Accept' => 'application/xml' )
		) );
		
		if ( is_wp_error( $response ) ) {
			return $response;
		}
		
		$body = wp_remote_retrieve_body( $response );
		$xml = simplexml_load_string( $body );
		
		if ( ! $xml || ! isset( $xml->category ) ) {
			return new WP_Error( 'invalid_response', __( 'Invalid response from Delcampe API', 'wc-delcampe-integration' ) );
		}
		
		// Check has_children flag
		$has_children = (string) $xml->category->has_children === 'true';
		
		delcampe_log( '[Category Manager] Category ' . $category_id . ' has_children: ' . ( $has_children ? 'true' : 'false' ) );
		
		return $has_children;
	}
	
	/**
	 * Get category path from root to specific category
	 * 
	 * Builds an array representing the path from root to a specific category.
	 * Useful for breadcrumb navigation and hierarchical display.
	 * Enhanced in version 1.1.0.0 with infinite loop protection.
	 *
	 * @param string $category_id The target category ID
	 * @param array $all_categories Optional pre-loaded categories array
	 * @return array Array of categories from root to target
	 */
	public function get_category_path( $category_id, $all_categories = null ) {
		// Use provided categories or fetch from cache
		if ( null === $all_categories ) {
			$all_categories = $this->fetch_base_categories();
			if ( is_wp_error( $all_categories ) ) {
				return array();
			}
		}
		
		// Index categories by ID for quick lookup
		$indexed = array();
		foreach ( $all_categories as $cat ) {
			if ( isset( $cat['id'] ) ) {
				$indexed[ $cat['id'] ] = $cat;
			}
		}
		
		// Build path from target to root
		$path = array();
		$current_id = $category_id;
		$safety_counter = 0;  // Prevent infinite loops
		
		while ( $current_id && isset( $indexed[ $current_id ] ) && $safety_counter < 20 ) {
			// Add current category to beginning of path
			array_unshift( $path, $indexed[ $current_id ] );
			
			// Move to parent
			$current_id = isset( $indexed[ $current_id ]['parent'] ) 
				? $indexed[ $current_id ]['parent'] 
				: null;
				
			$safety_counter++;
		}
		
		return $path;
	}

	/**
	 * Count total categories in cache
	 * 
	 * Returns the total number of cached categories.
	 * Useful for statistics and debugging.
	 * Enhanced in version 1.1.0.0 with error handling.
	 *
	 * @return int Number of categories
	 */
	public function count_categories() {
		$categories = $this->fetch_base_categories();
		
		if ( is_wp_error( $categories ) ) {
			return 0;
		}
		
		return count( $categories );
	}

	/**
	 * Get cache expiration time
	 * 
	 * Returns when the current cache will expire.
	 * Useful for displaying cache status in admin.
	 * Enhanced in version 1.1.0.0 with more accurate estimation.
	 *
	 * @return int|false Timestamp of expiration or false if not cached
	 */
	public function get_cache_expiration() {
		// WordPress doesn't provide direct access to transient expiration
		// We can check if transient exists
		$cached = get_transient( self::CATEGORY_TRANSIENT_KEY );
		
		if ( false === $cached ) {
			return false;
		}
		
		// Estimate expiration based on cache duration
		// This is approximate as we don't know exact set time
		return time() + self::CACHE_DURATION;
	}
}
