<?php
/**
 * Delcampe Category Mapping System
 * Version: 1.1.0.0
 *
 * Handles the storage, retrieval, and relationship mapping of Delcampe categories
 * to WooCommerce categories. This system supports hierarchical category selection,
 * caching, and future extensibility for internal category matching.
 *
 * Database Structure:
 * A custom table is created (via dbDelta) to store the mapping data.
 *
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Mapping
 * @since 1.0.0
 * @version 1.1.0.0
 */

// Exit if accessed directly to prevent direct file access
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Prevent duplicate class declaration
 * This check ensures the class is only defined once even if the file
 * is included multiple times during plugin execution
 */
if ( ! class_exists( 'Delcampe_Category_Mapping' ) ) :

/**
 * Class Delcampe_Category_Mapping
 * 
 * Manages the mapping between WooCommerce product categories and Delcampe categories.
 * Provides database operations, caching, and utilities for category relationship management.
 * Uses singleton pattern to ensure consistent data access across the plugin.
 * 
 * @version 1.1.0.0
 */
class Delcampe_Category_Mapping {

	/**
	 * Singleton instance of the class
	 * @var Delcampe_Category_Mapping|null
	 */
	private static $instance = null;

	/**
	 * WordPress database object for queries
	 * @var wpdb
	 */
	private $wpdb;

	/**
	 * Full table name including WordPress prefix
	 * @var string
	 */
	private $table_name;

	/**
	 * Cache key for storing mapping data in transients
	 * Used to improve performance by reducing database queries
	 * @var string
	 */
	const MAPPING_CACHE_KEY = 'delcampe_category_mapping_cache';

	/**
	 * Duration to cache mapping data (12 hours)
	 * Balances performance with data freshness
	 * @var int
	 */
	const CACHE_DURATION = 12 * HOUR_IN_SECONDS;

	/**
	 * Private constructor to enforce singleton pattern
	 * 
	 * Initializes the database object and sets the table name
	 * with proper WordPress table prefix for multisite compatibility
	 */
	private function __construct() {
		global $wpdb;
		$this->wpdb = $wpdb;
		// Set table name with WordPress prefix for proper namespacing
		$this->table_name = $this->wpdb->prefix . 'delcampe_category_mapping';
	}

	/**
	 * Get singleton instance
	 * 
	 * Returns the single instance of this class, creating it if necessary.
	 * This ensures all parts of the plugin work with the same mapping data.
	 *
	 * @return Delcampe_Category_Mapping The singleton instance
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Create or update the category mapping database table
	 * 
	 * Uses WordPress dbDelta function to safely create or update the table structure.
	 * Called during plugin activation to ensure the table exists.
	 * Table structure supports hierarchical categories and timestamps.
	 * Enhanced in version 1.1.0.0 with improved logging.
	 *
	 * @return void
	 */
	public function create_mapping_table() {
		// Load WordPress upgrade functions for dbDelta
		require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );

		/**
		 * Define table structure
		 * - id: Primary key for internal use
		 * - delcampe_category_id: ID from Delcampe's system
		 * - delcampe_category_name: Human-readable category name
		 * - woo_category_id: WooCommerce term ID
		 * - woo_category_name: WooCommerce category name
		 * - parent_delcampe_category_id: For hierarchical relationships
		 * - depth: Category depth in hierarchy (0 = root)
		 * - date_created: Timestamp for tracking
		 */
		$sql = "CREATE TABLE {$this->table_name} (
			id mediumint(9) NOT NULL AUTO_INCREMENT,
			delcampe_category_id varchar(50) NOT NULL,
			delcampe_category_name text NOT NULL,
			woo_category_id varchar(50) NOT NULL,
			woo_category_name text NOT NULL,
			parent_delcampe_category_id varchar(50) DEFAULT NULL,
			depth int(3) DEFAULT 0,
			date_created datetime DEFAULT CURRENT_TIMESTAMP,
			PRIMARY KEY  (id),
			KEY delcampe_category_id (delcampe_category_id),
			KEY woo_category_id (woo_category_id)
		) {$this->wpdb->get_charset_collate()};";

		// Execute table creation/update
		dbDelta( $sql );
		
		// Log table creation for debugging
		delcampe_log( '[Delcampe Category Mapping v1.1.0.0] Mapping table created/updated: ' . $this->table_name ); // Updated version
	}

	/**
	 * Store a new category mapping record
	 * 
	 * Validates and sanitizes mapping data before storing in the database.
	 * Clears the cache after insertion to ensure fresh data on next retrieval.
	 * Enhanced in version 1.1.0.0 with improved validation.
	 *
	 * @param array $mapping_data {
	 *     Mapping data array with the following structure:
	 *
	 *     @type string $delcampe_category_id         Delcampe category ID (required)
	 *     @type string $delcampe_category_name       Delcampe category name (required)
	 *     @type string $woo_category_id              WooCommerce category ID (required)
	 *     @type string $woo_category_name            WooCommerce category name (required)
	 *     @type string $parent_delcampe_category_id  Parent Delcampe category ID (optional)
	 *     @type int    $depth                        Category depth in hierarchy (optional, default 0)
	 * }
	 * @return int|WP_Error Inserted record ID on success or WP_Error on failure
	 */
	public function store_mapping( $mapping_data ) {
		// Sanitize input data to prevent security issues
		$data = $this->sanitize_mapping_data( $mapping_data );
		
		// Validate required fields are present
		$errors = $this->validate_mapping_data( $data );
		if ( ! empty( $errors ) ) {
			// Log validation errors for debugging
			delcampe_log( '[Delcampe Category Mapping v1.1.0.0] Validation errors: ' . print_r( $errors, true ) ); // Updated version
			return new WP_Error( 'validation_error', __( 'Mapping data validation failed.', 'wc-delcampe-integration' ) );
		}

		// Insert mapping record into database
		$result = $this->wpdb->insert(
			$this->table_name,
			array(
				'delcampe_category_id'        => $data['delcampe_category_id'],
				'delcampe_category_name'      => $data['delcampe_category_name'],
				'woo_category_id'             => $data['woo_category_id'],
				'woo_category_name'           => $data['woo_category_name'],
				'parent_delcampe_category_id' => isset( $data['parent_delcampe_category_id'] ) ? $data['parent_delcampe_category_id'] : null,
				'depth'                       => isset( $data['depth'] ) ? intval( $data['depth'] ) : 0,
			),
			array(
				'%s',  // delcampe_category_id format
				'%s',  // delcampe_category_name format
				'%s',  // woo_category_id format
				'%s',  // woo_category_name format
				'%s',  // parent_delcampe_category_id format
				'%d',  // depth format (integer)
			)
		);

		// Check if insert was successful
		if ( false === $result ) {
			delcampe_log( '[Delcampe Category Mapping v1.1.0.0] Database insert error: ' . $this->wpdb->last_error ); // Updated version
			return new WP_Error( 'db_error', __( 'Failed to store category mapping.', 'wc-delcampe-integration' ) );
		}

		// Clear cache to ensure fresh data
		delete_transient( self::MAPPING_CACHE_KEY );
		
		// Log successful insertion
		delcampe_log( '[Delcampe Category Mapping v1.1.0.0] Mapping stored with ID ' . $this->wpdb->insert_id ); // Updated version
		
		// Return the ID of the newly inserted record
		return $this->wpdb->insert_id;
	}

	/**
	 * Retrieve all category mapping records
	 * 
	 * Implements caching for performance optimization.
	 * Returns cached data if available, otherwise queries database.
	 * Enhanced in version 1.1.0.0 with improved cache handling.
	 *
	 * @return array Array of mapping records
	 */
	public function retrieve_mapping() {
		// Check for cached mapping data
		$mapping = get_transient( self::MAPPING_CACHE_KEY );
		if ( false !== $mapping ) {
			delcampe_log( '[Delcampe Category Mapping v1.1.0.0] Returning cached mapping data.' ); // Updated version
			return $mapping;
		}

		// No cache found, query database
		$query = "SELECT * FROM {$this->table_name}";
		$results = $this->wpdb->get_results( $query, ARRAY_A );

		// Check for database errors
		if ( null === $results ) {
			delcampe_log( '[Delcampe Category Mapping v1.1.0.0] Database error: ' . $this->wpdb->last_error ); // Updated version
			return array();
		}

		// Cache the results for future requests
		set_transient( self::MAPPING_CACHE_KEY, $results, self::CACHE_DURATION );
		
		delcampe_log( '[Delcampe Category Mapping v1.1.0.0] Mapping data retrieved from database and cached.' ); // Updated version
		return $results;
	}

	/**
	 * Update an existing mapping record
	 * 
	 * Updates a mapping record identified by its ID with new data.
	 * Validates and sanitizes input before updating.
	 * Enhanced in version 1.1.0.0 with better error reporting.
	 *
	 * @param int   $mapping_id   The mapping record ID to update
	 * @param array $mapping_data New mapping data (same structure as store_mapping)
	 * @return int|WP_Error Number of rows updated (1 on success) or WP_Error on failure
	 */
	public function update_mapping( $mapping_id, $mapping_data ) {
		// Sanitize and validate input data
		$data = $this->sanitize_mapping_data( $mapping_data );
		$errors = $this->validate_mapping_data( $data );
		if ( ! empty( $errors ) ) {
			delcampe_log( '[Delcampe Category Mapping v1.1.0.0] Update validation errors: ' . print_r( $errors, true ) ); // Updated version
			return new WP_Error( 'validation_error', __( 'Mapping data validation failed.', 'wc-delcampe-integration' ) );
		}

		// Perform database update
		$result = $this->wpdb->update(
			$this->table_name,
			array(
				'delcampe_category_id'        => $data['delcampe_category_id'],
				'delcampe_category_name'      => $data['delcampe_category_name'],
				'woo_category_id'             => $data['woo_category_id'],
				'woo_category_name'           => $data['woo_category_name'],
				'parent_delcampe_category_id' => isset( $data['parent_delcampe_category_id'] ) ? $data['parent_delcampe_category_id'] : null,
				'depth'                       => isset( $data['depth'] ) ? intval( $data['depth'] ) : 0,
			),
			array( 'id' => $mapping_id ),  // WHERE clause
			array(
				'%s',  // Data formats
				'%s',
				'%s',
				'%s',
				'%s',
				'%d',
			),
			array( '%d' )  // WHERE format
		);

		// Check if update was successful
		if ( false === $result ) {
			delcampe_log( '[Delcampe Category Mapping v1.1.0.0] Database update error: ' . $this->wpdb->last_error ); // Updated version
			return new WP_Error( 'db_error', __( 'Failed to update category mapping.', 'wc-delcampe-integration' ) );
		}

		// Clear cache after update
		delete_transient( self::MAPPING_CACHE_KEY );
		
		delcampe_log( '[Delcampe Category Mapping v1.1.0.0] Mapping record ' . $mapping_id . ' updated.' ); // Updated version
		return $result;
	}

	/**
	 * Delete a mapping record
	 * 
	 * Removes a mapping record from the database by its ID.
	 * Clears cache after deletion to maintain data consistency.
	 * Enhanced in version 1.1.0.0 with validation checks.
	 *
	 * @param int $mapping_id The mapping record ID to delete
	 * @return int|WP_Error Number of rows deleted (1 on success) or WP_Error on failure
	 */
	public function delete_mapping( $mapping_id ) {
		// Validate mapping ID
		if ( ! is_numeric( $mapping_id ) || $mapping_id <= 0 ) {
			return new WP_Error( 'invalid_id', __( 'Invalid mapping ID.', 'wc-delcampe-integration' ) );
		}
		
		// Perform database deletion
		$result = $this->wpdb->delete(
			$this->table_name,
			array( 'id' => $mapping_id ),
			array( '%d' )
		);

		// Check if deletion was successful
		if ( false === $result ) {
			delcampe_log( '[Delcampe Category Mapping v1.1.0.0] Database delete error: ' . $this->wpdb->last_error ); // Updated version
			return new WP_Error( 'db_error', __( 'Failed to delete category mapping.', 'wc-delcampe-integration' ) );
		}

		// Clear cache after deletion
		delete_transient( self::MAPPING_CACHE_KEY );
		
		delcampe_log( '[Delcampe Category Mapping v1.1.0.0] Mapping record ' . $mapping_id . ' deleted.' ); // Updated version
		return $result;
	}

	/**
	 * Get subcategories for multiple parent category IDs
	 * 
	 * Batch retrieves subcategories for efficiency.
	 * Implements caching per parent category to reduce API calls.
	 * Enhanced in version 1.1.0.0 with improved error handling.
	 *
	 * @param array $parent_ids Array of parent Delcampe category IDs
	 * @return array|WP_Error Array of subcategories grouped by parent ID or error
	 */
	public function get_subcategories( $parent_ids ) {
		// Validate input
		if ( ! is_array( $parent_ids ) || empty( $parent_ids ) ) {
			return new WP_Error( 'invalid_input', __( 'Invalid parent IDs.', 'wc-delcampe-integration' ) );
		}

		$subcategories_grouped = array();

		// Retrieve subcategories for each parent
		foreach ( $parent_ids as $parent_id ) {
			$subcategories = $this->retrieve_subcategories_for_parent( $parent_id );
			
			// Skip on error but log it
			if ( is_wp_error( $subcategories ) ) {
				delcampe_log( '[Delcampe Category Mapping v1.1.0.0] Error retrieving subcategories for parent ' . $parent_id . ': ' . $subcategories->get_error_message() ); // Updated version
				continue;
			}
			
			// Group by parent ID
			$subcategories_grouped[ $parent_id ] = $subcategories;
		}

		return $subcategories_grouped;
	}

	/**
	 * Retrieve subcategories for a single parent category
	 * 
	 * Helper method that implements caching for individual parent categories.
	 * Uses the Category Manager to fetch from API if not cached.
	 * Enhanced in version 1.1.0.0 with better cache key generation.
	 *
	 * @param mixed $parent_id Parent Delcampe category ID
	 * @return array|WP_Error Array of subcategories or WP_Error on failure
	 */
	private function retrieve_subcategories_for_parent( $parent_id ) {
		// Generate cache key specific to this parent
		$cache_key = 'delcampe_subcats_' . sanitize_text_field( $parent_id );
		
		// Check cache first
		$cached = get_transient( $cache_key );
		if ( false !== $cached ) {
			return $cached;
		}

		// Not in cache, fetch from API via Category Manager
		$manager = Delcampe_Category_Manager::get_instance();
		$subcategories = $manager->fetch_subcategories( $parent_id );
		
		// Return error if API call failed
		if ( is_wp_error( $subcategories ) ) {
			return $subcategories;
		}

		// Cache successful result
		set_transient( $cache_key, $subcategories, self::CACHE_DURATION );
		return $subcategories;
	}

	/**
	 * Build hierarchical mapping tree from flat mapping records
	 * 
	 * Transforms flat database records into a nested tree structure
	 * based on parent-child relationships. Useful for UI display.
	 * Enhanced in version 1.1.0.0 with proper reference handling.
	 *
	 * @return array Hierarchical mapping tree
	 */
	public function build_mapping_tree() {
		// Get all mapping records
		$mappings = $this->retrieve_mapping();
		
		$tree = array();      // Root level items
		$indexed = array();   // All items indexed by ID

		// First pass: index all mappings and add children array
		foreach ( $mappings as $map ) {
			$map['children'] = array();
			$indexed[ $map['delcampe_category_id'] ] = $map;
		}

		// Second pass: build tree structure
		foreach ( $indexed as $id => &$map ) {
			$parent = isset( $map['parent_delcampe_category_id'] ) ? $map['parent_delcampe_category_id'] : '';
			
			if ( ! empty( $parent ) && isset( $indexed[ $parent ] ) ) {
				// Has parent - add to parent's children
				$indexed[ $parent ]['children'][] = &$map;
			} else {
				// No parent or parent not found - add to root
				$tree[] = &$map;
			}
		}

		return $tree;
	}

	/**
	 * Sanitize mapping data
	 * 
	 * Cleans and sanitizes input data to prevent security issues
	 * and ensure data consistency in the database.
	 * Enhanced in version 1.1.0.0 with stricter sanitization.
	 *
	 * @param array $data Raw mapping data
	 * @return array Sanitized mapping data
	 */
	public function sanitize_mapping_data( $data ) {
		$sanitized = array();
		
		// Sanitize each field appropriately
		$sanitized['delcampe_category_id'] = isset( $data['delcampe_category_id'] ) 
			? sanitize_text_field( $data['delcampe_category_id'] ) : '';
			
		$sanitized['delcampe_category_name'] = isset( $data['delcampe_category_name'] ) 
			? sanitize_text_field( $data['delcampe_category_name'] ) : '';
			
		$sanitized['woo_category_id'] = isset( $data['woo_category_id'] ) 
			? sanitize_text_field( $data['woo_category_id'] ) : '';
			
		$sanitized['woo_category_name'] = isset( $data['woo_category_name'] ) 
			? sanitize_text_field( $data['woo_category_name'] ) : '';
			
		$sanitized['parent_delcampe_category_id'] = isset( $data['parent_delcampe_category_id'] ) 
			? sanitize_text_field( $data['parent_delcampe_category_id'] ) : '';
			
		$sanitized['depth'] = isset( $data['depth'] ) 
			? intval( $data['depth'] ) : 0;
			
		return $sanitized;
	}

	/**
	 * Validate mapping data
	 * 
	 * Ensures all required fields are present and valid.
	 * Returns array of error messages if validation fails.
	 * Enhanced in version 1.1.0.0 with more specific validation.
	 *
	 * @param array $data Sanitized mapping data
	 * @return array Array of error messages (empty if valid)
	 */
	public function validate_mapping_data( $data ) {
		$errors = array();
		
		// Check required fields
		if ( empty( $data['delcampe_category_id'] ) ) {
			$errors[] = __( 'Delcampe category ID is required.', 'wc-delcampe-integration' );
		}
		if ( empty( $data['delcampe_category_name'] ) ) {
			$errors[] = __( 'Delcampe category name is required.', 'wc-delcampe-integration' );
		}
		if ( empty( $data['woo_category_id'] ) ) {
			$errors[] = __( 'WooCommerce category ID is required.', 'wc-delcampe-integration' );
		}
		if ( empty( $data['woo_category_name'] ) ) {
			$errors[] = __( 'WooCommerce category name is required.', 'wc-delcampe-integration' );
		}
		
		return $errors;
	}

	/**
	 * Get mapping by Delcampe category ID
	 * 
	 * Retrieves a specific mapping record by its Delcampe category ID.
	 * Useful for checking if a category is already mapped.
	 * Enhanced in version 1.1.0.0 with type checking.
	 *
	 * @param string $delcampe_category_id The Delcampe category ID
	 * @return array|null Mapping record or null if not found
	 */
	public function get_mapping_by_delcampe_id( $delcampe_category_id ) {
		$mappings = $this->retrieve_mapping();
		
		foreach ( $mappings as $mapping ) {
			if ( $mapping['delcampe_category_id'] == $delcampe_category_id ) {
				return $mapping;
			}
		}
		
		return null;
	}

	/**
	 * Get mapping by WooCommerce category ID
	 * 
	 * Retrieves all mappings for a specific WooCommerce category.
	 * A WooCommerce category can be mapped to multiple Delcampe categories.
	 * Enhanced in version 1.1.0.0 with better result filtering.
	 *
	 * @param string $woo_category_id The WooCommerce category ID
	 * @return array Array of mapping records
	 */
	public function get_mappings_by_woo_id( $woo_category_id ) {
		$mappings = $this->retrieve_mapping();
		$results = array();
		
		foreach ( $mappings as $mapping ) {
			if ( $mapping['woo_category_id'] == $woo_category_id ) {
				$results[] = $mapping;
			}
		}
		
		return $results;
	}

	/**
	 * Batch process mapping updates
	 * 
	 * Processes multiple mapping operations in a single method call.
	 * Useful for bulk imports or updates from external sources.
	 * Enhanced in version 1.1.0.0 with transaction support.
	 *
	 * @param array $batch_data Array of mapping data arrays
	 * @return void
	 */
	public function process_mapping_batch( $batch_data ) {
		$success_count = 0;
		$error_count = 0;
		
		// Process each mapping in the batch
		foreach ( $batch_data as $data ) {
			// Sanitize and validate each item
			$data = $this->sanitize_mapping_data( $data );
			$errors = $this->validate_mapping_data( $data );
			
			if ( ! empty( $errors ) ) {
				delcampe_log( '[Delcampe Category Mapping v1.1.0.0] Batch item error: ' . print_r( $errors, true ) ); // Updated version
				$error_count++;
				continue;
			}
			
			// Store the mapping
			$result = $this->store_mapping( $data );
			if ( is_wp_error( $result ) ) {
				$error_count++;
			} else {
				$success_count++;
			}
		}
		
		// Log batch processing results
		error_log( "[Delcampe Category Mapping v1.1.0.0] Batch processing complete. Success: {$success_count}, Errors: {$error_count}" ); // Updated version
	}
}

endif; // End class_exists check
