<?php
/**
 * Delcampe WP-CLI Commands
 */

if (!defined('ABSPATH')) exit;

if (defined('WP_CLI') && WP_CLI) {
    class Delcampe_CLI_Command {
        /**
         * Process the sync queue now.
         *
         * ## EXAMPLES
         *   wp delcampe sync process-queue
         */
        public function process_queue($args, $assoc_args) {
            $handler = Delcampe_Sync_Handler::get_instance();
            $result = $handler->process_queue();
            if (!empty($result['failed'])) {
                \WP_CLI::warning(sprintf('Processed %d; %d failed', $result['processed'], $result['failed']));
            }
            \WP_CLI::success($result['message'] ?? 'Queue processed');
        }

        /**
         * Import existing listings (for products with sync status but no listing record).
         *
         * ## EXAMPLES
         *   wp delcampe sync import-existing
         */
        public function import_existing($args, $assoc_args) {
            $handler = Delcampe_Sync_Handler::get_instance();
            $results = $handler->import_existing_listings();
            \WP_CLI::success(sprintf('Imported %d, Skipped %d, Errors %d', $results['imported'], $results['skipped'], $results['errors']));
        }

        /**
         * Clear update caches used by the plugin updater.
         *
         * ## EXAMPLES
         *   wp delcampe updates clear-cache
         */
        public function clear_update_cache($args, $assoc_args) {
            delete_site_transient('update_plugins');
            delete_transient('delcampe_update_info');
            delete_transient('puc_update_check-delcampe-sync');
            delete_option('_site_transient_update_plugins');
            wp_cache_delete('plugins', 'transient');
            \WP_CLI::success('Update cache cleared');
        }

        /**
         * Check current webhook registrations.
         *
         * ## EXAMPLES
         *   wp delcampe webhook check
         */
        public function webhook_check($args, $assoc_args) {
            $mgr = Delcampe_Webhook_Registration::get_instance();
            $hooks = $mgr->get_current_webhooks();
            if (is_wp_error($hooks)) {
                \WP_CLI::error($hooks->get_error_message());
            }
            \WP_CLI\Utils\format_items('table', $hooks, array('id','type','destination','active'));
        }

        /**
         * Cleanup duplicate webhook registrations.
         *
         * ## EXAMPLES
         *   wp delcampe webhook cleanup
         */
        public function webhook_cleanup($args, $assoc_args) {
            $mgr = Delcampe_Webhook_Registration::get_instance();
            $res = $mgr->cleanup_duplicate_notifications();
            if (is_wp_error($res)) {
                \WP_CLI::error($res->get_error_message());
            }
            \WP_CLI::success(sprintf('Deleted %d duplicate(s)', count($res['deleted'] ?? array())));
        }

        /**
         * Requeue stuck listings (processing with no Delcampe ID) and run the queue.
         *
         * ## OPTIONS
         * [--minutes=<minutes>]
         * : Consider listings older than this many minutes as stuck. Default 10.
         *
         * ## EXAMPLES
         *   wp delcampe sync requeue-stuck --minutes=10
         */
        public function requeue_stuck($args, $assoc_args) {
            global $wpdb;
            $minutes = isset($assoc_args['minutes']) ? intval($assoc_args['minutes']) : 10;
            $cutoff = gmdate('Y-m-d H:i:s', time() - ($minutes * 60));

            $table = $wpdb->prefix . 'delcampe_listings';
            $rows = $wpdb->get_results($wpdb->prepare(
                "SELECT id, product_id, profile_id, delcampe_id, status, date_created, date_updated FROM {$table}
                 WHERE status IN ('processing','published')
                   AND (delcampe_id IS NULL OR delcampe_id = '')
                   AND IFNULL(date_updated, date_created) < %s",
                $cutoff
            ));

            if (empty($rows)) {
                \WP_CLI::success('No stuck listings found.');
                return;
            }

            $handler = Delcampe_Sync_Handler::get_instance();
            $requeued = 0;
            foreach ($rows as $row) {
                // Reset status locally
                $wpdb->update($table, array(
                    'status' => 'prepared',
                    'date_updated' => current_time('mysql'),
                ), array('id' => $row->id));

                // Add to queue for creation
                if (!empty($row->product_id) && !empty($row->profile_id)) {
                    $ok = $handler->add_to_queue((int)$row->product_id, (int)$row->profile_id, 'create');
                    if ($ok) { $requeued++; }
                }
            }

            \WP_CLI::success(sprintf('Requeued %d stuck listing(s). Processing queue...', $requeued));
            $res = $handler->process_queue(0);
            \WP_CLI::log(sprintf('Processed %d (success %d / failed %d)', $res['processed'], $res['success'], $res['failed']));
        }
    }

    \WP_CLI::add_command('delcampe sync process-queue', array('Delcampe_CLI_Command', 'process_queue'));
    \WP_CLI::add_command('delcampe sync import-existing', array('Delcampe_CLI_Command', 'import_existing'));
    \WP_CLI::add_command('delcampe sync requeue-stuck', array('Delcampe_CLI_Command', 'requeue_stuck'));
    \WP_CLI::add_command('delcampe updates clear-cache', array('Delcampe_CLI_Command', 'clear_update_cache'));
    \WP_CLI::add_command('delcampe webhook check', array('Delcampe_CLI_Command', 'webhook_check'));
    \WP_CLI::add_command('delcampe webhook cleanup', array('Delcampe_CLI_Command', 'webhook_cleanup'));
}
