<?php
/**
 * Delcampe Encryption Utility
 * 
 * Provides secure encryption/decryption for sensitive data like license keys
 * Uses WordPress salts and OpenSSL for cryptographically secure encryption
 * 
 * @package     WooCommerce_Delcampe_Integration
 * @subpackage  Security
 * @since       1.10.22.0
 * @version     1.0.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class Delcampe_Encryption
 * 
 * Handles encryption and decryption of sensitive data
 * 
 * @since 1.10.22.0
 */
class Delcampe_Encryption {
    
    /**
     * Singleton instance
     * @var Delcampe_Encryption|null
     */
    private static $instance = null;
    
    /**
     * Encryption method
     * @var string
     */
    const CIPHER_METHOD = 'AES-256-CBC';
    
    /**
     * Key derivation iterations
     * @var int
     */
    const KEY_ITERATIONS = 10000;
    
    /**
     * Get singleton instance
     * 
     * @return Delcampe_Encryption
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Private constructor
     */
    private function __construct() {
        // Check if OpenSSL is available
        if (!extension_loaded('openssl')) {
            error_log('[Delcampe Encryption] OpenSSL extension is not available. Encryption disabled.');
        }
    }
    
    /**
     * Get encryption key derived from WordPress salts
     * 
     * @return string
     */
    private function get_encryption_key() {
        // Use WordPress authentication salts for key generation
        $salt = '';
        
        // Combine multiple WordPress salts for stronger key material
        if (defined('AUTH_KEY')) {
            $salt .= AUTH_KEY;
        }
        if (defined('SECURE_AUTH_KEY')) {
            $salt .= SECURE_AUTH_KEY;
        }
        if (defined('LOGGED_IN_KEY')) {
            $salt .= LOGGED_IN_KEY;
        }
        if (defined('NONCE_KEY')) {
            $salt .= NONCE_KEY;
        }
        
        // Fallback if no salts are defined (should never happen in production)
        if (empty($salt)) {
            $salt = get_site_url() . 'delcampe-sync-default-salt-2025';
        }
        
        // Add plugin-specific salt component
        $salt .= 'delcampe-license-encryption-v1';
        
        // Derive a proper encryption key using PBKDF2
        $key = hash_pbkdf2(
            'sha256',
            $salt,
            'delcampe-sync-license',
            self::KEY_ITERATIONS,
            32,
            true
        );
        
        return $key;
    }
    
    /**
     * Encrypt data
     * 
     * @param string $data Plain text data to encrypt
     * @return string|false Encrypted data in base64 format or false on failure
     */
    public function encrypt($data) {
        // Check if encryption is available
        if (!extension_loaded('openssl')) {
            return $data; // Return unencrypted if OpenSSL not available
        }
        
        // Don't encrypt empty data
        if (empty($data)) {
            return $data;
        }
        
        // Check if already encrypted (starts with 'enc:')
        if (strpos($data, 'enc:') === 0) {
            return $data; // Already encrypted
        }
        
        try {
            // Get encryption key
            $key = $this->get_encryption_key();
            
            // Generate random IV
            $iv_length = openssl_cipher_iv_length(self::CIPHER_METHOD);
            $iv = openssl_random_pseudo_bytes($iv_length);
            
            // Encrypt the data
            $encrypted = openssl_encrypt(
                $data,
                self::CIPHER_METHOD,
                $key,
                OPENSSL_RAW_DATA,
                $iv
            );
            
            if ($encrypted === false) {
                error_log('[Delcampe Encryption] Encryption failed: ' . openssl_error_string());
                return false;
            }
            
            // Combine IV and encrypted data
            $combined = $iv . $encrypted;
            
            // Create HMAC for authentication
            $hmac = hash_hmac('sha256', $combined, $key, true);
            
            // Combine HMAC and encrypted data, encode and prefix
            $final = 'enc:' . base64_encode($hmac . $combined);
            
            return $final;
            
        } catch (Exception $e) {
            error_log('[Delcampe Encryption] Encryption error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Decrypt data
     * 
     * @param string $encrypted_data Encrypted data in base64 format
     * @return string|false Decrypted plain text or false on failure
     */
    public function decrypt($encrypted_data) {
        // Check if encryption is available
        if (!extension_loaded('openssl')) {
            return $encrypted_data; // Return as-is if OpenSSL not available
        }
        
        // Check if data is encrypted (starts with 'enc:')
        if (strpos($encrypted_data, 'enc:') !== 0) {
            return $encrypted_data; // Not encrypted, return as-is
        }
        
        try {
            // Remove prefix and decode
            $encrypted_data = substr($encrypted_data, 4); // Remove 'enc:'
            $decoded = base64_decode($encrypted_data);
            
            if ($decoded === false) {
                error_log('[Delcampe Encryption] Base64 decode failed');
                return false;
            }
            
            // Get encryption key
            $key = $this->get_encryption_key();
            
            // Extract HMAC (first 32 bytes)
            $hmac = substr($decoded, 0, 32);
            $combined = substr($decoded, 32);
            
            // Verify HMAC
            $calculated_hmac = hash_hmac('sha256', $combined, $key, true);
            if (!hash_equals($hmac, $calculated_hmac)) {
                error_log('[Delcampe Encryption] HMAC verification failed - data may be tampered');
                return false;
            }
            
            // Extract IV
            $iv_length = openssl_cipher_iv_length(self::CIPHER_METHOD);
            $iv = substr($combined, 0, $iv_length);
            $encrypted = substr($combined, $iv_length);
            
            // Decrypt the data
            $decrypted = openssl_decrypt(
                $encrypted,
                self::CIPHER_METHOD,
                $key,
                OPENSSL_RAW_DATA,
                $iv
            );
            
            if ($decrypted === false) {
                error_log('[Delcampe Encryption] Decryption failed: ' . openssl_error_string());
                return false;
            }
            
            return $decrypted;
            
        } catch (Exception $e) {
            error_log('[Delcampe Encryption] Decryption error: ' . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Check if a value is encrypted
     * 
     * @param string $value Value to check
     * @return bool True if encrypted, false otherwise
     */
    public function is_encrypted($value) {
        return strpos($value, 'enc:') === 0;
    }
    
    /**
     * Migrate a plain text value to encrypted format
     * 
     * @param string $value Value to migrate
     * @return string Encrypted value or original if already encrypted
     */
    public function migrate_to_encrypted($value) {
        if (empty($value)) {
            return $value;
        }
        
        // Check if already encrypted
        if ($this->is_encrypted($value)) {
            return $value;
        }
        
        // Encrypt the value
        $encrypted = $this->encrypt($value);
        
        // Return encrypted value or original on failure
        return ($encrypted !== false) ? $encrypted : $value;
    }
    
    /**
     * Test encryption/decryption functionality
     * 
     * @return array Test results
     */
    public function test_encryption() {
        $results = array(
            'openssl_available' => extension_loaded('openssl'),
            'encryption_test' => false,
            'decryption_test' => false,
            'hmac_test' => false
        );
        
        if (!$results['openssl_available']) {
            return $results;
        }
        
        // Test encryption
        $test_data = 'test-license-key-12345';
        $encrypted = $this->encrypt($test_data);
        $results['encryption_test'] = ($encrypted !== false && $encrypted !== $test_data);
        
        if ($results['encryption_test']) {
            // Test decryption
            $decrypted = $this->decrypt($encrypted);
            $results['decryption_test'] = ($decrypted === $test_data);
            
            // Test HMAC by modifying encrypted data
            $tampered = $encrypted . 'x';
            $tampered_result = $this->decrypt($tampered);
            $results['hmac_test'] = ($tampered_result === false);
        }
        
        return $results;
    }
}