<?php
/**
 * Delcampe Inventory Admin
 * 
 * Admin interface for inventory management and monitoring
 * 
 * @package     WooCommerce_Delcampe_Integration
 * @subpackage  Admin
 * @since       1.5.0.0
 * @version     1.6.2.1
 * 
 * @author      Frank Kahle
 * @copyright   2024 Frank Kahle
 * @license     Proprietary
 * 
 * Changelog:
 * 1.6.2.1 - Fixed duplicate menu registration and missing render_page() method
 *         - Removed duplicate submenu registration that was causing two inventory menu items
 *         - Added render_page() method for compatibility with admin menu class
 * 1.5.0.0 - Initial implementation of inventory management interface
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class Delcampe_Inventory_Admin
 * 
 * Handles the admin interface for inventory management
 * 
 * @since   1.5.0.0
 * @version 1.6.2.1
 */
class Delcampe_Inventory_Admin {
    
    /**
     * Singleton instance
     * @var Delcampe_Inventory_Admin|null
     */
    private static $instance = null;
    
    /**
     * Inventory manager instance
     * @var Delcampe_Inventory_Manager
     */
    private $inventory_manager;
    
    /**
     * Get singleton instance
     * 
     * @since  1.5.0.0
     * @return Delcampe_Inventory_Admin
     */
    public static function get_instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     * 
     * @since  1.5.0.0
     */
    private function __construct() {
        // Get inventory manager instance
        $this->inventory_manager = Delcampe_Inventory_Manager::get_instance();
        
        // Initialize hooks
        $this->init_hooks();
    }
    
    /**
     * Initialize hooks
     * 
     * @since  1.5.0.0
     * @version 1.6.2.1 - Removed duplicate submenu registration
     */
    private function init_hooks() {
        // Add inventory tab to admin interface
        add_filter( 'delcampe_admin_tabs', array( $this, 'add_inventory_tab' ) );
        add_action( 'delcampe_admin_tab_content_inventory', array( $this, 'render_inventory_tab' ) );

        // (Removed) Settings page Queue tab – Queue is shown under Inventory page instead
        
        // AJAX handlers
        add_action( 'wp_ajax_delcampe_check_inventory', array( $this, 'ajax_check_inventory' ) );
        add_action( 'wp_ajax_delcampe_sync_inventory', array( $this, 'ajax_sync_inventory' ) );
        add_action( 'wp_ajax_delcampe_resolve_alert', array( $this, 'ajax_resolve_alert' ) );
        add_action( 'wp_ajax_delcampe_bulk_inventory_update', array( $this, 'ajax_bulk_inventory_update' ) );
        add_action( 'wp_ajax_delcampe_run_stock_sync_batch', array( $this, 'ajax_run_stock_sync_batch' ) );
        add_action( 'wp_ajax_delcampe_cleanup_orphaned_ids', array( $this, 'ajax_cleanup_orphaned_ids' ) );
        add_action( 'wp_ajax_delcampe_refresh_stock_cache', array( $this, 'ajax_refresh_stock_cache' ) );
        // Integrity tools endpoints
        add_action( 'wp_ajax_delcampe_inventory_diagnose', array( $this, 'ajax_inventory_diagnose' ) );
        add_action( 'wp_ajax_delcampe_inventory_fix_duplicates', array( $this, 'ajax_inventory_fix_duplicates' ) );
        add_action( 'wp_ajax_delcampe_inventory_backfill_missing', array( $this, 'ajax_inventory_backfill_missing' ) );
        add_action( 'wp_ajax_delcampe_inventory_add_constraints', array( $this, 'ajax_inventory_add_constraints' ) );
        add_action( 'wp_ajax_delcampe_inventory_reconcile_verified', array( $this, 'ajax_inventory_reconcile_verified' ) );
        
        // REMOVED in v1.6.2.1: Duplicate submenu registration - this is handled by main admin menu class
        // add_action('admin_menu', array($this, 'add_inventory_submenu'), 20);
        
        // Enqueue scripts and styles
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_assets' ) );
    }
    
    /**
     * Add inventory tab to admin tabs
     * 
     * @since  1.5.0.0
     * @param  array $tabs Existing tabs
     * @return array Modified tabs
     */
    public function add_inventory_tab( $tabs ) {
        $tabs['inventory'] = __( 'Inventory', 'wc-delcampe-integration' );
        return $tabs;
    }

    // (Removed) add_queue_tab – Queue now lives under Inventory page
    
    /**
     * Render page method for admin menu compatibility
     * 
     * This method is called by the admin menu class.
     * It delegates to the actual render method.
     * 
     * @since  1.6.2.1
     * @return void
     */
    public function render_page() {
        $this->render_inventory_page();
    }
    
    /**
     * Render inventory tab content
     * 
     * @since  1.5.0.0
     */
    public function render_inventory_tab() {
        // Get statistics
        $stats = $this->inventory_manager->get_inventory_statistics();
        
        ?>
        <div class="delcampe-inventory-settings">
            <h2><?php _e('Inventory Management', 'wc-delcampe-integration'); ?></h2>
            
            <!-- Statistics Dashboard -->
            <div class="inventory-stats-dashboard">
                <div class="stats-grid">
                    <div class="stat-box">
                        <h3><?php _e('Synced Products', 'wc-delcampe-integration'); ?></h3>
                        <p class="stat-number"><?php echo esc_html($stats['total_synced']); ?></p>
                    </div>
                    <div class="stat-box">
                        <h3><?php _e('Recent Syncs (24h)', 'wc-delcampe-integration'); ?></h3>
                        <p class="stat-number"><?php echo esc_html($stats['recent_syncs']); ?></p>
                    </div>
                    <div class="stat-box <?php echo $stats['failed_syncs'] > 0 ? 'has-errors' : ''; ?>">
                        <h3><?php _e('Failed Syncs (7d)', 'wc-delcampe-integration'); ?></h3>
                        <p class="stat-number"><?php echo esc_html($stats['failed_syncs']); ?></p>
                    </div>
                    <div class="stat-box <?php echo $stats['active_alerts'] > 0 ? 'has-alerts' : ''; ?>">
                        <h3><?php _e('Active Alerts', 'wc-delcampe-integration'); ?></h3>
                        <p class="stat-number"><?php echo esc_html($stats['active_alerts']); ?></p>
                    </div>
                </div>
            </div>
            
            <!-- Settings Form -->
            <form method="post" action="options.php">
                <?php settings_fields('delcampe_inventory_settings'); ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="delcampe_auto_sync_inventory">
                                <?php _e('Auto-sync Inventory', 'wc-delcampe-integration'); ?>
                            </label>
                        </th>
                        <td>
                            <input type="checkbox" 
                                   name="delcampe_auto_sync_inventory" 
                                   id="delcampe_auto_sync_inventory" 
                                   value="yes" 
                                   <?php checked(get_option('delcampe_auto_sync_inventory', 'yes'), 'yes'); ?> />
                            <label for="delcampe_auto_sync_inventory">
                                <?php _e('Automatically sync inventory when discrepancies are found', 'wc-delcampe-integration'); ?>
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="delcampe_inventory_check_frequency">
                                <?php _e('Inventory Check Frequency', 'wc-delcampe-integration'); ?>
                            </label>
                        </th>
                        <td>
                            <select name="delcampe_inventory_check_frequency" id="delcampe_inventory_check_frequency">
                                <option value="hourly" <?php selected(get_option('delcampe_inventory_check_frequency', 'hourly'), 'hourly'); ?>>
                                    <?php _e('Hourly', 'wc-delcampe-integration'); ?>
                                </option>
                                <option value="twicedaily" <?php selected(get_option('delcampe_inventory_check_frequency'), 'twicedaily'); ?>>
                                    <?php _e('Twice Daily', 'wc-delcampe-integration'); ?>
                                </option>
                                <option value="daily" <?php selected(get_option('delcampe_inventory_check_frequency'), 'daily'); ?>>
                                    <?php _e('Daily', 'wc-delcampe-integration'); ?>
                                </option>
                            </select>
                            <p class="description">
                                <?php _e('How often to check for inventory discrepancies between WooCommerce and Delcampe', 'wc-delcampe-integration'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="delcampe_low_stock_alert">
                                <?php _e('Low Stock Alert Threshold', 'wc-delcampe-integration'); ?>
                            </label>
                        </th>
                        <td>
                            <input type="number" 
                                   name="delcampe_low_stock_alert" 
                                   id="delcampe_low_stock_alert" 
                                   value="<?php echo esc_attr(get_option('delcampe_low_stock_alert', '5')); ?>" 
                                   min="0" />
                            <p class="description">
                                <?php _e('Alert when Delcampe product stock falls below this number', 'wc-delcampe-integration'); ?>
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <label for="delcampe_prevent_oversell">
                                <?php _e('Prevent Overselling', 'wc-delcampe-integration'); ?>
                            </label>
                        </th>
                        <td>
                            <input type="checkbox" 
                                   name="delcampe_prevent_oversell" 
                                   id="delcampe_prevent_oversell" 
                                   value="yes" 
                                   <?php checked(get_option('delcampe_prevent_oversell', 'yes'), 'yes'); ?> />
                            <label for="delcampe_prevent_oversell">
                                <?php _e('Reserve stock across channels to prevent overselling', 'wc-delcampe-integration'); ?>
                            </label>
                        </td>
                    </tr>
                </table>
                
                <?php submit_button(); ?>
            </form>
            
            <!-- Quick Actions -->
            <div class="inventory-quick-actions">
                <h3><?php _e('Quick Actions', 'wc-delcampe-integration'); ?></h3>
                <p>
                    <button type="button" class="button button-primary" id="run-inventory-check">
                        <?php _e('Run Inventory Check Now', 'wc-delcampe-integration'); ?>
                    </button>
                    <button type="button" class="button" id="sync-all-inventory">
                        <?php _e('Sync All Inventory to Delcampe', 'wc-delcampe-integration'); ?>
                    </button>
                    <button type="button" class="button" id="cleanup-orphaned-ids">
                        <?php _e('Clean Orphaned Delcampe IDs', 'wc-delcampe-integration'); ?>
                    </button>
                    <a href="<?php echo admin_url('admin.php?page=delcampe-inventory'); ?>" class="button">
                        <?php _e('View Detailed Inventory Report', 'wc-delcampe-integration'); ?>
                    </a>
                </p>
                <div id="cleanup-results" style="display:none; margin-top:10px; padding:10px; background:#f0f0f0; border:1px solid #ddd;"></div>
            </div>

            <!-- Integrity Tools -->
            <div class="inventory-quick-actions" style="margin-top:20px;">
                <h3><?php _e('Inventory Integrity Tools', 'wc-delcampe-integration'); ?></h3>
                <p class="description"><?php _e('Diagnose and fix duplicate mappings and backfill missing listings using Delcampe data.', 'wc-delcampe-integration'); ?></p>
                <p>
                    <button type="button" class="button" id="delcampe-diagnose"><?php _e('Run Diagnostics', 'wc-delcampe-integration'); ?></button>
                    <button type="button" class="button" id="delcampe-fix-duplicates-dry"><?php _e('Fix Duplicates (Dry Run)', 'wc-delcampe-integration'); ?></button>
                    <button type="button" class="button button-primary" id="delcampe-fix-duplicates-apply"><?php _e('Fix Duplicates (Apply)', 'wc-delcampe-integration'); ?></button>
                    <button type="button" class="button" id="delcampe-backfill-dry"><?php _e('Backfill Missing (Dry Run)', 'wc-delcampe-integration'); ?></button>
                    <button type="button" class="button button-primary" id="delcampe-backfill-apply"><?php _e('Backfill Missing (Apply)', 'wc-delcampe-integration'); ?></button>
                    <button type="button" class="button" id="delcampe-constraints-check"><?php _e('Check Constraints', 'wc-delcampe-integration'); ?></button>
                    <button type="button" class="button" id="delcampe-constraints-apply"><?php _e('Apply Constraints', 'wc-delcampe-integration'); ?></button>
                </p>
                <div id="delcampe-integrity-output-main" class="delcampe-integrity-output" style="background:#fff;border:1px solid #ccd0d4;padding:12px;max-height:420px;overflow:auto"></div>
            </div>

            <!-- Recent Alerts -->
            <?php if (!empty($stats['alerts_by_type'])): ?>
            <div class="inventory-recent-alerts">
                <h3><?php _e('Alert Summary', 'wc-delcampe-integration'); ?></h3>
                <ul>
                    <?php foreach ($stats['alerts_by_type'] as $type => $count): ?>
                    <li>
                        <strong><?php echo esc_html(ucfirst(str_replace('_', ' ', $type))); ?>:</strong>
                        <?php echo esc_html($count); ?> <?php _e('alerts', 'wc-delcampe-integration'); ?>
                    </li>
                    <?php endforeach; ?>
                </ul>
            </div>
            <?php endif; ?>
        </div>
        
        <style>
        .inventory-stats-dashboard {
            margin: 20px 0;
        }
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        .stat-box {
            background: #fff;
            border: 1px solid #ccd0d4;
            padding: 20px;
            text-align: center;
            border-radius: 4px;
        }
        .stat-box h3 {
            margin: 0 0 10px;
            font-size: 14px;
            color: #666;
        }
        .stat-box .stat-number {
            font-size: 32px;
            font-weight: bold;
            margin: 0;
            color: #23282d;
        }
        .stat-box.has-errors .stat-number {
            color: #dc3232;
        }
        .stat-box.has-alerts .stat-number {
            color: #f56e28;
        }
        .inventory-quick-actions {
            margin-top: 30px;
            padding: 20px;
            background: #f1f1f1;
            border-radius: 4px;
        }
        .inventory-recent-alerts {
            margin-top: 30px;
        }
        </style>
        
        <script>
        jQuery(document).ready(function($) {
            // Run inventory check
            $('#run-inventory-check').on('click', function() {
                var $button = $(this);
                $button.prop('disabled', true).text('<?php _e('Checking...', 'wc-delcampe-integration'); ?>');
                
                $.post(ajaxurl, {
                    action: 'delcampe_check_inventory',
                    _ajax_nonce: '<?php echo wp_create_nonce('delcampe_check_inventory'); ?>'
                }, function(response) {
                    if (response.success) {
                        alert('<?php _e('Inventory check completed successfully!', 'wc-delcampe-integration'); ?>');
                        location.reload();
                    } else {
                        alert('<?php _e('Error:', 'wc-delcampe-integration'); ?> ' + response.data);
                    }
                    $button.prop('disabled', false).text('<?php _e('Run Inventory Check Now', 'wc-delcampe-integration'); ?>');
                });
            });
            
            // Sync all inventory
            $('#sync-all-inventory').on('click', function() {
                if (!confirm('<?php _e('This will sync all product inventory to Delcampe. Continue?', 'wc-delcampe-integration'); ?>')) {
                    return;
                }
                
                var $button = $(this);
                $button.prop('disabled', true).text('<?php _e('Syncing...', 'wc-delcampe-integration'); ?>');
                
                $.post(ajaxurl, {
                    action: 'delcampe_sync_inventory',
                    sync_all: true,
                    _ajax_nonce: '<?php echo wp_create_nonce('delcampe_sync_inventory'); ?>'
                }, function(response) {
                    if (response.success) {
                        alert('<?php _e('Inventory sync completed!', 'wc-delcampe-integration'); ?>');
                        location.reload();
                    } else {
                        alert('<?php _e('Error:', 'wc-delcampe-integration'); ?> ' + response.data);
                    }
                    $button.prop('disabled', false).text('<?php _e('Sync All Inventory to Delcampe', 'wc-delcampe-integration'); ?>');
                });
            });
            
            // Clean up orphaned Delcampe IDs
            $('#cleanup-orphaned-ids').on('click', function() {
                if (!confirm('<?php _e('This will remove Delcampe IDs from products with ended or closed listings. Continue?', 'wc-delcampe-integration'); ?>')) {
                    return;
                }
                
                var $button = $(this);
                var $results = $('#cleanup-results');
                $button.prop('disabled', true).text('<?php _e('Cleaning...', 'wc-delcampe-integration'); ?>');
                $results.html('<?php _e('Checking listing statuses...', 'wc-delcampe-integration'); ?>').show();
                
                $.post(ajaxurl, {
                    action: 'delcampe_cleanup_orphaned_ids',
                    _ajax_nonce: '<?php echo wp_create_nonce('delcampe_cleanup_orphaned_ids'); ?>'
                }, function(response) {
                    if (response.success) {
                        var msg = '<?php _e('Cleanup completed!', 'wc-delcampe-integration'); ?>\n\n' + response.data.message;
                        
                        $results.html('<strong><?php _e('Results:', 'wc-delcampe-integration'); ?></strong><br>' + 
                            '<?php _e('Products analyzed:', 'wc-delcampe-integration'); ?> ' + response.data.analyzed + '<br>' +
                            '<?php _e('Orphaned IDs removed:', 'wc-delcampe-integration'); ?> ' + response.data.cleaned + '<br>' +
                            '<?php _e('Active listings preserved:', 'wc-delcampe-integration'); ?> ' + response.data.kept);
                        
                        if (response.data.cleaned > 0) {
                            $results.css('background', '#d4f4dd');
                        }
                    } else {
                        $results.html('<span style="color:red;"><?php _e('Error:', 'wc-delcampe-integration'); ?> ' + response.data + '</span>');
                    }
                    $button.prop('disabled', false).text('<?php _e('Remove IDs from Ended Listings', 'wc-delcampe-integration'); ?>');
                });
            });

            function getOut(el){
                var $ctx = $(el).closest('.inventory-quick-actions');
                var $out = $ctx.find('.delcampe-integrity-output');
                if ($out.length) return $out.first();
                return $('.delcampe-integrity-output').last();
            }
            function printOutput(obj, el) {
                var $out = getOut(el);
                $out.text(JSON.stringify(obj, null, 2));
            }
            function renderBackfillPreview(data, el) {
                var $out = getOut(el);
                if (!data || !data.proposals || !data.proposals.length) { printOutput(data, el); return; }
                var html = '<table class="widefat fixed striped"><thead><tr>'+
                    '<th>Delcampe ID</th><th>Ref (SKU)</th><th>Title</th><th>Proposed Product</th><th>Reason</th>'+
                    '</tr></thead><tbody>';
                data.proposals.forEach(function(p){
                    html += '<tr>'+
                        '<td>'+ (p.delcampe_id||'') +'</td>'+
                        '<td>'+ (p.personal_reference||'') +'</td>'+
                        '<td>'+ (p.title||'') +'</td>'+
                        '<td>'+ (p.proposed_product_id ? ('#'+p.proposed_product_id+' '+(p.proposed_product_title||'')) : '<em>None</em>') +'</td>'+
                        '<td>'+ (p.match_reason||'<em>n/a</em>') +'</td>'+
                        '</tr>';
                });
                html += '</tbody></table>';
                $out.html(html);
            }
            function ajaxPost(action, data, done, el) {
                data = data || {}; data.action = action; data.nonce = '<?php echo wp_create_nonce('delcampe_inventory_integrity'); ?>';
                return $.post(ajaxurl, data)
                    .done(function(resp){
                        if (resp && resp.success) {
                            if (typeof done === 'function') done(resp.data, el);
                        } else {
                            printOutput(resp, el);
                            if (typeof done === 'function') done(null, el);
                        }
                    })
                    .fail(function(xhr){
                        printOutput({error:'network', status:xhr.status}, el);
                        if (typeof done === 'function') done(null, el);
                    });
            }

            $('#delcampe-diagnose').on('click', function(){
                var $b=$(this); $b.prop('disabled', true).text('<?php _e('Diagnosing...', 'wc-delcampe-integration'); ?>');
                var req = ajaxPost('delcampe_inventory_diagnose', {}, function(data, el){ printOutput(data, el); }, this);
                if (req && req.always) req.always(function(){ $b.prop('disabled', false).text('<?php _e('Run Diagnostics', 'wc-delcampe-integration'); ?>'); });
            });
            $('#delcampe-fix-duplicates-dry').on('click', function(){
                $(this).prop('disabled', true).text('<?php _e('Dry running...', 'wc-delcampe-integration'); ?>');
                ajaxPost('delcampe_inventory_fix_duplicates', {dry_run:'1'}, function(data){ printOutput(data); });
                $(this).prop('disabled', false).text('<?php _e('Fix Duplicates (Dry Run)', 'wc-delcampe-integration'); ?>');
            });
            $('#delcampe-fix-duplicates-apply').on('click', function(){
                if (!confirm('<?php _e('Apply duplicate fixes now?', 'wc-delcampe-integration'); ?>')) return;
                $(this).prop('disabled', true).text('<?php _e('Applying...', 'wc-delcampe-integration'); ?>');
                ajaxPost('delcampe_inventory_fix_duplicates', {dry_run:'0'}, function(data){ printOutput(data); });
                $(this).prop('disabled', false).text('<?php _e('Fix Duplicates (Apply)', 'wc-delcampe-integration'); ?>');
            });
            $('#delcampe-backfill-dry').on('click', function(){
                $(this).prop('disabled', true).text('<?php _e('Dry running...', 'wc-delcampe-integration'); ?>');
                ajaxPost('delcampe_inventory_backfill_missing', {dry_run:'1'}, function(data){ printOutput(data); });
                $(this).prop('disabled', false).text('<?php _e('Backfill Missing (Dry Run)', 'wc-delcampe-integration'); ?>');
            });
            $('#delcampe-backfill-apply').on('click', function(){
                if (!confirm('<?php _e('Backfill missing listings now?', 'wc-delcampe-integration'); ?>')) return;
                $(this).prop('disabled', true).text('<?php _e('Applying...', 'wc-delcampe-integration'); ?>');
                ajaxPost('delcampe_inventory_backfill_missing', {dry_run:'0'}, function(data){ printOutput(data); });
                $(this).prop('disabled', false).text('<?php _e('Backfill Missing (Apply)', 'wc-delcampe-integration'); ?>');
            });
            $('#delcampe-constraints-check').on('click', function(){
                $(this).prop('disabled', true).text('<?php _e('Checking...', 'wc-delcampe-integration'); ?>');
                ajaxPost('delcampe_inventory_add_constraints', {check_only:'1'}, function(data){ printOutput(data); });
                $(this).prop('disabled', false).text('<?php _e('Check Constraints', 'wc-delcampe-integration'); ?>');
            });
            $('#delcampe-constraints-apply').on('click', function(){
                if (!confirm('<?php _e('Apply database constraints now?', 'wc-delcampe-integration'); ?>')) return;
                $(this).prop('disabled', true).text('<?php _e('Applying...', 'wc-delcampe-integration'); ?>');
                ajaxPost('delcampe_inventory_add_constraints', {check_only:'0'}, function(data){ printOutput(data); });
                $(this).prop('disabled', false).text('<?php _e('Apply Constraints', 'wc-delcampe-integration'); ?>');
            });
        });
        </script>
        <?php
    }

    // (Removed) integrity-only helper – not needed without local sub-tabs

    /**
     * Render Queue Management tab content by embedding the Queue monitor UI.
     */
    public function render_sync_queue_tab() {
        if ( class_exists('Delcampe_Queue_Admin') ) {
            // Ensure assets are present
            wp_enqueue_style('delcampe-queue-admin', DWC_PLUGIN_URL . 'assets/css/queue-admin.css', array(), DWC_VERSION);
            wp_enqueue_script('delcampe-queue-admin', DWC_PLUGIN_URL . 'assets/js/queue-admin.js', array('jquery'), DWC_VERSION, true);
            echo '<div class="delcampe-section">';
            echo '<h3>' . esc_html__('Queue Monitor', 'wc-delcampe-integration') . '</h3>';
            Delcampe_Queue_Admin::get_instance()->render_embedded();
            echo '</div>';
        } else {
            echo '<p>' . esc_html__('Queue system not available.', 'wc-delcampe-integration') . '</p>';
        }
    }
    
    /**
     * Render inventory page
     * 
     * @since  1.5.0.0
     */
    public function render_inventory_page() {
        global $wpdb;
        
        // Handle bulk actions
        if (isset($_POST['action']) && $_POST['action'] === 'bulk_inventory_update') {
            $this->handle_bulk_inventory_update();
        }
        
        // Get current tab from URL parameter (v1.10.24.0 - fix tab persistence)
        $current_tab = isset($_GET['tab']) ? sanitize_text_field($_GET['tab']) : 'overview';
        
        ?>
        <div class="wrap">
            <h1><?php _e('Delcampe Inventory Management', 'wc-delcampe-integration'); ?></h1>
            
            <!-- Tabs -->
            <h2 class="nav-tab-wrapper">
                <a href="?page=delcampe-inventory&tab=overview" class="nav-tab <?php echo $current_tab === 'overview' ? 'nav-tab-active' : ''; ?>" data-tab="overview">
                    <?php _e('Overview', 'wc-delcampe-integration'); ?>
                </a>
                <a href="?page=delcampe-inventory&tab=inventory-check" class="nav-tab <?php echo $current_tab === 'inventory-check' ? 'nav-tab-active' : ''; ?>" data-tab="inventory-check">
                    <?php _e('Inventory Check', 'wc-delcampe-integration'); ?>
                </a>
                <a href="?page=delcampe-inventory&tab=sync-log" class="nav-tab <?php echo $current_tab === 'sync-log' ? 'nav-tab-active' : ''; ?>" data-tab="sync-log">
                    <?php _e('Sync Log', 'wc-delcampe-integration'); ?>
                </a>
                <a href="?page=delcampe-inventory&tab=alerts" class="nav-tab <?php echo $current_tab === 'alerts' ? 'nav-tab-active' : ''; ?>" data-tab="alerts">
                    <?php _e('Alerts', 'wc-delcampe-integration'); ?>
                </a>
                <a href="?page=delcampe-inventory&tab=batch-queue" class="nav-tab <?php echo $current_tab === 'batch-queue' ? 'nav-tab-active' : ''; ?>" data-tab="batch-queue">
                    <?php _e('Batch Queue', 'wc-delcampe-integration'); ?>
                </a>
                <a href="?page=delcampe-inventory&tab=sync-queue" class="nav-tab <?php echo $current_tab === 'sync-queue' ? 'nav-tab-active' : ''; ?>" data-tab="sync-queue">
                    <?php _e('Queue Management', 'wc-delcampe-integration'); ?>
                </a>
                <a href="?page=delcampe-inventory&tab=cleanup" class="nav-tab <?php echo $current_tab === 'cleanup' ? 'nav-tab-active' : ''; ?>" data-tab="cleanup">
                    <?php _e('Cleanup', 'wc-delcampe-integration'); ?>
                </a>
            </h2>
            
            <!-- Tab Content -->
            <div class="tab-content">
                <!-- Overview Tab -->
                <div id="overview-tab" class="tab-pane" <?php echo $current_tab !== 'overview' ? 'style="display:none;"' : ''; ?>>
                    <?php $this->render_overview_tab(); ?>
                </div>
                
                <!-- Inventory Check Tab -->
                <div id="inventory-check-tab" class="tab-pane" <?php echo $current_tab !== 'inventory-check' ? 'style="display:none;"' : ''; ?>>
                    <?php $this->render_inventory_check_tab(); ?>
                </div>
                
                <!-- Sync Log Tab -->
                <div id="sync-log-tab" class="tab-pane" <?php echo $current_tab !== 'sync-log' ? 'style="display:none;"' : ''; ?>>
                    <?php $this->render_sync_log_tab(); ?>
                </div>
                
                <!-- Alerts Tab -->
                <div id="alerts-tab" class="tab-pane" <?php echo $current_tab !== 'alerts' ? 'style="display:none;"' : ''; ?>>
                    <?php $this->render_alerts_tab(); ?>
                </div>
                
                <!-- Batch Queue Tab (v1.10.24.0 - moved from standalone menu) -->
                <div id="batch-queue-tab" class="tab-pane" <?php echo $current_tab !== 'batch-queue' ? 'style="display:none;"' : ''; ?>>
                    <?php $this->render_batch_queue_tab(); ?>
                </div>
                
                <!-- Queue Management Tab (new) -->
                <div id="sync-queue-tab" class="tab-pane" <?php echo $current_tab !== 'sync-queue' ? 'style="display:none;"' : ''; ?>>
                    <?php $this->render_sync_queue_tab(); ?>
                </div>
                
                <!-- Cleanup Tab -->
                <div id="cleanup-tab" class="tab-pane" <?php echo $current_tab !== 'cleanup' ? 'style="display:none;"' : ''; ?>>
                    <?php $this->render_cleanup_tab(); ?>
                </div>
            </div>
        </div>
        
        <style>
        .nav-tab-wrapper {
            margin-bottom: 20px;
        }
        .tab-pane {
            background: #fff;
            padding: 20px;
            border: 1px solid #ccd0d4;
            border-top: none;
        }
        .inventory-comparison-table {
            margin-top: 20px;
        }
        .inventory-comparison-table th,
        .inventory-comparison-table td {
            padding: 10px;
        }
        .stock-mismatch {
            background-color: #fff3cd;
        }
        .stock-low {
            background-color: #f8d7da;
        }
        .stock-ok {
            background-color: #d4edda;
        }
        </style>
        
        <script>
        jQuery(document).ready(function($) {
            // Tab switching with URL persistence (v1.10.24.0)
            $('.nav-tab').on('click', function(e) {
                e.preventDefault();
                var tab = $(this).data('tab');
                var href = $(this).attr('href');
                
                // Update URL without page reload
                if (history.pushState) {
                    history.pushState(null, null, href);
                }
                
                $('.nav-tab').removeClass('nav-tab-active');
                $(this).addClass('nav-tab-active');
                
                $('.tab-pane').hide();
                $('#' + tab + '-tab').show();
                
                // Special handling for batch queue tab to load its content
                if (tab === 'batch-queue' && !$('#batch-queue-tab').data('loaded')) {
                    $('#batch-queue-tab').data('loaded', true);
                    // Trigger refresh if batch queue monitor is available
                    if (typeof refreshQueueStats === 'function') {
                        refreshQueueStats();
                    }
                }
            });
        });
        </script>
        <?php
    }
    
    /**
     * Render overview tab
     * 
     * @since  1.5.0.0
     */
    private function render_overview_tab() {
        $stats = $this->inventory_manager->get_inventory_statistics();
        
        ?>
        <h2><?php _e('Inventory Overview', 'wc-delcampe-integration'); ?></h2>
        
        <div class="stats-grid">
            <div class="stat-box">
                <h3><?php _e('Total Synced Products', 'wc-delcampe-integration'); ?></h3>
                <p class="stat-number"><?php echo esc_html($stats['total_synced']); ?></p>
            </div>
            <div class="stat-box">
                <h3><?php _e('Inventory Mismatches', 'wc-delcampe-integration'); ?></h3>
                <p class="stat-number <?php echo $stats['mismatches'] > 0 ? 'has-errors' : ''; ?>">
                    <?php echo esc_html($stats['mismatches']); ?>
                </p>
            </div>
            <div class="stat-box">
                <h3><?php _e('Recent Syncs (24h)', 'wc-delcampe-integration'); ?></h3>
                <p class="stat-number"><?php echo esc_html($stats['recent_syncs']); ?></p>
            </div>
            <div class="stat-box">
                <h3><?php _e('Active Alerts', 'wc-delcampe-integration'); ?></h3>
                <p class="stat-number <?php echo $stats['active_alerts'] > 0 ? 'has-alerts' : ''; ?>">
                    <?php echo esc_html($stats['active_alerts']); ?>
                </p>
            </div>
        </div>
        
        <?php if (!empty($stats['alerts_by_type'])): ?>
        <h3><?php _e('Alert Breakdown', 'wc-delcampe-integration'); ?></h3>
        <table class="widefat">
            <thead>
                <tr>
                    <th><?php _e('Alert Type', 'wc-delcampe-integration'); ?></th>
                    <th><?php _e('Count', 'wc-delcampe-integration'); ?></th>
                </tr>
            </thead>
            <tbody>
                <?php foreach ($stats['alerts_by_type'] as $type => $count): ?>
                <tr>
                    <td><?php echo esc_html(ucfirst(str_replace('_', ' ', $type))); ?></td>
                    <td><?php echo esc_html($count); ?></td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
        <?php endif; ?>
        
        <div class="inventory-actions" style="margin-top: 20px;">
            <button type="button" class="button button-primary" id="check-all-inventory">
                <?php _e('Check All Inventory', 'wc-delcampe-integration'); ?>
            </button>
            <button type="button" class="button" id="cleanup-orphaned-ids">
                <?php _e('Remove IDs from Ended Listings', 'wc-delcampe-integration'); ?>
            </button>
            <button type="button" class="button" id="export-inventory-report">
                <?php _e('Export Inventory Report', 'wc-delcampe-integration'); ?>
            </button>
        </div>
        <div id="cleanup-results" style="display:none; margin-top:10px; padding:10px; background:#f0f0f0; border:1px solid #ddd;"></div>
        
        <script>
        jQuery(document).ready(function($) {
            $('#check-all-inventory').on('click', function() {
                if (confirm('<?php _e('This will check all product inventory. This may take a while. Continue?', 'wc-delcampe-integration'); ?>')) {
                    var $button = $(this);
                    $button.prop('disabled', true).text('<?php _e('Checking...', 'wc-delcampe-integration'); ?>');
                    
                    $.post(ajaxurl, {
                        action: 'delcampe_check_inventory',
                        check_all: true,
                        _ajax_nonce: '<?php echo wp_create_nonce('delcampe_check_inventory'); ?>'
                    }, function(response) {
                        alert(response.data);
                        location.reload();
                    });
                }
            });
            
            // Clean up orphaned Delcampe IDs
            $('#cleanup-orphaned-ids').on('click', function() {
                if (!confirm('<?php _e('This will remove Delcampe IDs from products with ended or closed listings. Continue?', 'wc-delcampe-integration'); ?>')) {
                    return;
                }
                
                var $button = $(this);
                var $results = $('#cleanup-results');
                $button.prop('disabled', true).text('<?php _e('Cleaning...', 'wc-delcampe-integration'); ?>');
                $results.html('<?php _e('Checking listing statuses...', 'wc-delcampe-integration'); ?>').show();
                
                $.post(ajaxurl, {
                    action: 'delcampe_cleanup_orphaned_ids',
                    _ajax_nonce: '<?php echo wp_create_nonce('delcampe_cleanup_orphaned_ids'); ?>'
                }, function(response) {
                    if (response.success) {
                        var html = '<strong><?php _e('Results:', 'wc-delcampe-integration'); ?></strong><br>' + response.data.message;
                        
                        $results.html(html);
                        
                        if (response.data.cleaned > 0) {
                            $results.css('background', '#d4f4dd');
                        }
                    } else {
                        $results.html('<span style="color:red;"><?php _e('Error:', 'wc-delcampe-integration'); ?> ' + (response.data || 'Unknown error') + '</span>');
                    }
                    $button.prop('disabled', false).text('<?php _e('Remove IDs from Ended Listings', 'wc-delcampe-integration'); ?>');
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * Render inventory check tab
     * 
     * @since  1.5.0.0
     */
    private function render_inventory_check_tab() {
        global $wpdb;
        
        // Get products with Delcampe listings
        $products = $wpdb->get_results(
            "SELECT DISTINCT p.ID, p.post_title, pm.meta_value as delcampe_item_id
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            WHERE p.post_type = 'product'
            AND p.post_status = 'publish'
            AND pm.meta_key = '_delcampe_item_id'
            AND pm.meta_value IS NOT NULL
            AND pm.meta_value != ''
            ORDER BY p.post_title"
        );
        
        ?>
        <h2><?php _e('Inventory Comparison', 'wc-delcampe-integration'); ?></h2>
        <p><?php _e('Compare inventory levels between WooCommerce and Delcampe', 'wc-delcampe-integration'); ?></p>

        <!-- Stock Sync Tools (v1.10.35.10 - Manual control) -->
        <div class="stock-sync-tools" style="margin:15px 0; padding: 15px; background: #f8f9fa; border: 1px solid #e1e1e1;">
            <h3><?php _e('Stock Synchronization', 'wc-delcampe-integration'); ?></h3>
            <p class="description"><?php _e('Manual stock synchronization. Automatic hourly sync disabled for better control.', 'wc-delcampe-integration'); ?></p>
            <p>
                <button type="button" class="button button-primary" id="delcampe-sync-all-stock">
                    <?php _e('Start Full Stock Sync', 'wc-delcampe-integration'); ?>
                </button>
                <button type="button" class="button" id="delcampe-refresh-stock-cache" style="margin-left: 10px;">
                    <?php _e('Refresh Stock Cache', 'wc-delcampe-integration'); ?>
                    <span class="dashicons dashicons-update" style="vertical-align: middle;"></span>
                </button>
                <button type="button" class="button" id="delcampe-stop-stock-sync" style="display:none;">
                    <?php _e('Stop Sync', 'wc-delcampe-integration'); ?>
                </button>
                <span class="spinner" style="display:none; float:none;"></span>
            </p>
            <div id="stock-sync-progress" style="display:none; margin-top: 10px;">
                <div style="background: #fff; border: 1px solid #ccd0d4; padding: 10px;">
                    <div class="progress-bar" style="height: 20px; background: #f0f0f0; border-radius: 3px; overflow: hidden;">
                        <div class="progress-fill" style="height: 100%; background: #46b450; width: 0%; transition: width 0.3s;"></div>
                    </div>
                    <p class="progress-text" style="margin-top: 5px;">
                        <?php _e('Processing...', 'wc-delcampe-integration'); ?> <span class="progress-count">0/0</span>
                    </p>
                    <div class="sync-log" style="max-height: 200px; overflow-y: auto; margin-top: 10px; padding: 10px; background: #f8f9fa; font-family: monospace; font-size: 12px;"></div>
                </div>
            </div>
        </div>

        <!-- Integrity Tools (Discrepancy Fixes) -->
        <div class="inventory-quick-actions" style="margin:15px 0;">
            <h3><?php _e('Inventory Integrity Tools', 'wc-delcampe-integration'); ?></h3>
            <p class="description"><?php _e('Diagnose and fix duplicate Delcampe IDs and backfill missing listings.', 'wc-delcampe-integration'); ?></p>
            <p>
                <button type="button" class="button" id="delcampe-diagnose"><?php _e('Run Diagnostics', 'wc-delcampe-integration'); ?></button>
                <button type="button" class="button" id="delcampe-fix-duplicates-dry"><?php _e('Fix Duplicates (Dry Run)', 'wc-delcampe-integration'); ?></button>
                <button type="button" class="button button-primary" id="delcampe-fix-duplicates-apply"><?php _e('Fix Duplicates (Apply)', 'wc-delcampe-integration'); ?></button>
                <button type="button" class="button" id="delcampe-backfill-dry"><?php _e('Backfill Missing (Dry Run)', 'wc-delcampe-integration'); ?></button>
                <button type="button" class="button button-primary" id="delcampe-backfill-apply"><?php _e('Backfill Missing (Apply)', 'wc-delcampe-integration'); ?></button>
                <button type="button" class="button" id="delcampe-constraints-check"><?php _e('Check Constraints', 'wc-delcampe-integration'); ?></button>
                <button type="button" class="button" id="delcampe-constraints-apply"><?php _e('Apply Constraints', 'wc-delcampe-integration'); ?></button>
                <button type="button" class="button" id="delcampe-promote-verified-dry"><?php _e('Promote Verified (Dry Run)', 'wc-delcampe-integration'); ?></button>
                <button type="button" class="button button-primary" id="delcampe-promote-verified-apply"><?php _e('Promote Verified (Apply)', 'wc-delcampe-integration'); ?></button>
            </p>
            <div id="delcampe-integrity-output-secondary" class="delcampe-integrity-output" style="background:#fff;border:1px solid #ccd0d4;padding:12px;max-height:420px;overflow:auto"></div>
        </div>
        
        <?php if (empty($products)): ?>
            <p><?php _e('No products with Delcampe listings found.', 'wc-delcampe-integration'); ?></p>
        <?php else: ?>
            <form method="post" id="inventory-check-form">
                <table class="wp-list-table widefat fixed striped inventory-comparison-table">
                    <thead>
                        <tr>
                            <th style="width: 50px;">
                                <input type="checkbox" id="select-all-products" />
                            </th>
                            <th><?php _e('Product', 'wc-delcampe-integration'); ?></th>
                            <th><?php _e('SKU', 'wc-delcampe-integration'); ?></th>
                            <th><?php _e('WooCommerce Stock', 'wc-delcampe-integration'); ?></th>
                            <th><?php _e('Delcampe Stock', 'wc-delcampe-integration'); ?></th>
                            <th><?php _e('Status', 'wc-delcampe-integration'); ?></th>
                            <th><?php _e('Last Check', 'wc-delcampe-integration'); ?></th>
                            <th><?php _e('Actions', 'wc-delcampe-integration'); ?></th>
                        </tr>
                    </thead>
                    <tbody id="inventory-check-results">
                        <?php foreach ($products as $product_data): 
                            $product = wc_get_product($product_data->ID);
                            if (!$product) continue;
                            
                            // Get last sync data
                            $last_sync = $wpdb->get_row($wpdb->prepare(
                                "SELECT * FROM {$wpdb->prefix}delcampe_inventory_sync
                                WHERE product_id = %d
                                ORDER BY created_at DESC
                                LIMIT 1",
                                $product_data->ID
                            ));
                        ?>
                        <tr data-product-id="<?php echo esc_attr($product_data->ID); ?>">
                            <td>
                                <input type="checkbox" 
                                       name="products[]" 
                                       value="<?php echo esc_attr($product_data->ID); ?>" 
                                       class="product-checkbox" />
                            </td>
                            <td>
                                <strong>
                                    <a href="<?php echo get_edit_post_link($product_data->ID); ?>">
                                        <?php echo esc_html($product->get_name()); ?>
                                    </a>
                                </strong>
                            </td>
                            <td><?php echo esc_html($product->get_sku()); ?></td>
                            <td class="wc-stock"><?php echo esc_html($product->get_stock_quantity()); ?></td>
                            <td class="delcampe-stock">
                                <?php 
                                if ($last_sync && $last_sync->delcampe_stock !== null) {
                                    echo esc_html($last_sync->delcampe_stock);
                                } else {
                                    echo '<span class="dashicons dashicons-minus"></span>';
                                }
                                ?>
                            </td>
                            <td class="sync-status">
                                <?php 
                                if ($last_sync) {
                                    if ($last_sync->wc_stock == $last_sync->delcampe_stock) {
                                        echo '<span class="status-ok">' . __('Synced', 'wc-delcampe-integration') . '</span>';
                                    } else {
                                        echo '<span class="status-mismatch">' . __('Mismatch', 'wc-delcampe-integration') . '</span>';
                                    }
                                } else {
                                    echo '<span class="status-unknown">' . __('Not checked', 'wc-delcampe-integration') . '</span>';
                                }
                                ?>
                            </td>
                            <td class="last-check">
                                <?php 
                                if ($last_sync) {
                                    echo esc_html(human_time_diff(strtotime($last_sync->created_at), current_time('timestamp')) . ' ' . __('ago', 'wc-delcampe-integration'));
                                } else {
                                    echo __('Never', 'wc-delcampe-integration');
                                }
                                ?>
                            </td>
                            <td class="actions">
                                <button type="button" 
                                        class="button button-small check-single" 
                                        data-product-id="<?php echo esc_attr($product_data->ID); ?>">
                                    <?php _e('Check', 'wc-delcampe-integration'); ?>
                                </button>
                                <button type="button" 
                                        class="button button-small sync-single" 
                                        data-product-id="<?php echo esc_attr($product_data->ID); ?>">
                                    <?php _e('Sync', 'wc-delcampe-integration'); ?>
                                </button>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
                
                <div class="tablenav bottom">
                    <div class="alignleft actions bulkactions">
                        <select name="bulk_action" id="bulk-action-selector">
                            <option value=""><?php _e('Bulk Actions', 'wc-delcampe-integration'); ?></option>
                            <option value="check"><?php _e('Check Selected', 'wc-delcampe-integration'); ?></option>
                            <option value="sync"><?php _e('Sync Selected', 'wc-delcampe-integration'); ?></option>
                        </select>
                        <button type="button" class="button action" id="do-bulk-action">
                            <?php _e('Apply', 'wc-delcampe-integration'); ?>
                        </button>
                    </div>
                </div>
            </form>
        <?php endif; ?>
        
        <style>
        .status-ok { color: #46b450; }
        .status-mismatch { color: #dc3232; }
        .status-unknown { color: #999; }
        </style>
        
        <script>
        jQuery(document).ready(function($) {
            // Select all checkbox
            $('#select-all-products').on('change', function() {
                $('.product-checkbox').prop('checked', $(this).prop('checked'));
            });
            
            // Check single product
            $('.check-single').on('click', function() {
                var $button = $(this);
                var productId = $button.data('product-id');
                
                $button.prop('disabled', true).text('<?php _e('Checking...', 'wc-delcampe-integration'); ?>');
                
                $.post(ajaxurl, {
                    action: 'delcampe_check_inventory',
                    product_id: productId,
                    _ajax_nonce: '<?php echo wp_create_nonce('delcampe_check_inventory'); ?>'
                }, function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert('<?php _e('Error:', 'wc-delcampe-integration'); ?> ' + response.data);
                    }
                    $button.prop('disabled', false).text('<?php _e('Check', 'wc-delcampe-integration'); ?>');
                });
            });
            
            // Sync single product
            $('.sync-single').on('click', function() {
                var $button = $(this);
                var productId = $button.data('product-id');
                
                $button.prop('disabled', true).text('<?php _e('Syncing...', 'wc-delcampe-integration'); ?>');
                
                $.post(ajaxurl, {
                    action: 'delcampe_sync_inventory',
                    product_id: productId,
                    _ajax_nonce: '<?php echo wp_create_nonce('delcampe_sync_inventory'); ?>'
                }, function(response) {
                    if (response.success) {
                        location.reload();
                    } else {
                        alert('<?php _e('Error:', 'wc-delcampe-integration'); ?> ' + response.data);
                    }
                    $button.prop('disabled', false).text('<?php _e('Sync', 'wc-delcampe-integration'); ?>');
                });
            });
            
            // Bulk actions
            $('#do-bulk-action').on('click', function() {
                var action = $('#bulk-action-selector').val();
                var products = [];
                
                $('.product-checkbox:checked').each(function() {
                    products.push($(this).val());
                });
                
                if (!action || products.length === 0) {
                    return;
                }
                
                var $button = $(this);
                $button.prop('disabled', true);
                
                $.post(ajaxurl, {
                    action: 'delcampe_bulk_inventory_' + action,
                    products: products,
                    _ajax_nonce: '<?php echo wp_create_nonce('delcampe_bulk_inventory'); ?>'
                }, function(response) {
                    if (response.success) {
                        alert(response.data);
                        location.reload();
                    } else {
                        alert('<?php _e('Error:', 'wc-delcampe-integration'); ?> ' + response.data);
                    }
                    $button.prop('disabled', false);
                });
            });

            // Integrity Tools JS
            function printOutput(obj) {
                $('#delcampe-integrity-output').text(JSON.stringify(obj, null, 2));
            }
            function ajaxPost(action, data, done) {
                data = data || {}; data.action = action; data.nonce = '<?php echo wp_create_nonce('delcampe_inventory_integrity'); ?>';
                return $.post(ajaxurl, data)
                    .done(function(resp){
                        if (resp && resp.success) {
                            if (typeof done === 'function') done(resp.data);
                        } else {
                            printOutput(resp);
                            if (typeof done === 'function') done(null);
                        }
                    })
                    .fail(function(xhr){
                        printOutput({error:'network', status:xhr.status});
                        if (typeof done === 'function') done(null);
                    });
            }
            $('#delcampe-diagnose').on('click', function(){
                var $b=$(this); $b.prop('disabled',true).text('<?php _e('Diagnosing...', 'wc-delcampe-integration'); ?>');
                ajaxPost('delcampe_inventory_diagnose', {}, function(data){ printOutput(data); $b.prop('disabled',false).text('<?php _e('Run Diagnostics', 'wc-delcampe-integration'); ?>'); });
            });
            $('#delcampe-fix-duplicates-dry').on('click', function(){
                var $b=$(this); $b.prop('disabled',true).text('<?php _e('Dry running...', 'wc-delcampe-integration'); ?>');
                ajaxPost('delcampe_inventory_fix_duplicates', {dry_run:'1'}, function(data){ printOutput(data); $b.prop('disabled',false).text('<?php _e('Fix Duplicates (Dry Run)', 'wc-delcampe-integration'); ?>'); });
            });
            $('#delcampe-fix-duplicates-apply').on('click', function(){
                if (!confirm('<?php _e('Apply duplicate fixes now?', 'wc-delcampe-integration'); ?>')) return;
                var $b=$(this); $b.prop('disabled',true).text('<?php _e('Applying...', 'wc-delcampe-integration'); ?>');
                ajaxPost('delcampe_inventory_fix_duplicates', {dry_run:'0'}, function(data){ printOutput(data); $b.prop('disabled',false).text('<?php _e('Fix Duplicates (Apply)', 'wc-delcampe-integration'); ?>'); });
            });
            $('#delcampe-backfill-dry').on('click', function(){
                var $b=$(this); $b.prop('disabled',true).text('<?php _e('Dry running...', 'wc-delcampe-integration'); ?>');
                var req = ajaxPost('delcampe_inventory_backfill_missing', {dry_run:'1'}, function(data, el){ renderBackfillPreview(data, el); }, this);
                if (req && typeof req.always === 'function') {
                    req.always(function(){ $b.prop('disabled',false).text('<?php _e('Backfill Missing (Dry Run)', 'wc-delcampe-integration'); ?>'); });
                } else {
                    $b.prop('disabled',false).text('<?php _e('Backfill Missing (Dry Run)', 'wc-delcampe-integration'); ?>');
                }
            });
            $('#delcampe-backfill-apply').on('click', function(){
                if (!confirm('<?php _e('Backfill missing listings now?', 'wc-delcampe-integration'); ?>')) return;
                var $b=$(this); $b.prop('disabled',true).text('<?php _e('Applying...', 'wc-delcampe-integration'); ?>');
                var req = ajaxPost('delcampe_inventory_backfill_missing', {dry_run:'0'}, function(data, el){ printOutput(data, el); }, this);
                if (req && typeof req.always === 'function') {
                    req.always(function(){ $b.prop('disabled',false).text('<?php _e('Backfill Missing (Apply)', 'wc-delcampe-integration'); ?>'); });
                } else {
                    $b.prop('disabled',false).text('<?php _e('Backfill Missing (Apply)', 'wc-delcampe-integration'); ?>');
                }
            });
            $('#delcampe-constraints-check').on('click', function(){
                var $b=$(this); $b.prop('disabled',true).text('<?php _e('Checking...', 'wc-delcampe-integration'); ?>');
                ajaxPost('delcampe_inventory_add_constraints', {check_only:'1'}, function(data){ printOutput(data); $b.prop('disabled',false).text('<?php _e('Check Constraints', 'wc-delcampe-integration'); ?>'); });
            });
            $('#delcampe-constraints-apply').on('click', function(){
                if (!confirm('<?php _e('Apply database constraints now?', 'wc-delcampe-integration'); ?>')) return;
                var $b=$(this); $b.prop('disabled',true).text('<?php _e('Applying...', 'wc-delcampe-integration'); ?>');
                ajaxPost('delcampe_inventory_add_constraints', {check_only:'0'}, function(data){ printOutput(data); $b.prop('disabled',false).text('<?php _e('Apply Constraints', 'wc-delcampe-integration'); ?>'); });
            });
            $('#delcampe-promote-verified-dry').on('click', function(){
                var $b=$(this); $b.prop('disabled',true).text('<?php _e('Dry running...', 'wc-delcampe-integration'); ?>');
                ajaxPost('delcampe_inventory_reconcile_verified', {dry_run:'1'}, function(data){ printOutput(data); $b.prop('disabled',false).text('<?php _e('Promote Verified (Dry Run)', 'wc-delcampe-integration'); ?>'); });
            });
            $('#delcampe-promote-verified-apply').on('click', function(){
                if (!confirm('<?php _e('Link Delcampe IDs to verified listings now?', 'wc-delcampe-integration'); ?>')) return;
                var $b=$(this); $b.prop('disabled',true).text('<?php _e('Applying...', 'wc-delcampe-integration'); ?>');
                ajaxPost('delcampe_inventory_reconcile_verified', {dry_run:'0'}, function(data){ printOutput(data); $b.prop('disabled',false).text('<?php _e('Promote Verified (Apply)', 'wc-delcampe-integration'); ?>'); });
            });
            
            // Stock Sync functionality (v1.10.34.7)
            var stockSyncInProgress = false;
            var stockSyncCancelled = false;
            
            $('#delcampe-sync-all-stock').on('click', function() {
                if (!confirm('<?php _e('This will sync ALL product stock levels to Delcampe. This may take several minutes. Continue?', 'wc-delcampe-integration'); ?>')) {
                    return;
                }
                
                stockSyncInProgress = true;
                stockSyncCancelled = false;
                
                var $button = $(this);
                var $stopButton = $('#delcampe-stop-stock-sync');
                var $spinner = $button.siblings('.spinner');
                var $progress = $('#stock-sync-progress');
                var $progressFill = $progress.find('.progress-fill');
                var $progressText = $progress.find('.progress-count');
                var $syncLog = $progress.find('.sync-log');
                
                $button.hide();
                $stopButton.show();
                $spinner.addClass('is-active').show();
                $progress.show();
                $syncLog.empty();
                
                function addLogEntry(message, isError) {
                    var timestamp = new Date().toLocaleTimeString();
                    var cssClass = isError ? 'color: #dc3232;' : '';
                    $syncLog.append('<div style="' + cssClass + '">[' + timestamp + '] ' + message + '</div>');
                    $syncLog.scrollTop($syncLog[0].scrollHeight);
                }
                
                function runBatch() {
                    if (stockSyncCancelled) {
                        addLogEntry('<?php _e('Stock sync cancelled by user', 'wc-delcampe-integration'); ?>', true);
                        finishSync();
                        return;
                    }
                    
                    $.post(ajaxurl, {
                        action: 'delcampe_run_stock_sync_batch',
                        _ajax_nonce: '<?php echo wp_create_nonce('delcampe_run_stock_sync'); ?>'
                    })
                    .done(function(response) {
                        if (response.success && response.data) {
                            var data = response.data;
                            
                            // Update progress
                            if (data.total > 0) {
                                var progress = (data.processed / data.total) * 100;
                                $progressFill.css('width', progress + '%');
                                $progressText.text(data.processed + '/' + data.total);
                            }
                            
                            // Add log entries
                            if (data.log && data.log.length > 0) {
                                data.log.forEach(function(entry) {
                                    addLogEntry(entry.message, entry.error);
                                });
                            }
                            
                            // Continue if not complete
                            if (!data.complete && !stockSyncCancelled) {
                                setTimeout(runBatch, 1000); // Wait 1 second between batches
                            } else {
                                addLogEntry('<?php _e('Stock sync completed!', 'wc-delcampe-integration'); ?>');
                                finishSync();
                            }
                        } else {
                            addLogEntry('<?php _e('Error: ', 'wc-delcampe-integration'); ?>' + (response.data || 'Unknown error'), true);
                            finishSync();
                        }
                    })
                    .fail(function() {
                        addLogEntry('<?php _e('Network error occurred', 'wc-delcampe-integration'); ?>', true);
                        finishSync();
                    });
                }
                
                function finishSync() {
                    stockSyncInProgress = false;
                    $spinner.removeClass('is-active').hide();
                    $stopButton.hide();
                    $button.show();
                }
                
                // Start the first batch
                addLogEntry('<?php _e('Starting stock synchronization...', 'wc-delcampe-integration'); ?>');
                runBatch();
            });
            
            $('#delcampe-stop-stock-sync').on('click', function() {
                if (confirm('<?php _e('Stop the stock sync process?', 'wc-delcampe-integration'); ?>')) {
                    stockSyncCancelled = true;
                    $(this).prop('disabled', true).text('<?php _e('Stopping...', 'wc-delcampe-integration'); ?>');
                }
            });
            
            // Stock Cache Refresh button (v1.10.35.10)
            $('#delcampe-refresh-stock-cache').on('click', function() {
                var $button = $(this);
                var $spinner = $button.parent().find('.spinner');
                var $log = $('#stock-sync-progress .sync-log');
                
                if (!$spinner.length) {
                    $button.after('<span class="spinner" style="display:none; float:none; margin-left: 10px;"></span>');
                    $spinner = $button.parent().find('.spinner');
                }
                
                $button.prop('disabled', true);
                $spinner.show().css('visibility', 'visible');
                
                // Show progress area if hidden
                $('#stock-sync-progress').show();
                if ($log.length) {
                    $log.append('<div style="color: #0073aa; font-weight: bold;">--- Starting Stock Cache Refresh ---</div>');
                }
                
                $.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'delcampe_refresh_stock_cache',
                        security: '<?php echo wp_create_nonce('delcampe_ajax_nonce'); ?>'
                    },
                    success: function(response) {
                        $spinner.hide();
                        $button.prop('disabled', false);
                        
                        if (response.success) {
                            if ($log.length) {
                                $log.append('<div style="color: green;">✓ Stock cache refreshed successfully</div>');
                                if (response.data && response.data.results) {
                                    $log.append('<div>Processed: ' + response.data.results.total + ' items</div>');
                                    $log.append('<div>Success: ' + response.data.results.success + ', Failed: ' + response.data.results.failed + '</div>');
                                }
                            } else {
                                alert('Stock cache refreshed successfully!');
                            }
                        } else {
                            var error = response.data || 'Unknown error';
                            if ($log.length) {
                                $log.append('<div style="color: red;">✗ Error: ' + error + '</div>');
                            } else {
                                alert('Error refreshing stock cache: ' + error);
                            }
                        }
                        
                        // Scroll log to bottom
                        if ($log.length) {
                            $log.scrollTop($log[0].scrollHeight);
                        }
                    },
                    error: function(xhr, status, error) {
                        $spinner.hide();
                        $button.prop('disabled', false);
                        var errorMsg = 'AJAX Error: ' + error;
                        if ($log.length) {
                            $log.append('<div style="color: red;">✗ ' + errorMsg + '</div>');
                        } else {
                            alert(errorMsg);
                        }
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * Render sync log tab
     * 
     * @since  1.5.0.0
     */
    private function render_sync_log_tab() {
        global $wpdb;
        
        // Get recent sync logs
        $logs = $wpdb->get_results(
            "SELECT s.*, p.post_title 
            FROM {$wpdb->prefix}delcampe_inventory_sync s
            LEFT JOIN {$wpdb->posts} p ON s.product_id = p.ID
            ORDER BY s.created_at DESC
            LIMIT 100"
        );
        
        ?>
        <h2><?php _e('Inventory Sync Log', 'wc-delcampe-integration'); ?></h2>
        
        <?php if (empty($logs)): ?>
            <p><?php _e('No sync logs found.', 'wc-delcampe-integration'); ?></p>
        <?php else: ?>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php _e('Date/Time', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('Product', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('Type', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('WC Stock', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('Delcampe Stock', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('Status', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('Message', 'wc-delcampe-integration'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($logs as $log): ?>
                    <tr>
                        <td><?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($log->created_at))); ?></td>
                        <td>
                            <?php if ($log->post_title): ?>
                                <a href="<?php echo get_edit_post_link($log->product_id); ?>">
                                    <?php echo esc_html($log->post_title); ?>
                                </a>
                            <?php else: ?>
                                <?php _e('Product not found', 'wc-delcampe-integration'); ?>
                            <?php endif; ?>
                        </td>
                        <td><?php echo esc_html(ucfirst(str_replace('_', ' ', $log->sync_type))); ?></td>
                        <td><?php echo esc_html($log->wc_stock); ?></td>
                        <td><?php echo $log->delcampe_stock !== null ? esc_html($log->delcampe_stock) : '-'; ?></td>
                        <td>
                            <span class="status-<?php echo esc_attr($log->sync_status); ?>">
                                <?php echo esc_html(ucfirst($log->sync_status)); ?>
                            </span>
                        </td>
                        <td><?php echo esc_html($log->sync_message); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
        
        <style>
        .status-success { color: #46b450; }
        .status-failed { color: #dc3232; }
        .status-pending { color: #f56e28; }
        </style>
        <?php
    }
    
    /**
     * Render alerts tab
     * 
     * @since  1.5.0.0
     */
    private function render_alerts_tab() {
        global $wpdb;
        
        // Get alerts
        $alerts = $wpdb->get_results(
            "SELECT a.*, p.post_title 
            FROM {$wpdb->prefix}delcampe_inventory_alerts a
            LEFT JOIN {$wpdb->posts} p ON a.product_id = p.ID
            WHERE a.is_resolved = 0
            ORDER BY a.created_at DESC"
        );
        
        ?>
        <h2><?php _e('Inventory Alerts', 'wc-delcampe-integration'); ?></h2>
        
        <?php if (empty($alerts)): ?>
            <p><?php _e('No active alerts.', 'wc-delcampe-integration'); ?></p>
        <?php else: ?>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php _e('Date/Time', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('Product', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('Alert Type', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('Message', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('Current Stock', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('Actions', 'wc-delcampe-integration'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($alerts as $alert): ?>
                    <tr>
                        <td><?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($alert->created_at))); ?></td>
                        <td>
                            <?php if ($alert->post_title): ?>
                                <a href="<?php echo get_edit_post_link($alert->product_id); ?>">
                                    <?php echo esc_html($alert->post_title); ?>
                                </a>
                            <?php else: ?>
                                <?php _e('Product not found', 'wc-delcampe-integration'); ?>
                            <?php endif; ?>
                        </td>
                        <td>
                            <span class="alert-type-<?php echo esc_attr($alert->alert_type); ?>">
                                <?php echo esc_html(ucfirst(str_replace('_', ' ', $alert->alert_type))); ?>
                            </span>
                        </td>
                        <td><?php echo esc_html($alert->alert_message); ?></td>
                        <td><?php echo $alert->current_stock !== null ? esc_html($alert->current_stock) : '-'; ?></td>
                        <td>
                            <button type="button" 
                                    class="button button-small resolve-alert" 
                                    data-alert-id="<?php echo esc_attr($alert->id); ?>">
                                <?php _e('Resolve', 'wc-delcampe-integration'); ?>
                            </button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
        
        <style>
        .alert-type-low_stock { color: #f56e28; }
        .alert-type-no_stock { color: #dc3232; }
        .alert-type-sync_error { color: #dc3232; }
        .alert-type-oversell { color: #a00; }
        </style>
        
        <script>
        jQuery(document).ready(function($) {
            $('.resolve-alert').on('click', function() {
                var $button = $(this);
                var alertId = $button.data('alert-id');
                
                if (confirm('<?php _e('Mark this alert as resolved?', 'wc-delcampe-integration'); ?>')) {
                    $.post(ajaxurl, {
                        action: 'delcampe_resolve_alert',
                        alert_id: alertId,
                        _ajax_nonce: '<?php echo wp_create_nonce('delcampe_resolve_alert'); ?>'
                    }, function(response) {
                        if (response.success) {
                            $button.closest('tr').fadeOut();
                        } else {
                            alert('<?php _e('Error:', 'wc-delcampe-integration'); ?> ' + response.data);
                        }
                    });
                }
            });
        });
        </script>
        <?php
    }
    
    /**
     * Render batch queue tab
     * 
     * @since  1.10.24.0
     * 
     * Displays the batch queue monitor within the inventory management interface
     */
    private function render_batch_queue_tab() {
        // Load batch monitor if not already loaded
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-batch-monitor.php';
        $batch_monitor = Delcampe_Batch_Monitor::get_instance();
        
        // Render the batch queue content directly
        $batch_monitor->render_batch_queue_content();
    }
    
    /**
     * Render cleanup tab
     * 
     * @since  1.10.5.5
     */
    private function render_cleanup_tab() {
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/cleanup-orphaned-listings-page.php';
    }
    
    /**
     * AJAX handler for inventory check
     * 
     * @since  1.5.0.0
     */
    public function ajax_check_inventory() {
        check_ajax_referer('delcampe_check_inventory');
        
        if (isset($_POST['product_id'])) {
            // Check single product
            $result = $this->inventory_manager->manual_inventory_check($_POST['product_id']);
            
            if (isset($result['error'])) {
                wp_send_json_error($result['error']);
            } else {
                wp_send_json_success(__('Inventory check completed', 'wc-delcampe-integration'));
            }
        } elseif (isset($_POST['check_all'])) {
            // Check all products
            $this->inventory_manager->run_inventory_check();
            wp_send_json_success(__('All inventory checked successfully', 'wc-delcampe-integration'));
        } else {
            // Quick check
            $this->inventory_manager->run_inventory_check();
            wp_send_json_success();
        }
    }
    
    /**
     * AJAX handler for inventory sync
     * 
     * @since  1.5.0.0
     */
    public function ajax_sync_inventory() {
        check_ajax_referer('delcampe_sync_inventory');
        
        if (isset($_POST['product_id'])) {
            // Sync single product
            $product = wc_get_product($_POST['product_id']);
            if ($product) {
                $this->inventory_manager->sync_stock_to_delcampe($product);
                wp_send_json_success(__('Product synced successfully', 'wc-delcampe-integration'));
            } else {
                wp_send_json_error(__('Product not found', 'wc-delcampe-integration'));
            }
        } elseif (isset($_POST['sync_all'])) {
            // Sync all products
            $args = array(
                'post_type' => 'product',
                'posts_per_page' => -1,
                'meta_query' => array(
                    array(
                        'key' => '_delcampe_item_id',
                        'compare' => 'EXISTS'
                    )
                )
            );
            
            $products = get_posts($args);
            $synced = 0;
            
            foreach ($products as $product_post) {
                $product = wc_get_product($product_post->ID);
                if ($product) {
                    $this->inventory_manager->sync_stock_to_delcampe($product);
                    $synced++;
                }
            }
            
            wp_send_json_success(sprintf(__('Synced %d products', 'wc-delcampe-integration'), $synced));
        } else {
            wp_send_json_error(__('Invalid request', 'wc-delcampe-integration'));
        }
    }
    
    /**
     * AJAX handler for resolving alerts
     * 
     * @since  1.5.0.0
     */
    public function ajax_resolve_alert() {
        check_ajax_referer('delcampe_resolve_alert');
        
        if (!isset($_POST['alert_id'])) {
            wp_send_json_error(__('Alert ID required', 'wc-delcampe-integration'));
        }
        
        global $wpdb;
        
        $result = $wpdb->update(
            $wpdb->prefix . 'delcampe_inventory_alerts',
            array(
                'is_resolved' => 1,
                'resolved_at' => current_time('mysql')
            ),
            array('id' => $_POST['alert_id'])
        );
        
        if ($result !== false) {
            wp_send_json_success();
        } else {
            wp_send_json_error(__('Failed to resolve alert', 'wc-delcampe-integration'));
        }
    }
    
    /**
     * AJAX handler for bulk inventory operations
     * 
     * @since  1.5.0.0
     */
    public function ajax_bulk_inventory_update() {
        check_ajax_referer('delcampe_bulk_inventory');
        
        if (!isset($_POST['products']) || !is_array($_POST['products'])) {
            wp_send_json_error(__('No products selected', 'wc-delcampe-integration'));
        }
        
        $action = str_replace('delcampe_bulk_inventory_', '', $_POST['action']);
        $processed = 0;
        
        foreach ($_POST['products'] as $product_id) {
            if ($action === 'check') {
                $this->inventory_manager->manual_inventory_check($product_id);
            } elseif ($action === 'sync') {
                $product = wc_get_product($product_id);
                if ($product) {
                    $this->inventory_manager->sync_stock_to_delcampe($product);
                }
            }
            $processed++;
        }
        
        wp_send_json_success(sprintf(
            __('%d products processed', 'wc-delcampe-integration'),
            $processed
        ));
    }

    private function verify_integrity_permission() {
        if ( ! current_user_can('manage_options') ) {
            wp_send_json_error( __('Insufficient permissions', 'wc-delcampe-integration') );
        }
        check_ajax_referer('delcampe_inventory_integrity', 'nonce');
    }

    public function ajax_inventory_diagnose() {
        $this->verify_integrity_permission();
        require_once plugin_dir_path(__FILE__) . 'class-delcampe-inventory-integrity.php';
        $data = Delcampe_Inventory_Integrity::diagnose();
        wp_send_json_success($data);
    }

    public function ajax_inventory_fix_duplicates() {
        $this->verify_integrity_permission();
        $dry = isset($_POST['dry_run']) && $_POST['dry_run'] === '1';
        require_once plugin_dir_path(__FILE__) . 'class-delcampe-inventory-integrity.php';
        $data = Delcampe_Inventory_Integrity::fix_duplicates( $dry );
        wp_send_json_success($data);
    }

    public function ajax_inventory_backfill_missing() {
        $this->verify_integrity_permission();
        $dry = isset($_POST['dry_run']) && $_POST['dry_run'] === '1';
        require_once plugin_dir_path(__FILE__) . 'class-delcampe-inventory-integrity.php';
        $data = Delcampe_Inventory_Integrity::backfill_missing( $dry );
        wp_send_json_success($data);
    }

    public function ajax_inventory_add_constraints() {
        $this->verify_integrity_permission();
        $check_only = isset($_POST['check_only']) && $_POST['check_only'] === '1';
        require_once plugin_dir_path(__FILE__) . 'class-delcampe-inventory-integrity.php';
        $data = Delcampe_Inventory_Integrity::add_constraints( $check_only );
        wp_send_json_success($data);
    }
    public function ajax_inventory_reconcile_verified() {
        $this->verify_integrity_permission();
        $dry = isset($_POST['dry_run']) && $_POST['dry_run'] === '1';
        require_once plugin_dir_path(__FILE__) . 'class-delcampe-inventory-integrity.php';
        $data = Delcampe_Inventory_Integrity::reconcile_verified( $dry );
        wp_send_json_success($data);
    }
    
    /**
     * Enqueue admin assets
     * 
     * @since  1.5.0.0
     * @param  string $hook Hook suffix
     */
    public function enqueue_admin_assets($hook) {
        if (strpos($hook, 'delcampe') === false) {
            return;
        }
        // Load queue assets only when Queue Management tab is active
        if (isset($_GET['page'], $_GET['tab']) && $_GET['page'] === DELCAMPE_MENU_INVENTORY && $_GET['tab'] === 'sync-queue') {
            wp_enqueue_style('delcampe-queue-admin', DWC_PLUGIN_URL . 'assets/css/queue-admin.css', array(), DWC_VERSION);
            wp_enqueue_script('delcampe-queue-admin', DWC_PLUGIN_URL . 'assets/js/queue-admin.js', array('jquery'), DWC_VERSION, true);
        }

        // Add any specific CSS or JS for inventory management here
        wp_add_inline_style('wp-admin', '
            .delcampe-inventory-alerts { margin-top: 20px; }
            .delcampe-inventory-alerts ul { margin: 10px 0; }
            .delcampe-inventory-alerts li { margin: 5px 0; }
        ');
    }
    
    /**
     * AJAX handler for stock sync batch processing
     * 
     * @since 1.10.34.7
     */
    public function ajax_run_stock_sync_batch() {
        check_ajax_referer('delcampe_run_stock_sync');
        
        // Get the current batch state
        $batch_state = get_transient('delcampe_stock_sync_state');
        if (!$batch_state) {
            // Initialize batch state
            global $wpdb;
            
            // Count total products with ACTIVE Delcampe listings only
            // Only sync products that have published or verified listings
            $total = $wpdb->get_var(
                "SELECT COUNT(DISTINCT dl.product_id) 
                FROM {$wpdb->prefix}delcampe_listings dl
                WHERE dl.status IN ('published', 'verified')
                AND dl.delcampe_id IS NOT NULL
                AND dl.delcampe_id != ''"
            );
            
            $batch_state = array(
                'total' => $total,
                'processed' => 0,
                'last_id' => 0,
                'batch_size' => 10, // Smaller batch for AJAX to avoid timeouts
                'log' => array()
            );
            
            set_transient('delcampe_stock_sync_state', $batch_state, HOUR_IN_SECONDS);
        }
        
        // Process a batch - ONLY ACTIVE LISTINGS
        global $wpdb;
        $products = $wpdb->get_results($wpdb->prepare(
            "SELECT DISTINCT p.ID, p.post_title, dl.delcampe_id
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->prefix}delcampe_listings dl ON p.ID = dl.product_id
            WHERE p.post_type = 'product'
            AND p.post_status = 'publish'
            AND dl.status IN ('published', 'verified')
            AND dl.delcampe_id IS NOT NULL
            AND dl.delcampe_id != ''
            AND p.ID > %d
            ORDER BY p.ID ASC
            LIMIT %d",
            $batch_state['last_id'],
            $batch_state['batch_size']
        ));
        
        $log_entries = array();
        
        foreach ($products as $product_data) {
            $product = wc_get_product($product_data->ID);
            if (!$product) {
                $log_entries[] = array(
                    'message' => sprintf(__('Skipped: Product #%d not found', 'wc-delcampe-integration'), $product_data->ID),
                    'error' => true
                );
                continue;
            }
            
            // Update product meta with the correct Delcampe ID from listings table
            // This ensures consistency between the listings table and product meta
            update_post_meta($product_data->ID, '_delcampe_item_id', $product_data->delcampe_id);
            
            // Get WooCommerce stock
            $stock = $product->get_stock_quantity();
            
            // Sync to Delcampe
            $result = $this->inventory_manager->sync_stock_to_delcampe($product);
            
            if (is_wp_error($result)) {
                $log_entries[] = array(
                    'message' => sprintf(__('Failed: %s (Stock: %d) - %s', 'wc-delcampe-integration'), 
                        $product->get_name(), 
                        $stock,
                        $result->get_error_message()
                    ),
                    'error' => true
                );
            } else {
                $log_entries[] = array(
                    'message' => sprintf(__('Synced: %s (Stock: %d)', 'wc-delcampe-integration'), 
                        $product->get_name(), 
                        $stock
                    ),
                    'error' => false
                );
            }
            
            $batch_state['processed']++;
            $batch_state['last_id'] = $product_data->ID;
        }
        
        // Check if complete
        $complete = (count($products) < $batch_state['batch_size']) || 
                    ($batch_state['processed'] >= $batch_state['total']);
        
        if ($complete) {
            // Clean up
            delete_transient('delcampe_stock_sync_state');
            delete_transient('delcampe_inventory_last_processed');
        } else {
            // Save state for next batch
            set_transient('delcampe_stock_sync_state', $batch_state, HOUR_IN_SECONDS);
        }
        
        wp_send_json_success(array(
            'total' => $batch_state['total'],
            'processed' => $batch_state['processed'],
            'complete' => $complete,
            'log' => $log_entries
        ));
    }
    
    /**
     * AJAX handler for cleaning up orphaned Delcampe IDs
     * 
     * @since 1.10.34.12
     */
    public function ajax_cleanup_orphaned_ids() {
        check_ajax_referer('delcampe_cleanup_orphaned_ids');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('Permission denied', 'wc-delcampe-integration'));
        }
        
        global $wpdb;
        
        // SIMPLE APPROACH: Get all products with Delcampe IDs and check their listing status
        $products_with_ids = $wpdb->get_results(
            "SELECT DISTINCT p.ID as product_id, dl.status, dl.delcampe_id
            FROM {$wpdb->posts} p
            INNER JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
            LEFT JOIN {$wpdb->prefix}delcampe_listings dl ON p.ID = dl.product_id
            WHERE p.post_type = 'product' 
            AND pm.meta_key = '_delcampe_item_id'
            AND pm.meta_value != ''"
        );
        
        $cleaned = 0;
        $preserved = 0;
        $analyzed = count($products_with_ids);
        
        foreach ($products_with_ids as $product) {
            // If no listing record OR listing is ended, remove the IDs
            if (!$product->status || in_array($product->status, ['ended', 'closed', 'sold', 'closed_manually', 'closed_unsold'])) {
                // Remove all Delcampe-related meta
                delete_post_meta($product->product_id, '_delcampe_item_id');
                delete_post_meta($product->product_id, '_delcampe_listing_id');
                delete_post_meta($product->product_id, '_delcampe_personal_reference');
                delete_post_meta($product->product_id, '_delcampe_sync_status');
                
                delcampe_log(sprintf(
                    '[Cleanup] Removed IDs from product #%d (status: %s)',
                    $product->product_id,
                    $product->status ?: 'no listing'
                ));
                
                $cleaned++;
            } else {
                $preserved++;
            }
        }
        
        delcampe_log(sprintf(
            '[Cleanup] Completed - Analyzed: %d, Cleaned: %d, Preserved: %d',
            $analyzed,
            $cleaned,
            $preserved
        ));
        
        wp_send_json_success(array(
            'message' => sprintf(
                'Cleanup complete: Removed IDs from %d ended/orphaned products. %d active listings preserved.',
                $cleaned,
                $preserved
            ),
            'analyzed' => $analyzed,
            'cleaned' => $cleaned,
            'preserved' => $preserved
        ));
    }
    
    /**
     * AJAX handler for stock cache refresh (v1.10.35.10)
     * 
     * @since  1.10.35.10
     */
    public function ajax_refresh_stock_cache() {
        // Verify nonce
        check_ajax_referer('delcampe_ajax_nonce', 'security');
        
        // Check permissions
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('Permission denied', 'wc-delcampe-integration'));
        }
        
        // Get stock cache manager instance
        if (!class_exists('Delcampe_Stock_Cache_Manager')) {
            require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-stock-cache-manager.php';
        }
        
        $cache_manager = Delcampe_Stock_Cache_Manager::get_instance();
        
        // Run batch refresh (50 items by default)
        $results = $cache_manager->batch_refresh_stock_cache();
        
        // Log the manual refresh
        delcampe_log('[Stock Cache] Manual refresh triggered from admin - ' . json_encode($results));
        
        // Send response
        if (is_array($results)) {
            wp_send_json_success(array(
                'message' => sprintf(
                    __('Stock cache refreshed: %d items processed, %d successful, %d failed', 'wc-delcampe-integration'),
                    $results['total'],
                    $results['success'],
                    $results['failed']
                ),
                'results' => $results
            ));
        } else {
            wp_send_json_error(__('Failed to refresh stock cache', 'wc-delcampe-integration'));
        }
    }
}
