<?php
/**
 * Delcampe Inventory Integrity Tools
 *
 * Diagnostics and safe fixes for duplicate mappings and missing listings.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class Delcampe_Inventory_Integrity {

    public static function diagnose() {
        global $wpdb;
        $table = $wpdb->prefix . DELCAMPE_TABLE_LISTINGS;

        // Duplicates by delcampe_id
        $dups = $wpdb->get_results(
            "SELECT delcampe_id, COUNT(*) AS cnt
             FROM {$table}
             WHERE delcampe_id IS NOT NULL AND delcampe_id <> ''
             GROUP BY delcampe_id
             HAVING cnt > 1",
            ARRAY_A
        );

        // Detailed duplicate rows
        $duplicate_rows = array();
        if ($dups) {
            $ids = array_map(function($r){ return $r['delcampe_id']; }, $dups);
            $in  = implode(",", array_map(function($v){ return "'" . esc_sql($v) . "'"; }, $ids));
            $rows = $wpdb->get_results("SELECT * FROM {$table} WHERE delcampe_id IN ({$in}) ORDER BY delcampe_id, date_created ASC", ARRAY_A);
            $duplicate_rows = $rows ?: array();
        }

        // Remote open items from Delcampe (fetch full map and cache for reuse)
        $remote_map = self::fetch_open_items_map();
        if ( is_wp_error( $remote_map ) ) {
            $remote_error = $remote_map->get_error_message();
            $remote_ids = array();
            $remote_map = array();
        } else {
            $remote_error = '';
            $remote_ids = array_keys($remote_map);
            set_transient('delcampe_integrity_remote_cache', $remote_map, 10 * MINUTE_IN_SECONDS);
        }

        // Local unique ids
        $local_ids = $wpdb->get_col("SELECT DISTINCT delcampe_id FROM {$table} WHERE delcampe_id IS NOT NULL AND delcampe_id <> ''");
        $local_ids = array_map('strval', $local_ids ?: array());

        // Missing locally = in remote but not in local
        $missing_locally = array_values(array_diff($remote_ids, $local_ids));

        return array(
            'duplicate_counts' => $dups ?: array(),
            'duplicate_rows' => $duplicate_rows,
            'remote_total' => count($remote_ids),
            'local_unique_total' => count($local_ids),
            'missing_locally' => $missing_locally,
            'remote_error' => $remote_error,
        );
    }

    public static function fix_duplicates( $dry_run = true ) {
        global $wpdb;
        $table = $wpdb->prefix . DELCAMPE_TABLE_LISTINGS;
        $changes = array();

        $dups = $wpdb->get_results(
            "SELECT delcampe_id FROM {$table}
             WHERE delcampe_id IS NOT NULL AND delcampe_id <> ''
             GROUP BY delcampe_id HAVING COUNT(*) > 1",
            ARRAY_A
        );

        foreach ( $dups as $dup ) {
            $dc_id = $dup['delcampe_id'];
            $rows = $wpdb->get_results( $wpdb->prepare(
                "SELECT id, product_id, delcampe_id, date_created FROM {$table} WHERE delcampe_id = %s ORDER BY date_created ASC",
                $dc_id
            ), ARRAY_A );
            if ( count($rows) < 2 ) { continue; }
            // Keep the oldest
            $keep = array_shift($rows);
            foreach ( $rows as $lose ) {
                $changes[] = array(
                    'action' => 'nullify_delcampe_id',
                    'row_id' => $lose['id'],
                    'delcampe_id' => $dc_id,
                    'product_id' => $lose['product_id'],
                    'keep_row_id' => $keep['id'],
                );
                if ( ! $dry_run ) {
                    $wpdb->update(
                        $table,
                        array( 'delcampe_id' => null ),
                        array( 'id' => $lose['id'] ),
                        array( '%s' ),
                        array( '%d' )
                    );
                    if ( class_exists('Delcampe_Business_Logger') ) {
                        Delcampe_Business_Logger::get_instance()->log_event('integrity', 'duplicate_resolved', array(
                            'dup_delcampe_id' => $dc_id,
                            'kept_row_id' => $keep['id'],
                            'nulled_row_id' => $lose['id'],
                        ));
                    }
                }
            }
        }

        return array( 'dry_run' => $dry_run, 'changes' => $changes, 'count' => count($changes) );
    }

    public static function backfill_missing( $dry_run = true ) {
        global $wpdb;
        $table = $wpdb->prefix . DELCAMPE_TABLE_LISTINGS;

        // Get cached remote items map if available to avoid re-fetch
        $remote_items = get_transient('delcampe_integrity_remote_cache');
        if ( ! is_array($remote_items) || empty($remote_items) ) {
            $remote_items = self::fetch_open_items_map(); // id => ['id','personal_reference','title']
            if ( is_wp_error($remote_items) ) {
                return array('dry_run' => $dry_run, 'error' => $remote_items->get_error_message());
            }
            set_transient('delcampe_integrity_remote_cache', $remote_items, 10 * MINUTE_IN_SECONDS);
        }
        $diagnostic = self::diagnose();
        $missing_ids = $diagnostic['missing_locally'];

        // Prefer a fast path: for small sets, lookup by SKU/title directly per ID to avoid scanning all products
        $fast_path = count($missing_ids) > 0 && count($missing_ids) <= 1000;
        $index = array('sku' => array(), 'title' => array());
        if ( ! $fast_path ) {
            // Build local product indexes (SKU/title)
            $index = self::build_local_product_index();
        }

        $proposals = array();
        $linked = 0;
        $inserted = 0;

        foreach ( $missing_ids as $dc_id ) {
            $item = isset($remote_items[$dc_id]) ? $remote_items[$dc_id] : array('id'=>$dc_id, 'personal_reference'=>'', 'title'=>'');
            $match = null; $reason = '';
            // Try SKU/personal_reference
            $ref = isset($item['personal_reference']) ? trim((string)$item['personal_reference']) : '';
            if ( $ref !== '' ) {
                if ( $fast_path ) {
                    // Direct lookup by SKU meta for this reference
                    $pid = $wpdb->get_var( $wpdb->prepare(
                        "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key='_sku' AND meta_value=%s LIMIT 1",
                        $ref
                    ) );
                    if ( $pid ) {
                        $match = array('ID' => (int)$pid, 'post_title' => get_the_title((int)$pid), 'sku' => $ref);
                        $reason = 'sku';
                    }
                } elseif ( isset($index['sku'][$ref]) ) {
                    $match = $index['sku'][$ref];
                    $reason = 'sku';
                }
            }
            // If no SKU, try exact title match (case-insensitive)
            if ( ! $match ) {
                $title = isset($item['title']) ? trim((string)$item['title']) : '';
                $tkey = mb_strtolower($title);
                if ( $fast_path && $tkey !== '' ) {
                    $pid = $wpdb->get_var( $wpdb->prepare(
                        "SELECT ID FROM {$wpdb->posts} WHERE post_type='product' AND post_status='publish' AND LOWER(post_title)=%s LIMIT 1",
                        $tkey
                    ) );
                    if ( $pid ) {
                        $match = array('ID' => (int)$pid, 'post_title' => get_the_title((int)$pid), 'sku' => get_post_meta((int)$pid, '_sku', true));
                        $reason = 'title';
                    }
                } elseif ( $tkey !== '' && isset($index['title'][$tkey]) ) {
                    $match = $index['title'][$tkey];
                    $reason = 'title';
                }
            }

            $proposal = array(
                'delcampe_id' => $dc_id,
                'personal_reference' => $ref,
                'title' => isset($item['title']) ? $item['title'] : '',
                'proposed_product_id' => $match ? (int)$match['ID'] : null,
                'proposed_product_title' => $match ? $match['post_title'] : null,
                'proposed_product_sku' => $match ? $match['sku'] : null,
                'match_reason' => $reason ?: null,
            );
            $proposals[] = $proposal;

            if ( ! $dry_run ) {
                $wpdb->insert(
                    $table,
                    array(
                        'delcampe_id' => $dc_id,
                        'product_id' => $proposal['proposed_product_id'],
                        'status' => 'published',
                        'date_created' => current_time('mysql'),
                        'date_updated' => current_time('mysql'),
                    ),
                    array('%s','%d','%s','%s','%s')
                );
                if ( ! empty($proposal['proposed_product_id']) ) { $linked++; }
                $inserted++;
                if ( class_exists('Delcampe_Business_Logger') ) {
                    Delcampe_Business_Logger::get_instance()->log_event('integrity', 'backfill_missing', array(
                        'delcampe_id' => $dc_id,
                        'linked_product_id' => $proposal['proposed_product_id'],
                        'reason' => $reason,
                    ));
                }
            }
        }

        return array(
            'dry_run' => $dry_run,
            'proposals' => $proposals,
            'proposed_count' => count($proposals),
            'inserted' => $inserted,
            'linked' => $linked,
        );
    }

    public static function add_constraints( $check_only = false ) {
        global $wpdb;
        $table = $wpdb->prefix . DELCAMPE_TABLE_LISTINGS;
        $messages = array();

        // Unique on delcampe_id if not exists
        $has_unique = $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(1) FROM INFORMATION_SCHEMA.STATISTICS WHERE TABLE_SCHEMA=%s AND TABLE_NAME=%s AND INDEX_NAME='unique_delcampe_id'",
            DB_NAME, $table
        ) );
        if ( ! $has_unique ) {
            $messages[] = 'Will add unique index unique_delcampe_id(delcampe_id)';
            if ( ! $check_only ) {
                $wpdb->query( "ALTER TABLE {$table} ADD UNIQUE KEY unique_delcampe_id (delcampe_id)" );
            }
        } else {
            $messages[] = 'Unique index on delcampe_id already exists';
        }

        // Helpful composite index for active lookup (not unique due to MySQL limitations)
        $has_idx = $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(1) FROM INFORMATION_SCHEMA.STATISTICS WHERE TABLE_SCHEMA=%s AND TABLE_NAME=%s AND INDEX_NAME='idx_product_status'",
            DB_NAME, $table
        ) );
        if ( ! $has_idx ) {
            $messages[] = 'Will add index idx_product_status(product_id, status)';
            if ( ! $check_only ) {
                $wpdb->query( "CREATE INDEX idx_product_status ON {$table} (product_id, status)" );
            }
        } else {
            $messages[] = 'Index idx_product_status already exists';
        }

        return array( 'check_only' => $check_only, 'messages' => $messages );
    }

    private static function fetch_open_item_ids() {
        // Reuse reconciliation fetcher to avoid duplicating logic
        if ( ! class_exists('Delcampe_Reconciliation') ) {
            require_once plugin_dir_path( __FILE__ ) . 'class-delcampe-reconciliation.php';
        }
        $rec = Delcampe_Reconciliation::get_instance();
        $items = $rec ? self::call_private_fetch($rec) : new WP_Error('no_recon', 'Reconciliation unavailable');
        if ( is_wp_error($items) ) return $items;
        $ids = array();
        foreach ( $items as $it ) {
            if ( ! empty($it['id']) ) { $ids[] = (string)$it['id']; }
        }
        return array_values(array_unique($ids));
    }

    private static function fetch_open_items_map() {
        if ( ! class_exists('Delcampe_Reconciliation') ) {
            require_once plugin_dir_path( __FILE__ ) . 'class-delcampe-reconciliation.php';
        }
        $rec = Delcampe_Reconciliation::get_instance();
        $fn = function() { return $this->fetch_all_delcampe_items(); };
        $bound = $fn->bindTo( $rec, get_class($rec) );
        $items = $bound();
        if ( is_wp_error($items) ) return $items;
        $map = array();
        foreach ( (array)$items as $it ) {
            if ( ! empty($it['id']) ) {
                $map[(string)$it['id']] = $it;
            }
        }
        return $map;
    }

    private static function build_local_product_index() {
        global $wpdb;
        $rows = $wpdb->get_results(
            "SELECT p.ID, p.post_title,
                    MAX(CASE WHEN pm.meta_key = '_sku' THEN pm.meta_value END) AS sku
             FROM {$wpdb->posts} p
             LEFT JOIN {$wpdb->postmeta} pm ON pm.post_id = p.ID AND pm.meta_key = '_sku'
             WHERE p.post_type = 'product' AND p.post_status = 'publish'
             GROUP BY p.ID, p.post_title",
            ARRAY_A
        );
        $sku_map = array();
        $title_map = array();
        foreach ( $rows as $r ) {
            if ( ! empty($r['sku']) ) {
                $sku_map[(string)$r['sku']] = $r;
            }
            $tkey = mb_strtolower(trim((string)$r['post_title']));
            if ( $tkey !== '' && ! isset($title_map[$tkey]) ) {
                $title_map[$tkey] = $r;
            }
        }
        return array('sku' => $sku_map, 'title' => $title_map);
    }

    /**
     * Reconcile locally 'verified' listings by linking remote Delcampe IDs via SKU/personal_reference and promote to 'published'.
     * Returns a preview list in dry-run; applies updates when not dry-run.
     */
    public static function reconcile_verified( $dry_run = true ) {
        global $wpdb;
        $table = $wpdb->prefix . DELCAMPE_TABLE_LISTINGS;

        // Fetch remote items map and index by personal_reference
        $remote = self::fetch_open_items_map();
        if ( is_wp_error($remote) ) {
            return array('dry_run'=>$dry_run, 'error'=>$remote->get_error_message());
        }
        $remoteByRef = array();
        foreach ( $remote as $it ) {
            $ref = isset($it['personal_reference']) ? trim((string)$it['personal_reference']) : '';
            if ($ref !== '') $remoteByRef[$ref] = $it;
        }

        // Find local candidates: status verified (or processing) without delcampe_id
        $candidates = $wpdb->get_results(
            "SELECT id, product_id, delcampe_id, status FROM {$table}
             WHERE (status = 'verified' OR status = 'processing' OR status IS NULL)
             AND (delcampe_id IS NULL OR delcampe_id = '')",
            ARRAY_A
        );
        if (empty($candidates)) {
            return array('dry_run'=>$dry_run, 'promoted'=>0, 'preview'=>array());
        }

        $promoted = 0; $preview = array();
        foreach ( $candidates as $row ) {
            $pid = (int)$row['product_id'];
            $sku = $pid ? get_post_meta($pid, '_sku', true) : '';
            $found = null; $reason = '';
            if ($sku && isset($remoteByRef[$sku])) { $found = $remoteByRef[$sku]; $reason = 'sku'; }
            if (!$found && $pid) {
                $title = get_the_title($pid);
                if ($title) {
                    $tkey = mb_strtolower($title);
                    foreach ($remote as $it) {
                        if (isset($it['title']) && mb_strtolower($it['title']) === $tkey) { $found = $it; $reason = 'title'; break; }
                    }
                }
            }
            $preview[] = array(
                'listing_row_id' => (int)$row['id'],
                'product_id' => $pid,
                'sku' => $sku,
                'proposed_delcampe_id' => $found ? (string)$found['id'] : null,
                'reason' => $reason ?: null,
            );
            if ($found && ! $dry_run) {
                // Update listings table
                $wpdb->update(
                    $table,
                    array(
                        'delcampe_id' => (string)$found['id'],
                        'status' => 'published',
                        'date_published' => current_time('mysql'),
                        'date_updated' => current_time('mysql'),
                    ),
                    array('id' => (int)$row['id']),
                    array('%s','%s','%s','%s'),
                    array('%d')
                );
                // Update product meta
                if ($pid) {
                    update_post_meta($pid, '_delcampe_item_id', (string)$found['id']);
                }
                $promoted++;
                if ( class_exists('Delcampe_Business_Logger') ) {
                    Delcampe_Business_Logger::get_instance()->log_event('integrity', 'promote_verified', array(
                        'listing_row_id' => (int)$row['id'],
                        'product_id' => $pid,
                        'delcampe_id' => (string)$found['id'],
                        'reason' => $reason,
                    ));
                }
            }
        }
        return array('dry_run'=>$dry_run, 'promoted'=>$promoted, 'preview'=>$preview);
    }

    private static function call_private_fetch( $rec ) {
        // Call private method via closure binding for reuse
        $fn = function() { return $this->fetch_all_delcampe_items(); };
        $bound = $fn->bindTo( $rec, get_class($rec) );
        return $bound();
    }
}
