<?php
/**
 * Delcampe Listing Tracker
 * 
 * Provides audit logging, idempotency, and verification for listing operations
 * to prevent duplicates and ensure every Delcampe Item ID is captured.
 * 
 * @package     WooCommerce_Delcampe_Integration
 * @subpackage  Core
 * @since       1.10.18.13
 * @version     1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Listing_Tracker {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Log directory path
     */
    private $log_dir;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->log_dir = WP_CONTENT_DIR . '/delcampe-sync-logs';
        
        // Ensure log directory exists
        if (!file_exists($this->log_dir)) {
            wp_mkdir_p($this->log_dir);
        }
    }
    
    /**
     * Log API audit entry
     * 
     * Creates a write-ahead log entry for API operations
     * 
     * @param string $action Operation type (create, update, delete)
     * @param int $product_id Product ID
     * @param array $request Request data
     * @param mixed $response Response data or WP_Error
     * @return string|null Extracted Delcampe ID if found
     */
    public function log_api_audit($action, $product_id, $request, $response = null) {
        $log_entry = array(
            'timestamp' => current_time('mysql'),
            'action' => $action,
            'product_id' => $product_id,
            'sku' => get_post_meta($product_id, '_sku', true),
            'personal_reference' => isset($request['item']['personal_reference']) ? $request['item']['personal_reference'] : '',
            'idempotency_key' => $this->get_idempotency_key($product_id),
            'request_summary' => $this->summarize_request($request),
        );
        
        if ($response !== null) {
            if (is_wp_error($response)) {
                $log_entry['response_error'] = $response->get_error_message();
                $log_entry['response_code'] = $response->get_error_code();
            } else {
                $log_entry['response_code'] = isset($response['status']) ? $response['status'] : 200;
                $log_entry['delcampe_id'] = $this->extract_item_id($response);
                $log_entry['success'] = isset($response['success']) ? $response['success'] : false;
                
                // Store critical response data
                if (!empty($response['id'])) {
                    $log_entry['delcampe_id'] = $response['id'];
                }
            }
        }
        
        // Write to JSON lines log file
        $log_file = $this->log_dir . '/api-audit-' . date('Y-m-d') . '.jsonl';
        $result = file_put_contents($log_file, json_encode($log_entry) . "\n", FILE_APPEND | LOCK_EX);
        
        // Also write to WordPress transient for quick access
        if (!empty($log_entry['delcampe_id'])) {
            set_transient('delcampe_item_' . $product_id, $log_entry['delcampe_id'], DAY_IN_SECONDS);
        }
        
        return isset($log_entry['delcampe_id']) ? $log_entry['delcampe_id'] : null;
    }
    
    /**
     * Check idempotency before creating listing
     * 
     * Prevents duplicate creation attempts within a time window
     * 
     * v1.10.23.0 - Extended idempotency window from 5 minutes to 2 hours
     * This prevents duplicate listings when batch processing takes longer
     * 
     * @param int $product_id Product ID
     * @return bool True if should proceed with creation
     */
    public function should_create_listing($product_id) {
        $idempotency_key = $this->get_idempotency_key($product_id);
        
        // Check if recently attempted (2 hour window - was 5 minutes)
        // Extended to prevent duplicates during long batch processing
        $existing = get_transient('delcampe_idem_' . $idempotency_key);
        if ($existing) {
            // Log the duplicate attempt
            $this->log_api_audit('create_blocked', $product_id, array(
                'reason' => 'idempotency',
                'idempotency_key' => $idempotency_key,
                'time_remaining' => $existing - time()
            ));
            return false;
        }
        
        // Check if product already has a Delcampe ID
        $existing_id = get_post_meta($product_id, '_delcampe_item_id', true);
        if (!empty($existing_id)) {
            $this->log_api_audit('create_blocked', $product_id, array(
                'reason' => 'already_exists', 
                'delcampe_id' => $existing_id
            ));
            return false;
        }
        
        // Set idempotency flag for 2 hours (7200 seconds, was 300)
        // This ensures that even if batch processing takes a long time,
        // we won't create duplicates if the same product is processed again
        set_transient('delcampe_idem_' . $idempotency_key, time() + 7200, 7200);
        
        return true;
    }
    
    /**
     * Store listing with verification and retry
     * 
     * Ensures the Delcampe ID is properly stored with retry logic
     * 
     * @param int $product_id Product ID
     * @param string $delcampe_id Delcampe Item ID
     * @param array $additional_meta Additional metadata to store
     * @param int $max_retries Maximum retry attempts
     * @return bool Success status
     */
    public function store_listing_with_verification($product_id, $delcampe_id, $additional_meta = array(), $max_retries = 3) {
        if (empty($delcampe_id)) {
            return false;
        }
        
        for ($i = 0; $i < $max_retries; $i++) {
            // Update all relevant meta fields
            update_post_meta($product_id, '_delcampe_item_id', $delcampe_id);
            update_post_meta($product_id, '_delcampe_listing_id', $delcampe_id); // Redundancy for compatibility
            update_post_meta($product_id, '_delcampe_sync_status', 'published');
            update_post_meta($product_id, '_delcampe_last_sync', current_time('timestamp'));
            
            // Store additional meta
            foreach ($additional_meta as $key => $value) {
                update_post_meta($product_id, $key, $value);
            }
            
            // Verify write succeeded
            $stored_id = get_post_meta($product_id, '_delcampe_item_id', true);
            if ($stored_id == $delcampe_id) {
                // Also update/create listing record in table
                $this->ensure_listing_record($product_id, $delcampe_id, $additional_meta);
                
                // Log successful storage
                $this->log_api_audit('storage_verified', $product_id, array('delcampe_id' => $delcampe_id));
                
                return true;
            }
            
            // Brief pause before retry
            if ($i < $max_retries - 1) {
                sleep(1);
            }
        }
        
        // If all retries fail, schedule Action Scheduler job if available
        if (function_exists('as_enqueue_async_action')) {
            as_enqueue_async_action('delcampe_verify_listing', array(
                'product_id' => $product_id,
                'delcampe_id' => $delcampe_id,
                'additional_meta' => $additional_meta
            ));
        }
        
        // Log storage failure
        $this->log_api_audit('storage_failed', $product_id, array(
            'delcampe_id' => $delcampe_id,
            'retries' => $max_retries
        ));
        
        return false;
    }
    
    /**
     * Ensure listing record exists in database table
     * 
     * @param int $product_id Product ID
     * @param string $delcampe_id Delcampe Item ID
     * @param array $meta Additional metadata
     */
    private function ensure_listing_record($product_id, $delcampe_id, $meta = array()) {
        global $wpdb;
        
        $table = $wpdb->prefix . 'delcampe_listings';
        
        // Check if table exists
        if ($wpdb->get_var("SHOW TABLES LIKE '$table'") != $table) {
            return;
        }
        
        // Check if record exists
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM $table WHERE product_id = %d OR delcampe_id = %s",
            $product_id,
            $delcampe_id
        ));
        
        if (!$existing) {
            // Insert new record
            $wpdb->insert($table, array(
                'product_id' => $product_id,
                'delcampe_id' => $delcampe_id,
                'personal_reference' => isset($meta['_delcampe_personal_reference']) ? $meta['_delcampe_personal_reference'] : get_post_meta($product_id, '_sku', true),
                'status' => 'published',
                'quantity' => isset($meta['quantity']) ? $meta['quantity'] : 1,
                'price' => isset($meta['price']) ? $meta['price'] : get_post_meta($product_id, '_price', true),
                'date_created' => current_time('mysql'),
                'date_updated' => current_time('mysql')
            ));
        } else {
            // Update existing record
            $wpdb->update($table, array(
                'delcampe_id' => $delcampe_id,
                'status' => 'published',
                'date_updated' => current_time('mysql')
            ), array('product_id' => $product_id));
        }
    }
    
    /**
     * Get idempotency key for a product
     * 
     * @param int $product_id Product ID
     * @return string Idempotency key
     */
    private function get_idempotency_key($product_id) {
        $sku = get_post_meta($product_id, '_sku', true);
        $price = get_post_meta($product_id, '_price', true);
        return md5($product_id . '|' . $sku . '|' . $price);
    }
    
    /**
     * Extract Delcampe Item ID from response
     * 
     * @param mixed $response Response data
     * @return string|null Item ID if found
     */
    private function extract_item_id($response) {
        if (is_array($response)) {
            // Check various possible locations
            if (!empty($response['id'])) {
                return $response['id'];
            }
            if (!empty($response['item_id'])) {
                return $response['item_id'];
            }
            if (!empty($response['id_item'])) {
                return $response['id_item'];
            }
            if (!empty($response['listing_id'])) {
                return $response['listing_id'];
            }
        }
        return null;
    }
    
    /**
     * Summarize request for logging
     * 
     * Handles both field naming conventions used in our API calls:
     * - fixed_price vs price
     * - qty vs quantity
     * 
     * @param array $request Request data
     * @return array Summary
     */
    private function summarize_request($request) {
        $summary = array();
        
        if (isset($request['type'])) {
            $summary['type'] = $request['type'];
        }
        
        if (isset($request['item'])) {
            $item = $request['item'];
            
            // Title (truncate for summary)
            $summary['title'] = isset($item['title']) ? substr($item['title'], 0, 50) : '';
            
            // Price - check all possible field names used in our codebase
            if (isset($item['fixed_price'])) {
                $summary['price'] = $item['fixed_price'];
            } elseif (isset($item['price'])) {
                $summary['price'] = $item['price'];
            } elseif (isset($item['price_present'])) {
                $summary['price'] = $item['price_present'];
            } elseif (isset($item['best_bid'])) {
                $summary['price'] = $item['best_bid'];
            } else {
                $summary['price'] = '';
            }
            
            // Quantity - check both field names used in our API calls
            if (isset($item['qty'])) {
                $summary['quantity'] = $item['qty'];
            } elseif (isset($item['quantity'])) {
                $summary['quantity'] = $item['quantity'];
            } else {
                $summary['quantity'] = 1; // Default to 1 if not specified
            }
            
            // Personal reference (SKU)
            $summary['personal_reference'] = isset($item['personal_reference']) ? $item['personal_reference'] : '';
            
            // Category if present
            if (isset($item['id_category'])) {
                $summary['category_id'] = $item['id_category'];
            }
        }
        
        return $summary;
    }
    
    /**
     * Find missing Delcampe IDs from audit log
     * 
     * Searches audit logs for successful creates that may not be in database
     * 
     * @param int $days Number of days to look back
     * @return array Array of found IDs with product mapping
     */
    public function find_missing_ids_from_logs($days = 7) {
        $missing = array();
        
        for ($i = 0; $i < $days; $i++) {
            $date = date('Y-m-d', strtotime("-$i days"));
            $log_file = $this->log_dir . '/api-audit-' . $date . '.jsonl';
            
            if (!file_exists($log_file)) {
                continue;
            }
            
            $lines = file($log_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            foreach ($lines as $line) {
                $entry = json_decode($line, true);
                
                if (!$entry || $entry['action'] !== 'create' || empty($entry['delcampe_id'])) {
                    continue;
                }
                
                // Check if this ID is missing from database
                $product_id = $entry['product_id'];
                $stored_id = get_post_meta($product_id, '_delcampe_item_id', true);
                
                if (empty($stored_id)) {
                    $missing[$product_id] = array(
                        'delcampe_id' => $entry['delcampe_id'],
                        'sku' => $entry['sku'],
                        'timestamp' => $entry['timestamp']
                    );
                }
            }
        }
        
        return $missing;
    }
}

// Initialize on plugins_loaded
add_action('plugins_loaded', function() {
    Delcampe_Listing_Tracker::get_instance();
});