<?php
/**
 * Delcampe Listings Importer
 *
 * Handles importing existing listings from Delcampe account
 *
 * @package WC_Delcampe_Integration
 * @version 1.9.0.6
 * @since   1.9.0.6
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Delcampe_Listings_Importer class
 *
 * @since 1.9.0.6
 */
class Delcampe_Listings_Importer {
    
    /**
     * Auth instance
     * @var Delcampe_Auth
     */
    private $auth;
    
    /**
     * API base URL
     * @var string
     */
    const API_BASE_URL = 'https://rest.delcampe.net/';
    
    /**
     * Constructor
     */
    public function __construct() {
        // Don't initialize auth in constructor - do it lazily when needed
        
        // Add AJAX handlers
        add_action('wp_ajax_delcampe_fetch_listings', array($this, 'ajax_fetch_listings'));
        add_action('wp_ajax_delcampe_import_listing', array($this, 'ajax_import_listing'));
        add_action('wp_ajax_delcampe_match_products', array($this, 'ajax_match_products'));
    }
    
    /**
     * Get auth instance (lazy loading)
     */
    private function get_auth() {
        if (!$this->auth) {
            // Make sure auth class is loaded
            if (!class_exists('Delcampe_Auth')) {
                $auth_file = plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-auth.php';
                if (file_exists($auth_file)) {
                    require_once $auth_file;
                }
            }
            
            if (class_exists('Delcampe_Auth')) {
                $this->auth = Delcampe_Auth::get_instance();
            }
        }
        return $this->auth;
    }
    
    /**
     * Render the import page
     */
    public function render_page() {
        // Check license for import feature (Pro/Enterprise only)
        if (!delcampe_has_feature('import_listings')) {
            ?>
            <div class="wrap">
                <h1><?php esc_html_e('Import Delcampe Listings', 'wc-delcampe-integration'); ?></h1>
                <div class="notice notice-error">
                    <p>
                        <strong><?php esc_html_e('Professional or Enterprise License Required', 'wc-delcampe-integration'); ?></strong>
                    </p>
                    <p><?php 
                        printf(
                            esc_html__('The Import from Delcampe feature is only available with Professional or Enterprise licenses. %sUpgrade your license%s to access this feature.', 'wc-delcampe-integration'),
                            '<a href="' . admin_url('admin.php?page=delcampe-settings&tab=license') . '">',
                            '</a>'
                        ); 
                    ?></p>
                </div>
            </div>
            <?php
            return;
        }
        
        // Check if API is configured
        $api_key = get_option('delcampe_api_key', '');
        if (empty($api_key)) {
            ?>
            <div class="wrap">
                <h1><?php esc_html_e('Import Delcampe Listings', 'wc-delcampe-integration'); ?></h1>
                <div class="notice notice-error">
                    <p><?php 
                        printf(
                            esc_html__('Please configure your Delcampe API key in the %ssettings%s first.', 'wc-delcampe-integration'),
                            '<a href="' . admin_url('admin.php?page=delcampe-settings') . '">',
                            '</a>'
                        ); 
                    ?></p>
                </div>
            </div>
            <?php
            return;
        }
        
        // Enqueue scripts
        wp_enqueue_script(
            'delcampe-importer',
            plugin_dir_url(dirname(__FILE__)) . 'assets/js/listings-importer.js',
            array('jquery'),
            DELCAMPE_PLUGIN_VERSION,
            true
        );
        
        wp_localize_script('delcampe-importer', 'delcampe_importer', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('delcampe_importer_nonce'),
            'strings' => array(
                'fetching' => __('Fetching listings from Delcampe...', 'wc-delcampe-integration'),
                'importing' => __('Importing listing %d of %d...', 'wc-delcampe-integration'),
                'matching' => __('Matching products...', 'wc-delcampe-integration'),
                'complete' => __('Import complete!', 'wc-delcampe-integration'),
                'error' => __('An error occurred:', 'wc-delcampe-integration'),
                'confirm_import' => __('Are you sure you want to import these listings?', 'wc-delcampe-integration')
            )
        ));
        
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Import Delcampe Listings', 'wc-delcampe-integration'); ?></h1>
            
            <div class="delcampe-importer-container">
                
                <div class="card">
                    <h2><?php esc_html_e('Import Options', 'wc-delcampe-integration'); ?></h2>
                    
                    <p><?php esc_html_e('This tool will fetch your active listings from Delcampe and attempt to match them with products in your WooCommerce store.', 'wc-delcampe-integration'); ?></p>
                    
                    <table class="form-table">
                        <tr>
                            <th scope="row"><?php esc_html_e('Import Status', 'wc-delcampe-integration'); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" id="import-active" value="1" checked>
                                    <?php esc_html_e('Active listings', 'wc-delcampe-integration'); ?>
                                </label><br>
                                <label>
                                    <input type="checkbox" id="import-ended" value="1">
                                    <?php esc_html_e('Ended listings', 'wc-delcampe-integration'); ?>
                                </label><br>
                                <label>
                                    <input type="checkbox" id="import-sold" value="1">
                                    <?php esc_html_e('Sold listings', 'wc-delcampe-integration'); ?>
                                </label>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Matching Method', 'wc-delcampe-integration'); ?></th>
                            <td>
                                <select id="matching-method">
                                    <option value="sku"><?php esc_html_e('Match by SKU', 'wc-delcampe-integration'); ?></option>
                                    <option value="title"><?php esc_html_e('Match by Title', 'wc-delcampe-integration'); ?></option>
                                    <option value="personal_ref"><?php esc_html_e('Match by Personal Reference', 'wc-delcampe-integration'); ?></option>
                                    <option value="manual"><?php esc_html_e('Manual matching', 'wc-delcampe-integration'); ?></option>
                                </select>
                                <p class="description"><?php esc_html_e('How to match Delcampe listings with WooCommerce products.', 'wc-delcampe-integration'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Create Products', 'wc-delcampe-integration'); ?></th>
                            <td>
                                <label>
                                    <input type="checkbox" id="create-products" value="1">
                                    <?php esc_html_e('Create new products for unmatched listings', 'wc-delcampe-integration'); ?>
                                </label>
                                <p class="description"><?php esc_html_e('If checked, new WooCommerce products will be created for listings that cannot be matched.', 'wc-delcampe-integration'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th scope="row"><?php esc_html_e('Profile Assignment', 'wc-delcampe-integration'); ?></th>
                            <td>
                                <?php
                                global $wpdb;
                                $profiles = $wpdb->get_results("SELECT profile_id, profile_name FROM {$wpdb->prefix}delcampe_profiles ORDER BY profile_name");
                                ?>
                                <select id="default-profile">
                                    <option value=""><?php esc_html_e('-- No default profile --', 'wc-delcampe-integration'); ?></option>
                                    <?php foreach ($profiles as $profile) : ?>
                                        <option value="<?php echo esc_attr($profile->profile_id); ?>">
                                            <?php echo esc_html($profile->profile_name); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <p class="description"><?php esc_html_e('Default profile to assign to imported listings without a profile.', 'wc-delcampe-integration'); ?></p>
                            </td>
                        </tr>
                    </table>
                    
                    <p class="submit">
                        <button type="button" id="fetch-listings-btn" class="button button-primary">
                            <?php esc_html_e('Fetch Listings from Delcampe', 'wc-delcampe-integration'); ?>
                        </button>
                        <span class="spinner"></span>
                    </p>
                </div>
                
                <div id="import-progress" class="card" style="display:none;">
                    <h2><?php esc_html_e('Import Progress', 'wc-delcampe-integration'); ?></h2>
                    <div class="progress-bar">
                        <div class="progress-bar-inner" style="width: 0%;"></div>
                    </div>
                    <div class="progress-text"></div>
                </div>
                
                <div id="listings-preview" class="card" style="display:none;">
                    <h2><?php esc_html_e('Listings Preview', 'wc-delcampe-integration'); ?></h2>
                    <p class="listings-count"></p>
                    
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th class="check-column"><input type="checkbox" id="select-all-listings"></th>
                                <th><?php esc_html_e('Listing ID', 'wc-delcampe-integration'); ?></th>
                                <th><?php esc_html_e('Title', 'wc-delcampe-integration'); ?></th>
                                <th><?php esc_html_e('Status', 'wc-delcampe-integration'); ?></th>
                                <th><?php esc_html_e('Price', 'wc-delcampe-integration'); ?></th>
                                <th><?php esc_html_e('Matched Product', 'wc-delcampe-integration'); ?></th>
                            </tr>
                        </thead>
                        <tbody id="listings-table-body">
                            <!-- Listings will be inserted here -->
                        </tbody>
                    </table>
                    
                    <p class="submit">
                        <button type="button" id="import-selected-btn" class="button button-primary" disabled>
                            <?php esc_html_e('Import Selected Listings', 'wc-delcampe-integration'); ?>
                        </button>
                        <button type="button" id="cancel-import-btn" class="button">
                            <?php esc_html_e('Cancel', 'wc-delcampe-integration'); ?>
                        </button>
                        <span class="spinner"></span>
                    </p>
                </div>
                
                <div id="import-results" class="card" style="display:none;">
                    <h2><?php esc_html_e('Import Results', 'wc-delcampe-integration'); ?></h2>
                    <div class="results-summary"></div>
                    <div class="results-details"></div>
                    
                    <p class="submit">
                        <a href="<?php echo admin_url('admin.php?page=delcampe-listings'); ?>" class="button button-primary">
                            <?php esc_html_e('View Imported Listings', 'wc-delcampe-integration'); ?>
                        </a>
                        <button type="button" id="import-more-btn" class="button">
                            <?php esc_html_e('Import More Listings', 'wc-delcampe-integration'); ?>
                        </button>
                    </p>
                </div>
                
            </div>
        </div>
        
        <style>
            .delcampe-importer-container .card {
                max-width: 1200px;
                margin-top: 20px;
            }
            .progress-bar {
                width: 100%;
                height: 20px;
                background: #f0f0f0;
                border: 1px solid #ccc;
                margin: 10px 0;
            }
            .progress-bar-inner {
                height: 100%;
                background: #0073aa;
                transition: width 0.3s ease;
            }
            .progress-text {
                margin: 10px 0;
                font-style: italic;
            }
            #listings-table-body tr.matched {
                background-color: #e7f7e7;
            }
            #listings-table-body tr.no-match {
                background-color: #fff3cd;
            }
            .results-summary {
                padding: 15px;
                background: #f0f8ff;
                border-left: 4px solid #0073aa;
                margin: 10px 0;
            }
        </style>
        <?php
    }
    
    /**
     * AJAX handler to fetch listings from Delcampe
     */
    public function ajax_fetch_listings() {
        // Enable error reporting for debugging
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_reporting(E_ALL);
            ini_set('display_errors', 0);
            ini_set('log_errors', 1);
        }
        
        try {
            check_ajax_referer('delcampe_importer_nonce', 'nonce');
            
            if (!current_user_can('manage_options')) {
                wp_send_json_error(array('message' => 'Unauthorized'));
                return;
            }
            
            // Check license for import feature (Pro/Enterprise only)
            if (!delcampe_has_feature('import_listings')) {
                wp_send_json_error(array('message' => __('Professional or Enterprise license required for import feature.', 'wc-delcampe-integration')));
                return;
            }
            
            $include_active = !empty($_POST['include_active']);
            $include_ended = !empty($_POST['include_ended']);
            $include_sold = !empty($_POST['include_sold']);
            
            // Get authentication token
            $auth = $this->get_auth();
            if (!$auth) {
                wp_send_json_error(array('message' => __('Failed to load authentication module.', 'wc-delcampe-integration')));
                return;
            }
            
            $token = $auth->get_auth_token();
            if (!$token) {
                wp_send_json_error(array('message' => __('Failed to authenticate with Delcampe API.', 'wc-delcampe-integration')));
                return;
            }
        
        $listings = array();
        $page = 1;
        $per_page = 100;
        $has_more = true;
        
        while ($has_more && count($listings) < 1000) { // Limit to 1000 listings for safety
            $params = array(
                'token' => $token,
                'page' => $page,
                'perPage' => $per_page
            );
            
            // Add status filters
            $statuses = array();
            if ($include_active) $statuses[] = 'active';
            if ($include_ended) $statuses[] = 'ended';
            if ($include_sold) $statuses[] = 'sold';
            
            if (!empty($statuses)) {
                $params['status'] = implode(',', $statuses);
            }
            
            // Determine which endpoint to use based on status
            $endpoint = 'item/opened'; // default
            if (!empty($statuses)) {
                if (count($statuses) === 1) {
                    // Use specific endpoint for single status
                    switch ($statuses[0]) {
                        case 'active':
                            $endpoint = 'item/opened';
                            break;
                        case 'ended':
                            $endpoint = 'item/closed';
                            break;
                        case 'sold':
                            $endpoint = 'item/sold';
                            break;
                    }
                } else {
                    // For multiple statuses, we'll need to make multiple requests
                    // For now, just use opened items
                    $endpoint = 'item/opened';
                }
            }
            
            // Build query params for API
            $query_params = array(
                'token' => $token,
                'startingItem' => ($page - 1) * $per_page,
                'numberOfItems' => $per_page
            );
            
            $response = wp_remote_get(
                self::API_BASE_URL . $endpoint . '?' . http_build_query($query_params),
                array(
                    'headers' => array(
                        'Accept' => 'application/xml'
                    ),
                    'timeout' => 30
                )
            );
            
            if (is_wp_error($response)) {
                wp_send_json_error(array('message' => $response->get_error_message()));
                return;
            }
            
            $body = wp_remote_retrieve_body($response);
            $xml = simplexml_load_string($body);
            
            if (!$xml) {
                wp_send_json_error(array('message' => __('Failed to parse API response.', 'wc-delcampe-integration')));
                return;
            }
            
            // Check for API errors in the notification structure
            if (isset($xml->Notification_Data->body->error)) {
                $error_msg = (string)$xml->Notification_Data->body->error;
                wp_send_json_error(array('message' => $error_msg));
                return;
            }
            
            // Extract listings from response
            // The API returns items wrapped in Notification_Data->body
            if (isset($xml->Notification_Data->body->item)) {
                foreach ($xml->Notification_Data->body->item as $item) {
                    // Determine price based on item type
                    $price = 0;
                    if (!empty($item->fixed_price) && (float)$item->fixed_price > 0) {
                        $price = (float)$item->fixed_price;
                    } elseif (!empty($item->price_present)) {
                        $price = (float)$item->price_present;
                    } elseif (!empty($item->price_starting)) {
                        $price = (float)$item->price_starting;
                    }
                    
                    $listings[] = array(
                        'id' => (string)$item->id_item,
                        'title' => (string)$item->title,
                        'status' => (string)$item->status,
                        'price' => $price,
                        'currency' => (string)$item->currency,
                        'personal_reference' => (string)$item->personal_reference,
                        'category_id' => (string)$item->id_category,
                        'start_date' => (string)$item->date_start,
                        'end_date' => (string)$item->date_end,
                        'quantity' => isset($item->qty) ? (int)$item->qty : 1
                    );
                }
            }
            
            // Check if there are more pages
            // The API doesn't return total count, so check if we got a full page
            $items_in_response = isset($xml->Notification_Data->body->item) ? count($xml->Notification_Data->body->item) : 0;
            $has_more = ($items_in_response == $per_page);
            $page++;
        }
        
            wp_send_json_success(array(
                'listings' => $listings,
                'total' => count($listings)
            ));
            
        } catch (Exception $e) {
            error_log('[Delcampe Importer] Error in ajax_fetch_listings: ' . $e->getMessage());
            error_log('[Delcampe Importer] Stack trace: ' . $e->getTraceAsString());
            wp_send_json_error(array(
                'message' => 'Server error: ' . $e->getMessage(),
                'debug' => WP_DEBUG ? $e->getTraceAsString() : null
            ));
        }
    }
    
    /**
     * AJAX handler to match products
     */
    public function ajax_match_products() {
        check_ajax_referer('delcampe_importer_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $listings = isset($_POST['listings']) ? $_POST['listings'] : array();
        $method = isset($_POST['method']) ? sanitize_text_field($_POST['method']) : 'sku';
        
        $matched = array();
        
        foreach ($listings as $listing) {
            $product_id = null;
            
            switch ($method) {
                case 'sku':
                    // Try to match by SKU using personal reference
                    if (!empty($listing['personal_reference'])) {
                        $product_id = wc_get_product_id_by_sku($listing['personal_reference']);
                    }
                    break;
                    
                case 'title':
                    // Try to match by title
                    $args = array(
                        'post_type' => 'product',
                        'post_status' => 'publish',
                        's' => $listing['title'],
                        'posts_per_page' => 1,
                        'orderby' => 'relevance'
                    );
                    $products = get_posts($args);
                    if (!empty($products)) {
                        $product_id = $products[0]->ID;
                    }
                    break;
                    
                case 'personal_ref':
                    // Try to match by personal reference in meta
                    if (!empty($listing['personal_reference'])) {
                        $args = array(
                            'post_type' => 'product',
                            'post_status' => 'publish',
                            'meta_key' => '_delcampe_personal_reference',
                            'meta_value' => $listing['personal_reference'],
                            'posts_per_page' => 1
                        );
                        $products = get_posts($args);
                        if (!empty($products)) {
                            $product_id = $products[0]->ID;
                        }
                    }
                    break;
            }
            
            $matched[$listing['id']] = $product_id;
        }
        
        wp_send_json_success(array('matched' => $matched));
    }
    
    /**
     * AJAX handler to import a single listing
     */
    public function ajax_import_listing() {
        check_ajax_referer('delcampe_importer_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        // Check license for import feature (Pro/Enterprise only)
        if (!delcampe_has_feature('import_listings')) {
            wp_send_json_error(array('message' => __('Professional or Enterprise license required for import feature.', 'wc-delcampe-integration')));
            return;
        }
        
        $listing_data = isset($_POST['listing']) ? $_POST['listing'] : array();
        $product_id = isset($_POST['product_id']) ? intval($_POST['product_id']) : 0;
        $profile_id = isset($_POST['profile_id']) ? intval($_POST['profile_id']) : null;
        $create_products = isset($_POST['create_products']) ? filter_var($_POST['create_products'], FILTER_VALIDATE_BOOLEAN) : false;
        
        if (empty($listing_data['id'])) {
            wp_send_json_error(array('message' => __('Invalid listing data.', 'wc-delcampe-integration')));
            return;
        }
        
        // If no product matched and create_products is enabled, create a new product
        if ($product_id === 0 && $create_products) {
            $product = new WC_Product_Simple();
            $product->set_name($listing_data['title']);
            $product->set_status('publish');
            
            if (!empty($listing_data['personal_reference'])) {
                $product->set_sku($listing_data['personal_reference']);
            }
            
            if ($listing_data['price'] > 0) {
                $product->set_price($listing_data['price']);
                $product->set_regular_price($listing_data['price']);
            }
            
            if ($listing_data['quantity'] > 0) {
                $product->set_manage_stock(true);
                $product->set_stock_quantity($listing_data['quantity']);
                $product->set_stock_status('instock');
            }
            
            $product_id = $product->save();
            
            if (!$product_id) {
                wp_send_json_error(array('message' => __('Failed to create product.', 'wc-delcampe-integration')));
                return;
            }
            
            // Try to attach images from Delcampe
            if (!empty($listing_data['id'])) {
                $this->attach_product_images($product_id, $listing_data['id']);
            }
        }
        
        global $wpdb;
        $table = $wpdb->prefix . 'delcampe_listings';
        
        // Check if listing already exists
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM $table WHERE delcampe_id = %s",
            $listing_data['id']
        ));
        
        if ($existing) {
            // Update existing listing
            $result = $wpdb->update(
                $table,
                array(
                    'product_id' => $product_id,
                    'profile_id' => $profile_id,
                    'status' => $listing_data['status'],
                    'listing_title' => $listing_data['title'],
                    'price' => $listing_data['price'],
                    'currency' => $listing_data['currency'],
                    'quantity' => $listing_data['quantity'],
                    'start_date' => $listing_data['start_date'],
                    'end_date' => $listing_data['end_date'],
                    'category_id' => $listing_data['category_id'],
                    'personal_reference' => $listing_data['personal_reference'],
                    'date_updated' => current_time('mysql')
                ),
                array('delcampe_id' => $listing_data['id'])
            );
        } else {
            // Insert new listing
            $result = $wpdb->insert(
                $table,
                array(
                    'delcampe_id' => $listing_data['id'],
                    'product_id' => $product_id,
                    'profile_id' => $profile_id,
                    'status' => $listing_data['status'],
                    'listing_title' => $listing_data['title'],
                    'price' => $listing_data['price'],
                    'currency' => $listing_data['currency'],
                    'quantity' => $listing_data['quantity'],
                    'start_date' => $listing_data['start_date'],
                    'end_date' => $listing_data['end_date'],
                    'category_id' => $listing_data['category_id'],
                    'personal_reference' => $listing_data['personal_reference'],
                    'date_created' => current_time('mysql'),
                    'date_updated' => current_time('mysql')
                )
            );
        }
        
        if ($result === false) {
            wp_send_json_error(array('message' => __('Failed to import listing.', 'wc-delcampe-integration')));
            return;
        }
        
        // If product is matched, update product meta
        if ($product_id > 0) {
            update_post_meta($product_id, '_delcampe_listing_id', $listing_data['id']);
            update_post_meta($product_id, '_delcampe_personal_reference', $listing_data['personal_reference']);
            
            if ($profile_id) {
                update_post_meta($product_id, '_delcampe_profile_id', $profile_id);
            }
        }
        
        wp_send_json_success(array(
            'message' => __('Listing imported successfully.', 'wc-delcampe-integration'),
            'listing_id' => $listing_data['id']
        ));
    }
    
    /**
     * Attach images from Delcampe to product
     * 
     * @param int $product_id Product ID
     * @param string $delcampe_id Delcampe listing ID
     */
    private function attach_product_images($product_id, $delcampe_id) {
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        
        // Convert Delcampe ID to image URL path
        $padded = str_pad($delcampe_id, 12, '0', STR_PAD_LEFT);
        $parts = str_split($padded, 3);
        $path = implode('/', $parts);
        
        $attached = 0;
        
        // Try to get up to 5 images
        for ($i = 1; $i <= 5; $i++) {
            $image_num = str_pad($i, 3, '0', STR_PAD_LEFT);
            $image_url = "https://www.delcampe.net/static/img_large/auction/{$path}_{$image_num}.jpg";
            
            // Check if image exists
            $response = wp_remote_head($image_url, array('timeout' => 5));
            
            if (is_wp_error($response)) {
                continue;
            }
            
            $code = wp_remote_retrieve_response_code($response);
            if ($code !== 200) {
                break; // No more images
            }
            
            // Download and attach image
            $tmp = download_url($image_url . '?v=' . time());
            
            if (is_wp_error($tmp)) {
                continue;
            }
            
            $file_array = array(
                'name' => 'delcampe-' . $delcampe_id . '-' . $i . '.jpg',
                'tmp_name' => $tmp
            );
            
            $attachment_id = media_handle_sideload($file_array, $product_id);
            
            if (is_wp_error($attachment_id)) {
                @unlink($tmp);
                continue;
            }
            
            // First image becomes featured
            if ($attached === 0) {
                set_post_thumbnail($product_id, $attachment_id);
            } else {
                // Add to gallery
                $gallery = get_post_meta($product_id, '_product_image_gallery', true);
                $gallery = empty($gallery) ? $attachment_id : $gallery . ',' . $attachment_id;
                update_post_meta($product_id, '_product_image_gallery', $gallery);
            }
            
            $attached++;
        }
        
        return $attached;
    }
}