<?php
/**
 * Delcampe Listings Model
 *
 * Handles all database operations for Delcampe listings
 *
 * @package WooCommerce_Delcampe_Integration
 * @version 1.6.2.0
 * 
 * Changelog:
 * 1.6.2.0 - Added personal_reference column to database schema
 *         - This field is used by Delcampe API to store merchant's personal reference for the listing
 *         - Added migration logic to add column to existing installations
 *         - Updated version numbers and added comprehensive comments
 * 1.6.0.1 - Initial listings model implementation
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

/**
 * Delcampe_Listings_Model class
 * 
 * Provides database operations for managing Delcampe listings.
 * This includes creating, reading, updating, and deleting listings.
 * 
 * @since 1.6.0.0
 * @version 1.6.2.0
 */
class Delcampe_Listings_Model {
    
    /**
     * Table name
     *
     * @var string
     */
    private static $table_name;
    
    /**
     * Initialize the model
     * 
     * Sets up the table name for database operations.
     * 
     * @since 1.6.0.0
     * @version 1.6.2.0
     */
    public static function init() {
        global $wpdb;
        self::$table_name = $wpdb->prefix . 'delcampe_listings';
    }
    
    /**
     * Create the listings table
     * 
     * Creates or updates the database table structure for listings.
     * Now includes personal_reference column for Delcampe API integration.
     * 
     * @since 1.6.0.0
     * @version 1.6.2.0 - Added personal_reference column
     */
    public static function create_table() {
        global $wpdb;
        
        $table_name = self::get_table_name();
        $charset_collate = $wpdb->get_charset_collate();
        
        // Create table with all required columns including personal_reference
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id int(11) NOT NULL AUTO_INCREMENT,
            delcampe_id varchar(100) DEFAULT NULL,
            product_id int(11) DEFAULT NULL,
            parent_id int(11) DEFAULT NULL,
            listing_title varchar(255) DEFAULT NULL,
            status varchar(50) DEFAULT NULL,
            account_id int(11) DEFAULT 1,
            profile_id int(11) DEFAULT NULL,
            quantity int(11) DEFAULT NULL,
            quantity_sold int(11) DEFAULT 0,
            price decimal(10,2) DEFAULT NULL,
            currency varchar(10) DEFAULT 'EUR',
            start_date datetime DEFAULT NULL,
            end_date datetime DEFAULT NULL,
            date_published datetime DEFAULT NULL,
            date_finished datetime DEFAULT NULL,
            category_id varchar(50) DEFAULT NULL,
            category_name varchar(255) DEFAULT NULL,
            shipping_method varchar(50) DEFAULT NULL,
            condition_id varchar(50) DEFAULT NULL,
            personal_reference varchar(255) DEFAULT NULL,
            details longtext,
            profile_data longtext,
            last_errors longtext,
            history longtext,
            locked tinyint(1) DEFAULT 0,
            view_item_url varchar(255) DEFAULT NULL,
            relist_date datetime DEFAULT NULL,
            date_created datetime DEFAULT CURRENT_TIMESTAMP,
            date_updated datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY product_id (product_id),
            KEY delcampe_id (delcampe_id),
            KEY status (status),
            KEY account_id (account_id),
            KEY profile_id (profile_id),
            KEY personal_reference (personal_reference)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // Check if personal_reference column exists (for upgrades from older versions)
        self::ensure_personal_reference_column();
        
        // Log table creation
        // Table created/updated successfully
    }
    
    /**
     * Ensure personal_reference column exists
     * 
     * Adds the personal_reference column if it doesn't exist.
     * This handles upgrades from versions prior to 1.6.2.0.
     * 
     * @since 1.6.2.0
     */
    private static function ensure_personal_reference_column() {
        global $wpdb;
        $table_name = self::get_table_name();
        
        // Check if column exists
        $column_exists = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
                WHERE TABLE_SCHEMA = %s 
                AND TABLE_NAME = %s 
                AND COLUMN_NAME = 'personal_reference'",
                DB_NAME,
                $table_name
            )
        );
        
        // Add column if it doesn't exist
        if (!$column_exists) {
            $wpdb->query(
                "ALTER TABLE $table_name 
                ADD COLUMN personal_reference VARCHAR(255) DEFAULT NULL AFTER condition_id,
                ADD INDEX personal_reference (personal_reference)"
            );
            // Personal_reference column added successfully
        }
    }
    
    /**
     * Get table name
     * 
     * Returns the full table name with WordPress prefix.
     * 
     * @since 1.6.0.0
     * @version 1.6.2.0
     * @return string Full table name
     */
    public static function get_table_name() {
        global $wpdb;
        if (empty(self::$table_name)) {
            self::$table_name = $wpdb->prefix . 'delcampe_listings';
        }
        return self::$table_name;
    }
    
    /**
     * Get a single listing by ID
     * 
     * Retrieves a listing record by its ID.
     * 
     * @since 1.6.0.0
     * @version 1.6.2.0
     * @param int $id Listing ID
     * @return object|null Listing object or null if not found
     */
    public static function get_listing($id) {
        global $wpdb;
        $table_name = self::get_table_name();
        
        $listing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE id = %d",
            $id
        ));
        
        return $listing;
    }
    
    /**
     * Get listing by Delcampe ID
     * 
     * Retrieves a listing by its Delcampe marketplace ID.
     * 
     * @since 1.6.0.0
     * @version 1.6.2.0
     * @param string $delcampe_id Delcampe listing ID
     * @return object|null Listing object or null if not found
     */
    public static function get_listing_by_delcampe_id($delcampe_id) {
        global $wpdb;
        $table_name = self::get_table_name();
        
        $listing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE delcampe_id = %s",
            $delcampe_id
        ));
        
        return $listing;
    }
    
    /**
     * Get listing by personal reference
     * 
     * Retrieves a listing by its personal reference.
     * The personal reference is a merchant-defined identifier for the listing.
     * 
     * @since 1.6.2.0
     * @param string $personal_reference Personal reference
     * @return object|null Listing object or null if not found
     */
    public static function get_listing_by_personal_reference($personal_reference) {
        global $wpdb;
        $table_name = self::get_table_name();
        
        $listing = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM $table_name WHERE personal_reference = %s",
            $personal_reference
        ));
        
        return $listing;
    }
    
    /**
     * Get listings by product ID
     * 
     * Retrieves all listings associated with a WooCommerce product.
     * 
     * @since 1.6.0.0
     * @version 1.6.2.0
     * @param int $product_id WooCommerce product ID
     * @return array Array of listing objects
     */
    public static function get_listings_by_product_id($product_id) {
        global $wpdb;
        $table_name = self::get_table_name();
        
        $listings = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM $table_name WHERE product_id = %d OR parent_id = %d",
            $product_id,
            $product_id
        ));
        
        return $listings;
    }
    
    /**
     * Get all listings
     * 
     * Retrieves listings based on provided query arguments.
     * 
     * @since 1.6.0.0
     * @version 1.6.2.0
     * @param array $args Query arguments
     * @return array Array of listing objects
     */
    public static function get_listings($args = array()) {
        global $wpdb;
        $table_name = self::get_table_name();
        
        $defaults = array(
            'status' => null,
            'account_id' => null,
            'profile_id' => null,
            'orderby' => 'date_created',
            'order' => 'DESC',
            'limit' => null,
            'offset' => 0,
            'search' => null,
            'locked' => null,
        );
        
        $args = wp_parse_args($args, $defaults);
        
        $where = array('1=1');
        $values = array();
        
        // Build WHERE clause
        if (!empty($args['status'])) {
            if ($args['status'] == 'error') {
                // Special handling for error status - check for entries with errors
                $where[] = 'last_errors IS NOT NULL AND last_errors != %s';
                $values[] = '';
            } else {
                $where[] = 'status = %s';
                $values[] = $args['status'];
            }
        }
        
        if (!empty($args['account_id'])) {
            $where[] = 'account_id = %d';
            $values[] = $args['account_id'];
        }
        
        if (!empty($args['profile_id'])) {
            $where[] = 'profile_id = %d';
            $values[] = $args['profile_id'];
        }
        
        if ($args['locked'] !== null) {
            $where[] = 'locked = %d';
            $values[] = $args['locked'] ? 1 : 0;
        }
        
        if (!empty($args['search'])) {
            // Updated to include personal_reference in search
            $where[] = '(listing_title LIKE %s OR delcampe_id LIKE %s OR personal_reference LIKE %s)';
            $search_term = '%' . $wpdb->esc_like($args['search']) . '%';
            $values[] = $search_term;
            $values[] = $search_term;
            $values[] = $search_term;
        }
        
        $where_clause = implode(' AND ', $where);
        
        // Build ORDER BY clause
        $allowed_orderby = array('id', 'listing_title', 'status', 'price', 'quantity', 'quantity_sold', 'date_created', 'date_published', 'end_date', 'personal_reference');
        $orderby = in_array($args['orderby'], $allowed_orderby) ? $args['orderby'] : 'date_created';
        $order = strtoupper($args['order']) === 'ASC' ? 'ASC' : 'DESC';
        
        // Build LIMIT clause
        $limit_clause = '';
        if (!empty($args['limit'])) {
            $limit_clause = sprintf('LIMIT %d OFFSET %d', $args['limit'], $args['offset']);
        }
        
        // Build and execute query
        $query = "SELECT * FROM $table_name WHERE $where_clause ORDER BY $orderby $order $limit_clause";
        
        if (!empty($values)) {
            $query = $wpdb->prepare($query, $values);
        }
        
        $listings = $wpdb->get_results($query);
        
        return $listings;
    }
    
    /**
     * Get listings count
     * 
     * Returns the count of listings matching the query arguments.
     * 
     * @since 1.6.0.0
     * @version 1.6.2.0
     * @param array $args Query arguments
     * @return int Count of listings
     */
    public static function get_listings_count($args = array()) {
        global $wpdb;
        $table_name = self::get_table_name();
        
        $defaults = array(
            'status' => null,
            'account_id' => null,
            'profile_id' => null,
            'search' => null,
            'locked' => null,
        );
        
        $args = wp_parse_args($args, $defaults);
        
        $where = array('1=1');
        $values = array();
        
        // Build WHERE clause
        if (!empty($args['status'])) {
            if ($args['status'] == 'error') {
                // Special handling for error status
                $where[] = 'last_errors IS NOT NULL AND last_errors != %s';
                $values[] = '';
            } else {
                $where[] = 'status = %s';
                $values[] = $args['status'];
            }
        }
        
        if (!empty($args['account_id'])) {
            $where[] = 'account_id = %d';
            $values[] = $args['account_id'];
        }
        
        if (!empty($args['profile_id'])) {
            $where[] = 'profile_id = %d';
            $values[] = $args['profile_id'];
        }
        
        if ($args['locked'] !== null) {
            $where[] = 'locked = %d';
            $values[] = $args['locked'] ? 1 : 0;
        }
        
        if (!empty($args['search'])) {
            // Updated to include personal_reference in search
            $where[] = '(listing_title LIKE %s OR delcampe_id LIKE %s OR personal_reference LIKE %s)';
            $search_term = '%' . $wpdb->esc_like($args['search']) . '%';
            $values[] = $search_term;
            $values[] = $search_term;
            $values[] = $search_term;
        }
        
        $where_clause = implode(' AND ', $where);
        
        // Build and execute query
        $query = "SELECT COUNT(*) FROM $table_name WHERE $where_clause";
        
        if (!empty($values)) {
            $query = $wpdb->prepare($query, $values);
        }
        
        $count = $wpdb->get_var($query);
        
        return intval($count);
    }
    
    /**
     * Get status summary
     * 
     * Returns a summary of listings grouped by status.
     * 
     * @since 1.6.0.0
     * @version 1.6.2.0
     * @return object Summary object with counts for each status
     */
    public static function get_status_summary() {
        global $wpdb;
        $table_name = self::get_table_name();
        
        $query = "SELECT 
            COUNT(*) as total_items,
            SUM(CASE WHEN status = 'prepared' THEN 1 ELSE 0 END) as prepared,
            SUM(CASE WHEN status = 'verified' THEN 1 ELSE 0 END) as verified,
            SUM(CASE WHEN status = 'published' THEN 1 ELSE 0 END) as published,
            SUM(CASE WHEN status = 'changed' THEN 1 ELSE 0 END) as changed,
            SUM(CASE WHEN status = 'ended' THEN 1 ELSE 0 END) as ended,
            SUM(CASE WHEN status = 'sold' THEN 1 ELSE 0 END) as sold,
            SUM(CASE WHEN status = 'archived' THEN 1 ELSE 0 END) as archived,
            SUM(CASE WHEN last_errors IS NOT NULL AND last_errors != '' THEN 1 ELSE 0 END) as with_errors,
            SUM(CASE WHEN locked = 1 THEN 1 ELSE 0 END) as locked,
            SUM(CASE WHEN locked = 0 AND status = 'published' THEN 1 ELSE 0 END) as unlocked,
            SUM(CASE WHEN relist_date IS NOT NULL THEN 1 ELSE 0 END) as scheduled
        FROM $table_name";
        
        $summary = $wpdb->get_row($query);
        
        return $summary;
    }
    
    /**
     * Insert a new listing
     * 
     * Creates a new listing record in the database.
     * Now properly handles personal_reference field.
     * 
     * @since 1.6.0.0
     * @version 1.6.2.0 - Added personal_reference support
     * @param array $data Listing data
     * @return int|false The listing ID on success, false on error
     */
    public static function insert_listing($data) {
        global $wpdb;
        $table_name = self::get_table_name();
        
        // Set defaults
        $defaults = array(
            'status' => 'prepared',
            'quantity_sold' => 0,
            'locked' => 0,
            'date_created' => current_time('mysql'),
            'date_updated' => current_time('mysql'),
        );
        
        $data = wp_parse_args($data, $defaults);
        
        // Serialize complex data
        if (isset($data['details']) && is_array($data['details'])) {
            $data['details'] = maybe_serialize($data['details']);
        }
        if (isset($data['profile_data']) && is_array($data['profile_data'])) {
            $data['profile_data'] = maybe_serialize($data['profile_data']);
        }
        if (isset($data['last_errors']) && is_array($data['last_errors'])) {
            $data['last_errors'] = maybe_serialize($data['last_errors']);
        }
        if (isset($data['history']) && is_array($data['history'])) {
            $data['history'] = maybe_serialize($data['history']);
        }
        
        $result = $wpdb->insert($table_name, $data);
        
        if ($result !== false) {
            // Listing inserted successfully
            return $wpdb->insert_id;
        }
        
        // Log any database errors
        if ($wpdb->last_error) {
            // Database error occurred during insert
        }
        
        return false;
    }
    
    /**
     * Update a listing
     * 
     * Updates an existing listing record.
     * 
     * @since 1.6.0.0
     * @version 1.6.2.0
     * @param int $id Listing ID
     * @param array $data Data to update
     * @return int|false The number of rows updated, or false on error
     */
    public static function update_listing($id, $data) {
        global $wpdb;
        $table_name = self::get_table_name();
        
        // Update timestamp
        $data['date_updated'] = current_time('mysql');
        
        // Serialize complex data
        if (isset($data['details']) && is_array($data['details'])) {
            $data['details'] = maybe_serialize($data['details']);
        }
        if (isset($data['profile_data']) && is_array($data['profile_data'])) {
            $data['profile_data'] = maybe_serialize($data['profile_data']);
        }
        if (isset($data['last_errors']) && is_array($data['last_errors'])) {
            $data['last_errors'] = maybe_serialize($data['last_errors']);
        }
        if (isset($data['history']) && is_array($data['history'])) {
            $data['history'] = maybe_serialize($data['history']);
        }
        
        $result = $wpdb->update(
            $table_name,
            $data,
            array('id' => $id),
            null,
            array('%d')
        );
        
        if ($result !== false) {
            // Listing updated successfully
        }
        
        return $result;
    }
    
    /**
     * Delete a listing
     * 
     * Permanently removes a listing from the database.
     * 
     * @since 1.6.0.0
     * @version 1.6.2.0
     * @param int $id Listing ID
     * @return int|false The number of rows deleted, or false on error
     */
    public static function delete_listing($id) {
        global $wpdb;
        $table_name = self::get_table_name();
        
        $result = $wpdb->delete(
            $table_name,
            array('id' => $id),
            array('%d')
        );
        
        if ($result !== false) {
            // Listing deleted successfully
        }
        
        return $result;
    }
    
    /**
     * Prepare product for listing
     * 
     * Creates a new listing from a WooCommerce product.
     * 
     * @since 1.6.0.0
     * @version 1.6.2.0 - Added personal_reference generation
     * @param int $product_id WooCommerce product ID
     * @param int $profile_id Optional profile ID
     * @return int|false Listing ID or false on error
     */
    public static function prepare_product_for_listing($product_id, $profile_id = null) {
        $product = wc_get_product($product_id);
        
        if (!$product) {
            // Product not found for listing preparation
            return false;
        }
        
        // Check if listing already exists for this product
        $existing_listings = self::get_listings_by_product_id($product_id);
        if (!empty($existing_listings)) {
            // Listing already exists for this product
            return false;
        }
        
        // Generate personal reference from SKU or product ID
        $personal_reference = $product->get_sku();
        if (empty($personal_reference)) {
            $personal_reference = 'WC-' . $product_id;
        }
        
        // Prepare listing data
        $listing_data = array(
            'product_id' => $product_id,
            'listing_title' => $product->get_name(),
            'status' => 'prepared',
            'quantity' => $product->get_stock_quantity(),
            'price' => $product->get_price(),
            'profile_id' => $profile_id,
            'personal_reference' => $personal_reference,
        );
        
        // Handle variable products
        if ($product->is_type('variable')) {
            $listing_data['parent_id'] = $product_id;
            // For now, we'll create a single listing for the parent
            // In the future, we might want to create separate listings for each variation
        }
        
        // Insert the listing
        $listing_id = self::insert_listing($listing_data);
        
        return $listing_id;
    }
    
    /**
     * Update listing status
     * 
     * Convenience method to update only the status field.
     * 
     * @since 1.6.0.0
     * @version 1.6.2.0
     * @param int $id Listing ID
     * @param string $status New status value
     * @return bool True on success, false on failure
     */
    public static function update_status($id, $status) {
        $result = self::update_listing($id, array('status' => $status));
        return $result !== false;
    }
    
    /**
     * Lock/unlock listing
     * 
     * Sets the locked status of a listing.
     * 
     * @since 1.6.0.0
     * @version 1.6.2.0
     * @param int $id Listing ID
     * @param bool $locked Whether to lock (true) or unlock (false)
     * @return bool True on success, false on failure
     */
    public static function set_locked($id, $locked = true) {
        $result = self::update_listing($id, array('locked' => $locked ? 1 : 0));
        return $result !== false;
    }
}

// Initialize the model
Delcampe_Listings_Model::init();
