<?php
/**
 * Delcampe Order Inspector with Webhook Debugging
 * 
 * Enhanced inspector that shows notification logs alongside orders
 * Based on ChatGPT's community research findings
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.10.13.17
 */

if (!defined('ABSPATH')) exit;

class Delcampe_Order_Inspector {
    
    /**
     * Inspect an order with full webhook history
     * 
     * @param int $order_id WooCommerce order ID or Delcampe item ID
     * @return array Inspection results
     */
    public static function inspect_order($order_id) {
        $results = array(
            'order_data' => null,
            'delcampe_data' => null,
            'webhook_logs' => null,
            'recommendations' => array()
        );
        
        // Get order data
        $order = wc_get_order($order_id);
        if (!$order) {
            // Try finding by Delcampe ID
            global $wpdb;
            $table = $wpdb->prefix . 'delcampe_orders';
            $wc_order_id = $wpdb->get_var($wpdb->prepare(
                "SELECT wc_order_id FROM {$table} WHERE delcampe_order_id = %s OR delcampe_item_id = %s LIMIT 1",
                $order_id,
                $order_id
            ));
            
            if ($wc_order_id) {
                $order = wc_get_order($wc_order_id);
            }
        }
        
        if ($order) {
            $results['order_data'] = array(
                'id' => $order->get_id(),
                'status' => $order->get_status(),
                'total' => $order->get_total(),
                'created' => $order->get_date_created()->format('Y-m-d H:i:s'),
                'delcampe_order_id' => $order->get_meta('_delcampe_order_id'),
                'delcampe_item_id' => $order->get_meta('_delcampe_item_id'),
                'delcampe_bill_id' => $order->get_meta('_delcampe_bill_id'),
                'reconciliation_stage' => $order->get_meta('_delcampe_reconciliation_stage')
            );
            
            // Get webhook logs for this order's timeframe
            $results['webhook_logs'] = self::get_webhook_logs_for_order($order);
        }
        
        // Get Delcampe data
        if ($results['order_data']['delcampe_item_id']) {
            $results['delcampe_data'] = self::get_delcampe_item_status($results['order_data']['delcampe_item_id']);
        }
        
        // Generate recommendations
        $results['recommendations'] = self::generate_recommendations($results);
        
        return $results;
    }
    
    /**
     * Get webhook logs for order timeframe
     * 
     * @param WC_Order $order
     * @return array
     */
    private static function get_webhook_logs_for_order($order) {
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return array('error' => 'Authentication failed');
        }
        
        // Get order creation date and add buffer
        $created = $order->get_date_created();
        $from = $created->modify('-1 day')->format('Y-m-d');
        $to = date('Y-m-d');
        
        // Query notification logs
        $url = "https://rest.delcampe.net/notification/logs?token={$token}&from={$from}&to={$to}";
        
        $response = wp_remote_get($url, array(
            'timeout' => 30,
            'headers' => array('Accept' => 'application/xml')
        ));
        
        if (is_wp_error($response)) {
            return array('error' => $response->get_error_message());
        }
        
        $body = wp_remote_retrieve_body($response);
        $xml = simplexml_load_string($body);
        
        $logs = array();
        
        if ($xml && isset($xml->Notification_Data->body->logs)) {
            $log_entries = $xml->Notification_Data->body->logs->children();
            
            foreach ($log_entries as $log) {
                // Filter for relevant events
                $item_id = $order->get_meta('_delcampe_item_id');
                $is_relevant = false;
                
                // Check if log mentions this item
                if ($item_id) {
                    $log_xml = $log->asXML();
                    if (strpos($log_xml, $item_id) !== false) {
                        $is_relevant = true;
                    }
                }
                
                // Check notification type
                $type = isset($log->notification_type) ? (string)$log->notification_type : '';
                if (in_array($type, ['Seller_Item_Close_Sold', 'Seller_Payment_Received'])) {
                    $is_relevant = true;
                }
                
                if ($is_relevant) {
                    $logs[] = array(
                        'timestamp' => isset($log->timestamp) ? (string)$log->timestamp : '',
                        'type' => $type,
                        'status' => isset($log->status_code) ? (string)$log->status_code : '',
                        'method' => isset($log->method) ? (string)$log->method : 'POST',
                        'url' => isset($log->url) ? (string)$log->url : '',
                        'response_time' => isset($log->response_time) ? (string)$log->response_time : '',
                        'error' => isset($log->error) ? (string)$log->error : ''
                    );
                }
            }
        }
        
        return $logs;
    }
    
    /**
     * Get current Delcampe item status
     * 
     * @param string $item_id
     * @return array
     */
    private static function get_delcampe_item_status($item_id) {
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return array('error' => 'Authentication failed');
        }
        
        // Check item status
        $url = "https://rest.delcampe.net/item/{$item_id}?token={$token}";
        $response = wp_remote_get($url, array(
            'timeout' => 10,
            'headers' => array('Accept' => 'application/xml')
        ));
        
        if (is_wp_error($response)) {
            return array('error' => $response->get_error_message());
        }
        
        $body = wp_remote_retrieve_body($response);
        $xml = simplexml_load_string($body);
        
        $data = array();
        
        if ($xml && isset($xml->Notification_Data->body->item)) {
            $item = $xml->Notification_Data->body->item;
            $data['status'] = isset($item->status) ? (string)$item->status : '';
            $data['quantity'] = isset($item->quantity) ? (int)$item->quantity : 0;
            $data['quantity_sold'] = isset($item->quantity_sold) ? (int)$item->quantity_sold : 0;
        }
        
        // Check for buyer
        $buyer_url = "https://rest.delcampe.net/item/{$item_id}/buyer?token={$token}";
        $buyer_response = wp_remote_get($buyer_url, array(
            'timeout' => 10,
            'headers' => array('Accept' => 'application/xml')
        ));
        
        if (!is_wp_error($buyer_response)) {
            $buyer_body = wp_remote_retrieve_body($buyer_response);
            $buyer_xml = simplexml_load_string($buyer_body);
            
            if ($buyer_xml && isset($buyer_xml->Notification_Data->body->buyer)) {
                $buyer = $buyer_xml->Notification_Data->body->buyer;
                $data['buyer'] = array(
                    'id' => isset($buyer->buyer_id) ? (string)$buyer->buyer_id : '',
                    'nickname' => isset($buyer->buyer_nickname) ? (string)$buyer->buyer_nickname : '',
                    'payment_status' => isset($buyer->payment_status) ? (string)$buyer->payment_status : ''
                );
            }
        }
        
        return $data;
    }
    
    /**
     * Generate recommendations based on inspection
     * 
     * @param array $results
     * @return array
     */
    private static function generate_recommendations($results) {
        $recommendations = array();
        
        // Check webhook failures
        if (!empty($results['webhook_logs'])) {
            $failures = array_filter($results['webhook_logs'], function($log) {
                return !in_array($log['status'], ['200', '201', '204']);
            });
            
            if (!empty($failures)) {
                $recommendations[] = array(
                    'type' => 'warning',
                    'message' => 'Webhook failures detected',
                    'action' => 'Check webhook URL for redirects and ensure it returns 200 directly'
                );
            }
            
            // Check for slow responses
            $slow = array_filter($results['webhook_logs'], function($log) {
                return isset($log['response_time']) && intval($log['response_time']) > 5000;
            });
            
            if (!empty($slow)) {
                $recommendations[] = array(
                    'type' => 'warning',
                    'message' => 'Slow webhook responses detected (>5 seconds)',
                    'action' => 'Optimize webhook handler or use async processing'
                );
            }
        }
        
        // Check order ID pattern
        if ($results['order_data']) {
            $order_id = $results['order_data']['delcampe_order_id'];
            
            // Check for duplicate pattern
            if (preg_match('/^(\d+)-/', $order_id, $matches)) {
                $base_id = $matches[1];
                
                global $wpdb;
                $table = $wpdb->prefix . 'delcampe_orders';
                $duplicates = $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(*) FROM {$table} WHERE delcampe_order_id LIKE %s",
                    $base_id . '%'
                ));
                
                if ($duplicates > 1) {
                    $recommendations[] = array(
                        'type' => 'info',
                        'message' => "Multiple orders found for item {$base_id}",
                        'action' => 'This is normal for multi-quantity items'
                    );
                }
            }
            
            // Check reconciliation stage
            if ($results['order_data']['reconciliation_stage'] === 'initial_sale') {
                $recommendations[] = array(
                    'type' => 'info',
                    'message' => 'Order awaiting payment confirmation',
                    'action' => 'Will update when Seller_Payment_Received webhook arrives'
                );
            }
        }
        
        // Check Delcampe status
        if ($results['delcampe_data'] && !isset($results['delcampe_data']['error'])) {
            if ($results['delcampe_data']['status'] === 'open' && 
                isset($results['delcampe_data']['quantity_sold']) && 
                $results['delcampe_data']['quantity_sold'] > 0) {
                $recommendations[] = array(
                    'type' => 'info',
                    'message' => 'Item still open but has sales',
                    'action' => 'Multi-quantity item with remaining stock'
                );
            }
        }
        
        return $recommendations;
    }
    
    /**
     * Generate inspection report
     * 
     * @param int $order_id
     * @return string HTML report
     */
    public static function generate_report($order_id) {
        $inspection = self::inspect_order($order_id);
        
        ob_start();
        ?>
        <div class="delcampe-order-inspection">
            <h2>Order Inspection Report</h2>
            
            <?php if ($inspection['order_data']): ?>
            <div class="section">
                <h3>WooCommerce Order</h3>
                <table class="inspection-table">
                    <?php foreach ($inspection['order_data'] as $key => $value): ?>
                    <tr>
                        <th><?php echo esc_html(ucwords(str_replace('_', ' ', $key))); ?></th>
                        <td><?php echo esc_html($value ?: 'N/A'); ?></td>
                    </tr>
                    <?php endforeach; ?>
                </table>
            </div>
            <?php endif; ?>
            
            <?php if ($inspection['delcampe_data']): ?>
            <div class="section">
                <h3>Delcampe Status</h3>
                <pre><?php echo esc_html(json_encode($inspection['delcampe_data'], JSON_PRETTY_PRINT)); ?></pre>
            </div>
            <?php endif; ?>
            
            <?php if (!empty($inspection['webhook_logs'])): ?>
            <div class="section">
                <h3>Webhook History</h3>
                <table class="inspection-table">
                    <thead>
                        <tr>
                            <th>Time</th>
                            <th>Type</th>
                            <th>Status</th>
                            <th>Response Time</th>
                            <th>Error</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($inspection['webhook_logs'] as $log): ?>
                        <tr class="<?php echo in_array($log['status'], ['200','201','204']) ? 'success' : 'error'; ?>">
                            <td><?php echo esc_html($log['timestamp']); ?></td>
                            <td><?php echo esc_html($log['type']); ?></td>
                            <td><?php echo esc_html($log['status']); ?></td>
                            <td><?php echo esc_html($log['response_time']); ?>ms</td>
                            <td><?php echo esc_html($log['error']); ?></td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php endif; ?>
            
            <?php if (!empty($inspection['recommendations'])): ?>
            <div class="section">
                <h3>Recommendations</h3>
                <?php foreach ($inspection['recommendations'] as $rec): ?>
                <div class="recommendation <?php echo esc_attr($rec['type']); ?>">
                    <strong><?php echo esc_html($rec['message']); ?></strong><br>
                    <?php echo esc_html($rec['action']); ?>
                </div>
                <?php endforeach; ?>
            </div>
            <?php endif; ?>
        </div>
        
        <style>
        .delcampe-order-inspection {
            max-width: 1200px;
            margin: 20px;
        }
        .section {
            background: #fff;
            border: 1px solid #ccd0d4;
            padding: 20px;
            margin-bottom: 20px;
        }
        .inspection-table {
            width: 100%;
            border-collapse: collapse;
        }
        .inspection-table th {
            text-align: left;
            padding: 8px;
            background: #f0f0f1;
            border: 1px solid #ccd0d4;
        }
        .inspection-table td {
            padding: 8px;
            border: 1px solid #ccd0d4;
        }
        .success { background: #d4f4d4; }
        .error { background: #ffd4d4; }
        .recommendation {
            padding: 10px;
            margin: 10px 0;
            border-left: 4px solid;
        }
        .recommendation.warning {
            background: #fff3cd;
            border-color: #ffc107;
        }
        .recommendation.info {
            background: #d1ecf1;
            border-color: #0c5460;
        }
        </style>
        <?php
        
        return ob_get_clean();
    }
}