<?php
/**
 * Delcampe Order Reconciliation
 * 
 * Properly handles dollar amounts at each stage of the order lifecycle
 * Based on ChatGPT's analysis of Delcampe's data structure
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.10.13.16
 */

if (!defined('ABSPATH')) exit;

class Delcampe_Order_Reconciliation {
    
    /**
     * Stage A: Process initial sale (no payment yet)
     * Data from: /item/:id, /item/:id/buyer, or Seller_Item_Close_Sold
     * 
     * @param WC_Order $order WooCommerce order object
     * @param SimpleXMLElement $item Item data from Delcampe
     * @param array $buyer_info Buyer information
     * @return void
     */
    public static function reconcile_initial_sale($order, $item, $buyer_info = array()) {
        // Parse the fixed price safely
        $item_price = 0;
        if (isset($item->fixed_price)) {
            $price_string = (string)$item->fixed_price;
            // Remove any non-numeric characters except dots and commas
            $price_string = preg_replace('/[^0-9.,]/', '', $price_string);
            // Replace comma with dot for decimal
            $price_string = str_replace(',', '.', $price_string);
            $item_price = floatval($price_string);
        }
        
        // Determine quantity sold
        $qty_sold = 1;
        if (isset($item->quantity_sold)) {
            $qty_sold = intval($item->quantity_sold);
        } elseif (isset($buyer_info['quantity'])) {
            $qty_sold = intval($buyer_info['quantity']);
        }
        
        // Calculate items total
        $items_total = $item_price * $qty_sold;
        
        // At this stage, we know:
        // - Items total = sum of fixed_price × qty_sold
        // - Shipping = 0.00 (unknown until payment)
        // - Tax = 0.00 (Delcampe is tax-exempt channel)
        // - Fees = 0.00 (unknown until payment)
        
        // Clear any existing totals
        $order->set_cart_tax(0);
        $order->set_shipping_tax(0);
        $order->set_shipping_total(0);
        
        // Set the order currency
        $currency = isset($item->currency) ? (string)$item->currency : 'USD';
        $order->set_currency($currency);
        
        // Add order note about reconciliation
        $order->add_order_note(sprintf(
            __('Initial sale reconciliation:', 'wc-delcampe-integration') . "\n" .
            __('Items: %s %s (%d × %s)', 'wc-delcampe-integration') . "\n" .
            __('Shipping: %s (pending)', 'wc-delcampe-integration') . "\n" .
            __('Tax: %s (Delcampe tax-exempt)', 'wc-delcampe-integration') . "\n" .
            __('Status: On-Hold (awaiting payment)', 'wc-delcampe-integration'),
            number_format($items_total, 2),
            $currency,
            $qty_sold,
            number_format($item_price, 2),
            '0.00',
            '0.00'
        ), false, false);
        
        // Store meta for later payment reconciliation
        $order->update_meta_data('_delcampe_item_price', $item_price);
        $order->update_meta_data('_delcampe_qty_sold', $qty_sold);
        $order->update_meta_data('_delcampe_items_total', $items_total);
        $order->update_meta_data('_delcampe_reconciliation_stage', 'initial_sale');
        
        // Set order status to on-hold
        $order->set_status('on-hold', __('Awaiting payment confirmation from Delcampe', 'wc-delcampe-integration'));
        
        // Save order
        $order->save();
    }
    
    /**
     * Stage B: Process payment received
     * Data from: Seller_Payment_Received notification
     * 
     * @param WC_Order $order WooCommerce order object
     * @param SimpleXMLElement $payment_data Payment notification data
     * @return void
     */
    public static function reconcile_payment_received($order, $payment_data) {
        // Extract payment amounts
        $total_amount = 0;  // What buyer paid (items + shipping + add-ons)
        $seller_amount = 0; // What seller receives after platform fee
        
        if (isset($payment_data->total_amount)) {
            $total_amount = floatval($payment_data->total_amount);
        }
        
        if (isset($payment_data->seller_amount)) {
            $seller_amount = floatval($payment_data->seller_amount);
        }
        
        // Recompute items total from payment data if available
        $items_total = $order->get_meta('_delcampe_items_total', true);
        if (isset($payment_data->item_list) && isset($payment_data->item_list->item)) {
            $items_total = 0;
            foreach ($payment_data->item_list->item as $paid_item) {
                $price = floatval($paid_item->price);
                $qty = isset($paid_item->qty) ? intval($paid_item->qty) : 1;
                $items_total += $price * $qty;
            }
        }
        
        // Calculate fees and shipping
        // Platform fee (approximate) = total_amount - seller_amount
        $platform_fee = $total_amount - $seller_amount;
        
        // Inferred shipping = seller_amount - items_total
        // This works because: total_amount - platform_fee - items_total = shipping
        $shipping_cost = $seller_amount - $items_total;
        if ($shipping_cost < 0) {
            $shipping_cost = 0; // Safeguard against negative shipping
        }
        
        // Update order totals
        // IMPORTANT: Fees are NOT part of WooCommerce order total
        // They are stored as meta for reference only
        
        // Normalize amounts with WooCommerce helpers
        $shipping_cost = wc_format_decimal($shipping_cost, wc_get_price_decimals());
        $platform_fee = wc_format_decimal($platform_fee, wc_get_price_decimals());
        $total_amount = wc_format_decimal($total_amount, wc_get_price_decimals());
        $seller_amount = wc_format_decimal($seller_amount, wc_get_price_decimals());
        
        // Add shipping if we have it
        if ($shipping_cost > 0) {
            $shipping_item = new WC_Order_Item_Shipping();
            $shipping_item->set_method_title('Delcampe Shipping');
            $shipping_item->set_method_id('delcampe_shipping');
            $shipping_item->set_total($shipping_cost);
            $order->add_item($shipping_item);
        }
        
        // Store platform fee as meta (not added to order total)
        $order->update_meta_data('_delcampe_platform_fee', $platform_fee);
        $order->update_meta_data('_delcampe_total_amount', $total_amount);
        $order->update_meta_data('_delcampe_seller_amount', $seller_amount);
        
        // Add reconciliation note
        $order->add_order_note(sprintf(
            __('Payment reconciliation complete:', 'wc-delcampe-integration') . "\n" .
            __('Items total: %s', 'wc-delcampe-integration') . "\n" .
            __('Shipping: %s (inferred)', 'wc-delcampe-integration') . "\n" .
            __('Tax: %s', 'wc-delcampe-integration') . "\n" .
            __('Platform fee: %s (stored as meta)', 'wc-delcampe-integration') . "\n" .
            __('Buyer paid: %s', 'wc-delcampe-integration') . "\n" .
            __('Seller receives: %s', 'wc-delcampe-integration'),
            number_format($items_total, 2),
            number_format($shipping_cost, 2),
            '0.00',
            number_format($platform_fee, 2),
            number_format($total_amount, 2),
            number_format($seller_amount, 2)
        ), false, false);
        
        // Update reconciliation stage
        $order->update_meta_data('_delcampe_reconciliation_stage', 'payment_received');
        
        // Set order status to processing
        $order->set_status('processing', __('Payment confirmed by Delcampe', 'wc-delcampe-integration'));
        
        // Recalculate totals
        $order->calculate_totals(false); // false = don't calculate taxes
        $order->save();
    }
    
    /**
     * Helper: Calculate Delcampe fee based on item price
     * This is an approximation - actual fee comes in payment notification
     * 
     * @param float $price Item price
     * @return float Estimated fee
     */
    public static function estimate_delcampe_fee($price) {
        // Delcampe typical fee structure (approximate)
        // You should adjust based on your actual fee agreement
        $fee_percentage = 0.08; // 8% example
        $minimum_fee = 0.05;    // Minimum fee
        
        $fee = $price * $fee_percentage;
        if ($fee < $minimum_fee) {
            $fee = $minimum_fee;
        }
        
        return round($fee, 2);
    }
}