<?php
/**
 * Delcampe Orders Admin
 * 
 * Admin interface for managing Delcampe orders
 * Provides UI for order import, status sync, and reporting
 * 
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Admin
 * @since 1.4.0.0
 * @version 1.4.0.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

/**
 * Class Delcampe_Orders_Admin
 * 
 * Handles the admin interface for order management
 * 
 * @since 1.4.0.0
 */
class Delcampe_Orders_Admin {
    
    /**
     * Order manager instance
     * @var Delcampe_Order_Manager
     */
    private $order_manager;
    
    /**
     * Constructor
     * 
     * Initialize the admin interface
     */
    public function __construct() {
        $this->order_manager = Delcampe_Order_Manager::get_instance();
        
        // Add AJAX handlers
        add_action('wp_ajax_delcampe_import_orders', array($this, 'ajax_import_orders'));
        add_action('wp_ajax_delcampe_sync_order', array($this, 'ajax_sync_order'));
        add_action('wp_ajax_delcampe_get_order_details', array($this, 'ajax_get_order_details'));
        add_action('wp_ajax_delcampe_sync_closed_sales', array($this, 'ajax_sync_closed_sales'));
        add_action('wp_ajax_delcampe_cleanup_listings', array($this, 'ajax_cleanup_listings'));
        add_action('wp_ajax_delcampe_update_pending_orders', array($this, 'ajax_update_pending_orders'));
        
        // Enqueue scripts
        add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
    }
    
    /**
     * Enqueue admin scripts and styles
     * 
     * @param string $hook Current admin page hook
     * 
     * @since 1.4.0.0
     */
    public function enqueue_scripts($hook) {
        // Only load on our pages
        if (strpos($hook, 'delcampe-orders') === false) {
            return;
        }
        
        // Enqueue styles
        wp_enqueue_style(
            'delcampe-orders-admin',
            DWC_PLUGIN_URL . 'assets/css/orders-admin.css',
            array(),
            DELCAMPE_PLUGIN_VERSION
        );
        
        // Enqueue scripts
        wp_enqueue_script(
            'delcampe-orders-admin',
            DWC_PLUGIN_URL . 'assets/js/orders-admin.js',
            array('jquery', 'wp-util'),
            DELCAMPE_PLUGIN_VERSION,
            true
        );
        
        // Localize script
        wp_localize_script('delcampe-orders-admin', 'delcampe_orders', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('delcampe_orders_nonce'),
            'strings' => array(
                'importing' => __('Importing orders...', 'wc-delcampe-integration'),
                'import_complete' => __('Import complete!', 'wc-delcampe-integration'),
                'import_error' => __('Import failed. Please check the logs.', 'wc-delcampe-integration'),
                'confirm_import' => __('Are you sure you want to import orders from Delcampe?', 'wc-delcampe-integration'),
                'syncing' => __('Syncing order...', 'wc-delcampe-integration'),
                'sync_complete' => __('Sync complete!', 'wc-delcampe-integration'),
                'sync_error' => __('Sync failed. Please try again.', 'wc-delcampe-integration')
            )
        ));
    }
    
    /**
     * Render the orders page
     * 
     * @since 1.4.0.0
     */
    public function render_page() {
        // Get current tab
        $current_tab = isset($_GET['tab']) ? sanitize_key($_GET['tab']) : 'orders';
        
        ?>
        <div class="wrap">
            <h1><?php _e('Delcampe Orders', 'wc-delcampe-integration'); ?></h1>
            
            <!-- Tab Navigation -->
            <nav class="nav-tab-wrapper">
                <a href="?page=delcampe-orders&tab=orders" 
                   class="nav-tab <?php echo $current_tab === 'orders' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Orders', 'wc-delcampe-integration'); ?>
                </a>
                <a href="?page=delcampe-orders&tab=import" 
                   class="nav-tab <?php echo $current_tab === 'import' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Import', 'wc-delcampe-integration'); ?>
                </a>
                <a href="?page=delcampe-orders&tab=settings" 
                   class="nav-tab <?php echo $current_tab === 'settings' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Settings', 'wc-delcampe-integration'); ?>
                </a>
                <a href="?page=delcampe-orders&tab=reports" 
                   class="nav-tab <?php echo $current_tab === 'reports' ? 'nav-tab-active' : ''; ?>">
                    <?php _e('Reports', 'wc-delcampe-integration'); ?>
                </a>
            </nav>
            
            <!-- Tab Content -->
            <div class="tab-content">
                <?php
                switch ($current_tab) {
                    case 'orders':
                        $this->render_orders_tab();
                        break;
                    case 'import':
                        $this->render_import_tab();
                        break;
                    case 'settings':
                        $this->render_settings_tab();
                        break;
                    case 'reports':
                        $this->render_reports_tab();
                        break;
                }
                ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render orders list tab
     * 
     * @since 1.4.0.0
     */
    private function render_orders_tab() {
        // Display any admin notices
        settings_errors('delcampe_orders');
        
        // Create list table instance
        require_once DWC_PLUGIN_DIR . 'includes/tables/class-delcampe-orders-list-table.php';
        
        $list_table = new Delcampe_Orders_List_Table();
        $list_table->prepare_items();
        
        ?>
        <div class="delcampe-orders-list">
            <h2>
                <?php _e('Delcampe Orders', 'wc-delcampe-integration'); ?>
                <a href="?page=delcampe-orders&tab=import" class="page-title-action">
                    <?php _e('Import Orders', 'wc-delcampe-integration'); ?>
                </a>
                <button type="button" class="page-title-action" id="update-orders-btn-main">
                    <?php _e('Update Orders', 'wc-delcampe-integration'); ?>
                </button>
            </h2>
            
            <!-- Filters -->
            <form method="get">
                <input type="hidden" name="page" value="delcampe-orders">
                <input type="hidden" name="tab" value="orders">
                <?php $list_table->display(); ?>
            </form>
        </div>
        <?php
    }
    
    /**
     * Render import tab
     * 
     * @since 1.4.0.0
     */
    private function render_import_tab() {
        ?>
        <div class="delcampe-import-orders">
            <h2><?php _e('Import Orders from Delcampe', 'wc-delcampe-integration'); ?></h2>
            
            <div class="card">
                <h3><?php _e('Manual Import', 'wc-delcampe-integration'); ?></h3>
                <p><?php _e('Click the button below to manually import recent orders from Delcampe.', 'wc-delcampe-integration'); ?></p>
                
                <div class="import-controls">
                    <label for="import-limit"><?php _e('Number of orders to import:', 'wc-delcampe-integration'); ?></label>
                    <select id="import-limit" name="limit">
                        <option value="10">10</option>
                        <option value="25">25</option>
                        <option value="50" selected>50</option>
                        <option value="100">100</option>
                        <option value="250">250</option>
                    </select>
                    
                    <button type="button" class="button button-primary" id="import-orders-btn">
                        <?php _e('Import Orders', 'wc-delcampe-integration'); ?>
                    </button>
                    
                    <button type="button" class="button button-secondary" id="update-pending-orders-btn">
                        <?php _e('Update Orders', 'wc-delcampe-integration'); ?>
                    </button>
                </div>
                
                <div id="import-progress" style="display: none;">
                    <div class="spinner is-active"></div>
                    <span class="import-message"><?php _e('Importing orders...', 'wc-delcampe-integration'); ?></span>
                </div>
                
                <div id="import-results" style="display: none;">
                    <!-- Results will be displayed here -->
                </div>
            </div>
            
            <div class="card">
                <h3><?php _e('Sync Closed Listings', 'wc-delcampe-integration'); ?></h3>
                <p><?php _e('Sync closed/ended listings from Delcampe and update their status in the database.', 'wc-delcampe-integration'); ?></p>
                
                <div class="import-controls">
                    <label for="closed-sales-limit"><?php _e('Number of closed listings to sync:', 'wc-delcampe-integration'); ?></label>
                    <select id="closed-sales-limit" name="closed_limit">
                        <option value="10">10</option>
                        <option value="25">25</option>
                        <option value="50" selected>50</option>
                        <option value="100">100</option>
                        <option value="250">250</option>
                    </select>
                    
                    <button type="button" class="button button-primary" id="sync-closed-sales-btn">
                        <?php _e('Sync Closed Listings', 'wc-delcampe-integration'); ?>
                    </button>
                </div>
                
                <div id="closed-sales-progress" style="display: none;">
                    <div class="spinner is-active"></div>
                    <span class="import-message"><?php _e('Syncing closed listings...', 'wc-delcampe-integration'); ?></span>
                </div>
                
                <div id="closed-sales-results" style="display: none;">
                    <!-- Results will be displayed here -->
                </div>
            </div>
            
            <div class="card">
                <h3><?php _e('Cleanup All Listings', 'wc-delcampe-integration'); ?></h3>
                <p><?php _e('Check all listings against Delcampe and mark any that are no longer active as ended.', 'wc-delcampe-integration'); ?></p>
                <p class="description"><?php _e('This will fetch all open listings from Delcampe and compare with your database to find items that should be marked as ended.', 'wc-delcampe-integration'); ?></p>
                
                <div class="import-controls">
                    <button type="button" class="button button-primary" id="cleanup-listings-btn">
                        <?php _e('Cleanup All Listings', 'wc-delcampe-integration'); ?>
                    </button>
                </div>
                
                <div id="cleanup-progress" style="display: none;">
                    <div class="spinner is-active"></div>
                    <span class="import-message"><?php _e('Cleaning up listings...', 'wc-delcampe-integration'); ?></span>
                </div>
                
                <div id="cleanup-results" style="display: none;">
                    <!-- Results will be displayed here -->
                </div>
            </div>
            
            <div class="card">
                <h3><?php _e('Import History', 'wc-delcampe-integration'); ?></h3>
                <?php $this->render_import_history(); ?>
            </div>
            
            <div class="card">
                <h3><?php _e('Automatic Import', 'wc-delcampe-integration'); ?></h3>
                <p><?php _e('Orders are automatically imported every hour when enabled in settings.', 'wc-delcampe-integration'); ?></p>
                <?php
                $auto_import = get_option('delcampe_auto_import_orders', 'yes');
                if ($auto_import === 'yes') {
                    $next_run = wp_next_scheduled('delcampe_import_orders_cron');
                    if ($next_run) {
                        echo '<p class="description">';
                        printf(
                            __('Next automatic import: %s', 'wc-delcampe-integration'),
                            date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $next_run)
                        );
                        echo '</p>';
                    }
                } else {
                    echo '<p class="notice notice-warning">';
                    _e('Automatic import is currently disabled. Enable it in the settings tab.', 'wc-delcampe-integration');
                    echo '</p>';
                }
                ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render settings tab
     * 
     * @since 1.4.0.0
     */
    private function render_settings_tab() {
        // Handle form submission
        if (isset($_POST['save_settings']) && wp_verify_nonce($_POST['delcampe_orders_settings_nonce'], 'delcampe_orders_settings')) {
            update_option('delcampe_auto_import_orders', sanitize_text_field($_POST['auto_import_orders']));
            update_option('delcampe_order_status_mapping', $_POST['order_status_mapping']);
            update_option('delcampe_create_customers', sanitize_text_field($_POST['create_customers']));
            update_option('delcampe_order_notification_email', sanitize_email($_POST['notification_email']));
            
            echo '<div class="notice notice-success"><p>' . __('Settings saved.', 'wc-delcampe-integration') . '</p></div>';
        }
        
        // Get current settings
        $auto_import = get_option('delcampe_auto_import_orders', 'yes');
        $status_mapping = get_option('delcampe_order_status_mapping', array());
        $create_customers = get_option('delcampe_create_customers', 'yes');
        $notification_email = get_option('delcampe_order_notification_email', get_option('admin_email'));
        
        ?>
        <div class="delcampe-order-settings">
            <h2><?php _e('Order Settings', 'wc-delcampe-integration'); ?></h2>
            
            <form method="post" action="">
                <?php wp_nonce_field('delcampe_orders_settings', 'delcampe_orders_settings_nonce'); ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="auto_import_orders">
                                <?php _e('Automatic Import', 'wc-delcampe-integration'); ?>
                            </label>
                        </th>
                        <td>
                            <select name="auto_import_orders" id="auto_import_orders">
                                <option value="yes" <?php selected($auto_import, 'yes'); ?>>
                                    <?php _e('Yes', 'wc-delcampe-integration'); ?>
                                </option>
                                <option value="no" <?php selected($auto_import, 'no'); ?>>
                                    <?php _e('No', 'wc-delcampe-integration'); ?>
                                </option>
                            </select>
                            <p class="description">
                                <?php _e('Automatically import orders from Delcampe every hour.', 'wc-delcampe-integration'); ?>
                            </p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="create_customers">
                                <?php _e('Create Customers', 'wc-delcampe-integration'); ?>
                            </label>
                        </th>
                        <td>
                            <select name="create_customers" id="create_customers">
                                <option value="yes" <?php selected($create_customers, 'yes'); ?>>
                                    <?php _e('Yes', 'wc-delcampe-integration'); ?>
                                </option>
                                <option value="no" <?php selected($create_customers, 'no'); ?>>
                                    <?php _e('No', 'wc-delcampe-integration'); ?>
                                </option>
                            </select>
                            <p class="description">
                                <?php _e('Create WooCommerce customers for Delcampe buyers.', 'wc-delcampe-integration'); ?>
                            </p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="notification_email">
                                <?php _e('Notification Email', 'wc-delcampe-integration'); ?>
                            </label>
                        </th>
                        <td>
                            <input type="email" 
                                   name="notification_email" 
                                   id="notification_email" 
                                   value="<?php echo esc_attr($notification_email); ?>"
                                   class="regular-text">
                            <p class="description">
                                <?php _e('Email address to receive notifications about new orders.', 'wc-delcampe-integration'); ?>
                            </p>
                        </td>
                    </tr>
                </table>
                
                <h3><?php _e('Order Status Mapping', 'wc-delcampe-integration'); ?></h3>
                <p><?php _e('Map Delcampe order statuses to WooCommerce order statuses.', 'wc-delcampe-integration'); ?></p>
                
                <table class="form-table">
                    <tr>
                        <th><?php _e('Delcampe Status', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('WooCommerce Status', 'wc-delcampe-integration'); ?></th>
                    </tr>
                    <?php
                    $delcampe_statuses = array(
                        'sold' => __('Sold', 'wc-delcampe-integration'),
                        'paid' => __('Paid', 'wc-delcampe-integration'),
                        'shipped' => __('Shipped', 'wc-delcampe-integration'),
                        'completed' => __('Completed', 'wc-delcampe-integration')
                    );
                    
                    $wc_statuses = wc_get_order_statuses();
                    
                    foreach ($delcampe_statuses as $dc_status => $dc_label) {
                        $selected = isset($status_mapping[$dc_status]) ? $status_mapping[$dc_status] : '';
                        ?>
                        <tr>
                            <td><?php echo esc_html($dc_label); ?></td>
                            <td>
                                <select name="order_status_mapping[<?php echo esc_attr($dc_status); ?>]">
                                    <option value=""><?php _e('-- Select --', 'wc-delcampe-integration'); ?></option>
                                    <?php foreach ($wc_statuses as $wc_status => $wc_label): ?>
                                        <option value="<?php echo esc_attr($wc_status); ?>" 
                                                <?php selected($selected, $wc_status); ?>>
                                            <?php echo esc_html($wc_label); ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                            </td>
                        </tr>
                        <?php
                    }
                    ?>
                </table>
                
                <p class="submit">
                    <button type="submit" name="save_settings" class="button button-primary">
                        <?php _e('Save Settings', 'wc-delcampe-integration'); ?>
                    </button>
                </p>
            </form>
        </div>
        <?php
    }
    
    /**
     * Render reports tab
     * 
     * @since 1.4.0.0
     */
    private function render_reports_tab() {
        $stats = $this->order_manager->get_order_statistics();
        
        ?>
        <div class="delcampe-order-reports">
            <h2><?php _e('Order Reports', 'wc-delcampe-integration'); ?></h2>
            
            <!-- Statistics Cards -->
            <div class="stats-cards">
                <div class="card stat-card">
                    <h3><?php _e('Total Orders', 'wc-delcampe-integration'); ?></h3>
                    <div class="stat-value"><?php echo intval($stats['total']); ?></div>
                </div>
                
                <div class="card stat-card">
                    <h3><?php _e('Orders Today', 'wc-delcampe-integration'); ?></h3>
                    <div class="stat-value"><?php echo intval($stats['today']); ?></div>
                </div>
                
                <div class="card stat-card">
                    <h3><?php _e('This Month', 'wc-delcampe-integration'); ?></h3>
                    <div class="stat-value"><?php echo intval($stats['this_month']); ?></div>
                </div>
                
                <div class="card stat-card">
                    <h3><?php _e('Total Revenue', 'wc-delcampe-integration'); ?></h3>
                    <div class="stat-value">
                        <?php echo wc_price($stats['total_revenue']); ?>
                    </div>
                </div>
            </div>
            
            <!-- Order Status Breakdown -->
            <div class="card">
                <h3><?php _e('Orders by Status', 'wc-delcampe-integration'); ?></h3>
                <?php if (!empty($stats['by_status'])): ?>
                    <table class="widefat">
                        <thead>
                            <tr>
                                <th><?php _e('Status', 'wc-delcampe-integration'); ?></th>
                                <th><?php _e('Count', 'wc-delcampe-integration'); ?></th>
                                <th><?php _e('Percentage', 'wc-delcampe-integration'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($stats['by_status'] as $status => $count): ?>
                                <tr>
                                    <td><?php echo esc_html(ucfirst($status)); ?></td>
                                    <td><?php echo intval($count); ?></td>
                                    <td>
                                        <?php 
                                        $percentage = $stats['total'] > 0 ? ($count / $stats['total']) * 100 : 0;
                                        echo number_format($percentage, 1) . '%';
                                        ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php else: ?>
                    <p><?php _e('No order data available yet.', 'wc-delcampe-integration'); ?></p>
                <?php endif; ?>
            </div>
            
            <!-- Payment Status Breakdown -->
            <div class="card">
                <h3><?php _e('Orders by Payment Status', 'wc-delcampe-integration'); ?></h3>
                <?php if (!empty($stats['by_payment'])): ?>
                    <table class="widefat">
                        <thead>
                            <tr>
                                <th><?php _e('Payment Status', 'wc-delcampe-integration'); ?></th>
                                <th><?php _e('Count', 'wc-delcampe-integration'); ?></th>
                                <th><?php _e('Percentage', 'wc-delcampe-integration'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($stats['by_payment'] as $status => $count): ?>
                                <tr>
                                    <td><?php echo esc_html(ucfirst($status)); ?></td>
                                    <td><?php echo intval($count); ?></td>
                                    <td>
                                        <?php 
                                        $percentage = $stats['total'] > 0 ? ($count / $stats['total']) * 100 : 0;
                                        echo number_format($percentage, 1) . '%';
                                        ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php else: ?>
                    <p><?php _e('No payment data available yet.', 'wc-delcampe-integration'); ?></p>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render import history
     * 
     * @since 1.4.0.0
     */
    private function render_import_history() {
        global $wpdb;
        
        // Get recent imports
        $recent_imports = $wpdb->get_results(
            "SELECT DATE(created_at) as import_date, COUNT(*) as count 
            FROM {$wpdb->prefix}delcampe_orders 
            GROUP BY DATE(created_at) 
            ORDER BY import_date DESC 
            LIMIT 10"
        );
        
        if ($recent_imports) {
            ?>
            <table class="widefat">
                <thead>
                    <tr>
                        <th><?php _e('Date', 'wc-delcampe-integration'); ?></th>
                        <th><?php _e('Orders Imported', 'wc-delcampe-integration'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($recent_imports as $import): ?>
                        <tr>
                            <td><?php echo date_i18n(get_option('date_format'), strtotime($import->import_date)); ?></td>
                            <td><?php echo intval($import->count); ?></td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
            <?php
        } else {
            echo '<p>' . __('No import history available.', 'wc-delcampe-integration') . '</p>';
        }
    }
    
    /**
     * AJAX handler for importing orders
     * 
     * @since 1.4.0.0
     */
    public function ajax_import_orders() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'delcampe_orders_nonce')) {
            wp_die('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        $limit = isset($_POST['limit']) ? intval($_POST['limit']) : 50;
        
        // Import orders
        $result = $this->order_manager->import_orders(0, $limit);
        
        // v1.10.35.17: Also update pending orders to get current payment/shipping status
        $update_result = $this->order_manager->update_pending_orders_from_delcampe();
        
        if (is_wp_error($result)) {
            wp_send_json_error(array(
                'message' => $result->get_error_message()
            ));
        } else {
            $imported_count = isset($result['imported']) ? count($result['imported']) : count($result);
            $updated_count = isset($update_result['updated']) ? $update_result['updated'] : 0;
            
            $message = sprintf(
                __('Successfully imported %d new orders and updated %d pending orders.', 'wc-delcampe-integration'),
                $imported_count,
                $updated_count
            );
            
            wp_send_json_success(array(
                'message' => $message,
                'imported' => $imported_count,
                'updated' => $updated_count
            ));
        }
    }
    
    /**
     * AJAX handler for syncing single order
     * 
     * @since 1.4.0.0
     */
    public function ajax_sync_order() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'delcampe_orders_nonce')) {
            wp_die('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        $order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;
        
        if (!$order_id) {
            wp_send_json_error(array('message' => 'Invalid order ID'));
        }
        
        // Sync order
        // Implementation would go here
        
        wp_send_json_success(array(
            'message' => __('Order synced successfully.', 'wc-delcampe-integration')
        ));
    }
    
    /**
     * AJAX handler for getting order details
     * 
     * @since 1.4.0.0
     */
    public function ajax_get_order_details() {
        // Verify nonce
        if (!wp_verify_nonce($_POST['nonce'], 'delcampe_orders_nonce')) {
            wp_die('Security check failed');
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_die('Insufficient permissions');
        }
        
        $order_id = isset($_POST['order_id']) ? intval($_POST['order_id']) : 0;
        
        if (!$order_id) {
            wp_send_json_error(array('message' => 'Invalid order ID'));
        }
        
        global $wpdb;
        
        $order_data = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}delcampe_orders WHERE id = %d",
            $order_id
        ));
        
        if (!$order_data) {
            wp_send_json_error(array('message' => 'Order not found'));
        }
        
        // Get WooCommerce order if exists
        $wc_order = null;
        if ($order_data->wc_order_id) {
            $wc_order = wc_get_order($order_data->wc_order_id);
        }
        
        wp_send_json_success(array(
            'order' => $order_data,
            'wc_order' => $wc_order ? array(
                'id' => $wc_order->get_id(),
                'status' => $wc_order->get_status(),
                'total' => $wc_order->get_total(),
                'view_url' => $wc_order->get_edit_order_url()
            ) : null
        ));
    }
    
    /**
     * AJAX handler for updating pending orders
     * 
     * @since 1.10.13.5
     */
    public function ajax_update_pending_orders() {
        // Verify nonce
        check_ajax_referer('delcampe_orders_nonce', 'nonce');
        
        // Check permissions
        if (!current_user_can('manage_woocommerce') && !current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions', 'wc-delcampe-integration')));
        }
        
        try {
            // Update pending orders
            $result = $this->order_manager->update_pending_orders_from_delcampe();
            
            if ($result['updated'] > 0) {
                wp_send_json_success(array(
                    'message' => sprintf(
                        __('Updated %d of %d pending orders.', 'wc-delcampe-integration'),
                        $result['updated'],
                        $result['checked']
                    ),
                    'data' => $result
                ));
            } else if ($result['checked'] == 0) {
                wp_send_json_success(array(
                    'message' => __('No pending orders to update.', 'wc-delcampe-integration'),
                    'data' => $result
                ));
            } else {
                wp_send_json_success(array(
                    'message' => sprintf(
                        __('Checked %d orders but no updates were needed.', 'wc-delcampe-integration'),
                        $result['checked']
                    ),
                    'data' => $result
                ));
            }
        } catch (Exception $e) {
            wp_send_json_error(array(
                'message' => sprintf(
                    __('Error updating orders: %s', 'wc-delcampe-integration'),
                    $e->getMessage()
                )
            ));
        }
    }
    
    /**
     * AJAX handler for syncing closed sales
     * 
     * @since 1.10.2.0
     */
    public function ajax_sync_closed_sales() {
        // Verify nonce
        check_ajax_referer('delcampe_orders_nonce', 'nonce');
        
        // Check permissions
        if (!current_user_can('manage_woocommerce') && !current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
        }
        
        // Get limit
        $limit = isset($_POST['limit']) ? intval($_POST['limit']) : 50;
        
        try {
            // Load listings sync class
            if (!class_exists('Delcampe_Listings_Sync')) {
                require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-listings-sync.php';
            }
            
            $listings_sync = Delcampe_Listings_Sync::get_instance();
            // Use the new pagination-aware sync that processes up to $limit items
            // The method now handles pagination internally
            $max_pages = ceil($limit / 500); // 500 items per page
            $result = $listings_sync->sync_closed_listings(500, $max_pages);
            
            if (is_wp_error($result)) {
                error_log('[Delcampe Orders Admin] Closed listings sync error: ' . $result->get_error_message());
                wp_send_json_error(array(
                    'message' => $result->get_error_message()
                ));
            }
        } catch (Exception $e) {
            error_log('[Delcampe Orders Admin] Exception during closed listings sync: ' . $e->getMessage());
            wp_send_json_error(array(
                'message' => 'Error: ' . $e->getMessage()
            ));
        }
        
        // Return success with import details
        wp_send_json_success(array(
            'message' => sprintf(
                __('Successfully synced %d closed listings.', 'wc-delcampe-integration'),
                count($result)
            ),
            'count' => count($result),
            'listings' => $result
        ));
    }
    
    /**
     * AJAX handler for cleaning up listings
     * 
     * @since 1.10.2.0
     */
    public function ajax_cleanup_listings() {
        // Verify nonce
        check_ajax_referer('delcampe_orders_nonce', 'nonce');
        
        // Check permissions
        if (!current_user_can('manage_woocommerce') && !current_user_can('manage_options')) {
            wp_send_json_error(array('message' => 'Insufficient permissions'));
        }
        
        try {
            // Load cleanup class
            if (!class_exists('Delcampe_Listings_Cleanup')) {
                require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-listings-cleanup.php';
            }
            
            $cleanup = Delcampe_Listings_Cleanup::get_instance();
            $result = $cleanup->cleanup_all_listings();
            
            if (is_wp_error($result)) {
                error_log('[Delcampe Orders Admin] Cleanup error: ' . $result->get_error_message());
                wp_send_json_error(array(
                    'message' => $result->get_error_message()
                ));
            }
            
            // Return success with cleanup results
            wp_send_json_success(array(
                'message' => sprintf(
                    __('Cleanup complete: %d listings marked as ended, %d still active, %d errors.', 'wc-delcampe-integration'),
                    $result['ended'],
                    $result['still_active'],
                    $result['errors']
                ),
                'results' => $result
            ));
        } catch (Exception $e) {
            error_log('[Delcampe Orders Admin] Exception during cleanup: ' . $e->getMessage());
            wp_send_json_error(array(
                'message' => 'Error: ' . $e->getMessage()
            ));
        }
    }
}
