<?php
/**
 * Delcampe Queue Integration
 * 
 * Provides clean integration points for single item and bulk publish operations.
 * Replaces the existing complex sync handler and batch queue systems.
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 2.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Queue_Integration {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Hook into WordPress and WooCommerce
        add_action('init', array($this, 'init'));
        
        // AJAX handlers for admin interface
        add_action('wp_ajax_delcampe_publish_single', array($this, 'ajax_publish_single'));
        add_action('wp_ajax_delcampe_publish_bulk', array($this, 'ajax_publish_bulk'));
        add_action('wp_ajax_delcampe_queue_status', array($this, 'ajax_queue_status'));
        add_action('wp_ajax_delcampe_process_queue_now', array($this, 'ajax_process_queue_now'));
        add_action('wp_ajax_delcampe_backfill_listings_rows', array($this, 'ajax_backfill_listings_rows'));
        
        // WooCommerce product edit page integration
        add_action('woocommerce_product_data_panels', array($this, 'add_product_delcampe_panel'));
        add_action('woocommerce_process_product_meta', array($this, 'save_product_delcampe_data'));
        
        // Admin bulk actions
        add_filter('bulk_actions-edit-product', array($this, 'add_bulk_actions'));
        add_filter('handle_bulk_actions-edit-product', array($this, 'handle_bulk_actions'), 10, 3);
    }
    
    /**
     * Initialize integration
     */
    public function init() {
        // Register custom cron schedule for queue processing
        add_filter('cron_schedules', array($this, 'add_cron_schedules'));
        
        // Ensure queue worker is scheduled
        $queue = Delcampe_Queue::get_instance();
        $queue->schedule_worker();
    }
    
    /**
     * Add custom cron schedules
     * 
     * @param array $schedules Existing schedules
     * @return array Modified schedules
     */
    public function add_cron_schedules($schedules) {
        $schedules['delcampe_queue_interval'] = array(
            'interval' => 60, // Every minute
            'display' => __('Every Minute (Delcampe Queue)', 'wc-delcampe-integration')
        );
        
        return $schedules;
    }
    
    /**
     * Publish single product to Delcampe
     * 
     * @param int $product_id Product ID
     * @param string $action Action (create, update, delete)
     * @param int $priority Priority (higher = processed first)
     * @return bool|WP_Error Success or error
     */
    public function publish_single($product_id, $action = 'create', $priority = 0) {
        $queue = Delcampe_Queue::get_instance();
        
        // Validate product
        $product = wc_get_product($product_id);
        if (!$product) {
            return new WP_Error('invalid_product', 'Product not found');
        }
        
        // Check if product has required profile
        $profile_id = get_post_meta($product_id, '_delcampe_profile_id', true);
        if (!$profile_id && $action !== 'delete') {
            return new WP_Error('no_profile', 'Product must have a Delcampe profile assigned');
        }
        
        // Enqueue the product
        $result = $queue->enqueue($product_id, $action, $priority);
        
        if (is_wp_error($result)) {
            return $result;
        }
        
        delcampe_log("[Queue Integration] Enqueued single product {$product_id} for {$action}");
        
        return true;
    }
    
    /**
     * Publish multiple products to Delcampe
     * 
     * @param array $product_ids Array of product IDs
     * @param string $action Action (create, update, delete)
     * @param int $priority Priority (higher = processed first)
     * @return array Results with success/error counts
     */
    public function publish_bulk($product_ids, $action = 'create', $priority = 0) {
        $queue = Delcampe_Queue::get_instance();
        $results = array(
            'success' => 0,
            'errors' => 0,
            'messages' => array()
        );
        
        if (empty($product_ids) || !is_array($product_ids)) {
            $results['errors']++;
            $results['messages'][] = 'No products provided';
            return $results;
        }
        
        foreach ($product_ids as $product_id) {
            $product_id = intval($product_id);
            
            // Validate product
            $product = wc_get_product($product_id);
            if (!$product) {
                $results['errors']++;
                $results['messages'][] = "Product {$product_id}: Not found";
                continue;
            }
            
            // Check if product has required profile
            $profile_id = get_post_meta($product_id, '_delcampe_profile_id', true);
            if (!$profile_id && $action !== 'delete') {
                $results['errors']++;
                $results['messages'][] = "Product {$product_id}: No Delcampe profile assigned";
                continue;
            }
            
            // Enqueue the product
            $result = $queue->enqueue($product_id, $action, $priority);
            
            if (is_wp_error($result)) {
                $results['errors']++;
                $results['messages'][] = "Product {$product_id}: " . $result->get_error_message();
            } else {
                $results['success']++;
            }
        }
        
        delcampe_log("[Queue Integration] Bulk enqueued {$results['success']} products, {$results['errors']} errors");
        
        return $results;
    }
    
    /**
     * Get queue status and statistics
     * 
     * @return array Queue status
     */
    public function get_queue_status() {
        $queue = Delcampe_Queue::get_instance();
        $stats = $queue->get_stats();
        
        // Add estimated processing time
        if ($stats['pending'] > 0) {
            $avg_time = $stats['avg_processing_time'] ?: 30; // Default 30 seconds
            $stats['estimated_time'] = $stats['pending'] * $avg_time;
        } else {
            $stats['estimated_time'] = 0;
        }
        
        return $stats;
    }
    
    /**
     * AJAX handler for single product publish
     */
    public function ajax_publish_single() {
        // Check permissions
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('Insufficient permissions', 'wc-delcampe-integration'));
        }
        
        // Verify nonce
        check_ajax_referer('delcampe_publish_single', 'nonce');
        
        $product_id = intval($_POST['product_id']);
        $queue_action = sanitize_text_field($_POST['queue_action'] ?? 'create');
        
        $result = $this->publish_single($product_id, $queue_action, 10); // High priority for manual requests
        
        if (is_wp_error($result)) {
            wp_send_json_error(array(
                'message' => $result->get_error_message()
            ));
        } else {
            // Immediately process this single item to avoid waiting for cron
            try {
                // Ensure any backoff/locks are cleared so item is claimable now
                if (class_exists('Delcampe_Queue')) {
                    Delcampe_Queue::get_instance()->force_make_claimable();
                }
                $worker = Delcampe_Queue_Worker::get_instance();
                $worker->process_batch(1);
            } catch (\Throwable $e) {
                // Non-fatal for the response; cron will backstop
                delcampe_log('[Queue Integration] Immediate single-process error: ' . $e->getMessage());
            }
            wp_send_json_success(array(
                'message' => __('Product queued for publishing', 'wc-delcampe-integration'),
                'queue_status' => $this->get_queue_status()
            ));
        }
    }
    
    /**
     * AJAX handler for bulk publish
     */
    public function ajax_publish_bulk() {
        // Check permissions
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('Insufficient permissions', 'wc-delcampe-integration'));
        }
        
        // Verify nonce
        check_ajax_referer('delcampe_publish_bulk', 'nonce');
        
        $product_ids = array_map('intval', $_POST['product_ids'] ?? array());
        $queue_action = sanitize_text_field($_POST['queue_action'] ?? 'create');
        
        $results = $this->publish_bulk($product_ids, $queue_action, 5); // Medium priority for bulk

        // Kick off immediate background processing for all enqueued items
        $enqueued = max(0, intval($results['success']));
        if ($enqueued > 0) {
            // Make sure pending are claimable (clear backoffs, recover stale locks)
            try { if (class_exists('Delcampe_Queue')) { Delcampe_Queue::get_instance()->force_make_claimable(); } } catch (\Throwable $e) {}
            // For moderate batches, process inline to provide instant feedback
            if ($enqueued <= 500) {
                try {
                    $worker = Delcampe_Queue_Worker::get_instance();
                    $worker->process_batch($enqueued);
                } catch (\Throwable $e) {
                    delcampe_log('[Queue Integration] Immediate bulk-process error: ' . $e->getMessage());
                }
            } else {
                // For larger batches, trigger a non-blocking AJAX to process now
                $url = admin_url('admin-ajax.php');
                $body = array(
                    'action' => 'delcampe_process_queue_now',
                    'nonce'  => wp_create_nonce('delcampe_process_queue_now'),
                    'max'    => $enqueued,
                );
                wp_remote_post($url, array(
                    'timeout'  => 0.01,
                    'blocking' => false,
                    'body'     => $body,
                ));
            }
        }

        wp_send_json_success(array(
            'results' => $results,
            'queue_status' => $this->get_queue_status()
        ));
    }
    
    /**
     * AJAX handler for queue status
     */
    public function ajax_queue_status() {
        // Check permissions
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('Insufficient permissions', 'wc-delcampe-integration'));
        }
        
        wp_send_json_success($this->get_queue_status());
    }

    /**
     * AJAX: Backfill listing rows so Listings UI reflects new queue states.
     */
    public function ajax_backfill_listings_rows() {
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('Insufficient permissions', 'wc-delcampe-integration'));
        }
        check_ajax_referer('delcampe_admin_nonce', 'nonce');
        $limit = isset($_REQUEST['limit']) ? max(1, intval($_REQUEST['limit'])) : 500;
        try {
            if (class_exists('Delcampe_Queue')) {
                $count = Delcampe_Queue::get_instance()->run_backfill_listings_rows($limit);
                wp_send_json_success(array('backfilled' => $count));
                return;
            }
            wp_send_json_error(array('message' => 'Queue system not available'));
        } catch (\Throwable $e) {
            wp_send_json_error(array('message' => $e->getMessage()));
        }
    }

    /**
     * AJAX: Process queue immediately (non-blocking trigger from bulk enqueue)
     */
    public function ajax_process_queue_now() {
        // Check permissions
        if (!current_user_can('manage_woocommerce')) {
            wp_die(__('Insufficient permissions', 'wc-delcampe-integration'));
        }
        // Verify nonce
        check_ajax_referer('delcampe_process_queue_now', 'nonce');
        $max = isset($_REQUEST['max']) ? max(1, intval($_REQUEST['max'])) : 10;
        try {
            $worker = Delcampe_Queue_Worker::get_instance();
            $res = $worker->process_batch($max);
            wp_send_json_success($res);
        } catch (\Throwable $e) {
            wp_send_json_error(array('message' => $e->getMessage()));
        }
    }
    
    /**
     * Add Delcampe panel to product edit page
     */
    public function add_product_delcampe_panel() {
        global $post;
        
        $product_id = $post->ID;
        $profile_id = get_post_meta($product_id, '_delcampe_profile_id', true);
        $sync_status = get_post_meta($product_id, '_delcampe_sync_status', true);
        $listing_id = get_post_meta($product_id, '_delcampe_listing_id', true);
        
        ?>
        <div id="delcampe_product_data" class="panel woocommerce_options_panel">
            <div class="options_group">
                <h3><?php _e('Delcampe Integration', 'wc-delcampe-integration'); ?></h3>
                
                <?php
                // Profile selection
                $profiles = $this->get_available_profiles();
                woocommerce_wp_select(array(
                    'id' => '_delcampe_profile_id',
                    'label' => __('Delcampe Profile', 'wc-delcampe-integration'),
                    'description' => __('Select the Delcampe profile to use for this product', 'wc-delcampe-integration'),
                    'desc_tip' => true,
                    'value' => $profile_id,
                    'options' => $profiles
                ));
                ?>
                
                <div class="delcampe-status-info">
                    <p>
                        <strong><?php _e('Status:', 'wc-delcampe-integration'); ?></strong>
                        <span class="delcampe-status <?php echo esc_attr($sync_status); ?>">
                            <?php echo $this->get_status_label($sync_status); ?>
                        </span>
                    </p>
                    
                    <?php if ($listing_id): ?>
                    <p>
                        <strong><?php _e('Delcampe ID:', 'wc-delcampe-integration'); ?></strong>
                        <code><?php echo esc_html($listing_id); ?></code>
                    </p>
                    <?php endif; ?>
                </div>
                
                <div class="delcampe-actions">
                    <?php if ($sync_status !== 'published'): ?>
                    <button type="button" class="button button-primary" id="delcampe-publish-single" data-product-id="<?php echo $product_id; ?>">
                        <?php _e('Publish to Delcampe', 'wc-delcampe-integration'); ?>
                    </button>
                    <?php else: ?>
                    <button type="button" class="button" id="delcampe-update-single" data-product-id="<?php echo $product_id; ?>">
                        <?php _e('Update on Delcampe', 'wc-delcampe-integration'); ?>
                    </button>
                    <button type="button" class="button" id="delcampe-delete-single" data-product-id="<?php echo $product_id; ?>">
                        <?php _e('Remove from Delcampe', 'wc-delcampe-integration'); ?>
                    </button>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            // Single product publish buttons
            $('#delcampe-publish-single, #delcampe-update-single, #delcampe-delete-single').on('click', function() {
                var action = 'create';
                if ($(this).attr('id') === 'delcampe-update-single') action = 'update';
                if ($(this).attr('id') === 'delcampe-delete-single') action = 'delete';
                
                var productId = $(this).data('product-id');
                var button = $(this);
                
                button.prop('disabled', true).text('Processing...');
                
                $.post(ajaxurl, {
                    action: 'delcampe_publish_single',
                    product_id: productId,
                    queue_action: action,
                    nonce: '<?php echo wp_create_nonce('delcampe_publish_single'); ?>'
                }, function(response) {
                    if (response.success) {
                        alert(response.data.message);
                        location.reload();
                    } else {
                        alert('Error: ' + response.data.message);
                    }
                }).always(function() {
                    button.prop('disabled', false);
                });
            });
        });
        </script>
        
        <style>
        .delcampe-status.published { color: #46b450; }
        .delcampe-status.pending { color: #ffb900; }
        .delcampe-status.error { color: #dc3232; }
        .delcampe-actions { margin-top: 10px; }
        .delcampe-actions button { margin-right: 10px; }
        </style>
        <?php
    }
    
    /**
     * Save product Delcampe data
     * 
     * @param int $product_id Product ID
     */
    public function save_product_delcampe_data($product_id) {
        if (isset($_POST['_delcampe_profile_id'])) {
            $profile_id = intval($_POST['_delcampe_profile_id']);
            update_post_meta($product_id, '_delcampe_profile_id', $profile_id);
        }
    }
    
    /**
     * Add bulk actions to product list
     * 
     * @param array $actions Existing actions
     * @return array Modified actions
     */
    public function add_bulk_actions($actions) {
        $actions['delcampe_publish'] = __('Publish to Delcampe', 'wc-delcampe-integration');
        $actions['delcampe_update'] = __('Update on Delcampe', 'wc-delcampe-integration');
        $actions['delcampe_delete'] = __('Remove from Delcampe', 'wc-delcampe-integration');
        
        return $actions;
    }
    
    /**
     * Handle bulk actions
     * 
     * @param string $redirect_url Redirect URL
     * @param string $action Action being performed
     * @param array $product_ids Selected product IDs
     * @return string Modified redirect URL
     */
    public function handle_bulk_actions($redirect_url, $action, $product_ids) {
        if (!in_array($action, ['delcampe_publish', 'delcampe_update', 'delcampe_delete'])) {
            return $redirect_url;
        }
        
        $queue_action = 'create';
        if ($action === 'delcampe_update') $queue_action = 'update';
        if ($action === 'delcampe_delete') $queue_action = 'delete';
        
        $results = $this->publish_bulk($product_ids, $queue_action);
        
        // Add query args for admin notice
        $redirect_url = add_query_arg(array(
            'delcampe_bulk_action' => $action,
            'delcampe_success' => $results['success'],
            'delcampe_errors' => $results['errors']
        ), $redirect_url);
        
        return $redirect_url;
    }
    
    /**
     * Get available profiles for dropdown
     * 
     * @return array Profile options
     */
    private function get_available_profiles() {
        $profiles = array('' => __('Select a profile...', 'wc-delcampe-integration'));
        
        if (class_exists('Delcampe_Profiles_Model')) {
            $profiles_model = Delcampe_Profiles_Model::get_instance();
            $all_profiles = $profiles_model->get_all_profiles();
            
            foreach ($all_profiles as $profile) {
                $profiles[$profile->profile_id] = $profile->profile_name;
            }
        }
        
        return $profiles;
    }
    
    /**
     * Get human-readable status label
     * 
     * @param string $status Status
     * @return string Label
     */
    private function get_status_label($status) {
        $labels = array(
            'local' => __('Not Published', 'wc-delcampe-integration'),
            'pending' => __('Queued', 'wc-delcampe-integration'),
            'processing' => __('Publishing...', 'wc-delcampe-integration'),
            'verifying' => __('Verifying...', 'wc-delcampe-integration'),
            'published' => __('Published', 'wc-delcampe-integration'),
            'error' => __('Error', 'wc-delcampe-integration')
        );
        
        return $labels[$status] ?? $status;
    }
}
