<?php
/**
 * Delcampe Queue Migration System
 * 
 * Handles migration from the old complex queue systems to the new simplified queue.
 * Provides safe migration path with rollback capabilities.
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 2.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Queue_Migration {
    
    /**
     * Migration version
     */
    const MIGRATION_VERSION = '2.0.0';
    
    /**
     * Option key for migration status
     */
    const MIGRATION_STATUS_KEY = 'delcampe_queue_migration_status';
    
    /**
     * Option key for backup data
     */
    const BACKUP_DATA_KEY = 'delcampe_queue_migration_backup';
    
    /**
     * Initialize migration hooks
     */
    public static function init() {
        // Register AJAX handlers
        add_action('wp_ajax_delcampe_queue_migrate', array(__CLASS__, 'ajax_run_migration'));
        add_action('wp_ajax_delcampe_queue_rollback', array(__CLASS__, 'ajax_rollback_migration'));
    }
    
    /**
     * AJAX handler for running migration
     */
    public static function ajax_run_migration() {
        // Check permissions
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('Insufficient permissions', 'wc-delcampe-integration'));
        }
        
        // Verify nonce
        check_ajax_referer('delcampe_queue_action', 'nonce');
        
        // Run migration
        $result = self::run_migration();
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }
        
        wp_send_json_success(array(
            'message' => __('Migration completed successfully', 'wc-delcampe-integration')
        ));
    }
    
    /**
     * AJAX handler for rollback
     */
    public static function ajax_rollback_migration() {
        // Check permissions
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(__('Insufficient permissions', 'wc-delcampe-integration'));
        }
        
        // Verify nonce
        check_ajax_referer('delcampe_queue_action', 'nonce');
        
        // Run rollback
        $result = self::rollback_migration();
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }
        
        wp_send_json_success(array(
            'message' => __('Rollback completed successfully', 'wc-delcampe-integration')
        ));
    }
    
    /**
     * Run migration process
     * 
     * @return bool|WP_Error Success or error
     */
    public static function run_migration() {
        global $wpdb;
        
        delcampe_log('[Queue Migration] Starting migration to v2.0.0 queue system');
        
        // Check if migration already completed
        $status = get_option(self::MIGRATION_STATUS_KEY);
        if ($status === 'completed') {
            delcampe_log('[Queue Migration] Migration already completed');
            return true;
        }
        
        // Set migration in progress
        update_option(self::MIGRATION_STATUS_KEY, 'in_progress');
        
        try {
            // Step 1: Create new tables
            self::create_new_tables();
            
            // Step 2: Backup existing data
            $backup_data = self::backup_existing_data();
            update_option(self::BACKUP_DATA_KEY, $backup_data);
            
            // Step 3: Migrate data from old queues
            $migrated_count = self::migrate_queue_data();
            
            // Step 4: Migrate product metadata
            $meta_count = self::migrate_product_metadata();
            
            // Step 5: Clean up old tables (optional - keep for rollback)
            // self::cleanup_old_tables();
            
            // Mark migration as completed
            update_option(self::MIGRATION_STATUS_KEY, 'completed');
            update_option('delcampe_queue_migration_version', self::MIGRATION_VERSION);
            
            delcampe_log("[Queue Migration] Migration completed successfully. Migrated {$migrated_count} queue items and {$meta_count} product metadata records");
            
            return true;
            
        } catch (Exception $e) {
            // Mark migration as failed
            update_option(self::MIGRATION_STATUS_KEY, 'failed');
            delcampe_log('[Queue Migration] Migration failed: ' . $e->getMessage());
            
            return new WP_Error('migration_failed', $e->getMessage());
        }
    }
    
    /**
     * Create new queue tables
     */
    private static function create_new_tables() {
        $queue = Delcampe_Queue::get_instance();
        $queue->create_tables();
        
        delcampe_log('[Queue Migration] Created new queue tables');
    }
    
    /**
     * Backup existing queue data
     * 
     * @return array Backup data
     */
    private static function backup_existing_data() {
        global $wpdb;
        
        $backup = array(
            'timestamp' => time(),
            'old_sync_queue' => array(),
            'old_batch_queue' => array(),
            'option_queue' => array()
        );
        
        // Backup old sync queue table
        $sync_queue_table = $wpdb->prefix . 'delcampe_sync_queue';
        if ($wpdb->get_var("SHOW TABLES LIKE '{$sync_queue_table}'") === $sync_queue_table) {
            $backup['old_sync_queue'] = $wpdb->get_results("SELECT * FROM {$sync_queue_table}");
            delcampe_log('[Queue Migration] Backed up ' . count($backup['old_sync_queue']) . ' items from sync queue');
        }
        
        // Backup old batch queue table
        $batch_queue_table = $wpdb->prefix . 'delcampe_batch_queue';
        if ($wpdb->get_var("SHOW TABLES LIKE '{$batch_queue_table}'") === $batch_queue_table) {
            $backup['old_batch_queue'] = $wpdb->get_results("SELECT * FROM {$batch_queue_table}");
            delcampe_log('[Queue Migration] Backed up ' . count($backup['old_batch_queue']) . ' items from batch queue');
        }
        
        // Backup option-based queue (if any)
        $option_queue = get_option('delcampe_sync_queue', array());
        if (!empty($option_queue)) {
            $backup['option_queue'] = $option_queue;
            delcampe_log('[Queue Migration] Backed up ' . count($option_queue) . ' items from option queue');
        }
        
        return $backup;
    }
    
    /**
     * Migrate data from old queue systems
     * 
     * @return int Number of items migrated
     */
    private static function migrate_queue_data() {
        global $wpdb;
        
        $migrated_count = 0;
        $queue = Delcampe_Queue::get_instance();
        
        // Migrate from old sync queue table
        $sync_queue_table = $wpdb->prefix . 'delcampe_sync_queue';
        if ($wpdb->get_var("SHOW TABLES LIKE '{$sync_queue_table}'") === $sync_queue_table) {
            $old_items = $wpdb->get_results("SELECT * FROM {$sync_queue_table} WHERE state IN ('queued', 'pending', 'retry')");
            
            foreach ($old_items as $item) {
                $action = isset($item->action) ? $item->action : 'create';
                $priority = isset($item->priority) ? intval($item->priority) : 0;
                
                $result = $queue->enqueue($item->product_id, $action, $priority);
                if (!is_wp_error($result)) {
                    $migrated_count++;
                }
            }
            
            delcampe_log("[Queue Migration] Migrated {$migrated_count} items from sync queue");
        }
        
        // Migrate from old batch queue table
        $batch_queue_table = $wpdb->prefix . 'delcampe_batch_queue';
        if ($wpdb->get_var("SHOW TABLES LIKE '{$batch_queue_table}'") === $batch_queue_table) {
            $old_batches = $wpdb->get_results("SELECT * FROM {$batch_queue_table} WHERE status IN ('queued', 'pending', 'retry')");
            
            foreach ($old_batches as $batch) {
                $product_ids = json_decode($batch->product_ids, true);
                if (is_array($product_ids)) {
                    foreach ($product_ids as $product_id) {
                        $action = isset($batch->action) ? $batch->action : 'create';
                        
                        $result = $queue->enqueue($product_id, $action, 1); // Lower priority for batch items
                        if (!is_wp_error($result)) {
                            $migrated_count++;
                        }
                    }
                }
            }
            
            delcampe_log("[Queue Migration] Migrated batch queue items");
        }
        
        // Migrate from option-based queue
        $option_queue = get_option('delcampe_sync_queue', array());
        if (!empty($option_queue)) {
            foreach ($option_queue as $item) {
                if (isset($item['product_id'])) {
                    $action = isset($item['action']) ? $item['action'] : 'create';
                    $priority = isset($item['priority']) ? intval($item['priority']) : 0;
                    
                    $result = $queue->enqueue($item['product_id'], $action, $priority);
                    if (!is_wp_error($result)) {
                        $migrated_count++;
                    }
                }
            }
            
            delcampe_log("[Queue Migration] Migrated option queue items");
        }
        
        return $migrated_count;
    }
    
    /**
     * Migrate product metadata to new format
     * 
     * @return int Number of metadata records migrated
     */
    private static function migrate_product_metadata() {
        global $wpdb;
        
        $migrated_count = 0;
        
        // Get all products with Delcampe metadata
        $products_with_meta = $wpdb->get_results("
            SELECT DISTINCT p.post_id, 
                   MAX(CASE WHEN p.meta_key = '_delcampe_listing_id' THEN p.meta_value END) as listing_id,
                   MAX(CASE WHEN p.meta_key = '_delcampe_sync_status' THEN p.meta_value END) as sync_status
            FROM {$wpdb->postmeta} p
            WHERE p.meta_key IN ('_delcampe_listing_id', '_delcampe_sync_status')
            GROUP BY p.post_id
        ");
        
        foreach ($products_with_meta as $meta) {
            $product_id = $meta->post_id;
            $listing_id = $meta->listing_id;
            $sync_status = $meta->sync_status;
            
            // Determine state based on sync status
            $state = Delcampe_Queue::STATE_LOCAL;
            if ($sync_status === 'published' || $sync_status === 'active') {
                $state = Delcampe_Queue::STATE_PUBLISHED;
            } elseif ($sync_status === 'error' || $sync_status === 'failed') {
                $state = Delcampe_Queue::STATE_ERRORED;
            } elseif ($sync_status === 'pending' || $sync_status === 'queued') {
                $state = Delcampe_Queue::STATE_PENDING;
            }
            
            // Insert into new product meta table
            $queue_table = $wpdb->prefix . 'delcampe_product_meta';
            $result = $wpdb->replace(
                $queue_table,
                array(
                    'product_id' => $product_id,
                    'delcampe_listing_id' => $listing_id,
                    'state' => $state,
                    'last_synced_at' => ($state === Delcampe_Queue::STATE_PUBLISHED) ? current_time('mysql') : null
                )
            );
            
            if ($result !== false) {
                $migrated_count++;
            }
        }
        
        delcampe_log("[Queue Migration] Migrated {$migrated_count} product metadata records");
        
        return $migrated_count;
    }
    
    /**
     * Clean up old tables (use with caution)
     */
    private static function cleanup_old_tables() {
        global $wpdb;
        
        // Only clean up if migration is confirmed successful
        $confirmation = get_option('delcampe_queue_migration_cleanup_confirmed', false);
        if (!$confirmation) {
            delcampe_log('[Queue Migration] Skipping cleanup - not confirmed');
            return;
        }
        
        $tables_to_cleanup = array(
            $wpdb->prefix . 'delcampe_sync_queue',
            $wpdb->prefix . 'delcampe_batch_queue'
        );
        
        foreach ($tables_to_cleanup as $table) {
            if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'") === $table) {
                $wpdb->query("DROP TABLE {$table}");
                delcampe_log("[Queue Migration] Dropped old table {$table}");
            }
        }
        
        // Clear option-based queue
        delete_option('delcampe_sync_queue');
        
        delcampe_log('[Queue Migration] Cleanup completed');
    }
    
    /**
     * Rollback migration
     * 
     * @return bool|WP_Error Success or error
     */
    public static function rollback_migration() {
        global $wpdb;
        
        delcampe_log('[Queue Migration] Starting migration rollback');
        
        $backup_data = get_option(self::BACKUP_DATA_KEY);
        if (!$backup_data) {
            return new WP_Error('no_backup', 'No backup data found for rollback');
        }
        
        try {
            // Restore old sync queue table
            if (!empty($backup_data['old_sync_queue'])) {
                $sync_queue_table = $wpdb->prefix . 'delcampe_sync_queue';
                
                // Recreate table if needed
                $queue_instance = new Delcampe_Sync_Queue();
                $queue_instance->maybe_create_table();
                
                // Restore data
                foreach ($backup_data['old_sync_queue'] as $item) {
                    $wpdb->insert($sync_queue_table, (array) $item);
                }
                
                delcampe_log('[Queue Migration] Restored sync queue table');
            }
            
            // Restore old batch queue table
            if (!empty($backup_data['old_batch_queue'])) {
                $batch_queue_table = $wpdb->prefix . 'delcampe_batch_queue';
                
                // Recreate table if needed
                $batch_instance = Delcampe_Batch_Queue::get_instance();
                $batch_instance->maybe_create_table();
                
                // Restore data
                foreach ($backup_data['old_batch_queue'] as $item) {
                    $wpdb->insert($batch_queue_table, (array) $item);
                }
                
                delcampe_log('[Queue Migration] Restored batch queue table');
            }
            
            // Restore option queue
            if (!empty($backup_data['option_queue'])) {
                update_option('delcampe_sync_queue', $backup_data['option_queue']);
                delcampe_log('[Queue Migration] Restored option queue');
            }
            
            // Clear new queue tables
            $new_queue_table = $wpdb->prefix . 'delcampe_queue';
            $new_meta_table = $wpdb->prefix . 'delcampe_product_meta';
            
            $wpdb->query("TRUNCATE TABLE {$new_queue_table}");
            $wpdb->query("TRUNCATE TABLE {$new_meta_table}");
            
            // Reset migration status
            update_option(self::MIGRATION_STATUS_KEY, 'rolled_back');
            
            delcampe_log('[Queue Migration] Rollback completed successfully');
            
            return true;
            
        } catch (Exception $e) {
            delcampe_log('[Queue Migration] Rollback failed: ' . $e->getMessage());
            return new WP_Error('rollback_failed', $e->getMessage());
        }
    }
    
    /**
     * Get migration status
     * 
     * @return array Migration status information
     */
    public static function get_migration_status() {
        $status = get_option(self::MIGRATION_STATUS_KEY, 'not_started');
        $version = get_option('delcampe_queue_migration_version', '0.0.0');
        $backup_data = get_option(self::BACKUP_DATA_KEY);
        
        return array(
            'status' => $status,
            'version' => $version,
            'has_backup' => !empty($backup_data),
            'backup_timestamp' => $backup_data ? $backup_data['timestamp'] : null
        );
    }
    
    /**
     * Check if migration is needed
     * 
     * @return bool Whether migration is needed
     */
    public static function is_migration_needed() {
        global $wpdb;
        
        $status = get_option(self::MIGRATION_STATUS_KEY, 'not_started');
        if ($status === 'completed') {
            return false;
        }
        
        // Check if old tables exist with data
        $old_tables = array(
            $wpdb->prefix . 'delcampe_sync_queue',
            $wpdb->prefix . 'delcampe_batch_queue'
        );
        
        foreach ($old_tables as $table) {
            if ($wpdb->get_var("SHOW TABLES LIKE '{$table}'") === $table) {
                $count = $wpdb->get_var("SELECT COUNT(*) FROM {$table}");
                if ($count > 0) {
                    return true;
                }
            }
        }
        
        // Check option-based queue
        $option_queue = get_option('delcampe_sync_queue', array());
        if (!empty($option_queue)) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Add admin notice for migration
     */
    public static function add_migration_notice() {
        if (!self::is_migration_needed()) {
            return;
        }
        
        add_action('admin_notices', function() {
            ?>
            <div class="notice notice-warning">
                <p>
                    <strong><?php _e('Delcampe Queue Migration Required', 'wc-delcampe-integration'); ?></strong>
                </p>
                <p>
                    <?php _e('The Delcampe sync system has been redesigned for better reliability. A migration is required to move your existing queue data to the new system.', 'wc-delcampe-integration'); ?>
                </p>
                <p>
                    <a href="<?php echo wp_nonce_url(admin_url('admin.php?page=delcampe-queue&migrate=1'), 'delcampe_migrate'); ?>" class="button button-primary">
                        <?php _e('Run Migration', 'wc-delcampe-integration'); ?>
                    </a>
                    <a href="<?php echo admin_url('admin.php?page=delcampe-queue'); ?>" class="button">
                        <?php _e('View Queue Status', 'wc-delcampe-integration'); ?>
                    </a>
                </p>
            </div>
            <?php
        });
    }
}