<?php
/**
 * Delcampe Queue System - Main Initialization
 * 
 * This is the main entry point for the new queue system. It initializes all components
 * and provides a clean integration with the existing plugin structure.
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 2.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Queue_System {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * System version
     */
    const VERSION = '2.0.0';
    
    /**
     * Components
     */
    private $queue;
    private $worker;
    private $integration;
    private $admin;
    private $migration;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Initialize only if license is valid
        if (!delcampe_has_valid_license()) {
            return;
        }
        
        // Register AJAX handlers immediately for admin requests
        // This must happen before WordPress processes admin-ajax.php
        if (is_admin()) {
            require_once DWC_PLUGIN_DIR . 'includes/class-delcampe-queue-migration.php';
            Delcampe_Queue_Migration::init();
        }
        
        add_action('init', array($this, 'init'), 15);
    }
    
    /**
     * Initialize the queue system
     */
    public function init() {
        // Only log initialization in debug mode or on admin pages
        $should_log = defined('WP_DEBUG') && WP_DEBUG && is_admin();
        
        if ($should_log) {
            delcampe_log('[Queue System v' . self::VERSION . '] Initializing new queue system');
        }
        
        // Load required files
        $this->load_files();
        
        // CRITICAL: Ensure tables exist before any operations
        $this->ensure_tables_exist();
        
        // Check if migration is needed
        if (Delcampe_Queue_Migration::is_migration_needed()) {
            // Show migration notice
            Delcampe_Queue_Migration::add_migration_notice();
            
            // Handle migration request
            if (isset($_GET['migrate']) && $_GET['migrate'] === '1' && current_user_can('manage_woocommerce')) {
                $this->handle_migration_request();
            }
        }
        
        // Initialize components
        $this->initialize_components();
        
        // Hook into WordPress
        $this->setup_hooks();
        
        if ($should_log) {
            delcampe_log('[Queue System v' . self::VERSION . '] Initialization complete');
        }
    }
    
    /**
     * Ensure database tables exist
     * Critical for production sites with custom prefixes
     */
    private function ensure_tables_exist() {
        global $wpdb;
        
        $tables_to_check = array(
            'delcampe_queue' => 'Main queue table',
            'delcampe_product_meta' => 'Product metadata table'
        );
        
        $tables_missing = false;
        
        foreach ($tables_to_check as $table_suffix => $description) {
            $table_name = $wpdb->prefix . $table_suffix;
            $table_exists = $wpdb->get_var("SHOW TABLES LIKE '$table_name'");
            
            if (!$table_exists) {
                delcampe_log("[Queue System] CRITICAL: $description ($table_name) does not exist");
                $tables_missing = true;
            }
        }
        
        if ($tables_missing) {
            delcampe_log('[Queue System] Creating missing tables now...');
            
            // Force immediate table creation
            if (class_exists('Delcampe_Queue')) {
                $queue = Delcampe_Queue::get_instance();
                if (method_exists($queue, 'create_tables')) {
                    $queue->create_tables();
                    delcampe_log('[Queue System] All tables created successfully');
                }
            }
        }
    }
    
    /**
     * Load required files
     */
    private function load_files() {
        $files = array(
            'class-delcampe-queue.php',
            'class-delcampe-queue-worker.php',
            'class-delcampe-queue-integration.php',
            'class-delcampe-queue-admin.php',
            'class-delcampe-queue-migration.php'
        );
        
        foreach ($files as $file) {
            $path = DWC_PLUGIN_DIR . 'includes/' . $file;
            if (file_exists($path)) {
                require_once $path;
            } else {
                delcampe_log('[Queue System] Warning: Could not load ' . $file);
            }
        }
    }
    
    /**
     * Initialize all components
     */
    private function initialize_components() {
        // Core queue
        $this->queue = Delcampe_Queue::get_instance();
        
        // Worker
        $this->worker = Delcampe_Queue_Worker::get_instance();
        
        // Integration layer
        $this->integration = Delcampe_Queue_Integration::get_instance();
        
        // Admin interface (only in admin)
        if (is_admin()) {
            $this->admin = Delcampe_Queue_Admin::get_instance();
            
            // Migration handlers are already initialized in constructor
            // to ensure they're registered before admin-ajax.php runs
        }
        
        // Only log in debug mode
        if (defined('WP_DEBUG') && WP_DEBUG && is_admin()) {
            delcampe_log('[Queue System] All components initialized');
        }
    }
    
    /**
     * Setup WordPress hooks
     */
    private function setup_hooks() {
        // Override old sync handler methods
        add_filter('delcampe_use_new_queue_system', '__return_true');
        
        // Disable old queue systems
        $this->disable_old_systems();
        
        // Add upgrade routine
        add_action('upgrader_process_complete', array($this, 'check_for_updates'), 10, 2);
        
        // Cleanup routine
        add_action('wp_scheduled_delete', array($this, 'daily_cleanup'));
    }
    
    /**
     * Disable old queue systems to prevent conflicts
     */
    private function disable_old_systems() {
        // Prevent old sync handler from processing
        add_filter('delcampe_sync_handler_enabled', '__return_false');
        
        // Prevent old batch queue from processing
        add_filter('delcampe_batch_queue_enabled', '__return_false');
        
        // v1.10.35.11: Only clear cron jobs once per day to avoid database lock issues
        // Use a transient to track when we last cleared the cron jobs
        $last_cleared = get_transient('delcampe_crons_cleared');
        
        if (!$last_cleared) {
            // Check before clearing to prevent unnecessary database writes
            if (wp_next_scheduled('delcampe_process_sync_queue')) {
                wp_clear_scheduled_hook('delcampe_process_sync_queue');
            }
            
            if (wp_next_scheduled('delcampe_process_batch_queue')) {
                wp_clear_scheduled_hook('delcampe_process_batch_queue');
            }
            
            // Set transient for 24 hours
            set_transient('delcampe_crons_cleared', true, DAY_IN_SECONDS);
        }
        
        // No need to log this on every init - it's just cleanup
        // delcampe_log('[Queue System] Disabled old queue systems');
    }
    
    /**
     * Handle migration request
     */
    private function handle_migration_request() {
        // Verify nonce for security
        if (!wp_verify_nonce($_GET['_wpnonce'] ?? '', 'delcampe_migrate')) {
            wp_die(__('Security check failed', 'wc-delcampe-integration'));
        }
        
        delcampe_log('[Queue System] Starting migration process');
        
        $result = Delcampe_Queue_Migration::run_migration();
        
        if (is_wp_error($result)) {
            wp_die(__('Migration failed: ', 'wc-delcampe-integration') . $result->get_error_message());
        }
        
        // Redirect to queue page with success message
        wp_redirect(add_query_arg(array(
            'page' => 'delcampe-queue',
            'migrated' => '1'
        ), admin_url('admin.php')));
        exit;
    }
    
    /**
     * Check for system updates
     */
    public function check_for_updates($upgrader, $options) {
        if ($options['action'] == 'update' && $options['type'] == 'plugin') {
            foreach ($options['plugins'] as $plugin) {
                if ($plugin == plugin_basename(DWC_PLUGIN_DIR . 'main.php')) {
                    // Plugin was updated - check if we need to migrate
                    if (Delcampe_Queue_Migration::is_migration_needed()) {
                        delcampe_log('[Queue System] Update detected - migration may be needed');
                    }
                    break;
                }
            }
        }
    }
    
    /**
     * Daily cleanup routine
     */
    public function daily_cleanup() {
        if ($this->queue) {
            $cleaned = $this->queue->cleanup(7); // Clean items older than 7 days
            if ($cleaned > 0) {
                delcampe_log("[Queue System] Daily cleanup: removed {$cleaned} old queue items");
            }
        }
    }
    
    /**
     * Get system status
     * 
     * @return array System status information
     */
    public function get_system_status() {
        $migration_status = Delcampe_Queue_Migration::get_migration_status();
        $queue_stats = $this->queue ? $this->queue->get_stats() : array();
        $worker_stats = $this->worker ? $this->worker->get_stats() : array();
        
        return array(
            'version' => self::VERSION,
            'migration' => $migration_status,
            'queue' => $queue_stats,
            'worker' => $worker_stats,
            'components' => array(
                'queue' => $this->queue !== null,
                'worker' => $this->worker !== null,
                'integration' => $this->integration !== null,
                'admin' => $this->admin !== null
            )
        );
    }
    
    /**
     * Emergency stop - pause all queue processing
     */
    public function emergency_stop() {
        set_transient('delcampe_queue_paused', true, HOUR_IN_SECONDS);
        delcampe_log('[Queue System] EMERGENCY STOP - All queue processing paused');
    }
    
    /**
     * Resume queue processing
     */
    public function resume() {
        delete_transient('delcampe_queue_paused');
        delcampe_log('[Queue System] Queue processing resumed');
    }
    
    /**
     * Legacy compatibility methods
     */
    
    /**
     * Legacy method for adding items to queue (compatibility)
     * 
     * @param int $product_id Product ID
     * @param int $profile_id Profile ID (ignored in new system)
     * @param string $action Action
     * @return bool Success
     */
    public function legacy_add_to_queue($product_id, $profile_id, $action = 'create') {
        if (!$this->integration) {
            return false;
        }
        
        $result = $this->integration->publish_single($product_id, $action);
        return !is_wp_error($result);
    }
    
    /**
     * Legacy method for processing queue (compatibility)
     */
    public function legacy_process_queue() {
        if (!$this->worker) {
            return;
        }
        
        // Process all items, not just 10
        $this->worker->process_batch(9999);
    }
    
    /**
     * Get public API for other plugins/themes
     * 
     * @return array Public API methods
     */
    public function get_public_api() {
        return array(
            'enqueue' => array($this->integration, 'publish_single'),
            'enqueue_bulk' => array($this->integration, 'publish_bulk'),
            'get_status' => array($this->integration, 'get_queue_status'),
            'pause' => array($this, 'emergency_stop'),
            'resume' => array($this, 'resume')
        );
    }
}

/**
 * Global function to get queue system instance
 * 
 * @return Delcampe_Queue_System
 */
function delcampe_queue_system() {
    return Delcampe_Queue_System::get_instance();
}

/**
 * Legacy compatibility functions
 */

/**
 * Legacy function to add item to queue
 * 
 * @param int $product_id Product ID
 * @param int $profile_id Profile ID
 * @param string $action Action
 * @return bool Success
 */
function delcampe_add_to_queue($product_id, $profile_id, $action = 'create') {
    $system = delcampe_queue_system();
    return $system->legacy_add_to_queue($product_id, $profile_id, $action);
}

/**
 * Legacy function to process queue
 */
function delcampe_process_queue() {
    $system = delcampe_queue_system();
    $system->legacy_process_queue();
}