<?php
/**
 * Delcampe Queue System - Complete Redesign
 * 
 * A simple, reliable FIFO queue system that replaces all existing queue implementations.
 * 
 * Core Design Principles:
 * 1. Single FIFO Queue - One table handles everything
 * 2. Clear States - Simple, well-defined states with strict transitions  
 * 3. Idempotency - Never create duplicates on Delcampe
 * 4. Same Path - Single item and bulk publish use identical queue logic
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 2.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Queue {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Database table names
     */
    private $queue_table;
    private $product_meta_table;
    
    /**
     * Queue states
     */
    const STATE_LOCAL = 'local';        // Product exists in WC but never queued
    const STATE_PENDING = 'pending';    // In queue waiting to be processed
    const STATE_PROCESSING = 'processing'; // Worker has locked it
    const STATE_VERIFYING = 'verifying';   // Accepted by API; awaiting Delcampe ID
    const STATE_PUBLISHED = 'published';   // Successfully on Delcampe
    const STATE_ERRORED = 'errored';       // Failed after max retries
    
    /**
     * Queue actions
     */
    const ACTION_CREATE = 'create';
    const ACTION_UPDATE = 'update';
    const ACTION_DELETE = 'delete';
    
    /**
     * Configuration
     */
    const MAX_RETRIES = 5;
    const WORKER_TIMEOUT = 300; // 5 minutes
    const RATE_LIMIT_PER_MINUTE = 150;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        global $wpdb;
        $this->queue_table = $wpdb->prefix . 'delcampe_queue';
        $this->product_meta_table = $wpdb->prefix . 'delcampe_product_meta';
        
        // v1.10.35.6: Fixed - Initialize tables immediately if needed
        $this->maybe_create_tables();
        
        // Add custom cron schedules EARLY
        add_filter('cron_schedules', array($this, 'add_cron_schedules'));
        
        // Schedule worker process
        add_action('delcampe_process_queue', array($this, 'process_queue'));
        add_action('init', array($this, 'schedule_worker'), 20);
        // Resolver to populate listing IDs for verifying items
        add_action('delcampe_resolve_listing_ids', array($this, 'cron_resolve_ids'));
    }
    
    /**
     * Check if tables need to be created
     * v1.10.35.6: Added to prevent duplicate creation attempts
     */
    private function maybe_create_tables() {
        global $wpdb;
        
        // Check if queue table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$this->queue_table}'");
        
        if (!$table_exists) {
            $this->create_tables();
        }
    }
    
    /**
     * Create database tables
     * v1.10.35.6: Made public so it can be called during activation
     */
    public function create_tables() {
        global $wpdb;
        
        $charset_collate = $wpdb->get_charset_collate();
        
        // Main queue table - single source of truth
        $queue_sql = "CREATE TABLE IF NOT EXISTS {$this->queue_table} (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            product_id bigint(20) UNSIGNED NOT NULL,
            action varchar(20) NOT NULL DEFAULT 'create',
            state varchar(20) NOT NULL DEFAULT 'pending',
            priority int(11) NOT NULL DEFAULT 0,
            attempts int(11) NOT NULL DEFAULT 0,
            max_attempts int(11) NOT NULL DEFAULT " . self::MAX_RETRIES . ",
            idempotency_key varchar(64) NOT NULL,
            worker_id varchar(64) DEFAULT NULL,
            locked_at datetime DEFAULT NULL,
            next_attempt_at datetime DEFAULT NULL,
            error_message text DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY unique_product_action (product_id, action),
            UNIQUE KEY unique_idempotency (idempotency_key),
            KEY state_priority_idx (state, priority, id),
            KEY next_attempt_idx (next_attempt_at),
            KEY worker_timeout_idx (worker_id, locked_at)
        ) {$charset_collate};";
        
        // Product metadata table - stores Delcampe listing info
        $meta_sql = "CREATE TABLE IF NOT EXISTS {$this->product_meta_table} (
            id bigint(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            product_id bigint(20) UNSIGNED NOT NULL,
            delcampe_listing_id varchar(64) DEFAULT NULL,
            state varchar(20) NOT NULL DEFAULT 'local',
            listing_data longtext DEFAULT NULL,
            last_synced_at datetime DEFAULT NULL,
            created_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY product_id (product_id),
            KEY state_idx (state),
            KEY listing_id_idx (delcampe_listing_id)
        ) {$charset_collate};";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($queue_sql);
        dbDelta($meta_sql);
        
        delcampe_log('[Delcampe Queue v2.0] Database tables created/updated');
    }
    
    /**
     * Enqueue a product for processing
     * 
     * @param int $product_id WooCommerce product ID
     * @param string $action Action to perform (create, update, delete)
     * @param int $priority Priority (higher = processed first)
     * @return bool|WP_Error True on success, WP_Error on failure
     */
    public function enqueue($product_id, $action = self::ACTION_CREATE, $priority = 0) {
        global $wpdb;
        
        // Validate inputs
        $product_id = intval($product_id);
        if ($product_id <= 0) {
            return new WP_Error('invalid_product', 'Invalid product ID');
        }
        
        if (!in_array($action, [self::ACTION_CREATE, self::ACTION_UPDATE, self::ACTION_DELETE])) {
            return new WP_Error('invalid_action', 'Invalid action');
        }
        
        // Check if product exists
        if (!get_post($product_id)) {
            return new WP_Error('product_not_found', 'Product not found');
        }
        
        // Generate idempotency key based on product data
        $idempotency_key = $this->generate_idempotency_key($product_id, $action);
        
        // Start transaction for atomic operation
        $wpdb->query('START TRANSACTION');
        
        try {
            // Check for existing queue entry
            $existing = $wpdb->get_row($wpdb->prepare(
                "SELECT id, state FROM {$this->queue_table} WHERE product_id = %d AND action = %s",
                $product_id, $action
            ));
            
            if ($existing) {
                if (in_array($existing->state, [self::STATE_PENDING, self::STATE_PROCESSING])) {
                    $wpdb->query('COMMIT');
                    return true; // Already queued, no need to duplicate
                }
                
                // Update existing entry to retry
                $result = $wpdb->update(
                    $this->queue_table,
                    array(
                        'state' => self::STATE_PENDING,
                        'priority' => max($priority, 0),
                        'idempotency_key' => $idempotency_key,
                        'worker_id' => null,
                        'locked_at' => null,
                        'next_attempt_at' => null,
                        'updated_at' => current_time('mysql')
                    ),
                    array('id' => $existing->id)
                );
            } else {
                // Insert new queue entry
                $result = $wpdb->insert(
                    $this->queue_table,
                    array(
                        'product_id' => $product_id,
                        'action' => $action,
                        'state' => self::STATE_PENDING,
                        'priority' => max($priority, 0),
                        'idempotency_key' => $idempotency_key
                    )
                );
            }
            
            if ($result === false) {
                $wpdb->query('ROLLBACK');
                return new WP_Error('db_error', 'Failed to enqueue product: ' . $wpdb->last_error);
            }
            
            // Update product meta state
            $this->update_product_meta_state($product_id, self::STATE_PENDING);
            
            $wpdb->query('COMMIT');
            
            delcampe_log("[Delcampe Queue] Enqueued product {$product_id} for {$action} (priority: {$priority})");
            
            return true;
            
        } catch (Exception $e) {
            $wpdb->query('ROLLBACK');
            return new WP_Error('exception', $e->getMessage());
        }
    }
    
    /**
     * Get next item from queue for processing
     * Uses atomic locking to prevent race conditions
     * 
     * @param string $worker_id Unique worker identifier
     * @return object|null Queue item or null if none available
     */
    public function dequeue($worker_id) {
        global $wpdb;
        
        $worker_id = sanitize_text_field($worker_id);
        $timeout_threshold = date('Y-m-d H:i:s', time() - self::WORKER_TIMEOUT);
        
        // Visibility into claimable items
        try {
            $total_pending = (int)$wpdb->get_var("SELECT COUNT(*) FROM {$this->queue_table} WHERE state='pending'");
            $pending_now  = (int)$wpdb->get_var("SELECT COUNT(*) FROM {$this->queue_table} WHERE state='pending' AND (next_attempt_at IS NULL OR next_attempt_at <= NOW())");
            $processing   = (int)$wpdb->get_var("SELECT COUNT(*) FROM {$this->queue_table} WHERE state='processing'");
            delcampe_log("[Delcampe Queue] Stats before claim: pending_now={$pending_now}, pending_total={$total_pending}, processing={$processing}");
        } catch (\Throwable $e) {}
        
        // Use transaction with proper isolation
        $wpdb->query('SET TRANSACTION ISOLATION LEVEL READ COMMITTED');
        $wpdb->query('START TRANSACTION');
        
        try {
            // First, recover any stuck items from this worker or timed out workers
            $wpdb->query($wpdb->prepare(
                "UPDATE {$this->queue_table} 
                 SET state = %s, worker_id = NULL, locked_at = NULL, next_attempt_at = NULL 
                 WHERE state = %s AND (worker_id = %s OR locked_at < %s)",
                self::STATE_PENDING, self::STATE_PROCESSING, $worker_id, $timeout_threshold
            ));
            
            // Get next available item with atomic locking
            $sql = $wpdb->prepare(
                "SELECT * FROM {$this->queue_table} 
                 WHERE state = %s AND (next_attempt_at IS NULL OR next_attempt_at <= NOW())
                 ORDER BY priority DESC, id ASC 
                 LIMIT 1 
                 FOR UPDATE SKIP LOCKED",
                self::STATE_PENDING
            );
            
            $item = $wpdb->get_row($sql);
            // Fallback for environments without SKIP LOCKED support
            if (!$item && !empty($wpdb->last_error)) {
                $sql_fallback = $wpdb->prepare(
                    "SELECT * FROM {$this->queue_table} 
                     WHERE state = %s AND (next_attempt_at IS NULL OR next_attempt_at <= NOW())
                     ORDER BY priority DESC, id ASC 
                     LIMIT 1 
                     FOR UPDATE",
                    self::STATE_PENDING
                );
                $item = $wpdb->get_row($sql_fallback);
            }
            
            if (!$item) {
                $wpdb->query('COMMIT');
                return null;
            }
            
            // Immediately claim the item
            $update_result = $wpdb->update(
                $this->queue_table,
                array(
                    'state' => self::STATE_PROCESSING,
                    'worker_id' => $worker_id,
                    'locked_at' => current_time('mysql'),
                    'updated_at' => current_time('mysql')
                ),
                array('id' => $item->id)
            );
            
            if ($update_result === false) {
                $wpdb->query('ROLLBACK');
                return null;
            }
            
            $wpdb->query('COMMIT');
            
            delcampe_log("[Delcampe Queue] Worker {$worker_id} claimed item {$item->id} (product {$item->product_id})");
            
            return $item;
            
        } catch (Exception $e) {
            $wpdb->query('ROLLBACK');
            delcampe_log("[Delcampe Queue] Error in dequeue: " . $e->getMessage());
            return null;
        }
    }

    /**
     * Force make all pending items claimable now (clear backoff), and recover stale processing
     */
    public function force_make_claimable() {
        global $wpdb;
        // Clear backoff on pending
        $wpdb->query("UPDATE {$this->queue_table} SET next_attempt_at = NULL WHERE state='pending'");
        // Recover any processing older than 1 minute
        $wpdb->query("UPDATE {$this->queue_table} SET state='pending', worker_id=NULL, locked_at=NULL, next_attempt_at=NULL WHERE state='processing' AND locked_at < DATE_SUB(NOW(), INTERVAL 1 MINUTE)");
        delcampe_log('[Delcampe Queue] force_make_claimable executed');
    }
    
    /**
     * Mark queue item as completed successfully
     * 
     * @param int $queue_id Queue item ID
     * @param string $delcampe_listing_id Delcampe listing ID (for creates)
     * @return bool Success
     */
    public function mark_completed($queue_id, $delcampe_listing_id = null) {
        global $wpdb;
        
        $wpdb->query('START TRANSACTION');
        
        try {
            // Get queue item
            $item = $wpdb->get_row($wpdb->prepare(
                "SELECT * FROM {$this->queue_table} WHERE id = %d",
                $queue_id
            ));
            
            if (!$item) {
                $wpdb->query('ROLLBACK');
                return false;
            }
            
            // v1.10.35.13: Don't just update to 'published' state - we'll delete it instead
            // This was the bug causing items to remain in queue after processing
            
            // Update product meta
            $meta_data = array(
                'state' => self::STATE_PUBLISHED,
                'last_synced_at' => current_time('mysql')
            );
            
            if ($delcampe_listing_id) {
                $meta_data['delcampe_listing_id'] = $delcampe_listing_id;
            }
            
            $this->update_product_meta($item->product_id, $meta_data);
            
            // Update WooCommerce meta
            update_post_meta($item->product_id, '_delcampe_sync_status', 'published');
            if ($delcampe_listing_id) {
                update_post_meta($item->product_id, '_delcampe_listing_id', $delcampe_listing_id);
            }

            // Ensure a row exists in the legacy listings table for admin UI
            $this->upsert_listing_row((int)$item->product_id, 'published', $delcampe_listing_id);
            
            // v1.10.35.13: DELETE the queue item after successful processing
            // This prevents items from staying in queue with 'published' state
            $deleted = $wpdb->delete(
                $this->queue_table,
                array('id' => $queue_id)
            );
            
            if (!$deleted) {
                delcampe_log("[Delcampe Queue] Warning: Failed to delete completed queue item {$queue_id}");
            }
            
            $wpdb->query('COMMIT');
            
            delcampe_log("[Delcampe Queue] Completed and removed queue item {$queue_id} (product {$item->product_id})");
            
            return true;
            
        } catch (Exception $e) {
            $wpdb->query('ROLLBACK');
            delcampe_log("[Delcampe Queue] Error marking completed: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Mark queue item as verifying (accepted by API; awaiting Delcampe ID).
     */
    public function mark_verifying($queue_id, $product_id, $personal_reference = null) {
        global $wpdb;
        $wpdb->update(
            $this->queue_table,
            array(
                'state' => self::STATE_VERIFYING,
                'worker_id' => null,
                'locked_at' => null,
                'updated_at' => current_time('mysql')
            ),
            array('id' => (int)$queue_id)
        );
        $this->update_product_meta($product_id, array('state' => self::STATE_VERIFYING));
        if ($personal_reference) {
            update_post_meta($product_id, '_delcampe_personal_reference', $personal_reference);
        }
        // Reflect verifying state in listings table so admin UI doesn't prompt for import
        $this->upsert_listing_row((int)$product_id, 'verified', null);
        delcampe_log("[Delcampe Queue] Item {$queue_id} set to verifying (product {$product_id})");
        return true;
    }

    /**
     * Create or update the legacy listings table row for a product.
     * Keeps the Listings admin UI in sync with the new queue system.
     */
    private function upsert_listing_row($product_id, $status, $delcampe_id = null) {
        try {
            if (!function_exists('wc_get_product')) return;
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-delcampe-listings-model.php';
            $existing = Delcampe_Listings_Model::get_listings_by_product_id($product_id);
            $product = wc_get_product($product_id);
            $title = $product ? $product->get_name() : '';
            $personal_ref = get_post_meta($product_id, '_delcampe_personal_reference', true) ?: get_post_meta($product_id, '_sku', true);
            
            // Get profile_id from product meta, default to 2 (plate blocks) if not set
            $profile_id = get_post_meta($product_id, '_delcampe_profile_id', true);
            if (!$profile_id) {
                $profile_id = 2; // Default to plate blocks profile
            }
            
            $data = array(
                'listing_title' => $title,
                'status' => $status,
                'profile_id' => $profile_id,
            );
            if ($delcampe_id) { $data['delcampe_id'] = (string)$delcampe_id; }
            if ($personal_ref) { $data['personal_reference'] = (string)$personal_ref; }
            if ($status === 'published') { $data['date_published'] = current_time('mysql'); }
            if (!empty($existing)) {
                // Update first row for this product
                Delcampe_Listings_Model::update_listing((int)$existing[0]->id, $data);
            } else {
                // Insert new row
                $data['product_id'] = (int)$product_id;
                if ($product && $product->is_type('variable')) {
                    $data['parent_id'] = (int)$product_id;
                }
                Delcampe_Listings_Model::insert_listing($data);
            }
        } catch (\Throwable $e) {
            delcampe_log('[Delcampe Queue] upsert_listing_row error: ' . $e->getMessage());
        }
    }

    /**
     * Resolve verified items from legacy listings table
     * 
     * @param int $limit Number of items to process
     * @return int Number of items resolved
     */
    public function resolve_legacy_verified_items($limit = 25) {
        global $wpdb;
        
        $listings_table = $wpdb->prefix . 'delcampe_listings';
        
        // First fix any verified items with missing data
        $this->fix_verified_listings_data();
        
        // Get verified items without listing IDs
        $items = $wpdb->get_results($wpdb->prepare(
            "SELECT l.id, l.product_id, l.personal_reference 
             FROM {$listings_table} l
             WHERE l.status = 'verified' 
             AND (l.delcampe_id IS NULL OR l.delcampe_id = '')
             LIMIT %d",
            $limit
        ));
        
        if (empty($items)) {
            return 0;
        }
        
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-listing-api.php';
        $api = Delcampe_Listing_API::get_instance();
        $count = 0;
        
        foreach ($items as $item) {
            $product_id = (int)$item->product_id;
            
            // Get SKU/personal reference
            $ref = $item->personal_reference;
            if (!$ref) {
                $product = wc_get_product($product_id);
                if ($product) {
                    $ref = $product->get_sku();
                }
            }
            
            if (!$ref) {
                delcampe_log("[Queue Resolver] No SKU for verified product {$product_id}, skipping");
                continue;
            }
            
            // Check if item exists on Delcampe
            $found = $api->check_sku_exists_on_delcampe($ref);
            
            if (is_array($found) && !empty($found['id'])) {
                // Update listing with Delcampe ID
                $wpdb->update(
                    $listings_table,
                    array(
                        'delcampe_id' => $found['id'],
                        'status' => 'published',
                        'date_updated' => current_time('mysql')
                    ),
                    array('id' => $item->id),
                    array('%s', '%s', '%s'),
                    array('%d')
                );
                
                // Also update post meta
                update_post_meta($product_id, '_delcampe_listing_id', $found['id']);
                update_post_meta($product_id, '_delcampe_sync_status', 'active');
                
                delcampe_log("[Queue Resolver] Resolved legacy verified item: Product {$product_id}, SKU {$ref}, Delcampe ID {$found['id']}");
                $count++;
            } else {
                // Item doesn't exist on Delcampe - re-queue for publishing
                delcampe_log("[Queue Resolver] Verified product {$product_id} (SKU {$ref}) not found on Delcampe - re-queuing for publish");
                
                // Add to queue for creation
                $this->enqueue($product_id, self::ACTION_CREATE);
                
                // Update listing status to pending
                $wpdb->update(
                    $listings_table,
                    array(
                        'status' => 'pending',
                        'date_updated' => current_time('mysql')
                    ),
                    array('id' => $item->id),
                    array('%s', '%s'),
                    array('%d')
                );
                
                delcampe_log("[Queue Resolver] Re-queued product {$product_id} for publishing");
            }
        }
        
        return $count;
    }
    
    /**
     * Fix missing data for verified listings
     */
    private function fix_verified_listings_data() {
        global $wpdb;
        $listings_table = $wpdb->prefix . 'delcampe_listings';
        
        // Get verified listings with missing data
        $listings = $wpdb->get_results("
            SELECT id, product_id, listing_title 
            FROM {$listings_table}
            WHERE status = 'verified' 
            AND (profile_id IS NULL OR quantity IS NULL OR price IS NULL)
        ");
        
        if (empty($listings)) {
            return;
        }
        
        delcampe_log("[Queue Resolver] Fixing " . count($listings) . " verified listings with missing data");
        
        foreach ($listings as $listing) {
            $product = wc_get_product($listing->product_id);
            if (!$product) {
                continue;
            }
            
            // Get product data
            $price = $product->get_regular_price();
            $stock = $product->get_stock_quantity();
            $profile_id = get_post_meta($listing->product_id, '_delcampe_profile_id', true);
            $currency = get_option('woocommerce_currency', 'USD');
            
            // Update the listing with proper data
            $wpdb->update(
                $listings_table,
                array(
                    'profile_id' => $profile_id ?: null,
                    'quantity' => $stock ?: 0,
                    'price' => $price ?: 0.00,
                    'currency' => $currency,
                    'date_updated' => current_time('mysql')
                ),
                array('id' => $listing->id),
                array('%d', '%d', '%f', '%s', '%s'),
                array('%d')
            );
            
            delcampe_log("[Queue Resolver] Fixed data for listing {$listing->id} (Product {$listing->product_id}): Profile={$profile_id}, Price={$currency} {$price}, Stock={$stock}");
        }
    }
    
    /**
     * Resolve Delcampe IDs for items in verifying state.
     */
    public function resolve_ids_batch($limit = 25) {
        global $wpdb;
        $rows = $wpdb->get_results($wpdb->prepare(
            "SELECT id, product_id FROM {$this->queue_table} WHERE state=%s ORDER BY id ASC LIMIT %d",
            self::STATE_VERIFYING, (int)$limit
        ));
        if (empty($rows)) return 0;
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-delcampe-listing-api.php';
        $api = Delcampe_Listing_API::get_instance();
        $count = 0;
        foreach ($rows as $row) {
            $pid = (int)$row->product_id;
            // If local listings table already has an ID, complete without API call
            $local_id = $wpdb->get_var($wpdb->prepare(
                "SELECT delcampe_id FROM {$wpdb->prefix}delcampe_listings WHERE product_id=%d AND delcampe_id IS NOT NULL AND delcampe_id <> '' LIMIT 1",
                $pid
            ));
            if (!empty($local_id)) {
                $this->mark_completed($row->id, (string)$local_id);
                if (function_exists('delcampe_publish_audit')) {
                    delcampe_publish_audit(array(
                        'phase' => 'resolve_success_local',
                        'product_id' => $pid,
                        'listing_id' => (string)$local_id,
                    ));
                }
                $count++;
                continue;
            }
            // Also check product meta as a fallback
            $meta_id = get_post_meta($pid, '_delcampe_item_id', true);
            if (!empty($meta_id)) {
                $this->mark_completed($row->id, (string)$meta_id);
                if (function_exists('delcampe_publish_audit')) {
                    delcampe_publish_audit(array(
                        'phase' => 'resolve_success_meta',
                        'product_id' => $pid,
                        'listing_id' => (string)$meta_id,
                    ));
                }
                $count++;
                continue;
            }
            $ref = get_post_meta($pid, '_delcampe_personal_reference', true);
            if (!$ref) { continue; }
            $found = $api->check_sku_exists_on_delcampe($ref);
            if (is_array($found) && !empty($found['id'])) {
                $this->mark_completed($row->id, (string)$found['id']);
                if (function_exists('delcampe_publish_audit')) {
                    delcampe_publish_audit(array(
                        'phase' => 'resolve_success',
                        'product_id' => $pid,
                        'listing_id' => (string)$found['id'],
                        'personal_reference' => $ref
                    ));
                }
                $count++;
            } else {
                if (function_exists('delcampe_publish_audit')) {
                    delcampe_publish_audit(array(
                        'phase' => 'resolve_not_found',
                        'product_id' => $pid,
                        'personal_reference' => $ref
                    ));
                }
                // Backoff next attempt to reduce API chatter
                $delay = (int) apply_filters('delcampe_resolver_retry_delay', 3600); // default 60 minutes
                $next = date('Y-m-d H:i:s', time() + max(300, $delay));
                $wpdb->update(
                    $this->queue_table,
                    array('next_attempt_at' => $next, 'updated_at' => current_time('mysql')),
                    array('id' => (int)$row->id)
                );
            }
        }
        return $count;
    }
    
    /**
     * Mark queue item as failed or schedule retry
     * 
     * @param int $queue_id Queue item ID
     * @param string $error_message Error message
     * @param bool $is_permanent Whether this is a permanent failure
     * @return bool Success
     */
    public function mark_failed($queue_id, $error_message, $is_permanent = false) {
        global $wpdb;
        
        $item = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$this->queue_table} WHERE id = %d",
            $queue_id
        ));
        
        if (!$item) {
            return false;
        }
        
        $attempts = $item->attempts + 1;
        
        // Check if we should retry or fail permanently
        if ($is_permanent || $attempts >= $item->max_attempts) {
            // Permanent failure
            $wpdb->update(
                $this->queue_table,
                array(
                    'state' => self::STATE_ERRORED,
                    'attempts' => $attempts,
                    'error_message' => sanitize_textarea_field($error_message),
                    'worker_id' => null,
                    'locked_at' => null,
                    'updated_at' => current_time('mysql')
                ),
                array('id' => $queue_id)
            );
            
            $this->update_product_meta_state($item->product_id, self::STATE_ERRORED);
            update_post_meta($item->product_id, '_delcampe_sync_status', 'error');
            
            delcampe_log("[Delcampe Queue] Failed item {$queue_id} permanently: {$error_message}");
            
        } else {
            // Schedule retry with exponential backoff
            $delays = [60, 300, 900, 3600, 7200]; // 1min, 5min, 15min, 1hr, 2hr
            $delay_index = min($attempts - 1, count($delays) - 1);
            $delay = $delays[$delay_index] + rand(0, 60); // Add jitter
            $next_attempt = date('Y-m-d H:i:s', time() + $delay);
            
            $wpdb->update(
                $this->queue_table,
                array(
                    'state' => self::STATE_PENDING,
                    'attempts' => $attempts,
                    'error_message' => sanitize_textarea_field($error_message),
                    'next_attempt_at' => $next_attempt,
                    'worker_id' => null,
                    'locked_at' => null,
                    'updated_at' => current_time('mysql')
                ),
                array('id' => $queue_id)
            );
            
            delcampe_log("[Delcampe Queue] Scheduled retry {$attempts}/{$item->max_attempts} for item {$queue_id} at {$next_attempt}: {$error_message}");
        }
        
        return true;
    }
    
    /**
     * Get queue statistics
     * 
     * @return array Queue statistics
     */
    public function get_stats() {
        global $wpdb;
        
        $stats = $wpdb->get_results(
            "SELECT state, COUNT(*) as count FROM {$this->queue_table} GROUP BY state"
        );
        
        $result = array(
            'pending' => 0,
            'processing' => 0,
            'published' => 0,
            'errored' => 0,
            'total' => 0
        );
        
        foreach ($stats as $stat) {
            $result[$stat->state] = intval($stat->count);
            $result['total'] += intval($stat->count);
        }
        
        // Add processing metrics
        $result['avg_processing_time'] = $wpdb->get_var(
            "SELECT AVG(TIMESTAMPDIFF(SECOND, locked_at, updated_at)) 
             FROM {$this->queue_table} 
             WHERE state = 'published' AND locked_at IS NOT NULL 
             AND updated_at > DATE_SUB(NOW(), INTERVAL 7 DAY)"
        );
        
        return $result;
    }
    
    /**
     * Clear completed and errored items older than specified days
     * 
     * @param int $days Number of days to keep
     * @return int Number of items cleared
     */
    public function cleanup($days = 7) {
        global $wpdb;
        
        $result = $wpdb->query($wpdb->prepare(
            "DELETE FROM {$this->queue_table} 
             WHERE state IN ('published', 'errored') 
             AND updated_at < DATE_SUB(NOW(), INTERVAL %d DAY)",
            $days
        ));
        
        if ($result > 0) {
            delcampe_log("[Delcampe Queue] Cleaned up {$result} old queue items");
        }
        
        return $result;
    }
    
    /**
     * Get the queue table name
     * 
     * @return string Queue table name
     */
    public function get_table_name() {
        return $this->queue_table;
    }
    
    /**
     * Add custom cron schedules
     */
    public function add_cron_schedules($schedules) {
        if (!isset($schedules['five_minutes'])) {
            $schedules['five_minutes'] = array(
                'interval' => 300,
                'display' => __('Every 5 Minutes', 'wc-delcampe-integration')
            );
        }
        if (!isset($schedules['delcampe_queue_interval'])) {
            $schedules['delcampe_queue_interval'] = array(
                'interval' => 60,
                'display' => __('Every Minute', 'wc-delcampe-integration')
            );
        }
        return $schedules;
    }
    
    /**
     * Schedule the worker process
     */
    public function schedule_worker() {
        // v1.10.35.17: Disabled automatic scheduling - these should be manual or controlled by settings
        // Queue processing and resolver were causing excessive API calls
        /*
        if (!wp_next_scheduled('delcampe_process_queue')) {
            wp_schedule_event(time(), 'delcampe_queue_interval', 'delcampe_process_queue');
        }
        if (!wp_next_scheduled('delcampe_resolve_listing_ids')) {
            // Register a 15-minute schedule if missing (filterable)
            add_filter('cron_schedules', function($s){ if (!isset($s['fifteen_minutes'])) $s['fifteen_minutes']=['interval'=>900,'display'=>'Every 15 Minutes']; return $s;});
            $interval = apply_filters('delcampe_resolver_schedule', 'fifteen_minutes');
            wp_schedule_event(time() + 300, $interval, 'delcampe_resolve_listing_ids');
        }
        */
        // Opportunistic backfill to keep legacy listings table in sync
        if (!get_transient('delcampe_backfill_listings_rows')) {
            $this->backfill_missing_listings_rows(200);
            set_transient('delcampe_backfill_listings_rows', 1, 300); // at most every 5 minutes
        }
    }

    /**
     * Cron callback to resolve verifying items
     */
    public function cron_resolve_ids() {
        // Resolve items from new queue system
        $resolved = $this->resolve_ids_batch(apply_filters('delcampe_resolver_chunk', 25));
        
        // ALSO resolve verified items from legacy listings table
        $resolved_legacy = $this->resolve_legacy_verified_items(25);
        
        $total = $resolved + $resolved_legacy;
        if ($total > 0) {
            delcampe_log("[Delcampe Queue] Resolver populated {$total} listing ID(s) ({$resolved} queue, {$resolved_legacy} legacy)");
        }
    }
    
    /**
     * Process queue items (called by cron)
     */
    public function process_queue() {
        // Set sync processing lock to prevent reconciliation during processing
        set_transient('delcampe_sync_processing_lock', true, 600); // 10 minute lock
        
        try {
            $worker = Delcampe_Queue_Worker::get_instance();
            // Process ALL pending items in the queue (no batch limit)
            // This ensures the FIFO queue is fully processed each run
            $batch_size = (int) apply_filters('delcampe_queue_batch_size', 9999);
            if ($batch_size < 1) { $batch_size = 9999; }
            delcampe_log('[Queue] Processing all pending items (no batch limit)');
            $worker->process_batch($batch_size);
        } finally {
            // Always release lock when done
            delete_transient('delcampe_sync_processing_lock');
        }
    }

    /**
     * Backfill missing rows in legacy listings table from product_meta state.
     */
    private function backfill_missing_listings_rows($limit = 200) {
        global $wpdb;
        $meta = $this->product_meta_table;
        $listings = $wpdb->prefix . 'delcampe_listings';
        $rows = $wpdb->get_results($wpdb->prepare(
            "SELECT pm.product_id, pm.delcampe_listing_id, pm.state
             FROM {$meta} pm
             LEFT JOIN {$listings} l ON l.product_id = pm.product_id
             WHERE l.id IS NULL AND pm.state IN (%s,%s)
             ORDER BY pm.updated_at DESC
             LIMIT %d",
            self::STATE_VERIFYING, self::STATE_PUBLISHED, (int)$limit
        ));
        if (empty($rows)) return 0;
        $count = 0;
        foreach ($rows as $r) {
            $status = ($r->state === self::STATE_PUBLISHED) ? 'published' : 'verified';
            $this->upsert_listing_row((int)$r->product_id, $status, $r->delcampe_listing_id ? (string)$r->delcampe_listing_id : null);
            $count++;
        }
        // Also backfill from legacy postmeta for older records
        $pm = $wpdb->postmeta;
        $rows2 = $wpdb->get_results(
            "SELECT pm1.post_id AS product_id,
                    MAX(CASE WHEN pm1.meta_key = '_delcampe_listing_id' THEN pm1.meta_value END) AS delcampe_id,
                    MAX(CASE WHEN pm2.meta_key = '_delcampe_sync_status' THEN pm2.meta_value END) AS sync_status
             FROM {$pm} pm1
             LEFT JOIN {$pm} pm2 ON pm2.post_id = pm1.post_id AND pm2.meta_key = '_delcampe_sync_status'
             LEFT JOIN {$listings} l ON l.product_id = pm1.post_id
             WHERE l.id IS NULL
               AND pm1.meta_key IN ('_delcampe_listing_id','_delcampe_sync_status')
             GROUP BY pm1.post_id
             HAVING (sync_status IN ('published','active','pending') OR delcampe_id IS NOT NULL)
             ORDER BY MAX(pm1.post_id) DESC
             LIMIT " . intval($limit)
        );
        if (!empty($rows2)) {
            foreach ($rows2 as $r) {
                $status = (!empty($r->delcampe_id) || $r->sync_status === 'published') ? 'published' : 'verified';
                $this->upsert_listing_row((int)$r->product_id, $status, !empty($r->delcampe_id) ? (string)$r->delcampe_id : null);
                $count++;
            }
        }
        if ($count > 0) {
            delcampe_log("[Delcampe Queue] Backfilled {$count} listing row(s)");
        }
        return $count;
    }

    /** Public wrapper to run backfill on demand. */
    public function run_backfill_listings_rows($limit = 200) {
        return $this->backfill_missing_listings_rows($limit);
    }

    /**
     * Requeue an item to run at a specific time (rate limit/backoff helper)
     *
     * @param int $queue_id
     * @param string $next_datetime MySQL datetime (UTC or WP time OK)
     * @return bool
     */
    public function requeue_at($queue_id, $next_datetime) {
        global $wpdb;
        $res = $wpdb->update(
            $this->queue_table,
            array(
                'state' => self::STATE_PENDING,
                'next_attempt_at' => $next_datetime,
                'worker_id' => null,
                'locked_at' => null,
                'updated_at' => current_time('mysql')
            ),
            array('id' => (int)$queue_id)
        );
        return $res !== false;
    }
    
    /**
     * Generate idempotency key for product
     * 
     * @param int $product_id Product ID
     * @param string $action Action
     * @return string Idempotency key
     */
    private function generate_idempotency_key($product_id, $action) {
        $product = wc_get_product($product_id);
        if (!$product) {
            return hash('sha256', "{$product_id}:{$action}:" . time());
        }
        
        // Include key product data that would affect the listing
        $data = array(
            'product_id' => $product_id,
            'action' => $action,
            'title' => $product->get_name(),
            'price' => $product->get_price(),
            'sku' => $product->get_sku(),
            'stock' => $product->get_stock_quantity(),
            'modified' => $product->get_date_modified()->getTimestamp()
        );
        
        return hash('sha256', serialize($data));
    }
    
    /**
     * Update product meta state
     * 
     * @param int $product_id Product ID
     * @param string $state New state
     */
    private function update_product_meta_state($product_id, $state) {
        $this->update_product_meta($product_id, array('state' => $state));
    }
    
    /**
     * Update product meta data
     * 
     * @param int $product_id Product ID
     * @param array $data Data to update
     */
    private function update_product_meta($product_id, $data) {
        global $wpdb;
        
        $data['updated_at'] = current_time('mysql');
        
        $existing = $wpdb->get_var($wpdb->prepare(
            "SELECT id FROM {$this->product_meta_table} WHERE product_id = %d",
            $product_id
        ));
        
        if ($existing) {
            $wpdb->update(
                $this->product_meta_table,
                $data,
                array('product_id' => $product_id)
            );
        } else {
            $data['product_id'] = $product_id;
            $wpdb->insert($this->product_meta_table, $data);
        }
    }
}
