<?php
/**
 * Delcampe Reconciliation System
 * 
 * Handles daily reconciliation between local database and Delcampe listings
 * to identify and fix missing IDs, orphaned listings, and duplicates.
 * 
 * @package     WooCommerce_Delcampe_Integration
 * @subpackage  Core
 * @since       1.10.18.13
 * @version     1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Reconciliation {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Schedule daily reconciliation if not already scheduled
        add_action('init', array($this, 'schedule_reconciliation'));
        
        // Hook for the scheduled event
        add_action('delcampe_daily_reconciliation', array($this, 'run_reconciliation'));
        
        // Add WP-CLI command if available
        if (defined('WP_CLI') && WP_CLI) {
            WP_CLI::add_command('delcampe reconcile', array($this, 'cli_reconcile'));
        }
    }
    
    /**
     * Schedule daily reconciliation
     */
    public function schedule_reconciliation() {
        if (!wp_next_scheduled('delcampe_daily_reconciliation')) {
            // Schedule for 3 AM local time
            $timestamp = strtotime('tomorrow 3:00am');
            wp_schedule_event($timestamp, 'daily', 'delcampe_daily_reconciliation');
        }
    }
    
    /**
     * Run reconciliation process
     * 
     * @return array Results summary
     */
    public function run_reconciliation() {
        // Check if batch processing is currently running
        if (get_transient('delcampe_batch_processing_lock')) {
            delcampe_log('[Reconciliation] Skipped - batch processing is currently active');
            return array(
                'skipped' => true,
                'reason' => 'Batch processing is currently active',
                'timestamp' => current_time('mysql')
            );
        }
        
        // Check if sync queue is actively processing
        if (get_transient('delcampe_sync_processing_lock')) {
            delcampe_log('[Reconciliation] Skipped - sync queue is currently processing');
            return array(
                'skipped' => true,
                'reason' => 'Sync queue is currently processing',
                'timestamp' => current_time('mysql')
            );
        }
        
        // Set reconciliation lock to prevent concurrent runs
        if (get_transient('delcampe_reconciliation_lock')) {
            delcampe_log('[Reconciliation] Already running - skipping duplicate run');
            return array(
                'skipped' => true,
                'reason' => 'Reconciliation is already running',
                'timestamp' => current_time('mysql')
            );
        }
        
        // Set lock for 30 minutes max
        set_transient('delcampe_reconciliation_lock', true, 1800);
        
        $start_time = microtime(true);
        $results = array(
            'started_at' => current_time('mysql'),
            'missing_locally' => 0,
            'missing_on_delcampe' => 0,
            'fixed_missing_ids' => 0,
            'orphaned_marked' => 0,
            'duplicates_resolved' => 0,
            'errors' => array()
        );
        
        try {
            // Step 1: Fetch all open items from Delcampe
            $delcampe_items = $this->fetch_all_delcampe_items();
            
            if (is_wp_error($delcampe_items)) {
                $results['errors'][] = 'Failed to fetch Delcampe items: ' . $delcampe_items->get_error_message();
                $this->log_results($results);
                delete_transient('delcampe_reconciliation_lock'); // Release lock
                return $results;
            }
            
            $results['delcampe_total'] = count($delcampe_items);
            
            // Step 2: Build lookup maps
            $delcampe_by_id = array();
            $delcampe_by_ref = array();
            
            foreach ($delcampe_items as $item) {
                if (!empty($item['id'])) {
                    $delcampe_by_id[$item['id']] = $item;
                }
                if (!empty($item['personal_reference'])) {
                    $delcampe_by_ref[$item['personal_reference']] = $item;
                }
            }
            
            // Step 3: Get all local published products
            $local_products = $this->get_local_published_products();
            $results['local_total'] = count($local_products);
            
            // Step 4: Fix missing IDs (with queue and status checks to prevent duplicates)
            foreach ($local_products as $product_id => $product_data) {
                $stored_id = $product_data['delcampe_id'];
                $sku = $product_data['sku'];
                $personal_ref = $product_data['personal_reference'];
                
                // Check if we have ID locally
                if (empty($stored_id)) {
                    // CRITICAL: Check if product is currently being processed in queues
                    global $wpdb;
                    
                    // Check sync queue
                    $in_sync_queue = $wpdb->get_var($wpdb->prepare(
                        "SELECT COUNT(*) FROM {$wpdb->prefix}delcampe_queue 
                         WHERE product_id = %d AND state IN ('pending', 'processing', 'queued')",
                        $product_id
                    ));
                    
                    if ($in_sync_queue > 0) {
                        delcampe_log('[Reconciliation] Skipping product ' . $product_id . ' - currently in sync queue');
                        continue;
                    }
                    
                    // Check batch queue
                    $in_batch_queue = $wpdb->get_var($wpdb->prepare(
                        "SELECT COUNT(*) FROM {$wpdb->prefix}delcampe_batch_queue 
                         WHERE product_id = %d AND status IN ('pending', 'processing')",
                        $product_id
                    ));
                    
                    if ($in_batch_queue > 0) {
                        delcampe_log('[Reconciliation] Skipping product ' . $product_id . ' - currently in batch queue');
                        continue;
                    }
                    
                    // Check sync status meta
                    $sync_status = get_post_meta($product_id, '_delcampe_sync_status', true);
                    if (in_array($sync_status, ['pending', 'processing', 'queued', 'in_progress'])) {
                        delcampe_log('[Reconciliation] Skipping product ' . $product_id . ' - sync status: ' . $sync_status);
                        continue;
                    }
                    
                    // Check if product was recently modified (cooling period)
                    $modified_time = get_post_modified_time('U', false, $product_id);
                    $hours_since_modified = (time() - $modified_time) / 3600;
                    if ($hours_since_modified < 24) {
                        delcampe_log('[Reconciliation] Skipping product ' . $product_id . ' - modified ' . round($hours_since_modified, 1) . ' hours ago');
                        continue;
                    }
                    
                    // Try to find by personal reference or SKU
                    $found_item = null;
                    
                    if (!empty($personal_ref) && isset($delcampe_by_ref[$personal_ref])) {
                        $found_item = $delcampe_by_ref[$personal_ref];
                    } elseif (!empty($sku) && isset($delcampe_by_ref[$sku])) {
                        $found_item = $delcampe_by_ref[$sku];
                    }
                    
                    if ($found_item) {
                        // Found on Delcampe - update local record
                        delcampe_log('[Reconciliation] Fixing missing ID for product ' . $product_id . ' - found Delcampe ID: ' . $found_item['id']);
                        $this->fix_missing_id($product_id, $found_item['id'], $found_item);
                        $results['fixed_missing_ids']++;
                    } else {
                        // Not found on Delcampe - mark as orphaned
                        $results['missing_on_delcampe']++;
                    }
                } else {
                    // Have ID - verify it exists on Delcampe
                    if (!isset($delcampe_by_id[$stored_id])) {
                        // Orphaned locally
                        $this->mark_as_orphaned($product_id, $stored_id);
                        $results['orphaned_marked']++;
                    }
                }
            }
            
            // Step 5: Find items on Delcampe not tracked locally
            foreach ($delcampe_by_id as $delcampe_id => $item) {
                $found_locally = false;
                
                // Check if any local product has this ID
                foreach ($local_products as $product_data) {
                    if ($product_data['delcampe_id'] == $delcampe_id) {
                        $found_locally = true;
                        break;
                    }
                }
                
                if (!$found_locally) {
                    // Try to find by SKU/reference
                    $matched_product = $this->find_product_by_reference($item['personal_reference']);
                    
                    if ($matched_product) {
                        // Check if product is in queue or recently modified before fixing
                        global $wpdb;
                        
                        // Check sync queue
                        $in_sync_queue = $wpdb->get_var($wpdb->prepare(
                            "SELECT COUNT(*) FROM {$wpdb->prefix}delcampe_queue 
                             WHERE product_id = %d AND state IN ('pending', 'processing', 'queued')",
                            $matched_product
                        ));
                        
                        // Check batch queue
                        $in_batch_queue = $wpdb->get_var($wpdb->prepare(
                            "SELECT COUNT(*) FROM {$wpdb->prefix}delcampe_batch_queue 
                             WHERE product_id = %d AND status IN ('pending', 'processing')",
                            $matched_product
                        ));
                        
                        // Check sync status
                        $sync_status = get_post_meta($matched_product, '_delcampe_sync_status', true);
                        
                        // Check modification time
                        $modified_time = get_post_modified_time('U', false, $matched_product);
                        $hours_since_modified = (time() - $modified_time) / 3600;
                        
                        if ($in_sync_queue > 0 || $in_batch_queue > 0 || 
                            in_array($sync_status, ['pending', 'processing', 'queued', 'in_progress']) ||
                            $hours_since_modified < 24) {
                            delcampe_log('[Reconciliation] Skipping reverse fix for product ' . $matched_product . ' - currently processing or recently modified');
                        } else {
                            // Safe to link - product not being processed
                            delcampe_log('[Reconciliation] Fixing missing ID (reverse) for product ' . $matched_product . ' - Delcampe ID: ' . $delcampe_id);
                            $this->fix_missing_id($matched_product, $delcampe_id, $item);
                            $results['fixed_missing_ids']++;
                        }
                    } else {
                        $results['missing_locally']++;
                    }
                }
            }
            
            // Step 6: Resolve duplicates
            $duplicates = $this->find_duplicate_listings();
            foreach ($duplicates as $duplicate_set) {
                $this->resolve_duplicates($duplicate_set);
                $results['duplicates_resolved']++;
            }
            
        } catch (Exception $e) {
            $results['errors'][] = 'Exception: ' . $e->getMessage();
            // Always release lock on exception
            delete_transient('delcampe_reconciliation_lock');
            delcampe_log('[Reconciliation] Exception occurred: ' . $e->getMessage());
        }
        
        $results['duration'] = round(microtime(true) - $start_time, 2);
        
        // Log results
        $this->log_results($results);
        
        // Send notification if issues found
        $this->notify_if_needed($results);
        
        // Release reconciliation lock
        delete_transient('delcampe_reconciliation_lock');
        
        return $results;
    }
    
    /**
     * Fetch all open items from Delcampe
     * 
     * @return array|WP_Error Array of items or error
     */
    private function fetch_all_delcampe_items() {
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-auth.php';
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return $token;
        }
        
        $all_items = array();
        $starting_item = 0;
        $batch_size = 100;
        $max_items = 10000; // Safety limit
        
        do {
            // Use proven endpoint format with startingItem/numberOfItems (like in sync handler)
            $query_params = array(
                'token' => $token,
                'startingItem' => $starting_item,
                'numberOfItems' => $batch_size
            );
            
            $url = DELCAMPE_API_BASE_URL . '/item/opened?' . http_build_query($query_params);
            
            $response = wp_remote_get($url, array(
                'timeout' => 30,
                'headers' => array(
                    'Accept' => 'application/xml',
                    'User-Agent' => defined('DELCAMPE_USER_AGENT') ? DELCAMPE_USER_AGENT : 'Delcampe-Sync/1.0'
                )
            ));
            
            if (is_wp_error($response)) {
                error_log('[Delcampe Reconciliation] API error at offset ' . $starting_item . ': ' . $response->get_error_message());
                break;
            }
            
            $code = wp_remote_retrieve_response_code($response);
            if ($code !== 200) {
                error_log('[Delcampe Reconciliation] HTTP ' . $code . ' at offset ' . $starting_item);
                break;
            }
            
            $body = wp_remote_retrieve_body($response);
            if (empty($body)) {
                break;
            }
            
            // Suppress XML errors
            libxml_use_internal_errors(true);
            $xml = simplexml_load_string($body);
            
            if (!$xml) {
                $errors = libxml_get_errors();
                error_log('[Delcampe Reconciliation] XML parse error: ' . print_r($errors, true));
                libxml_clear_errors();
                break;
            }
            
            $items_found = 0;
            
            // Handle multiple possible response formats based on what we've seen
            // Format 1: Direct items
            if (isset($xml->items->item)) {
                foreach ($xml->items->item as $item) {
                    $all_items[] = array(
                        'id' => (string)(isset($item->id_item) ? $item->id_item : $item->id),
                        'personal_reference' => (string)$item->personal_reference,
                        'title' => (string)$item->title,
                        'quantity' => isset($item->qty) ? (int)$item->qty : (isset($item->quantity) ? (int)$item->quantity : 1),
                        'price' => isset($item->fixed_price) ? (float)$item->fixed_price : (isset($item->price) ? (float)$item->price : 0)
                    );
                    $items_found++;
                }
            }
            // Format 2: Notification envelope
            elseif (isset($xml->Notification_Data->items->item)) {
                foreach ($xml->Notification_Data->items->item as $item) {
                    $all_items[] = array(
                        'id' => (string)(isset($item->id_item) ? $item->id_item : $item->id),
                        'personal_reference' => (string)$item->personal_reference,
                        'title' => (string)$item->title,
                        'quantity' => isset($item->qty) ? (int)$item->qty : (isset($item->quantity) ? (int)$item->quantity : 1),
                        'price' => isset($item->fixed_price) ? (float)$item->fixed_price : (isset($item->price) ? (float)$item->price : 0)
                    );
                    $items_found++;
                }
            }
            // Format 3: Direct item nodes (seen in some responses)
            elseif (isset($xml->item)) {
                foreach ($xml->item as $item) {
                    $all_items[] = array(
                        'id' => (string)(isset($item->id_item) ? $item->id_item : $item->id),
                        'personal_reference' => (string)$item->personal_reference,
                        'title' => (string)$item->title,
                        'quantity' => isset($item->qty) ? (int)$item->qty : (isset($item->quantity) ? (int)$item->quantity : 1),
                        'price' => isset($item->fixed_price) ? (float)$item->fixed_price : (isset($item->price) ? (float)$item->price : 0)
                    );
                    $items_found++;
                }
            }
            // Format 4: Try XPath for any item nodes
            else {
                $items = $xml->xpath('//item');
                if ($items) {
                    foreach ($items as $item) {
                        if (isset($item->id_item) || isset($item->id)) {
                            $all_items[] = array(
                                'id' => (string)(isset($item->id_item) ? $item->id_item : $item->id),
                                'personal_reference' => (string)$item->personal_reference,
                                'title' => (string)$item->title,
                                'quantity' => isset($item->qty) ? (int)$item->qty : (isset($item->quantity) ? (int)$item->quantity : 1),
                                'price' => isset($item->fixed_price) ? (float)$item->fixed_price : (isset($item->price) ? (float)$item->price : 0)
                            );
                            $items_found++;
                        }
                    }
                }
            }
            
            // Log progress
            if ($items_found > 0) {
                error_log('[Delcampe Reconciliation] Fetched ' . $items_found . ' items at offset ' . $starting_item);
            }
            
            // If no items found or less than batch size, we're done
            if ($items_found == 0 || $items_found < $batch_size) {
                break;
            }
            
            // Move to next batch
            $starting_item += $batch_size;
            
            // Safety check
            if (count($all_items) >= $max_items) {
                error_log('[Delcampe Reconciliation] Reached max items limit of ' . $max_items);
                break;
            }
            
        } while ($items_found == $batch_size); // Continue if we got a full batch
        
        error_log('[Delcampe Reconciliation] Total items fetched: ' . count($all_items));
        
        return $all_items;
    }
    
    /**
     * Get all local published products
     * 
     * @return array Product data indexed by ID
     */
    private function get_local_published_products() {
        global $wpdb;
        
        $products = array();
        
        // Get all products with sync status
        $results = $wpdb->get_results("
            SELECT p.ID, 
                   pm1.meta_value as sync_status,
                   pm2.meta_value as delcampe_id,
                   pm3.meta_value as sku,
                   pm4.meta_value as personal_reference
            FROM {$wpdb->posts} p
            LEFT JOIN {$wpdb->postmeta} pm1 ON p.ID = pm1.post_id AND pm1.meta_key = '_delcampe_sync_status'
            LEFT JOIN {$wpdb->postmeta} pm2 ON p.ID = pm2.post_id AND pm2.meta_key = '_delcampe_item_id'
            LEFT JOIN {$wpdb->postmeta} pm3 ON p.ID = pm3.post_id AND pm3.meta_key = '_sku'
            LEFT JOIN {$wpdb->postmeta} pm4 ON p.ID = pm4.post_id AND pm4.meta_key = '_delcampe_personal_reference'
            WHERE p.post_type = 'product'
            AND p.post_status = 'publish'
            AND pm1.meta_value IN ('published', 'active', 'pending')
        ");
        
        foreach ($results as $row) {
            $products[$row->ID] = array(
                'sync_status' => $row->sync_status,
                'delcampe_id' => $row->delcampe_id,
                'sku' => $row->sku,
                'personal_reference' => $row->personal_reference
            );
        }
        
        return $products;
    }
    
    /**
     * Fix missing Delcampe ID
     * 
     * @param int $product_id Product ID
     * @param string $delcampe_id Delcampe Item ID
     * @param array $item_data Additional item data
     */
    private function fix_missing_id($product_id, $delcampe_id, $item_data = array()) {
        // Use tracker to store with verification
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-listing-tracker.php';
        $tracker = Delcampe_Listing_Tracker::get_instance();
        
        $additional_meta = array();
        if (!empty($item_data['personal_reference'])) {
            $additional_meta['_delcampe_personal_reference'] = $item_data['personal_reference'];
        }
        
        $tracker->store_listing_with_verification($product_id, $delcampe_id, $additional_meta);
        
        delcampe_log('Reconciliation: Fixed missing ID for product ' . $product_id . ' - Delcampe ID: ' . $delcampe_id);
    }
    
    /**
     * Mark product as orphaned
     * 
     * @param int $product_id Product ID
     * @param string $delcampe_id Delcampe ID that no longer exists
     */
    private function mark_as_orphaned($product_id, $delcampe_id) {
        update_post_meta($product_id, '_delcampe_sync_status', 'orphaned');
        update_post_meta($product_id, '_delcampe_orphaned_at', current_time('mysql'));
        
        delcampe_log('Reconciliation: Marked product ' . $product_id . ' as orphaned - Delcampe ID ' . $delcampe_id . ' not found');
    }
    
    /**
     * Find product by SKU or personal reference
     * 
     * @param string $reference Reference to search
     * @return int|null Product ID if found
     */
    private function find_product_by_reference($reference) {
        if (empty($reference)) {
            return null;
        }
        
        global $wpdb;
        
        // Try by SKU first
        $product_id = $wpdb->get_var($wpdb->prepare("
            SELECT post_id FROM {$wpdb->postmeta}
            WHERE meta_key = '_sku' AND meta_value = %s
            LIMIT 1
        ", $reference));
        
        if ($product_id) {
            return $product_id;
        }
        
        // Try by personal reference
        $product_id = $wpdb->get_var($wpdb->prepare("
            SELECT post_id FROM {$wpdb->postmeta}
            WHERE meta_key = '_delcampe_personal_reference' AND meta_value = %s
            LIMIT 1
        ", $reference));
        
        return $product_id;
    }
    
    /**
     * Find duplicate listings
     * 
     * @return array Array of duplicate sets
     */
    private function find_duplicate_listings() {
        global $wpdb;
        
        // Find products with same Delcampe ID
        $duplicates = $wpdb->get_results("
            SELECT meta_value as delcampe_id, GROUP_CONCAT(post_id) as product_ids
            FROM {$wpdb->postmeta}
            WHERE meta_key = '_delcampe_item_id'
            AND meta_value != ''
            GROUP BY meta_value
            HAVING COUNT(*) > 1
        ");
        
        $duplicate_sets = array();
        foreach ($duplicates as $dup) {
            $duplicate_sets[] = explode(',', $dup->product_ids);
        }
        
        return $duplicate_sets;
    }
    
    /**
     * Resolve duplicate listings
     * 
     * @param array $product_ids Array of product IDs with same Delcampe ID
     */
    private function resolve_duplicates($product_ids) {
        // Keep the most recently updated one
        $keep_id = null;
        $latest_time = 0;
        
        foreach ($product_ids as $product_id) {
            $last_sync = get_post_meta($product_id, '_delcampe_last_sync', true);
            if ($last_sync > $latest_time) {
                $latest_time = $last_sync;
                $keep_id = $product_id;
            }
        }
        
        // Clear Delcampe data from others
        foreach ($product_ids as $product_id) {
            if ($product_id != $keep_id) {
                delete_post_meta($product_id, '_delcampe_item_id');
                delete_post_meta($product_id, '_delcampe_sync_status');
                
                delcampe_log('Reconciliation: Removed duplicate Delcampe data from product ' . $product_id);
            }
        }
    }
    
    /**
     * Log reconciliation results
     * 
     * @param array $results Results array
     */
    private function log_results($results) {
        $log_entry = 'Reconciliation completed: ' . json_encode($results);
        delcampe_log($log_entry, 'info');
        
        // Also save to option for dashboard display
        update_option('delcampe_last_reconciliation', $results);
    }
    
    /**
     * Send notification if issues found
     * 
     * @param array $results Results array
     */
    private function notify_if_needed($results) {
        $has_issues = ($results['missing_locally'] > 0 || 
                      $results['missing_on_delcampe'] > 0 || 
                      $results['orphaned_marked'] > 0 ||
                      !empty($results['errors']));
        
        if ($has_issues) {
            $message = "Delcampe Reconciliation Report\n";
            $message .= "================================\n\n";
            $message .= "Missing locally: " . $results['missing_locally'] . "\n";
            $message .= "Missing on Delcampe: " . $results['missing_on_delcampe'] . "\n";
            $message .= "Fixed missing IDs: " . $results['fixed_missing_ids'] . "\n";
            $message .= "Orphaned marked: " . $results['orphaned_marked'] . "\n";
            $message .= "Duplicates resolved: " . $results['duplicates_resolved'] . "\n";
            
            if (!empty($results['errors'])) {
                $message .= "\nErrors:\n";
                foreach ($results['errors'] as $error) {
                    $message .= "- " . $error . "\n";
                }
            }
            
            wp_mail(
                get_option('admin_email'),
                'Delcampe Reconciliation Issues Found',
                $message
            );
        }
    }
    
    /**
     * WP-CLI command handler
     * 
     * @param array $args Arguments
     * @param array $assoc_args Associative arguments
     */
    public function cli_reconcile($args, $assoc_args) {
        WP_CLI::line('Starting Delcampe reconciliation...');
        
        $results = $this->run_reconciliation();
        
        WP_CLI::line('');
        WP_CLI::line('Results:');
        WP_CLI::line('--------');
        WP_CLI::line('Delcampe total: ' . (isset($results['delcampe_total']) ? $results['delcampe_total'] : 'N/A'));
        WP_CLI::line('Local total: ' . (isset($results['local_total']) ? $results['local_total'] : 'N/A'));
        WP_CLI::line('Missing locally: ' . $results['missing_locally']);
        WP_CLI::line('Missing on Delcampe: ' . $results['missing_on_delcampe']);
        WP_CLI::line('Fixed missing IDs: ' . $results['fixed_missing_ids']);
        WP_CLI::line('Orphaned marked: ' . $results['orphaned_marked']);
        WP_CLI::line('Duplicates resolved: ' . $results['duplicates_resolved']);
        
        if (!empty($results['errors'])) {
            WP_CLI::line('');
            WP_CLI::error('Errors encountered:');
            foreach ($results['errors'] as $error) {
                WP_CLI::line('- ' . $error);
            }
        }
        
        WP_CLI::success('Reconciliation completed in ' . $results['duration'] . ' seconds');
    }
}

// Initialize
add_action('plugins_loaded', function() {
    Delcampe_Reconciliation::get_instance();
});