<?php
/**
 * Delcampe Shipping Helper
 * 
 * Since Delcampe doesn't provide shipping API, this helps manage shipping on WooCommerce side
 * and provides reminders to update Delcampe manually
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.10.13.8
 */

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

class Delcampe_Shipping_Helper {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Add shipping actions to order actions
        add_filter('woocommerce_order_actions', array($this, 'add_shipping_actions'), 10, 2);
        add_action('woocommerce_order_action_delcampe_mark_shipped', array($this, 'handle_mark_shipped'));
        
        // Add metabox for Delcampe orders
        add_action('add_meta_boxes', array($this, 'add_shipping_metabox'), 10, 2);
    }
    
    /**
     * Add shipping actions to order actions dropdown
     */
    public function add_shipping_actions($actions, $order) {
        // Only for Delcampe orders
        $delcampe_id = $order->get_meta('_delcampe_order_id');
        if (!$delcampe_id) {
            return $actions;
        }
        
        // Only for processing orders
        if ($order->get_status() === 'processing') {
            $actions['delcampe_mark_shipped'] = __('Mark as Shipped (Delcampe)', 'wc-delcampe-integration');
        }
        
        return $actions;
    }
    
    /**
     * Handle mark as shipped action
     */
    public function handle_mark_shipped($order) {
        $delcampe_id = $order->get_meta('_delcampe_order_id');
        
        if (!$delcampe_id) {
            return;
        }
        
        // Update order status to completed
        $order->update_status('completed', __('Order marked as shipped', 'wc-delcampe-integration'));
        
        // Add tracking info if available
        $tracking_number = isset($_POST['tracking_number']) ? sanitize_text_field($_POST['tracking_number']) : '';
        $carrier = isset($_POST['carrier']) ? sanitize_text_field($_POST['carrier']) : '';
        
        if ($tracking_number) {
            $order->update_meta_data('_tracking_number', $tracking_number);
            $order->update_meta_data('_tracking_carrier', $carrier);
        }
        
        // Add order note with Delcampe reminder
        $delcampe_url = 'https://www.delcampe.net/en_GB/collectibles/seller/items/sold';
        $note = sprintf(
            __('Order marked as shipped in WooCommerce.', 'wc-delcampe-integration') . "\n\n" .
            __('⚠️ IMPORTANT: You must also mark this as shipped on Delcampe:', 'wc-delcampe-integration') . "\n" .
            __('1. Go to: %s', 'wc-delcampe-integration') . "\n" .
            __('2. Find item #%s', 'wc-delcampe-integration') . "\n" .
            __('3. Click "Mark as Sent"', 'wc-delcampe-integration') . "\n\n" .
            __('Tracking: %s (%s)', 'wc-delcampe-integration'),
            $delcampe_url,
            $delcampe_id,
            $tracking_number ?: 'Not provided',
            $carrier ?: 'N/A'
        );
        
        $order->add_order_note($note, false, true); // Customer note
        
        // Update our database
        global $wpdb;
        $table = $wpdb->prefix . 'delcampe_orders';
        $wpdb->update(
            $table,
            array(
                'shipping_status' => 'shipped',
                'last_sync' => current_time('mysql')
            ),
            array('wc_order_id' => $order->get_id()),
            array('%s', '%s'),
            array('%d')
        );
        
        // Send email to customer if enabled
        if (get_option('delcampe_send_shipping_email', 'yes') === 'yes') {
            $this->send_shipping_notification($order, $tracking_number, $carrier);
        }
        
        $order->save();
    }
    
    /**
     * Add shipping metabox for Delcampe orders
     */
    public function add_shipping_metabox($post_type, $post_or_order) {
        if ($post_type !== 'shop_order' && $post_type !== 'woocommerce_page_wc-orders') {
            return;
        }
        
        $order = ($post_or_order instanceof WC_Order) ? $post_or_order : wc_get_order($post_or_order->ID);
        if (!$order) {
            return;
        }
        
        $delcampe_id = $order->get_meta('_delcampe_order_id');
        if (!$delcampe_id) {
            return;
        }
        
        $screen = class_exists('\Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController') 
            && wc_get_container()->get(\Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController::class)->custom_orders_table_usage_is_enabled()
            ? wc_get_page_screen_id('shop-order')
            : 'shop_order';
            
        add_meta_box(
            'delcampe_shipping_info',
            __('Delcampe Shipping', 'wc-delcampe-integration'),
            array($this, 'render_shipping_metabox'),
            $screen,
            'side',
            'high'
        );
    }
    
    /**
     * Render shipping metabox
     */
    public function render_shipping_metabox($post_or_order) {
        $order = ($post_or_order instanceof WC_Order) ? $post_or_order : wc_get_order($post_or_order->ID);
        $delcampe_id = $order->get_meta('_delcampe_order_id');
        $tracking_number = $order->get_meta('_tracking_number');
        $carrier = $order->get_meta('_tracking_carrier');
        
        ?>
        <div class="delcampe-shipping-info">
            <p><strong><?php _e('Delcampe Item:', 'wc-delcampe-integration'); ?></strong> #<?php echo esc_html($delcampe_id); ?></p>
            
            <?php if ($order->get_status() === 'processing'): ?>
                <div class="shipping-form">
                    <p>
                        <label><?php _e('Tracking Number:', 'wc-delcampe-integration'); ?></label>
                        <input type="text" name="tracking_number" value="<?php echo esc_attr($tracking_number); ?>" style="width: 100%;" />
                    </p>
                    <p>
                        <label><?php _e('Carrier:', 'wc-delcampe-integration'); ?></label>
                        <select name="carrier" style="width: 100%;">
                            <option value=""><?php _e('Select carrier...', 'wc-delcampe-integration'); ?></option>
                            <option value="canada-post" <?php selected($carrier, 'canada-post'); ?>>Canada Post</option>
                            <option value="usps" <?php selected($carrier, 'usps'); ?>>USPS</option>
                            <option value="ups" <?php selected($carrier, 'ups'); ?>>UPS</option>
                            <option value="fedex" <?php selected($carrier, 'fedex'); ?>>FedEx</option>
                            <option value="dhl" <?php selected($carrier, 'dhl'); ?>>DHL</option>
                            <option value="other" <?php selected($carrier, 'other'); ?>>Other</option>
                        </select>
                    </p>
                    <p class="delcampe-shipping-notice" style="background: #fff3cd; padding: 10px; border-left: 4px solid #ffc107;">
                        <strong>⚠️ <?php _e('Remember:', 'wc-delcampe-integration'); ?></strong><br>
                        <?php _e('After marking as shipped here, you MUST also mark it as sent on Delcampe website.', 'wc-delcampe-integration'); ?><br>
                        <a href="https://www.delcampe.net/en_GB/collectibles/seller/items/sold" target="_blank">
                            <?php _e('Go to Delcampe Sold Items →', 'wc-delcampe-integration'); ?>
                        </a>
                    </p>
                </div>
            <?php elseif ($order->get_status() === 'completed'): ?>
                <p><strong><?php _e('Shipped', 'wc-delcampe-integration'); ?></strong></p>
                <?php if ($tracking_number): ?>
                    <p><?php _e('Tracking:', 'wc-delcampe-integration'); ?> <?php echo esc_html($tracking_number); ?></p>
                    <p><?php _e('Carrier:', 'wc-delcampe-integration'); ?> <?php echo esc_html($carrier ?: 'N/A'); ?></p>
                <?php endif; ?>
            <?php endif; ?>
        </div>
        
        <style>
            .delcampe-shipping-info input,
            .delcampe-shipping-info select {
                margin-top: 5px;
            }
        </style>
        <?php
    }
    
    /**
     * Send shipping notification email
     */
    private function send_shipping_notification($order, $tracking_number, $carrier) {
        $to = $order->get_billing_email();
        $subject = sprintf(__('Your order #%s has been shipped', 'wc-delcampe-integration'), $order->get_id());
        
        $message = sprintf(
            __('Hello %s,', 'wc-delcampe-integration') . "\n\n" .
            __('Your order #%s has been shipped.', 'wc-delcampe-integration') . "\n\n",
            $order->get_billing_first_name(),
            $order->get_id()
        );
        
        if ($tracking_number) {
            $message .= sprintf(
                __('Tracking Information:', 'wc-delcampe-integration') . "\n" .
                __('Tracking Number: %s', 'wc-delcampe-integration') . "\n" .
                __('Carrier: %s', 'wc-delcampe-integration') . "\n\n",
                $tracking_number,
                $carrier ?: 'N/A'
            );
        }
        
        $message .= __('Thank you for your order!', 'wc-delcampe-integration');
        
        wp_mail($to, $subject, $message);
    }
}

// Initialize
Delcampe_Shipping_Helper::get_instance();