<?php
/**
 * Delcampe Status Manager
 * 
 * Provides unified status management across all components
 * Ensures consistent status transitions and prevents conflicts
 * 
 * @package WC_Delcampe_Integration
 * @since 1.10.28.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Status_Manager {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Valid status values and their meanings
     */
    const STATUS_DRAFT = 'draft';           // Not yet submitted
    const STATUS_PENDING = 'pending';       // Waiting to be processed
    const STATUS_QUEUED = 'queued';         // In batch or sync queue
    const STATUS_PROCESSING = 'processing'; // Currently being processed
    const STATUS_VERIFIED = 'verified';     // Successfully created on Delcampe
    const STATUS_PUBLISHED = 'published';   // Active on Delcampe
    const STATUS_ACTIVE = 'active';         // Alias for published
    const STATUS_FAILED = 'failed';         // Failed to create/update
    const STATUS_ENDED = 'ended';           // Listing ended on Delcampe
    const STATUS_SOLD = 'sold';             // Item sold on Delcampe
    const STATUS_ARCHIVED = 'archived';     // Archived listing
    
    /**
     * Status transition rules
     * Defines valid transitions from one status to another
     */
    private $allowed_transitions = array(
        self::STATUS_DRAFT => array(self::STATUS_PENDING, self::STATUS_QUEUED),
        self::STATUS_PENDING => array(self::STATUS_QUEUED, self::STATUS_PROCESSING, self::STATUS_FAILED),
        self::STATUS_QUEUED => array(self::STATUS_PROCESSING, self::STATUS_FAILED),
        self::STATUS_PROCESSING => array(self::STATUS_VERIFIED, self::STATUS_PUBLISHED, self::STATUS_FAILED),
        self::STATUS_VERIFIED => array(self::STATUS_PUBLISHED, self::STATUS_ACTIVE),
        self::STATUS_PUBLISHED => array(self::STATUS_ACTIVE, self::STATUS_ENDED, self::STATUS_SOLD),
        self::STATUS_ACTIVE => array(self::STATUS_ENDED, self::STATUS_SOLD),
        self::STATUS_FAILED => array(self::STATUS_PENDING, self::STATUS_QUEUED),
        self::STATUS_ENDED => array(self::STATUS_ARCHIVED, self::STATUS_PENDING),
        self::STATUS_SOLD => array(self::STATUS_ARCHIVED),
        self::STATUS_ARCHIVED => array(self::STATUS_PENDING)
    );
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Hook into status change events
        add_action('delcampe_status_changed', array($this, 'log_status_change'), 10, 4);
    }
    
    /**
     * Get the current status of a product across all systems
     * 
     * @param int $product_id Product ID
     * @return array Status information from all sources
     */
    public function get_product_status($product_id) {
        global $wpdb;
        
        $status = array(
            'meta_status' => get_post_meta($product_id, '_delcampe_sync_status', true),
            'listing_status' => null,
            'batch_status' => null,
            'sync_status' => null,
            'consolidated' => null
        );
        
        // Check listings table
        $listings_table = $wpdb->prefix . 'delcampe_listings';
        $listing = $wpdb->get_row($wpdb->prepare(
            "SELECT status FROM {$listings_table} WHERE product_id = %d ORDER BY date_created DESC LIMIT 1",
            $product_id
        ));
        if ($listing) {
            $status['listing_status'] = $listing->status;
        }
        
        // Check batch queue
        $batch_table = $wpdb->prefix . 'delcampe_batch_queue';
        $batch = $wpdb->get_row($wpdb->prepare(
            "SELECT status FROM {$batch_table} 
            WHERE JSON_CONTAINS(product_ids, %s) 
            AND status IN ('pending', 'queued', 'processing', 'retry')
            ORDER BY created_at DESC LIMIT 1",
            json_encode($product_id)
        ));
        if ($batch) {
            $status['batch_status'] = $batch->status;
        }
        
        // Check sync queue
        $sync_table = $wpdb->prefix . 'delcampe_sync_queue';
        $sync = $wpdb->get_row($wpdb->prepare(
            "SELECT status FROM {$sync_table} WHERE product_id = %d ORDER BY added_date DESC LIMIT 1",
            $product_id
        ));
        if ($sync) {
            $status['sync_status'] = $sync->status;
        }
        
        // Determine consolidated status (priority order)
        if ($status['batch_status'] === 'processing' || $status['sync_status'] === 'processing') {
            $status['consolidated'] = self::STATUS_PROCESSING;
        } elseif ($status['batch_status'] === 'queued' || $status['sync_status'] === 'pending') {
            $status['consolidated'] = self::STATUS_QUEUED;
        } elseif ($status['listing_status'] === 'published' || $status['listing_status'] === 'active') {
            $status['consolidated'] = self::STATUS_PUBLISHED;
        } elseif ($status['listing_status'] === 'verified') {
            $status['consolidated'] = self::STATUS_VERIFIED;
        } elseif ($status['meta_status']) {
            $status['consolidated'] = $status['meta_status'];
        } else {
            $status['consolidated'] = self::STATUS_DRAFT;
        }
        
        return $status;
    }
    
    /**
     * Update product status across all systems atomically
     * 
     * @param int $product_id Product ID
     * @param string $new_status New status to set
     * @param array $context Additional context (batch_id, sync_id, etc.)
     * @return bool|WP_Error True on success, WP_Error on failure
     */
    public function update_product_status($product_id, $new_status, $context = array()) {
        global $wpdb;
        
        // Validate status
        if (!$this->is_valid_status($new_status)) {
            return new WP_Error('invalid_status', 'Invalid status: ' . $new_status);
        }
        
        // Get current status
        $current = $this->get_product_status($product_id);
        $current_status = $current['consolidated'];
        
        // Check if transition is allowed
        if (!$this->can_transition($current_status, $new_status)) {
            return new WP_Error(
                'invalid_transition',
                sprintf('Cannot transition from %s to %s', $current_status, $new_status)
            );
        }
        
        // Start transaction for atomic update
        $wpdb->query('START TRANSACTION');
        
        try {
            // Update product meta
            update_post_meta($product_id, '_delcampe_sync_status', $new_status);
            
            // Update listings table if exists
            if (!empty($context['listing_id'])) {
                $listings_table = $wpdb->prefix . 'delcampe_listings';
                $wpdb->update(
                    $listings_table,
                    array('status' => $new_status, 'date_updated' => current_time('mysql')),
                    array('id' => $context['listing_id']),
                    array('%s', '%s'),
                    array('%d')
                );
            }
            
            // Remove from sync queue if completed
            if (in_array($new_status, array(self::STATUS_PUBLISHED, self::STATUS_ACTIVE, self::STATUS_FAILED))) {
                $sync_table = $wpdb->prefix . 'delcampe_sync_queue';
                $wpdb->delete($sync_table, array('product_id' => $product_id), array('%d'));
            }
            
            // Commit transaction
            $wpdb->query('COMMIT');
            
            // Fire status change action
            do_action('delcampe_status_changed', $product_id, $current_status, $new_status, $context);
            
            return true;
            
        } catch (Exception $e) {
            // Rollback on error
            $wpdb->query('ROLLBACK');
            return new WP_Error('update_failed', $e->getMessage());
        }
    }
    
    /**
     * Check if a status value is valid
     * 
     * @param string $status Status to check
     * @return bool
     */
    public function is_valid_status($status) {
        $valid_statuses = array(
            self::STATUS_DRAFT,
            self::STATUS_PENDING,
            self::STATUS_QUEUED,
            self::STATUS_PROCESSING,
            self::STATUS_VERIFIED,
            self::STATUS_PUBLISHED,
            self::STATUS_ACTIVE,
            self::STATUS_FAILED,
            self::STATUS_ENDED,
            self::STATUS_SOLD,
            self::STATUS_ARCHIVED
        );
        
        return in_array($status, $valid_statuses, true);
    }
    
    /**
     * Check if a status transition is allowed
     * 
     * @param string $from Current status
     * @param string $to New status
     * @return bool
     */
    public function can_transition($from, $to) {
        // Same status is always allowed
        if ($from === $to) {
            return true;
        }
        
        // Check transition rules
        if (isset($this->allowed_transitions[$from])) {
            return in_array($to, $this->allowed_transitions[$from], true);
        }
        
        // If no current status (new item), allow initial states
        if (empty($from) || $from === self::STATUS_DRAFT) {
            return in_array($to, array(self::STATUS_PENDING, self::STATUS_QUEUED), true);
        }
        
        return false;
    }
    
    /**
     * Check if a product is currently being processed
     * 
     * @param int $product_id Product ID
     * @return bool
     */
    public function is_product_in_flight($product_id) {
        $status = $this->get_product_status($product_id);
        $in_flight_statuses = array(
            self::STATUS_PENDING,
            self::STATUS_QUEUED,
            self::STATUS_PROCESSING,
            self::STATUS_VERIFIED
        );
        
        return in_array($status['consolidated'], $in_flight_statuses, true);
    }
    
    /**
     * Check if a product is published on Delcampe
     * 
     * @param int $product_id Product ID
     * @return bool
     */
    public function is_product_published($product_id) {
        $status = $this->get_product_status($product_id);
        $published_statuses = array(
            self::STATUS_PUBLISHED,
            self::STATUS_ACTIVE,
            self::STATUS_ENDED,
            self::STATUS_SOLD
        );
        
        return in_array($status['consolidated'], $published_statuses, true);
    }
    
    /**
     * Clean up conflicting statuses
     * Resolves conflicts when multiple systems have different statuses
     * 
     * @param int $product_id Product ID
     * @return bool
     */
    public function reconcile_status($product_id) {
        $status = $this->get_product_status($product_id);
        $consolidated = $status['consolidated'];
        
        // Update meta to match consolidated status
        update_post_meta($product_id, '_delcampe_sync_status', $consolidated);
        
        // Log reconciliation
        if (function_exists('delcampe_log')) {
            delcampe_log(sprintf(
                '[Status Manager] Reconciled status for product %d: %s',
                $product_id,
                $consolidated
            ));
        }
        
        return true;
    }
    
    /**
     * Log status changes for audit trail
     * 
     * @param int $product_id Product ID
     * @param string $from Previous status
     * @param string $to New status
     * @param array $context Additional context
     */
    public function log_status_change($product_id, $from, $to, $context = array()) {
        if (function_exists('delcampe_log')) {
            delcampe_log(sprintf(
                '[Status Change] Product %d: %s → %s (Context: %s)',
                $product_id,
                $from ?: 'none',
                $to,
                json_encode($context)
            ));
        }
    }
    
    /**
     * Get human-readable status label
     * 
     * @param string $status Status code
     * @return string Human-readable label
     */
    public function get_status_label($status) {
        $labels = array(
            self::STATUS_DRAFT => __('Draft', 'wc-delcampe-integration'),
            self::STATUS_PENDING => __('Pending', 'wc-delcampe-integration'),
            self::STATUS_QUEUED => __('Queued', 'wc-delcampe-integration'),
            self::STATUS_PROCESSING => __('Processing', 'wc-delcampe-integration'),
            self::STATUS_VERIFIED => __('Verified', 'wc-delcampe-integration'),
            self::STATUS_PUBLISHED => __('Published', 'wc-delcampe-integration'),
            self::STATUS_ACTIVE => __('Active', 'wc-delcampe-integration'),
            self::STATUS_FAILED => __('Failed', 'wc-delcampe-integration'),
            self::STATUS_ENDED => __('Ended', 'wc-delcampe-integration'),
            self::STATUS_SOLD => __('Sold', 'wc-delcampe-integration'),
            self::STATUS_ARCHIVED => __('Archived', 'wc-delcampe-integration')
        );
        
        return isset($labels[$status]) ? $labels[$status] : $status;
    }
}