<?php
/**
 * Delcampe Sync Dashboard Widget
 * 
 * Provides admin dashboard visibility into sync status and reconciliation
 * 
 * @package     WooCommerce_Delcampe_Integration
 * @subpackage  Admin
 * @since       1.10.18.13
 * @version     1.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Sync_Dashboard {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Add dashboard widget
        add_action('wp_dashboard_setup', array($this, 'add_dashboard_widget'));
        
        // AJAX handlers
        add_action('wp_ajax_delcampe_run_reconciliation', array($this, 'ajax_run_reconciliation'));
        add_action('wp_ajax_delcampe_check_audit_log', array($this, 'ajax_check_audit_log'));
    }
    
    /**
     * Add dashboard widget
     */
    public function add_dashboard_widget() {
        wp_add_dashboard_widget(
            'delcampe_sync_status',
            __('Delcampe Sync Status', 'wc-delcampe-integration'),
            array($this, 'render_widget')
        );
    }
    
    /**
     * Render dashboard widget
     */
    public function render_widget() {
        global $wpdb;
        
        // Get quick stats
        $stats = $this->get_sync_stats();
        $last_reconciliation = get_option('delcampe_last_reconciliation', array());
        
        ?>
        <div class="delcampe-sync-widget">
            <div class="sync-stats">
                <h4><?php _e('Current Status', 'wc-delcampe-integration'); ?></h4>
                <table class="widefat">
                    <tr>
                        <td><?php _e('Published Products', 'wc-delcampe-integration'); ?>:</td>
                        <td><strong><?php echo $stats['published']; ?></strong></td>
                    </tr>
                    <tr>
                        <td><?php _e('With Delcampe ID', 'wc-delcampe-integration'); ?>:</td>
                        <td><strong><?php echo $stats['with_id']; ?></strong></td>
                    </tr>
                    <tr>
                        <td><?php _e('Missing IDs', 'wc-delcampe-integration'); ?>:</td>
                        <td>
                            <strong class="<?php echo ($stats['missing_id'] > 0) ? 'error' : 'success'; ?>">
                                <?php echo $stats['missing_id']; ?>
                            </strong>
                        </td>
                    </tr>
                    <tr>
                        <td><?php _e('Pending Sync', 'wc-delcampe-integration'); ?>:</td>
                        <td><strong><?php echo $stats['pending']; ?></strong></td>
                    </tr>
                    <tr>
                        <td><?php _e('Error Status', 'wc-delcampe-integration'); ?>:</td>
                        <td>
                            <strong class="<?php echo ($stats['errors'] > 0) ? 'error' : 'success'; ?>">
                                <?php echo $stats['errors']; ?>
                            </strong>
                        </td>
                    </tr>
                </table>
            </div>
            
            <?php if (!empty($last_reconciliation)): ?>
            <div class="last-reconciliation">
                <h4><?php _e('Last Reconciliation', 'wc-delcampe-integration'); ?></h4>
                <p>
                    <?php 
                    $time_ago = human_time_diff(strtotime($last_reconciliation['started_at']), current_time('timestamp'));
                    echo sprintf(__('Run %s ago', 'wc-delcampe-integration'), $time_ago);
                    ?>
                </p>
                <ul>
                    <?php if ($last_reconciliation['fixed_missing_ids'] > 0): ?>
                    <li class="success">
                        <?php echo sprintf(__('Fixed %d missing IDs', 'wc-delcampe-integration'), $last_reconciliation['fixed_missing_ids']); ?>
                    </li>
                    <?php endif; ?>
                    
                    <?php if ($last_reconciliation['orphaned_marked'] > 0): ?>
                    <li class="warning">
                        <?php echo sprintf(__('Marked %d as orphaned', 'wc-delcampe-integration'), $last_reconciliation['orphaned_marked']); ?>
                    </li>
                    <?php endif; ?>
                    
                    <?php if ($last_reconciliation['duplicates_resolved'] > 0): ?>
                    <li class="success">
                        <?php echo sprintf(__('Resolved %d duplicates', 'wc-delcampe-integration'), $last_reconciliation['duplicates_resolved']); ?>
                    </li>
                    <?php endif; ?>
                    
                    <?php if (!empty($last_reconciliation['errors'])): ?>
                    <li class="error">
                        <?php echo sprintf(__('%d errors encountered', 'wc-delcampe-integration'), count($last_reconciliation['errors'])); ?>
                    </li>
                    <?php endif; ?>
                </ul>
            </div>
            <?php endif; ?>
            
            <div class="sync-actions">
                <h4><?php _e('Actions', 'wc-delcampe-integration'); ?></h4>
                <p>
                    <button type="button" class="button button-primary" id="run-reconciliation">
                        <?php _e('Run Reconciliation Now', 'wc-delcampe-integration'); ?>
                    </button>
                    <button type="button" class="button" id="check-audit-log">
                        <?php _e('Check Audit Log', 'wc-delcampe-integration'); ?>
                    </button>
                </p>
                <p>
                    <a href="<?php echo admin_url('admin.php?page=delcampe'); ?>" class="button">
                        <?php _e('Go to Delcampe Settings', 'wc-delcampe-integration'); ?>
                    </a>
                    <a href="<?php echo admin_url('admin.php?page=delcampe-listings'); ?>" class="button">
                        <?php _e('View Listings', 'wc-delcampe-integration'); ?>
                    </a>
                </p>
            </div>
            
            <div id="reconciliation-status" style="display:none; margin-top: 10px;">
                <div class="spinner is-active" style="float: none;"></div>
                <p><?php _e('Running reconciliation...', 'wc-delcampe-integration'); ?></p>
            </div>
            
            <div id="audit-log-status" style="display:none; margin-top: 10px;">
                <h4><?php _e('Recent Audit Log Entries', 'wc-delcampe-integration'); ?></h4>
                <div id="audit-log-content"></div>
            </div>
        </div>
        
        <style>
        .delcampe-sync-widget .sync-stats table {
            margin: 10px 0;
        }
        .delcampe-sync-widget .sync-stats td:first-child {
            width: 60%;
        }
        .delcampe-sync-widget .error {
            color: #dc3232;
        }
        .delcampe-sync-widget .success {
            color: #46b450;
        }
        .delcampe-sync-widget .warning {
            color: #f56e28;
        }
        .delcampe-sync-widget h4 {
            margin-top: 15px;
            margin-bottom: 5px;
        }
        .delcampe-sync-widget .last-reconciliation ul {
            margin: 5px 0 5px 20px;
        }
        .delcampe-sync-widget .sync-actions .button {
            margin-right: 5px;
            margin-bottom: 5px;
        }
        </style>
        
        <script>
        jQuery(document).ready(function($) {
            // Run reconciliation
            $('#run-reconciliation').on('click', function() {
                var $button = $(this);
                var $status = $('#reconciliation-status');
                
                $button.prop('disabled', true);
                $status.show();
                
                $.post(ajaxurl, {
                    action: 'delcampe_run_reconciliation',
                    _ajax_nonce: '<?php echo wp_create_nonce('delcampe_reconciliation'); ?>'
                }, function(response) {
                    if (response.success) {
                        $status.html('<p class="success"><?php _e('Reconciliation completed!', 'wc-delcampe-integration'); ?></p>');
                        
                        // Show results
                        var results = response.data;
                        var html = '<ul>';
                        if (results.fixed_missing_ids > 0) {
                            html += '<li>Fixed ' + results.fixed_missing_ids + ' missing IDs</li>';
                        }
                        if (results.orphaned_marked > 0) {
                            html += '<li>Marked ' + results.orphaned_marked + ' as orphaned</li>';
                        }
                        if (results.duplicates_resolved > 0) {
                            html += '<li>Resolved ' + results.duplicates_resolved + ' duplicates</li>';
                        }
                        html += '</ul>';
                        
                        $status.append(html);
                        
                        // Reload widget after 3 seconds
                        setTimeout(function() {
                            location.reload();
                        }, 3000);
                    } else {
                        $status.html('<p class="error"><?php _e('Reconciliation failed:', 'wc-delcampe-integration'); ?> ' + response.data + '</p>');
                    }
                    
                    $button.prop('disabled', false);
                });
            });
            
            // Check audit log
            $('#check-audit-log').on('click', function() {
                var $button = $(this);
                var $status = $('#audit-log-status');
                var $content = $('#audit-log-content');
                
                $button.prop('disabled', true);
                $status.show();
                $content.html('<div class="spinner is-active" style="float: none;"></div>');
                
                $.post(ajaxurl, {
                    action: 'delcampe_check_audit_log',
                    _ajax_nonce: '<?php echo wp_create_nonce('delcampe_audit'); ?>'
                }, function(response) {
                    if (response.success) {
                        var entries = response.data;
                        var html = '<table class="widefat"><thead><tr><th>Time</th><th>Action</th><th>Product</th><th>Result</th></tr></thead><tbody>';
                        
                        for (var i = 0; i < entries.length && i < 10; i++) {
                            var entry = entries[i];
                            html += '<tr>';
                            html += '<td>' + entry.timestamp + '</td>';
                            html += '<td>' + entry.action + '</td>';
                            html += '<td>' + entry.sku + '</td>';
                            html += '<td>' + (entry.delcampe_id ? 'ID: ' + entry.delcampe_id : 'N/A') + '</td>';
                            html += '</tr>';
                        }
                        
                        html += '</tbody></table>';
                        $content.html(html);
                    } else {
                        $content.html('<p class="error">Failed to load audit log</p>');
                    }
                    
                    $button.prop('disabled', false);
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * Get sync statistics
     * 
     * @return array Statistics
     */
    private function get_sync_stats() {
        global $wpdb;
        
        $stats = array();
        
        // Published products
        $stats['published'] = $wpdb->get_var("
            SELECT COUNT(DISTINCT post_id) 
            FROM {$wpdb->postmeta} 
            WHERE meta_key = '_delcampe_sync_status' 
            AND meta_value IN ('published', 'active')
        ");
        
        // With Delcampe ID
        $stats['with_id'] = $wpdb->get_var("
            SELECT COUNT(DISTINCT post_id) 
            FROM {$wpdb->postmeta} 
            WHERE meta_key = '_delcampe_item_id' 
            AND meta_value != ''
        ");
        
        // Missing IDs (published but no ID)
        $stats['missing_id'] = $wpdb->get_var("
            SELECT COUNT(DISTINCT pm1.post_id)
            FROM {$wpdb->postmeta} pm1
            LEFT JOIN {$wpdb->postmeta} pm2 ON pm1.post_id = pm2.post_id AND pm2.meta_key = '_delcampe_item_id'
            WHERE pm1.meta_key = '_delcampe_sync_status'
            AND pm1.meta_value IN ('published', 'active')
            AND (pm2.meta_value IS NULL OR pm2.meta_value = '')
        ");
        
        // Pending
        $stats['pending'] = $wpdb->get_var("
            SELECT COUNT(DISTINCT post_id) 
            FROM {$wpdb->postmeta} 
            WHERE meta_key = '_delcampe_sync_status' 
            AND meta_value = 'pending'
        ");
        
        // Errors
        $stats['errors'] = $wpdb->get_var("
            SELECT COUNT(DISTINCT post_id) 
            FROM {$wpdb->postmeta} 
            WHERE meta_key = '_delcampe_sync_status' 
            AND meta_value = 'error'
        ");
        
        return $stats;
    }
    
    /**
     * AJAX handler for running reconciliation
     */
    public function ajax_run_reconciliation() {
        check_ajax_referer('delcampe_reconciliation');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Permission denied', 'wc-delcampe-integration'));
        }
        
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-reconciliation.php';
        $reconciliation = Delcampe_Reconciliation::get_instance();
        
        $results = $reconciliation->run_reconciliation();
        
        wp_send_json_success($results);
    }
    
    /**
     * AJAX handler for checking audit log
     */
    public function ajax_check_audit_log() {
        check_ajax_referer('delcampe_audit');
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error(__('Permission denied', 'wc-delcampe-integration'));
        }
        
        // Read today's audit log
        $log_dir = WP_CONTENT_DIR . '/delcampe-sync-logs';
        $log_file = $log_dir . '/api-audit-' . date('Y-m-d') . '.jsonl';
        
        $entries = array();
        
        if (file_exists($log_file)) {
            $lines = file($log_file, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
            
            // Get last 10 entries
            $lines = array_slice($lines, -10);
            
            foreach ($lines as $line) {
                $entry = json_decode($line, true);
                if ($entry) {
                    $entries[] = $entry;
                }
            }
            
            // Reverse to show newest first
            $entries = array_reverse($entries);
        }
        
        wp_send_json_success($entries);
    }
}

// Initialize
add_action('plugins_loaded', function() {
    Delcampe_Sync_Dashboard::get_instance();
});