<?php
/**
 * Delcampe Sync Handler
 * Version: 1.2.2.0
 *
 * Handles the synchronization queue and processing of product listings
 * Manages batch processing, error handling, and retry logic
 * 
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Sync
 * @since 1.2.0.0
 * @version 1.2.2.0
 * 
 * Changelog:
 * 1.2.2.0 - Fixed database error by handling personal_reference field properly
 *         - Added check for column existence before inserting personal_reference
 *         - Improved error handling for database operations
 *         - Updated version number and added comprehensive comments
 * 1.2.1.0 - Fixed Import Existing Listings button functionality
 *         - Enhanced debug logging with custom log file at wp-content/uploads/delcampe-debug.log
 *         - Added JSON response headers to ensure proper AJAX responses
 *         - Improved error handling and user feedback
 *         - Added admin notice to show debug log location
 * 1.2.0.6 - Added custom debug logging to wp-content/uploads/delcampe-debug.log
 *         - Added admin notices for debugging
 * 1.2.0.5 - Added debugging for import functionality troubleshooting
 *         - Added error logging at multiple stages of the import process
 * 1.2.0.4 - Updated version number and added comprehensive comments
 *         - Improved code documentation for clarity
 *         - Added detailed comments explaining import functionality
 * 1.2.0.3 - Added integration with listings table
 *         - Now creates listings table records when syncing
 *         - Added import function for existing synced items
 * 1.2.0.2 - Fixed undefined array key warnings in ajax_process_queue
 * 1.2.0.1 - Added AJAX handlers for queue status and clear queue functions
 * 1.2.0.0 - Initial implementation of sync handler
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Class Delcampe_Sync_Handler
 * 
 * Manages the synchronization of products to Delcampe
 * 
 * @since 1.2.0.0
 * @version 1.2.2.0
 */
class Delcampe_Sync_Handler {
    
    /**
     * Singleton instance
     * @var Delcampe_Sync_Handler|null
     * @since 1.2.0.0
     */
    private static $instance = null;
    
    /**
     * Sync queue option name
     * @var string
     * @since 1.2.0.0
     */
    const SYNC_QUEUE_OPTION = 'delcampe_sync_queue';
    
    /**
     * Sync lock transient name
     * @var string
     * @since 1.2.0.0
     */
    const SYNC_LOCK_TRANSIENT = 'delcampe_sync_lock';
    
    /**
     * Max items to process per batch
     * Set to 0 to process all items in queue
     * @var int
     * @since 1.2.0.0
     * @version 1.9.0.3
     */
    const BATCH_SIZE = 0; // Deprecated in favor of option 'delcampe_batch_size'

    /**
     * Resolve current batch size from settings with sane defaults
     *
     * @return int
     */
    private function get_batch_size() {
        // Process entire queue regardless of source. Daily new-listing limits are enforced elsewhere.
        return PHP_INT_MAX;
    }
    
    /**
     * Custom debug log file
     * @var string
     * @since 1.2.1.0
     */
    private $debug_log_file;
    
    /**
     * Get singleton instance
     * 
     * @since 1.2.0.0
     * @return Delcampe_Sync_Handler
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     * 
     * @since 1.2.0.0
     * @version 1.2.2.0 - Enhanced debug logging
     */
    private function __construct() {
        // Set up custom debug log file in dedicated logs directory
        if (defined('DELCAMPE_DEBUG_LOG_FILE')) {
            $this->debug_log_file = DELCAMPE_DEBUG_LOG_FILE;
            // Ensure log directory exists
            $log_dir = dirname($this->debug_log_file);
            if (!file_exists($log_dir)) {
                wp_mkdir_p($log_dir);
            }
        } else {
            // Fallback to uploads directory for backward compatibility
            $upload_dir = wp_upload_dir();
            $this->debug_log_file = $upload_dir['basedir'] . '/delcampe-debug.log';
        }
        
        // Register custom cron schedules EARLY so scheduling below can find them
        add_filter('cron_schedules', array($this, 'add_cron_interval'));

        // Hook into queue processing (both WP-Cron and Action Scheduler target the same hook)
        add_action('delcampe_process_sync_queue', array($this, 'process_queue'));
        
        // v1.10.35.17: Disabled automatic sync queue processing
        // Ensure at least one background trigger is scheduled (WP‑Cron fallback) - DISABLED
        /*
        if (!wp_next_scheduled('delcampe_process_sync_queue')) {
            $schedules = wp_get_schedules();
            if (isset($schedules['delcampe_sync_interval'])) {
                if (wp_schedule_event(time() + 60, 'delcampe_sync_interval', 'delcampe_process_sync_queue')) {
                    $this->debug_log('Cron job scheduled successfully (delcampe_sync_interval)');
                } else {
                    $this->debug_log('Failed to schedule cron job (delcampe_sync_interval)');
                }
            } else {
                // Fallback: schedule a single immediate run; future runs will be nudged by events
                if (wp_schedule_single_event(time() + 30, 'delcampe_process_sync_queue')) {
                    $this->debug_log('Scheduled single sync queue run (fallback)');
                } else {
                    $this->debug_log('Failed to schedule single sync queue run (fallback)');
                }
            }
        }
        */

        // Also register Action Scheduler group (no-op if library not present)
        if (function_exists('as_enqueue_async_action')) {
            $this->debug_log('[Init] Action Scheduler detected; queue runs will use async actions when possible');
        }
        
        // Hook for automatic status checking (v1.10.8.0)
        add_action('delcampe_check_processing_status', array($this, 'check_processing_listings'));
        
        // v1.10.35.17: Disabled automatic status checking - causes unnecessary API calls
        // Schedule automatic status checking every 2 minutes for faster updates (v1.10.10.4) - DISABLED
        /*
        if (!wp_next_scheduled('delcampe_check_processing_status')) {
            wp_schedule_event(time() + 120, 'delcampe_quick_status_check', 'delcampe_check_processing_status');
            $this->debug_log('[Auto Status Check] Scheduled automatic status checking (every 2 minutes)');
        }
        */
        
        // DISABLED: User doesn't want old/ended listings imported (v1.10.18.10)
        // Hook for syncing closed listings from Delcampe (v1.10.18.8)
        // add_action('delcampe_sync_closed_listings', array($this, 'sync_closed_listings_cron'));
        
        // Schedule closed listings sync every 15 minutes (v1.10.18.8)
        // if (!wp_next_scheduled('delcampe_sync_closed_listings')) {
        //     wp_schedule_event(time() + 900, 'every_15_minutes', 'delcampe_sync_closed_listings');
        //     $this->debug_log('[Closed Listings Sync] Scheduled automatic sync (every 15 minutes)');
        // }
        
        // Add custom cron interval (already registered above)
        
        // AJAX handlers for various sync operations
        add_action('wp_ajax_delcampe_process_sync_queue', array($this, 'ajax_process_queue'));
        add_action('wp_ajax_delcampe_get_queue_status', array($this, 'ajax_get_queue_status'));
        add_action('wp_ajax_delcampe_clear_sync_queue', array($this, 'ajax_clear_queue'));
        add_action('wp_ajax_delcampe_clear_sync_lock', array($this, 'ajax_clear_sync_lock'));
        
        // AJAX handler for importing existing listings (v1.2.0.3)
        // This handles the import button click from the listings admin page
        add_action('wp_ajax_delcampe_import_existing_listings', array($this, 'ajax_import_existing_listings'));
        
        // AJAX handler for checking listing status on Delcampe
        add_action('wp_ajax_delcampe_check_listing_status', array($this, 'ajax_check_listing_status'));
        
        // Log initialization
        $this->debug_log('[Delcampe Sync Handler v1.2.2.0] Initialized with enhanced debug logging');
        
        // Debug notice disabled - moved to Developer tab in Settings
        // add_action('admin_notices', array($this, 'show_debug_notice'));
    }

    /**
     * Schedule a queue run using Action Scheduler when available, otherwise WP‑Cron.
     *
     * @param int $delay_seconds Delay in seconds before running (0 for immediate async)
     * @param string $context Optional log context
     */
    private function schedule_queue_run($delay_seconds = 0, $context = '') {
        try {
            if (function_exists('as_enqueue_async_action')) {
                if ($delay_seconds <= 0) {
                    as_enqueue_async_action('delcampe_process_sync_queue', array(), 'delcampe');
                    $this->debug_log('[Scheduler] Enqueued async Action Scheduler run' . ($context ? " ({$context})" : ''));
                } else {
                    as_schedule_single_action(time() + absint($delay_seconds), 'delcampe_process_sync_queue', array(), 'delcampe');
                    $this->debug_log('[Scheduler] Scheduled Action Scheduler run in ' . $delay_seconds . 's' . ($context ? " ({$context})" : ''));
                }
                return;
            }
        } catch (\Throwable $e) {
            $this->debug_log('[Scheduler] Action Scheduler scheduling failed: ' . $e->getMessage());
        }

        // Fallback to WP‑Cron
        wp_schedule_single_event(time() + max(0, (int)$delay_seconds), 'delcampe_process_sync_queue');
        $this->debug_log('[Scheduler] Scheduled WP‑Cron run in ' . $delay_seconds . 's' . ($context ? " ({$context})" : ''));
    }
    
    /**
     * Custom debug logging function
     * 
     * Writes to a custom log file for easier debugging
     * 
     * @since 1.2.1.0
     * @param string $message Message to log
     */
    private function debug_log($message) {
        $timestamp = date('Y-m-d H:i:s');
        $log_message = "[{$timestamp}] {$message}\n";
        
        // Write to custom log file
        $written = file_put_contents($this->debug_log_file, $log_message, FILE_APPEND | LOCK_EX);
        if ($written === false) {
            error_log('[Delcampe Debug Fallback] ' . $log_message);
        }
        
        // Don't pollute WordPress debug log - only write to our dedicated log file
    }
    
    /**
     * Show debug notice in admin
     * 
     * Shows the location of the debug log file on Delcampe admin pages
     * 
     * @since 1.2.1.0
     */
    public function show_debug_notice() {
        // Only show on Delcampe admin pages
        if (!isset($_GET['page']) || strpos($_GET['page'], 'delcampe') === false) {
            return;
        }
        
        // Only show to administrators
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Check if debug log exists and has content
        if (file_exists($this->debug_log_file) && filesize($this->debug_log_file) > 0) {
            ?>
            <div class="notice notice-info is-dismissible">
                <p><strong>Delcampe Debug Log:</strong> <?php echo esc_html($this->debug_log_file); ?></p>
                <p><small>Last modified: <?php echo date('Y-m-d H:i:s', filemtime($this->debug_log_file)); ?></small></p>
            </div>
            <?php
        }
    }
    
    /**
     * Add custom cron interval
     * 
     * @since 1.2.0.0
     * @version 1.10.5.13 - Use hardcoded string to avoid early translation loading
     * @param array $schedules Existing schedules
     * @return array Modified schedules
     */
    public function add_cron_interval($schedules) {
        $schedules['delcampe_sync_interval'] = array(
            'interval' => 30, // 30 seconds for fast processing (v1.10.18.7)
            'display' => 'Every 30 seconds (Delcampe Sync)' // Hardcoded to avoid early translation
        );
        
        $schedules['delcampe_status_check_interval'] = array(
            'interval' => 600, // 10 minutes
            'display' => 'Every 10 minutes (Delcampe Status Check)' // Hardcoded to avoid early translation
        );
        
        // Added in v1.10.10.4 for faster status checking
        $schedules['delcampe_quick_status_check'] = array(
            'interval' => 120, // 2 minutes
            'display' => 'Every 2 minutes (Delcampe Quick Status Check)'
        );
        
        // Added in v1.10.18.0 for batch queue processing
        $schedules['delcampe_batch_interval'] = array(
            'interval' => 30, // 30 seconds for faster processing (v1.10.18.6)
            'display' => 'Every 30 seconds (Delcampe Batch Queue)'
        );
        
        $schedules['delcampe_stuck_check_interval'] = array(
            'interval' => 300, // 5 minutes
            'display' => 'Every 5 minutes (Delcampe Stuck Batch Check)'
        );
        
        // Added in v1.10.18.8 for closed listings sync
        $schedules['every_15_minutes'] = array(
            'interval' => 900, // 15 minutes
            'display' => 'Every 15 minutes'
        );
        
        return $schedules;
    }
    
    /**
     * Add product to sync queue
     * 
     * @since 1.2.0.0
     * @param int $product_id Product ID
     * @param int $profile_id Profile ID
     * @param string $action Action to perform (create, update, close)
     * @return bool Success
     */
    public function add_to_queue($product_id, $profile_id, $action = 'create') {
        // Use new Queue System v2.0 if available
        if (class_exists('Delcampe_Queue_Integration')) {
            $integration = Delcampe_Queue_Integration::get_instance();
            $result = $integration->publish_single($product_id, $action, 0);
            
            if (!is_wp_error($result)) {
                // Update product status
                $product_integration = Delcampe_Product_Integration::get_instance();
                $product_integration->update_sync_status($product_id, 'pending');
                
                delcampe_log("[Sync Handler] Added product {$product_id} to new queue system");
                return true;
            } else {
                delcampe_log("[Sync Handler] Failed to add product {$product_id}: " . $result->get_error_message());
                return false;
            }
        }
        
        // Fallback to old table-backed queue
        if (get_option('delcampe_queue_v2_enabled', 'yes') === 'yes' && class_exists('Delcampe_Sync_Queue')) {
            // Update product status first
            $integration = Delcampe_Product_Integration::get_instance();
            $integration->update_sync_status($product_id, 'pending');
            // Enqueue to table-backed FIFO
            $ok = Delcampe_Sync_Queue::get_instance()->enqueue($product_id, $profile_id, $action, 0, null);
            if ($ok) {
                $this->schedule_queue_run(0, 'v2-enqueue');
            }
            return (bool)$ok;
        }
        // Check if product is published before adding to queue (v1.10.17.8)
        $product_status = get_post_status($product_id);
        if ($product_status !== 'publish') {
            $this->debug_log('[Add to Queue] Product ' . $product_id . ' is ' . $product_status . ', not adding to Delcampe queue');
            return false;
        }
        
        $queue = get_option(self::SYNC_QUEUE_OPTION, array());
        
        // Create queue item
        $item = array(
            'product_id' => $product_id,
            'profile_id' => $profile_id,
            'action' => $action,
            'added' => current_time('timestamp'),
            'attempts' => 0,
            'last_attempt' => null,
            'status' => 'pending'
        );
        
        // Use product ID as key to avoid duplicates
        $queue[$product_id] = $item;
        
        // Update queue
        update_option(self::SYNC_QUEUE_OPTION, $queue);
        
        // Update product status
        $integration = Delcampe_Product_Integration::get_instance();
        $integration->update_sync_status($product_id, 'pending');
        
        $this->debug_log('[Sync Queue] Added product ' . $product_id . ' to sync queue with action: ' . $action);
        
        // Trigger processing soon to catch concurrent bulk publishes
        // If queue is currently being processed, schedule a follow-up run
        if (get_transient(self::SYNC_LOCK_TRANSIENT)) {
            // Queue is locked, schedule a follow-up run after current processing
            $this->schedule_queue_run(65, 'add_to_queue:locked');
            $this->debug_log('[Sync Queue] Queue is locked, scheduled follow-up processing in 65 seconds');
        } else {
            // Queue is not locked, schedule immediate processing
            $this->schedule_queue_run(0, 'add_to_queue:immediate');
        }
        
        return true;
    }
    
    /**
     * Remove product from sync queue
     * 
     * @since 1.2.0.0
     * @param int $product_id Product ID
     * @return bool Success
     */
    public function remove_from_queue($product_id) {
        $queue = get_option(self::SYNC_QUEUE_OPTION, array());
        
        $this->debug_log('[Remove from Queue] Current queue has ' . count($queue) . ' items');
        
        if (isset($queue[$product_id])) {
            unset($queue[$product_id]);
            $result = update_option(self::SYNC_QUEUE_OPTION, $queue);
            
            $this->debug_log('[Remove from Queue] Removed product ' . $product_id . ' from sync queue. Update result: ' . ($result ? 'success' : 'failed'));
            $this->debug_log('[Remove from Queue] Queue now has ' . count($queue) . ' items');
            
            // Verify the removal
            $verify_queue = get_option(self::SYNC_QUEUE_OPTION, array());
            if (isset($verify_queue[$product_id])) {
                $this->debug_log('[Remove from Queue] ERROR: Product ' . $product_id . ' still exists in queue after removal!');
            }
            
            return true;
        }
        
        $this->debug_log('[Remove from Queue] Product ' . $product_id . ' not found in queue');
        return false;
    }
    
    /**
     * Process sync queue
     * 
     * @since 1.2.0.0
     * @version 1.2.2.0 - Enhanced error handling
     * @return array Processing results
     */
    public function process_queue($max_items = null) {
        // Use new Queue System v2.0 if available
        if (class_exists('Delcampe_Queue_Worker')) {
            // For manual AJAX runs (button), aggressively make items claimable now
            if (defined('DOING_AJAX') && DOING_AJAX && class_exists('Delcampe_Queue')) {
                try { Delcampe_Queue::get_instance()->force_make_claimable(); } catch (\Throwable $e) {}
            }
            $worker = Delcampe_Queue_Worker::get_instance();
            
            // Determine batch size
            // - null: default 10
            // - 0: process all currently claimable pending items
            // - >0: process up to that many
            if ($max_items === null) {
                $batch_size = 10;
            } elseif ($max_items === 0) {
                $batch_size = $this->count_claimable_pending();
                if ($batch_size < 1) { $batch_size = 10; }
            } else {
                $batch_size = max(1, (int)$max_items);
            }
            $result = $worker->process_batch($batch_size);
            
            if ($result === false) {
                return array(
                    'status' => 'error',
                    'message' => 'Failed to process queue',
                    'processed' => 0,
                    'success' => 0,
                    'failed' => 0,
                    'errors' => array('Queue processing failed'),
                );
            }
            
            // v1.10.35.13: Fixed undefined array key warnings
            // process_batch returns 'processed' and 'errors' count, not 'success' and 'failed'
            $success_count = isset($result['processed']) ? $result['processed'] : 0;
            $error_count = isset($result['errors']) ? $result['errors'] : 0;
            
            return array(
                'status' => 'completed',
                'message' => sprintf('Processed %d item(s) via new queue system', $success_count),
                'processed' => $success_count + $error_count,
                'success' => $success_count,
                'failed' => $error_count,
                'errors' => isset($result['error_messages']) ? $result['error_messages'] : array(),
            );
        }
        
        // Fallback to old v2 queue if new system not available
        if (get_option('delcampe_queue_v2_enabled', 'yes') === 'yes' && class_exists('Delcampe_Sync_Queue')) {
            // One-time migrate legacy option queue into table-backed queue
            $legacy = get_option(self::SYNC_QUEUE_OPTION, array());
            if (!empty($legacy) && is_array($legacy)) {
                $queue_v2 = Delcampe_Sync_Queue::get_instance();
                foreach ($legacy as $legacy_item) {
                    $pid = isset($legacy_item['product_id']) ? intval($legacy_item['product_id']) : 0;
                    $prf = isset($legacy_item['profile_id']) ? intval($legacy_item['profile_id']) : 0;
                    $act = isset($legacy_item['action']) ? $legacy_item['action'] : 'create';
                    if ($pid && $prf) {
                        $queue_v2->enqueue($pid, $prf, $act, 0, null);
                    }
                }
                // Clear legacy after migration
                update_option(self::SYNC_QUEUE_OPTION, array());
            }

            // For manual AJAX runs, reclaim all processing and drain now
            $force = (defined('DOING_AJAX') && DOING_AJAX) ? true : false;
            Delcampe_Sync_Queue::get_instance()->process_queue($force);
            return array(
                'status' => 'completed',
                'message' => 'Processed via v2 queue',
                'processed' => 0,
                'success' => 0,
                'failed' => 0,
                'errors' => array(),
            );
        }
        // Initialize results array with all expected keys
        $results = array(
            'status' => 'processing',
            'message' => '',
            'processed' => 0,
            'success' => 0,
            'failed' => 0,
            'errors' => array()
        );
        
        // Check if already processing; auto-recover stale locks (>180s)
        if (get_transient(self::SYNC_LOCK_TRANSIENT)) {
            $locked_at = (int) get_transient(self::SYNC_LOCK_TRANSIENT . '_at');
            $age = $locked_at ? (time() - $locked_at) : 0;
            if ($locked_at && $age > 180) {
                $this->debug_log('[Queue Processing] Detected stale sync lock (' . $age . 's). Clearing and continuing.');
                delete_transient(self::SYNC_LOCK_TRANSIENT);
                delete_transient(self::SYNC_LOCK_TRANSIENT . '_at');
            } else {
                $this->debug_log('[Queue Processing] Sync already in progress, skipping');
                $results['status'] = 'locked';
                $results['message'] = 'Sync already in progress';
                return $results;
            }
        }
        
        // Set lock (short TTL) to allow quick follow-ups if a run crashes
        set_transient(self::SYNC_LOCK_TRANSIENT, true, 60);
        set_transient(self::SYNC_LOCK_TRANSIENT . '_at', time(), 600);
        
        // Get queue
        $queue = get_option(self::SYNC_QUEUE_OPTION, array());
        
        if (empty($queue)) {
            delete_transient(self::SYNC_LOCK_TRANSIENT);
            delete_transient(self::SYNC_LOCK_TRANSIENT . '_at');
            $results['status'] = 'empty';
            $results['message'] = 'No items in queue';
            return $results;
        }
        
        // Log queue processing start
        $this->debug_log('[Queue Processing] Starting to process queue with ' . count($queue) . ' items');
        
        // Get API instance
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-listing-api.php';
        $api = Delcampe_Listing_API::get_instance();
        
        // Process batch
        $count = 0;
        // Track items that need persisted attempt count updates without clobbering new queue additions
        $failed_updates = array();
        // Allow override via parameter; 0 means unlimited
        if ($max_items === null) {
            $batch_limit = $this->get_batch_size();
        } else {
            $batch_limit = ($max_items <= 0) ? PHP_INT_MAX : intval($max_items);
        }
        $this->debug_log('[Queue Processing] Using batch size: ' . $batch_limit);
        while (true) {
            // Stop if we've hit the batch limit
            if ($count >= $batch_limit) {
                break;
            }

            // Refresh queue each iteration to avoid foreach/option side-effects
            $queue = get_option(self::SYNC_QUEUE_OPTION, array());
            if (empty($queue)) {
                break; // Nothing left to process
            }

            // Get first item in queue (PHP 7.3+: array_key_first)
            if (function_exists('array_key_first')) {
                $product_id = array_key_first($queue);
            } else {
                $product_id = null;
                foreach ($queue as $pid => $_) { $product_id = $pid; break; }
                if ($product_id === null) { break; }
            }
            $item = $queue[$product_id];
            // Limit to configured batch size per run
            // (batch limit checked above)

            // Memory safety guard (relaxed): check every 10 items and only if usage exceeds ~95%
            if ($count > 0 && ($count % 10) === 0) {
                $memLimit = ini_get('memory_limit');
                $bytes = -1;
                if ($memLimit && $memLimit !== '-1') {
                    $unit = strtolower(substr($memLimit, -1));
                    $val = (int)$memLimit;
                    switch ($unit) {
                        case 'g': $bytes = $val * 1024 * 1024 * 1024; break;
                        case 'm': $bytes = $val * 1024 * 1024; break;
                        case 'k': $bytes = $val * 1024; break;
                        default: $bytes = (int)$memLimit; break;
                    }
                }
                if ($bytes > 0) {
                    $usage = memory_get_usage(true);
                    if ($usage > (0.95 * $bytes)) {
                        $this->debug_log('[Queue Processing] Memory guard tripped at 95%. Usage ' . $usage . ' of ' . $bytes . '. Scheduling continuation.');
                        $this->schedule_queue_run(5, 'memory-guard');
                        break;
                    }
                }
            }
            
            // Skip if too many attempts
            if (!empty($item['attempts']) && $item['attempts'] >= 3) {
                $this->mark_as_failed($product_id, 'Max attempts reached');
                $results['failed']++;
                continue;
            }
            
            // Process item
            $result = $this->process_item($item);
            
            if ($result === true) {
                // Success - remove from queue
                $this->remove_from_queue($product_id);
                $results['success']++;
                
                // Log success
                $this->debug_log('[Queue Processing] Successfully processed product ' . $product_id);
            } else {
                // Failed - update attempts
                $item['attempts'] = isset($item['attempts']) ? ($item['attempts'] + 1) : 1;
                $item['last_attempt'] = current_time('timestamp');
                $item['last_error'] = is_wp_error($result) ? $result->get_error_message() : 'Unknown error';
                $failed_updates[$product_id] = $item; // stage for merge-safe persistence
                
                $results['failed']++;
                $results['errors'][] = array(
                    'product_id' => $product_id,
                    'error' => $item['last_error']
                );
                
                // Update product status
                $integration = Delcampe_Product_Integration::get_instance();
                $integration->update_sync_status($product_id, 'error');
                
                // Store error message
                update_post_meta($product_id, '_delcampe_last_error', $item['last_error']);
                
                // Log error
                $this->debug_log('[Queue Processing] Failed to process product ' . $product_id . ': ' . $item['last_error']);
            }
            
            $results['processed']++;
            $count++;
        }
        
        // Persist attempt count updates without overwriting concurrently-added items
        if (!empty($failed_updates)) {
            // Merge staged updates into the latest queue from the database to avoid clobbering
            $current_queue = get_option(self::SYNC_QUEUE_OPTION, array());
            foreach ($failed_updates as $fid => $fitem) {
                $current_queue[$fid] = $fitem;
            }
            update_option(self::SYNC_QUEUE_OPTION, $current_queue);
            $this->debug_log('[Queue Processing] Persisted ' . count($failed_updates) . ' failed attempt update(s) with merge-safe write');
        }
        
        // If new items were added during this run, schedule a quick follow-up
        $remaining_queue = get_option(self::SYNC_QUEUE_OPTION, array());
        if (!empty($remaining_queue)) {
            $this->debug_log('[Queue Processing] Detected remaining items after run; scheduling follow-up');
            $this->schedule_queue_run(5, 'remaining-items');
        }

        // Release lock
        delete_transient(self::SYNC_LOCK_TRANSIENT);
        delete_transient(self::SYNC_LOCK_TRANSIENT . '_at');
        
        // Set final status and message
        $results['status'] = 'completed';
        $results['message'] = sprintf(
            'Processed %d items: %d successful, %d failed',
            $results['processed'],
            $results['success'],
            $results['failed']
        );
        
        // Log results
        $this->debug_log('[Queue Processing] Completed: ' . $results['message']);
        
        return $results;
    }
    
    /**
     * Process a single queue item
     * 
     * This method handles the actual synchronization of a product to Delcampe.
     * It creates listings table records to track the listing status.
     * 
     * @since 1.2.0.0
     * @version 1.2.2.0 - Enhanced error handling
     * @param array $item Queue item
     * @return true|WP_Error Success or error
     */
    private function process_item($item) {
        $product_id = $item['product_id'];
        $profile_id = $item['profile_id'];
        $action = $item['action'];
        
        $this->debug_log('[Process Item] Processing product ' . $product_id . ' with action: ' . $action);
        
        // Check if product still exists and is published (v1.10.17.8)
        $product = wc_get_product($product_id);
        if (!$product) {
            $this->debug_log('[Process Item] Product ' . $product_id . ' not found, skipping');
            return new WP_Error('product_not_found', 'Product not found');
        }
        
        $product_status = get_post_status($product_id);
        if ($product_status !== 'publish') {
            $this->debug_log('[Process Item] Product ' . $product_id . ' is ' . $product_status . ', skipping Delcampe operations');
            // Clean up any Delcampe metadata from non-published products
            delete_post_meta($product_id, '_delcampe_listing_id');
            delete_post_meta($product_id, '_delcampe_sync_status');
            delete_post_meta($product_id, '_delcampe_last_sync');
            return new WP_Error('product_not_published', 'Product is not published');
        }
        
        // Get API instance
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-listing-api.php';
        $api = Delcampe_Listing_API::get_instance();
        
        // Perform action based on type
        switch ($action) {
            case 'create':
                // Create new listing on Delcampe
                $result = $api->create_listing($product_id, $profile_id);
                
                // If successful, create listings table record (v1.2.0.3)
                // This ensures the listing is tracked in our listings management system
                if (!is_wp_error($result)) {
                    $this->create_listing_record($product_id, $profile_id, $result);
                }
                break;
                
            case 'update':
                // Update existing listing on Delcampe
                $listing_id = get_post_meta($product_id, '_delcampe_listing_id', true);
                if (!$listing_id) {
                    return new WP_Error('no_listing_id', 'No listing ID found for update');
                }
                $result = $api->update_listing($product_id, $listing_id);
                
                // If successful, update listings table record (v1.2.0.3)
                if (!is_wp_error($result)) {
                    $this->update_listing_record($product_id, $result);
                }
                break;
                
            case 'close':
                // Close listing on Delcampe
                $listing_id = get_post_meta($product_id, '_delcampe_listing_id', true);
                if (!$listing_id) {
                    return new WP_Error('no_listing_id', 'No listing ID found for closure');
                }
                $result = $api->close_listing($listing_id);
                
                // If successful, update listings table status (v1.2.0.3)
                if (!is_wp_error($result)) {
                    $this->close_listing_record($product_id);
                }
                break;
                
            default:
                return new WP_Error('invalid_action', 'Invalid sync action: ' . $action);
        }
        
        // Check result
        if (is_wp_error($result)) {
            $this->debug_log('[Process Item] Error processing product ' . $product_id . ': ' . $result->get_error_message());
            return $result;
        }
        
        $this->debug_log('[Process Item] Successfully processed product ' . $product_id);
        return true;
    }

    /**
     * Public wrapper to process a single queue item (for v2 table-backed queue)
     *
     * @param array $item ['product_id','profile_id','action']
     * @return true|WP_Error
     */
    public function process_item_public($item) {
        // Verbose log for diagnostics
        if (function_exists('delcampe_log')) {
            delcampe_log('[SyncV2] Processing product ' . $item['product_id'] . ' action ' . $item['action'], 'info');
        }
        return $this->process_item($item);
    }
    
    /**
     * Create listing record in listings table
     * 
     * Creates a new record in the listings table to track the listing.
     * This integrates with the listings management system introduced in v1.6.0.
     * 
     * @since 1.2.0.3
     * @version 1.2.2.0 - Fixed personal_reference handling
     * @param int $product_id Product ID
     * @param int $profile_id Profile ID
     * @param array $api_response API response data
     */
    private function create_listing_record($product_id, $profile_id, $api_response) {
        // Load listings model if needed
        if (!class_exists('Delcampe_Listings_Model')) {
            require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-listings-model.php';
        }
        
        $product = wc_get_product($product_id);
        if (!$product) {
            $this->debug_log('[Create Listing Record] Product ' . $product_id . ' not found');
            return;
        }
        
        // Get profile data
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/profiles/class-delcampe-profiles-model.php';
        $profiles_model = Delcampe_Profiles_Model::get_instance();
        $profile = $profiles_model->get_profile($profile_id);
        
        // Decide initial status based on API response
        // When queued (no ID yet), use 'verified' status to indicate it's been sent to Delcampe
        // This prevents re-selection while still using a valid listing status
        $initial_status = 'published';
        if (empty($api_response['id'])) {
            // Use 'verified' for queued items - means "sent to Delcampe, awaiting confirmation"
            $initial_status = 'verified';
        }

        // Prepare listing data
        $listing_data = array(
            'product_id' => $product_id,
            'profile_id' => $profile_id,
            'listing_title' => $product->get_name(),
            'quantity' => $product->get_stock_quantity() ?: 1,
            'price' => $product->get_price(),
            'currency' => $profile['listing_details']['currency'] ?? get_woocommerce_currency(),
            'status' => $initial_status,
            'date_published' => $initial_status === 'published' ? current_time('mysql') : null,
        );
        
        // Add Delcampe ID if available
        if (!empty($api_response['id'])) {
            $listing_data['delcampe_id'] = $api_response['id'];
        }
        
        // Add personal reference if available (v1.2.2.0)
        // First check if the column exists in the database
        global $wpdb;
        $table_name = $wpdb->prefix . 'delcampe_listings';
        $column_exists = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
                WHERE TABLE_SCHEMA = %s 
                AND TABLE_NAME = %s 
                AND COLUMN_NAME = 'personal_reference'",
                DB_NAME,
                $table_name
            )
        );
        
        if ($column_exists && !empty($api_response['personal_reference'])) {
            $listing_data['personal_reference'] = $api_response['personal_reference'];
            $this->debug_log('[Create Listing Record] Added personal_reference: ' . $api_response['personal_reference']);
        } else if (!$column_exists) {
            $this->debug_log('[Create Listing Record] Warning: personal_reference column does not exist in database. Run plugin activation or update to fix.');
        }
        
        // Check if listing already exists
        $existing_listings = Delcampe_Listings_Model::get_listings_by_product_id($product_id);
        if (!empty($existing_listings)) {
            // Prefer most recent record if multiples exist
            $existing_listing = $existing_listings[0];
            // Clean nulls for safe update
            $update_payload = array_filter($listing_data, function($v) { return $v !== null; });
            Delcampe_Listings_Model::update_listing($existing_listing->id, $update_payload);
            $this->debug_log('[Create Listing Record] Updated existing listing record ' . $existing_listing->id . ' for product ' . $product_id);
        } else {
            // Create new listing record
            $insert_payload = array_filter($listing_data, function($v) { return $v !== null; });
            $listing_id = Delcampe_Listings_Model::insert_listing($insert_payload);
            if ($listing_id) {
                $this->debug_log('[Create Listing Record] Created new listing record ' . $listing_id . ' for product ' . $product_id);
            } else {
                $this->debug_log('[Create Listing Record] Failed to create listing record for product ' . $product_id);
            }
        }
    }
    
    /**
     * Update listing record in listings table
     * 
     * @since 1.2.0.3
     * @version 1.2.2.0 - Enhanced error handling
     * @param int $product_id Product ID
     * @param array $api_response API response data
     */
    private function update_listing_record($product_id, $api_response) {
        // Load listings model if needed
        if (!class_exists('Delcampe_Listings_Model')) {
            require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-listings-model.php';
        }
        
        // Get existing listing
        $listings = Delcampe_Listings_Model::get_listings_by_product_id($product_id);
        if (empty($listings)) {
            // No listing record exists, create one
            $profile_id = get_post_meta($product_id, '_delcampe_profile_id', true);
            if ($profile_id) {
                $this->create_listing_record($product_id, $profile_id, $api_response);
            } else {
                $this->debug_log('[Update Listing Record] No profile ID found for product ' . $product_id);
            }
            return;
        }
        
        // Update the listing
        $listing = $listings[0];
        $update_data = array(
            'date_updated' => current_time('mysql'),
        );
        
        // Update any fields from API response
        if (!empty($api_response['id'])) {
            $update_data['delcampe_id'] = $api_response['id'];
        }
        
        Delcampe_Listings_Model::update_listing($listing->id, $update_data);
        $this->debug_log('[Update Listing Record] Updated listing record ' . $listing->id . ' for product ' . $product_id);
    }
    
    /**
     * Close listing record in listings table
     * 
     * @since 1.2.0.3
     * @param int $product_id Product ID
     */
    private function close_listing_record($product_id) {
        // Load listings model if needed
        if (!class_exists('Delcampe_Listings_Model')) {
            require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-listings-model.php';
        }
        
        // Get existing listing
        $listings = Delcampe_Listings_Model::get_listings_by_product_id($product_id);
        if (empty($listings)) {
            $this->debug_log('[Close Listing Record] No listing found for product ' . $product_id);
            return;
        }
        
        // Update status to ended
        $listing = $listings[0];
        Delcampe_Listings_Model::update_listing($listing->id, array(
            'status' => 'ended',
            'date_finished' => current_time('mysql'),
        ));
        
        $this->debug_log('[Close Listing Record] Closed listing record ' . $listing->id . ' for product ' . $product_id);
    }
    
    /**
     * Import existing synced products into listings table
     * 
     * This method handles the import of products that were synced to Delcampe
     * before the listings table was created. It creates listing records for
     * these products so they appear in the listings management interface.
     * 
     * @since 1.2.0.3
     * @version 1.2.2.0 - Fixed personal_reference handling
     * @return array Import results
     */
    public function import_existing_listings() {
        $this->debug_log('[Import Existing] Starting import_existing_listings function');
        
        global $wpdb;
        
        $results = array(
            'imported' => 0,
            'skipped' => 0,
            'errors' => 0,
            'details' => array()
        );
        
        // Load listings model if needed
        if (!class_exists('Delcampe_Listings_Model')) {
            $this->debug_log('[Import Existing] Loading Delcampe_Listings_Model class');
            require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-listings-model.php';
        }
        
        // Find all products with sync status but no listing record
        // These are products that were synced before the listings table existed
        // Only consider published products (v1.10.17.8)
        $query = "SELECT p1.post_id, p1.meta_value as sync_status, p2.meta_value as profile_id
                  FROM {$wpdb->postmeta} p1
                  LEFT JOIN {$wpdb->postmeta} p2 ON p1.post_id = p2.post_id AND p2.meta_key = '_delcampe_profile_id'
                  INNER JOIN {$wpdb->posts} p ON p1.post_id = p.ID
                  WHERE p1.meta_key = '_delcampe_sync_status' 
                  AND p1.meta_value IN ('active', 'pending')
                  AND p.post_type = 'product'
                  AND p.post_status = 'publish'";
        
        $this->debug_log('[Import Existing] Running query to find synced products');
        $synced_products = $wpdb->get_results($query);
        
        $this->debug_log('[Import Existing] Found ' . count($synced_products) . ' synced products to check');
        
        // Check if personal_reference column exists (v1.2.2.0)
        $table_name = $wpdb->prefix . 'delcampe_listings';
        $column_exists = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM INFORMATION_SCHEMA.COLUMNS 
                WHERE TABLE_SCHEMA = %s 
                AND TABLE_NAME = %s 
                AND COLUMN_NAME = 'personal_reference'",
                DB_NAME,
                $table_name
            )
        );
        
        if (!$column_exists) {
            $this->debug_log('[Import Existing] Warning: personal_reference column does not exist. Running without it.');
        }
        
        foreach ($synced_products as $synced) {
            $product_id = $synced->post_id;
            $profile_id = $synced->profile_id;
            
            $this->debug_log('[Import Existing] Checking product ID: ' . $product_id);
            
            // Check if listing already exists
            $existing_listings = Delcampe_Listings_Model::get_listings_by_product_id($product_id);
            if (!empty($existing_listings)) {
                $results['skipped']++;
                $this->debug_log('[Import Existing] Skipped product ' . $product_id . ' - listing already exists');
                continue;
            }
            
            // Get product
            $product = wc_get_product($product_id);
            if (!$product) {
                $results['errors']++;
                $results['details'][] = "Product $product_id not found";
                $this->debug_log('[Import Existing] Error: Product ' . $product_id . ' not found');
                continue;
            }
            
            // Get profile if we have one
            $profile = null;
            if ($profile_id) {
                require_once plugin_dir_path(dirname(__FILE__)) . 'includes/profiles/class-delcampe-profiles-model.php';
                $profiles_model = Delcampe_Profiles_Model::get_instance();
                $profile = $profiles_model->get_profile($profile_id);
            }
            
            // Prepare listing data
            $listing_data = array(
                'product_id' => $product_id,
                'profile_id' => $profile_id ?: 2,  // Default to plate blocks profile instead of 0
                'listing_title' => $product->get_name(),
                'quantity' => $product->get_stock_quantity() ?: 1,
                'price' => $product->get_price(),
                'currency' => ($profile && isset($profile['listing_details']['currency'])) ? $profile['listing_details']['currency'] : get_woocommerce_currency(),
                'status' => ($synced->sync_status == 'active') ? 'published' : 'prepared',
                'date_published' => ($synced->sync_status == 'active') ? current_time('mysql') : null,
            );
            
            // Get any stored Delcampe data
            $delcampe_id = get_post_meta($product_id, '_delcampe_listing_id', true);
            if ($delcampe_id) {
                $listing_data['delcampe_id'] = $delcampe_id;
            }
            
            // Only add personal_reference if column exists (v1.2.2.0)
            if ($column_exists) {
                $personal_ref = get_post_meta($product_id, '_delcampe_personal_reference', true);
                if (!$personal_ref) {
                    // Generate from SKU or product ID
                    $personal_ref = $product->get_sku();
                    if (empty($personal_ref)) {
                        $personal_ref = 'WC-' . $product_id;
                    }
                }
                if ($personal_ref) {
                    $listing_data['personal_reference'] = $personal_ref;
                }
            }
            
            $this->debug_log('[Import Existing] Creating listing for product ' . $product_id . ' with title: ' . $listing_data['listing_title']);
            
            // Create listing record
            $listing_id = Delcampe_Listings_Model::insert_listing($listing_data);
            if ($listing_id) {
                $results['imported']++;
                $results['details'][] = "Imported product $product_id as listing $listing_id";
                $this->debug_log('[Import Existing] Successfully imported product ' . $product_id . ' as listing ' . $listing_id);
            } else {
                $results['errors']++;
                $results['details'][] = "Failed to import product $product_id - database insert failed";
                $this->debug_log('[Import Existing] Failed to import product ' . $product_id . ' - database insert failed');
            }
        }
        
        $this->debug_log('[Import Existing] Import complete. Imported: ' . $results['imported'] . ', Skipped: ' . $results['skipped'] . ', Errors: ' . $results['errors']);
        return $results;
    }
    
    /**
     * AJAX handler for importing existing listings
     * 
     * This is called when the user clicks the "Import X Existing Listings" button
     * in the listings admin interface.
     * 
     * @since 1.2.0.3
     * @version 1.2.2.0 - Enhanced error handling and JSON response
     */
    public function ajax_import_existing_listings() {
        $this->debug_log('[AJAX Import] ajax_import_existing_listings called');
        
        // Add response header to ensure JSON
        @header('Content-Type: application/json; charset=utf-8');
        
        // Verify nonce for security
        if (!isset($_POST['nonce'])) {
            $this->debug_log('[AJAX Import] Error: No nonce provided');
            wp_send_json_error(array('message' => __('Security check failed: No nonce provided.', 'wc-delcampe-integration')));
            return;
        }
        
        if (!wp_verify_nonce($_POST['nonce'], 'delcampe_admin_nonce')) {
            $this->debug_log('[AJAX Import] Error: Nonce verification failed');
            wp_send_json_error(array('message' => __('Security check failed: Invalid nonce.', 'wc-delcampe-integration')));
            return;
        }
        
        // SECURITY FIX v1.10.21.0: Consistent permission check
        // Check permissions - user must be able to manage WooCommerce
        if (!current_user_can('manage_woocommerce')) {
            $this->debug_log('[AJAX Import] Error: User lacks manage_woocommerce capability');
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'wc-delcampe-integration')));
            return;
        }
        
        $this->debug_log('[AJAX Import] Security checks passed, running import');
        
        try {
            // Run import
            $results = $this->import_existing_listings();
            
            // Create user-friendly message
            $message = sprintf(
                __('Import complete: %d imported, %d skipped, %d errors', 'wc-delcampe-integration'),
                $results['imported'],
                $results['skipped'],
                $results['errors']
            );
            
            $this->debug_log('[AJAX Import] Sending success response: ' . $message);
            
            // Send JSON response
            wp_send_json_success(array(
                'message' => $message,
                'results' => $results
            ));
            
        } catch (Exception $e) {
            $this->debug_log('[AJAX Import] Exception caught: ' . $e->getMessage());
            wp_send_json_error(array(
                'message' => __('Import failed: ', 'wc-delcampe-integration') . $e->getMessage()
            ));
        }
    }
    
    /**
     * Mark product as failed
     * 
     * @since 1.2.0.0
     * @param int $product_id Product ID
     * @param string $reason Failure reason
     */
    private function mark_as_failed($product_id, $reason) {
        // Update product status
        $integration = Delcampe_Product_Integration::get_instance();
        $integration->update_sync_status($product_id, 'error');
        
        // Store error message
        update_post_meta($product_id, '_delcampe_last_error', $reason);
        
        // Remove from queue
        $this->remove_from_queue($product_id);
        
        $this->debug_log('[Mark Failed] Product ' . $product_id . ' marked as failed: ' . $reason);
    }

    /**
     * AJAX: Requeue stuck listings (processing with no Delcampe ID)
     *
     * @since 1.10.15.6
     */
    public function ajax_requeue_stuck_listings() {
        // Add response header
        @header('Content-Type: application/json; charset=utf-8');

        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'delcampe_admin_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'wc-delcampe-integration')));
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'wc-delcampe-integration')));
        }

        $minutes = isset($_POST['minutes']) ? intval($_POST['minutes']) : 10;
        $minutes = max(0, $minutes);

        global $wpdb;
        $table = $wpdb->prefix . 'delcampe_listings';
        $cutoff = gmdate('Y-m-d H:i:s', time() - ($minutes * 60));

        $rows = $wpdb->get_results($wpdb->prepare(
            "SELECT id, product_id, profile_id, delcampe_id, status, date_created, date_updated
             FROM {$table}
             WHERE status IN ('processing','published')
               AND (delcampe_id IS NULL OR delcampe_id = '')
               AND IFNULL(date_updated, date_created) < %s",
            $cutoff
        ));

        if (empty($rows)) {
            wp_send_json_success(array('requeued' => 0, 'scheduled' => false, 'message' => __('No stuck listings found.', 'wc-delcampe-integration')));
        }

        $requeued = 0;
        foreach ($rows as $row) {
            // Reset local status
            $wpdb->update($table, array(
                'status' => 'prepared',
                'date_updated' => current_time('mysql'),
            ), array('id' => $row->id));

            if (!empty($row->product_id) && !empty($row->profile_id)) {
                $this->add_to_queue((int)$row->product_id, (int)$row->profile_id, 'create');
                $requeued++;
            }
        }

        // Schedule immediate processing
        $this->schedule_queue_run(5, 'ajax_requeue_stuck');

        wp_send_json_success(array(
            'requeued' => $requeued,
            'scheduled' => true,
            'message' => sprintf(__('Requeued %d stuck listings; processing will start shortly.', 'wc-delcampe-integration'), $requeued)
        ));
    }

    /**
     * AJAX: Reconcile open listings from Delcampe (fills missing delcampe_id, updates statuses)
     *
     * @since 1.10.15.8
     */
    public function ajax_reconcile_open_listings() {
        @header('Content-Type: application/json; charset=utf-8');
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'delcampe_admin_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'wc-delcampe-integration')));
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'wc-delcampe-integration')));
        }

        // Authenticate
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-auth.php';
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        if (is_wp_error($token) || empty($token)) {
            wp_send_json_error(array('message' => __('Failed to authenticate with Delcampe.', 'wc-delcampe-integration')));
        }

        $url = DELCAMPE_API_BASE_URL . '/item/opened?token=' . urlencode($token);
        $res = wp_remote_get($url, array('timeout' => 30, 'headers' => array('Accept' => 'application/xml')));
        if (is_wp_error($res)) {
            wp_send_json_error(array('message' => $res->get_error_message()));
        }
        $code = wp_remote_retrieve_response_code($res);
        $body = wp_remote_retrieve_body($res);
        if ($code !== 200 || empty($body)) {
            wp_send_json_error(array('message' => __('Unexpected response from Delcampe.', 'wc-delcampe-integration')));
        }

        libxml_use_internal_errors(true);
        $xml = simplexml_load_string($body);
        if (!$xml) {
            wp_send_json_error(array('message' => __('Failed to parse XML.', 'wc-delcampe-integration')));
        }

        // Collect items from multiple possible structures
        $items = array();
        // Pattern A: Notification_Data->body->items->item
        if (isset($xml->Notification_Data->body->items->item)) {
            foreach ($xml->Notification_Data->body->items->item as $it) { $items[] = $it; }
        }
        // Pattern B: Notification_Data->body->item (single)
        if (empty($items) && isset($xml->Notification_Data->body->item)) {
            foreach ($xml->Notification_Data->body->item as $it) { $items[] = $it; }
        }
        // Pattern C: root->item
        if (empty($items) && isset($xml->item)) {
            foreach ($xml->item as $it) { $items[] = $it; }
        }

        if (empty($items)) {
            wp_send_json_error(array('message' => __('No open items found in API response.', 'wc-delcampe-integration')));
        }

        global $wpdb;
        $table = $wpdb->prefix . 'delcampe_listings';
        $updated = 0; $inserted = 0; $matched = 0;

        foreach ($items as $it) {
            $id = '';
            if (isset($it->id_item)) { $id = (string)$it->id_item; }
            elseif (isset($it->id)) { $id = (string)$it->id; }
            $pref = '';
            if (isset($it->personal_reference)) { $pref = (string)$it->personal_reference; }
            elseif (isset($it->personalReference)) { $pref = (string)$it->personalReference; }

            if (empty($id) && empty($pref)) { continue; }

            // If we have delcampe_id, update status to published
            if (!empty($id)) {
                $existing = $wpdb->get_row($wpdb->prepare("SELECT id, product_id FROM {$table} WHERE delcampe_id = %s", $id));
                if ($existing) {
                    $wpdb->update($table, array('status' => 'published', 'date_updated' => current_time('mysql')), array('id' => $existing->id));
                    if (!empty($existing->product_id)) {
                        update_post_meta($existing->product_id, '_delcampe_listing_id', $id);
                        update_post_meta($existing->product_id, '_delcampe_sync_status', 'active');
                    }
                    $updated++; $matched++; continue;
                }
            }

            // If we have personal reference, map to listing and fill delcampe_id
            if (!empty($pref)) {
                $existing = $wpdb->get_row($wpdb->prepare("SELECT id, product_id FROM {$table} WHERE personal_reference = %s", $pref));
                if ($existing) {
                    $data = array('status' => 'published', 'date_updated' => current_time('mysql'));
                    if (!empty($id)) { $data['delcampe_id'] = $id; }
                    $wpdb->update($table, $data, array('id' => $existing->id));
                    if (!empty($existing->product_id)) {
                        if (!empty($id)) { update_post_meta($existing->product_id, '_delcampe_listing_id', $id); }
                        update_post_meta($existing->product_id, '_delcampe_sync_status', 'active');
                    }
                    $updated++; $matched++; continue;
                }
            }

            // As a last resort, try SKU match for personal_reference
            if (!empty($pref)) {
                $product_id = wc_get_product_id_by_sku($pref);
                if ($product_id) {
                    // Get profile_id from product meta, default to 2 (plate blocks) if not set
                    $profile_id = get_post_meta($product_id, '_delcampe_profile_id', true);
                    if (!$profile_id) {
                        $profile_id = 2; // Default to plate blocks profile
                    }
                    $wpdb->insert($table, array(
                        'product_id' => $product_id,
                        'profile_id' => $profile_id,
                        'delcampe_id' => $id,
                        'listing_title' => get_the_title($product_id),
                        'status' => 'published',
                        'personal_reference' => $pref,
                        'date_created' => current_time('mysql'),
                        'date_updated' => current_time('mysql'),
                    ));
                    if (!empty($id)) { update_post_meta($product_id, '_delcampe_listing_id', $id); }
                    update_post_meta($product_id, '_delcampe_sync_status', 'active');
                    $inserted++; continue;
                }
            }
        }

        wp_send_json_success(array(
            'updated' => $updated,
            'inserted' => $inserted,
            'matched' => $matched,
            'message' => sprintf(__('Reconciled: %d updated, %d inserted', 'wc-delcampe-integration'), $updated, $inserted)
        ));
    }

    /**
     * AJAX: Rebuild local state (listings ⇄ product meta)
     * - Pass 1: For listings(status=published) → ensure product meta (_delcampe_listing_id, _delcampe_sync_status=active)
     * - Pass 2: For product meta(sync=active) → ensure listings row exists (status=published)
     *
     * @since 1.10.17.0
     */
    public function ajax_rebuild_local_state() {
        @header('Content-Type: application/json; charset=utf-8');
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'delcampe_admin_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'wc-delcampe-integration')));
        }
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'wc-delcampe-integration')));
        }

        $dry_run = !empty($_POST['dry_run']);
        global $wpdb;
        $table = $wpdb->prefix . 'delcampe_listings';

        $report = array(
            'pass1' => array('examined' => 0, 'meta_set' => 0, 'skipped' => 0),
            'pass2' => array('examined' => 0, 'inserted' => 0, 'updated' => 0),
        );

        // Pass 1: listings -> product meta
        $rows = $wpdb->get_results("SELECT id, product_id, delcampe_id, personal_reference, status FROM {$table} WHERE status = 'published'");
        foreach ($rows as $row) {
            $report['pass1']['examined']++;
            $pid = intval($row->product_id);
            if ($pid <= 0) {
                // Try to map via personal_reference to SKU → product
                $sku = trim((string)$row->personal_reference);
                if ($sku !== '') {
                    $pid = wc_get_product_id_by_sku($sku);
                }
                if ($pid <= 0) { $report['pass1']['skipped']++; continue; }
                if (!$dry_run) {
                    $wpdb->update($table, array('product_id' => $pid, 'date_updated' => current_time('mysql')), array('id' => $row->id));
                }
            }
            // Ensure product meta
            $existing_sync = get_post_meta($pid, '_delcampe_sync_status', true);
            $existing_id = get_post_meta($pid, '_delcampe_listing_id', true);
            $needs = 0;
            if ($existing_sync !== 'active') { $needs++; }
            if ($row->delcampe_id && $existing_id !== (string)$row->delcampe_id) { $needs++; }
            if ($needs > 0) {
                if (!$dry_run) {
                    update_post_meta($pid, '_delcampe_sync_status', 'active');
                    if (!empty($row->delcampe_id)) {
                        update_post_meta($pid, '_delcampe_listing_id', (string)$row->delcampe_id);
                    }
                }
                $report['pass1']['meta_set']++;
            } else {
                $report['pass1']['skipped']++;
            }
        }

        // Pass 2: product meta -> listings
        $products = $wpdb->get_results($wpdb->prepare(
            "SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = %s AND meta_value = %s",
            '_delcampe_sync_status', 'active'
        ));
        foreach ($products as $pm) {
            $report['pass2']['examined']++;
            $pid = intval($pm->post_id);
            if ($pid <= 0) { continue; }
            // Check if listing row exists for product
            $existing = $wpdb->get_row($wpdb->prepare("SELECT id FROM {$table} WHERE product_id = %d", $pid));
            if ($existing) {
                // Make sure status is published
                if (!$dry_run) {
                    $wpdb->update($table, array('status' => 'published', 'date_updated' => current_time('mysql')), array('id' => $existing->id));
                }
                $report['pass2']['updated']++;
                continue;
            }
            // Insert a listing record
            $sku = get_post_meta($pid, '_sku', true);
            $lid = get_post_meta($pid, '_delcampe_listing_id', true);
            $title = get_the_title($pid);
            // Get profile_id from product meta, default to 2 (plate blocks) if not set
            $profile_id = get_post_meta($pid, '_delcampe_profile_id', true);
            if (!$profile_id) {
                $profile_id = 2; // Default to plate blocks profile
            }
            if (!$dry_run) {
                $wpdb->insert($table, array(
                    'product_id' => $pid,
                    'profile_id' => $profile_id,
                    'delcampe_id' => $lid,
                    'listing_title' => $title,
                    'status' => 'published',
                    'personal_reference' => $sku ?: '',
                    'date_created' => current_time('mysql'),
                    'date_updated' => current_time('mysql'),
                ));
            }
            $report['pass2']['inserted']++;
        }

        wp_send_json_success(array(
            'report' => $report,
            'message' => sprintf(__('Rebuild complete. Pass1 set meta: %d, Pass2 inserted: %d', 'wc-delcampe-integration'), $report['pass1']['meta_set'], $report['pass2']['inserted'])
        ));
    }
    /**
     * AJAX handler for manual queue processing
     * 
     * @since 1.2.0.0
     * @version 1.2.2.0 - Enhanced error handling
     */
    public function ajax_process_queue() {
        // Add response header
        @header('Content-Type: application/json; charset=utf-8');
        
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'delcampe_admin_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'wc-delcampe-integration')));
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'wc-delcampe-integration')));
        }
        
        // Process entire queue on manual trigger
        $results = $this->process_queue(0);
        
        // Get status from results (now guaranteed to exist)
        $status = isset($results['status']) ? $results['status'] : 'unknown';
        
        // Format response based on status
        if ($status === 'locked') {
            $response = array(
                'message' => __('Sync is already in progress. Please wait.', 'wc-delcampe-integration'),
                'results' => $results
            );
        } elseif ($status === 'empty') {
            $response = array(
                'message' => __('No items in the sync queue.', 'wc-delcampe-integration'),
                'results' => $results
            );
        } else {
            // Use the message from results or create one
            $message = !empty($results['message']) ? $results['message'] : sprintf(
                __('Processed %d items: %d successful, %d failed', 'wc-delcampe-integration'),
                $results['processed'],
                $results['success'],
                $results['failed']
            );
            
            $response = array(
                'message' => $message,
                'results' => $results
            );
        }
        
        wp_send_json_success($response);
    }
    
    /**
     * AJAX handler for getting queue status
     * 
     * @since 1.2.0.1
     * @version 1.2.2.0 - Added response header
     */
    public function ajax_get_queue_status() {
        // Add response header
        @header('Content-Type: application/json; charset=utf-8');
        
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'delcampe_admin_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'wc-delcampe-integration')));
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'wc-delcampe-integration')));
        }
        
        // Get queue status
        $status = $this->get_queue_status();
        
        // Add next run time for queue processor (prefer new v2 hook, fallback to legacy)
        $next_run = wp_next_scheduled('delcampe_process_queue');
        if (!$next_run) {
            $next_run = wp_next_scheduled('delcampe_process_sync_queue');
        }
        if ($next_run) {
            $status['next_run'] = sprintf(
                __('%s (in %s)', 'wc-delcampe-integration'),
                date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $next_run),
                human_time_diff(time(), $next_run)
            );
        } else {
            // If Action Scheduler is available, check for a scheduled async action
            if (function_exists('as_next_scheduled_action')) {
                $as_next = as_next_scheduled_action('delcampe_process_queue', array(), 'delcampe');
                if (!$as_next) {
                    $as_next = as_next_scheduled_action('delcampe_process_sync_queue', array(), 'delcampe');
                }
                if ($as_next) {
                    $status['next_run'] = sprintf(
                        __('AS: %s (in %s)', 'wc-delcampe-integration'),
                        date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $as_next),
                        human_time_diff(time(), $as_next)
                    );
                } else {
                    $status['next_run'] = __('Not scheduled', 'wc-delcampe-integration');
                }
            } else {
                $status['next_run'] = __('Not scheduled', 'wc-delcampe-integration');
            }
        }

        // Also report Master Sync (orders/listings/inventory) next run based on Advanced settings
        $master_next = wp_next_scheduled('delcampe_master_sync');
        if ($master_next) {
            $status['master_next_run'] = sprintf(
                __('%s (in %s)', 'wc-delcampe-integration'),
                date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $master_next),
                human_time_diff(time(), $master_next)
            );
        } else {
            $status['master_next_run'] = __('Not scheduled', 'wc-delcampe-integration');
        }
        
        wp_send_json_success($status);
    }
    
    /**
     * AJAX handler for clearing the sync queue
     * 
     * @since 1.2.0.1
     * @version 1.2.2.0 - Added response header
     */
    public function ajax_clear_queue() {
        // Add response header
        @header('Content-Type: application/json; charset=utf-8');
        
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'delcampe_admin_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'wc-delcampe-integration')));
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'wc-delcampe-integration')));
        }
        
        // Clear queue
        $this->clear_queue();
        
        wp_send_json_success(array(
            'message' => __('Queue cleared successfully.', 'wc-delcampe-integration')
        ));
    }

    /**
     * AJAX handler to forcibly clear the sync lock
     */
    public function ajax_clear_sync_lock() {
        // Add response header
        @header('Content-Type: application/json; charset=utf-8');
        
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'delcampe_admin_nonce')) {
            wp_send_json_error(array('message' => __('Security check failed.', 'wc-delcampe-integration')));
        }
        
        // Check permissions
        if (!current_user_can('manage_options')) {
            wp_send_json_error(array('message' => __('Insufficient permissions.', 'wc-delcampe-integration')));
        }
        
        $had_lock = get_transient(self::SYNC_LOCK_TRANSIENT);
        delete_transient(self::SYNC_LOCK_TRANSIENT);
        delete_transient(self::SYNC_LOCK_TRANSIENT . '_at');
        
        // Schedule an immediate run after clearing
        $this->schedule_queue_run(0, 'manual-clear-lock');
        
        wp_send_json_success(array(
            'message' => $had_lock ? __('Sync lock cleared. Queue run triggered.', 'wc-delcampe-integration') : __('No lock was set. Queue run triggered.', 'wc-delcampe-integration')
        ));
    }
    
    /**
     * Get queue status
     * 
     * @since 1.2.0.0
     * @return array Queue status
     */
    public function get_queue_status() {
        // Check if new queue system v2.0 is available
        if (class_exists('Delcampe_Queue')) {
            $queue = Delcampe_Queue::get_instance();
            $stats = $queue->get_stats();
            // Interpret UI "Total items" as items still in flight (pending + processing)
            $in_flight = (int)$stats['pending'] + (int)$stats['processing'];
            
            // Get current processing item
            $current_processing = get_transient('delcampe_current_processing');
            
            $status = array(
                'total' => $in_flight,
                'pending' => (int)$stats['pending'],
                'failed' => (int)$stats['errored'],
                'processing' => ((int)$stats['processing']) > 0,
                'queued' => (int)$stats['pending'],
                'done' => (int)$stats['published'],
                'errored' => (int)$stats['errored']
            );
            
            // Add current processing item details
            if ($current_processing && is_array($current_processing)) {
                $status['current_item'] = array(
                    'sku' => $current_processing['sku'],
                    'title' => $current_processing['title'],
                    'action' => $current_processing['action'],
                    'elapsed' => time() - $current_processing['timestamp']
                );
            }
            
            return $status;
        }
        
        // Fallback to old sync queue if new system not available
        if (get_option('delcampe_queue_v2_enabled', 'yes') === 'yes' && class_exists('Delcampe_Sync_Queue')) {
            global $wpdb;
            $table = $wpdb->prefix . 'delcampe_sync_queue';
            $total = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table} WHERE status IN ('queued','retry','processing')");
            $pending = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table} WHERE status IN ('queued','retry')");
            $failed = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table} WHERE status = 'failed'");
            $processing = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table} WHERE status = 'processing'") > 0;
            return array(
                'total' => $total,
                'pending' => $pending,
                'failed' => $failed,
                'processing' => $processing,
            );
        }

        $queue = get_option(self::SYNC_QUEUE_OPTION, array());
        $status = array(
            'total' => count($queue),
            'pending' => 0,
            'failed' => 0,
            'processing' => get_transient(self::SYNC_LOCK_TRANSIENT) ? true : false
        );
        foreach ($queue as $item) {
            if ($item['attempts'] >= 3) {
                $status['failed']++;
            } else {
                $status['pending']++;
            }
        }
        return $status;
    }

    /**
     * Count claimable (ready-to-run) pending items in new queue.
     * Uses next_attempt_at to exclude backoff windows.
     */
    private function count_claimable_pending() {
        if (!class_exists('Delcampe_Queue')) { return 0; }
        global $wpdb;
        $table = $wpdb->prefix . 'delcampe_queue';
        $count = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$table} WHERE state='pending' AND (next_attempt_at IS NULL OR next_attempt_at <= NOW())");
        return $count;
    }
    
    /**
     * Clear sync queue
     * 
     * @since 1.2.0.0
     * @return bool Success
     */
    public function clear_queue() {
        // Use new Queue System v2.0 if available
        if (class_exists('Delcampe_Queue')) {
            $queue = Delcampe_Queue::get_instance();
            $cleaned = $queue->cleanup(0); // Clear all items
            
            delcampe_log("[Sync Handler] Cleared {$cleaned} items from new queue system");
            return true;
        }
        
        // Fallback to old system
        // Get current queue size before clearing
        $queue = get_option(self::SYNC_QUEUE_OPTION, array());
        $queue_size = count($queue);
        
        // Clear the queue
        update_option(self::SYNC_QUEUE_OPTION, array());
        
        // Clear the sync lock
        $had_lock = get_transient(self::SYNC_LOCK_TRANSIENT);
        delete_transient(self::SYNC_LOCK_TRANSIENT);
        delete_transient(self::SYNC_LOCK_TRANSIENT . '_at');
        
        $this->debug_log('[Clear Queue] Queue cleared - removed ' . $queue_size . ' items. Lock status: ' . ($had_lock ? 'was locked, now cleared' : 'was not locked'));
        
        return true;
    }
    
    /**
     * Queue all products with profiles
     * 
     * @since 1.2.0.0
     * @return int Number of products queued
     */
    public function queue_all_products() {
        global $wpdb;
        
        // Get all products with profiles
        $products = $wpdb->get_results(
            "SELECT post_id, meta_value as profile_id 
             FROM {$wpdb->postmeta} 
             WHERE meta_key = '_delcampe_profile_id' 
             AND meta_value != ''"
        );
        
        $count = 0;
        foreach ($products as $product) {
            // Check if already has listing
            $listing_id = get_post_meta($product->post_id, '_delcampe_listing_id', true);
            $action = $listing_id ? 'update' : 'create';
            
            $this->add_to_queue($product->post_id, $product->profile_id, $action);
            $count++;
        }
        
        $this->debug_log('[Queue All] Queued ' . $count . ' products for sync');
        
        return $count;
    }
    
    /**
     * AJAX handler for checking listing status on Delcampe
     * SECURITY FIX v1.10.21.0: Enhanced authorization and validation
     */
    public function ajax_check_listing_status() {
        // Verify nonce
        if (!isset($_POST['nonce']) || !wp_verify_nonce($_POST['nonce'], 'delcampe_check_status')) {
            wp_send_json_error('Invalid nonce');
        }
        
        // SECURITY FIX v1.10.21.0: Enhanced permission check
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
        }
        
        // Get product reference
        if (!isset($_POST['reference'])) {
            wp_send_json_error('Missing product reference');
        }
        
        $reference = sanitize_text_field($_POST['reference']);
        
        // SECURITY FIX v1.10.21.0: Validate reference format and ownership
        // Reference should be in format: SKU or product ID
        if (!preg_match('/^[A-Za-z0-9\-_]+$/', $reference)) {
            wp_send_json_error('Invalid reference format');
        }
        
        // If reference is numeric, verify product ownership/visibility
        if (is_numeric($reference)) {
            $product = wc_get_product($reference);
            if (!$product || $product->get_status() === 'trash') {
                wp_send_json_error('Product not found or inaccessible');
            }
        }
        
        $this->debug_log('[AJAX Check Status] Checking status for reference: ' . $reference);
        
        // Get API instance
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-listing-api.php';
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-auth.php';
        
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (!$token) {
            wp_send_json_error('Not authenticated');
        }
        
        // Query Delcampe API for listings
        // Note: Delcampe API uses token in query params, not Authorization header
        $query_params = array(
            'token' => $token,
            'startingItem' => 0,
            'numberOfItems' => 100  // Get first 100 listings to check
        );
        
        $api_url = DELCAMPE_API_BASE_URL . '/item/opened?' . http_build_query($query_params);
        
        $response = wp_remote_get($api_url, array(
            'headers' => array(
                'Accept' => 'application/xml'
            ),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            $this->debug_log('[AJAX Check Status] API error: ' . $response->get_error_message());
            wp_send_json_error('API request failed: ' . $response->get_error_message());
        }
        
        $body = wp_remote_retrieve_body($response);
        $this->debug_log('[AJAX Check Status] API response received, checking for reference: ' . $reference);
        $this->debug_log('[AJAX Check Status] Response body length: ' . strlen($body));
        
        // Log first 500 chars of response for debugging
        if (!empty($body)) {
            $this->debug_log('[AJAX Check Status] Response preview: ' . substr($body, 0, 500));
        }
        
        // Parse XML response
        libxml_use_internal_errors(true);
        $xml = simplexml_load_string($body);
        
        if (!$xml) {
            $errors = libxml_get_errors();
            $error_msg = 'XML parsing failed';
            if (!empty($errors)) {
                $error_msg .= ': ' . $errors[0]->message;
            }
            $this->debug_log('[AJAX Check Status] ' . $error_msg);
            libxml_clear_errors();
            wp_send_json_error($error_msg);
        }
        
        // Check for API errors
        if (isset($xml->Notification_Data->body->error)) {
            $error_msg = (string)$xml->Notification_Data->body->error;
            $this->debug_log('[AJAX Check Status] API error: ' . $error_msg);
            wp_send_json_error('API error: ' . $error_msg);
        }
        
        // Search for our product
        $found = false;
        $delcampe_id = null;
        
        // The API returns items wrapped in Notification_Data->body
        if (isset($xml->Notification_Data->body->item)) {
            foreach ($xml->Notification_Data->body->item as $item) {
                if ((string)$item->personal_reference === $reference) {
                    $found = true;
                    $delcampe_id = (string)$item->id_item;  // Note: it's id_item not id
                    break;
                }
            }
        }
        
        if ($found && $delcampe_id) {
            $this->debug_log('[AJAX Check Status] Found listing on Delcampe! ID: ' . $delcampe_id);
            
            // Update our database
            global $wpdb;
            $listings_table = $wpdb->prefix . 'delcampe_listings';
            
            $updated = $wpdb->update(
                $listings_table,
                array('delcampe_id' => $delcampe_id),
                array('personal_reference' => $reference)
            );
            
            if ($updated !== false) {
                wp_send_json_success(array(
                    'found' => true,
                    'status' => 'live',
                    'delcampe_id' => $delcampe_id,
                    'message' => 'Listing is live on Delcampe!'
                ));
            } else {
                wp_send_json_error('Failed to update database');
            }
        } else {
            $this->debug_log('[AJAX Check Status] Listing not found on Delcampe yet');
            wp_send_json_success(array(
                'found' => false,
                'status' => 'processing',  // UI status only - not stored in DB
                'message' => 'Still processing on Delcampe'
            ));
        }
    }
    
    /**
     * Check status of processing listings automatically
     *
     * @since 1.10.8.0
     */
    public function check_processing_listings() {
        global $wpdb;
        $listings_table = $wpdb->prefix . 'delcampe_listings';
        
        $this->debug_log('[Auto Status Check] Starting automatic status check for processing listings');
        
        // Get all listings in 'processing' status
        $processing_listings = $wpdb->get_results(
            "SELECT * FROM {$listings_table} 
             WHERE status = 'processing' 
             AND date_created > DATE_SUB(NOW(), INTERVAL 24 HOUR)
             ORDER BY date_created ASC
             LIMIT 10"
        );
        
        if (empty($processing_listings)) {
            $this->debug_log('[Auto Status Check] No processing listings found');
            return;
        }
        
        $this->debug_log('[Auto Status Check] Found ' . count($processing_listings) . ' processing listings to check');
        
        // Get auth instance for authentication
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-auth.php';
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (!$token || is_wp_error($token)) {
            $this->debug_log('[Auto Status Check] Authentication failed: Unable to get auth token');
            return;
        }
        
        // Get all opened items from Delcampe using direct API call
        $api_url = DELCAMPE_API_BASE_URL . '/item/opened?token=' . urlencode($token);
        $response = wp_remote_get($api_url, array(
            'headers' => array('Accept' => 'application/xml'),
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            $this->debug_log('[Auto Status Check] Failed to get opened items: ' . $response->get_error_message());
            return;
        }
        
        $body = wp_remote_retrieve_body($response);
        libxml_use_internal_errors(true);
        $xml = simplexml_load_string($body);
        
        if (!$xml) {
            $this->debug_log('[Auto Status Check] Failed to parse API response');
            return;
        }
        
        // Extract items from XML response
        $items = array();
        if (isset($xml->Notification_Data->body->item)) {
            foreach ($xml->Notification_Data->body->item as $item) {
                $items[] = array(
                    'id' => (string)$item->id_item,
                    'personal_reference' => (string)$item->personal_reference
                );
            }
        }
        $updated_count = 0;
        
        foreach ($processing_listings as $listing) {
            $reference = $listing->personal_reference;
            $found = false;
            $delcampe_id = null;
            
            // Search for this listing in opened items
            foreach ($items as $item) {
                if (isset($item['personal_reference']) && $item['personal_reference'] === $reference) {
                    $found = true;
                    $delcampe_id = $item['id'];
                    break;
                }
            }
            
            if ($found && $delcampe_id) {
                $this->debug_log('[Auto Status Check] Found listing ' . $reference . ' on Delcampe! ID: ' . $delcampe_id);
                
                // Update status to 'published' and store Delcampe ID
                $updated = $wpdb->update(
                    $listings_table,
                    array(
                        'delcampe_id' => $delcampe_id,
                        'status' => 'published',
                        'updated_at' => current_time('mysql')
                    ),
                    array('id' => $listing->id)
                );
                
                if ($updated !== false) {
                    $updated_count++;
                    
                    // Update product meta
                    if ($listing->product_id) {
                        update_post_meta($listing->product_id, '_delcampe_id', $delcampe_id);
                        update_post_meta($listing->product_id, '_delcampe_sync_status', 'published');
                    }
                }
            } else {
                // Check if listing has been processing for too long (over 1 hour)
                $created_time = strtotime($listing->date_created);
                $current_time = time();
                $time_diff = $current_time - $created_time;
                
                if ($time_diff > 3600) { // 1 hour
                    $this->debug_log('[Auto Status Check] Listing ' . $reference . ' has been processing for over 1 hour, marking as failed');
                    
                    $wpdb->update(
                        $listings_table,
                        array(
                            'status' => 'failed',
                            'error_message' => 'Listing not found on Delcampe after 1 hour',
                            'updated_at' => current_time('mysql')
                        ),
                        array('id' => $listing->id)
                    );
                    
                    if ($listing->product_id) {
                        update_post_meta($listing->product_id, '_delcampe_sync_status', 'failed');
                    }
                }
            }
        }
        
        $this->debug_log('[Auto Status Check] Updated ' . $updated_count . ' listings to published status');
    }
    
    /**
     * Cron handler for syncing closed listings from Delcampe
     * 
     * Runs every 15 minutes to check for listings that have been closed on Delcampe
     * and updates their status in the local database
     * 
     * @since 1.10.18.8
     */
    public function sync_closed_listings_cron() {
        $this->debug_log('[Listings Sync] Starting scheduled sync of open and closed listings');
        
        // Load the sync class
        require_once plugin_dir_path(dirname(__FILE__)) . 'includes/class-delcampe-listings-sync.php';
        
        $sync = Delcampe_Listings_Sync::get_instance();
        
        // First sync open listings to catch any new ones
        $this->debug_log('[Listings Sync] Syncing open listings...');
        $open_result = $sync->sync_open_listings(100, 5);
        
        if (is_wp_error($open_result)) {
            $this->debug_log('[Listings Sync] Open sync error: ' . $open_result->get_error_message());
        } else {
            $this->debug_log('[Listings Sync] Open listings synced: ' . count($open_result));
        }
        
        // Then sync closed listings to mark ended ones
        $this->debug_log('[Listings Sync] Syncing closed listings...');
        $closed_result = $sync->sync_closed_listings(100, 5);
        
        if (is_wp_error($closed_result)) {
            $this->debug_log('[Listings Sync] Closed sync error: ' . $closed_result->get_error_message());
        } else {
            $this->debug_log('[Listings Sync] Closed listings sync completed');
        }
        
        $this->debug_log('[Listings Sync] Full sync completed');
    }
}

// Initialize the sync handler
Delcampe_Sync_Handler::get_instance();
