<?php
/**
 * Delcampe Table-backed FIFO Sync Queue
 *
 * Minimal v1 scaffold: enqueue, claim, process using existing handler logic.
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Sync_Queue {
    private static $instance = null;
    private $table;
    const STATUS_QUEUED = 'queued';
    const STATUS_PROCESSING = 'processing';
    const STATUS_DONE = 'done';
    const STATUS_FAILED = 'failed';
    const STATUS_RETRY = 'retry';

    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        global $wpdb;
        $this->table = $wpdb->prefix . 'delcampe_sync_queue';

        add_action('init', [$this, 'maybe_create_table']);
        if (did_action('init')) {
            $this->maybe_create_table();
        }
        add_action('delcampe_process_sync_queue', [$this, 'process_queue']);
        add_action('delcampe_recover_sync_queue', [$this, 'recover_stuck']);
        add_action('init', [$this, 'schedule_cron'], 20);
        if (did_action('init')) {
            $this->schedule_cron();
        }
    }

    public function maybe_create_table() {
        global $wpdb;
        $charset = $wpdb->get_charset_collate();
        $sql = "CREATE TABLE IF NOT EXISTS {$this->table} (
            id BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
            product_id BIGINT UNSIGNED NOT NULL,
            profile_id BIGINT UNSIGNED NOT NULL,
            action VARCHAR(20) NOT NULL,
            status VARCHAR(20) NOT NULL DEFAULT 'queued',
            attempts INT NOT NULL DEFAULT 0,
            last_error TEXT NULL,
            next_attempt_at DATETIME NULL,
            priority INT NOT NULL DEFAULT 0,
            batch_id VARCHAR(64) NULL,
            created_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            updated_at DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY status_idx (status, next_attempt_at, priority, id),
            KEY product_idx (product_id),
            KEY batch_idx (batch_id)
        ) {$charset};";
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta($sql);
    }

    public function schedule_cron() {
        // Ensure processors exist (same hook as legacy for compatibility)
        if (!wp_next_scheduled('delcampe_process_sync_queue')) {
            wp_schedule_event(time() + 30, 'delcampe_sync_interval', 'delcampe_process_sync_queue');
        }
        if (!wp_next_scheduled('delcampe_recover_sync_queue')) {
            wp_schedule_event(time() + 300, 'delcampe_stuck_check_interval', 'delcampe_recover_sync_queue');
        }
    }

    public function enqueue($product_id, $profile_id, $action = 'create', $priority = 0, $batch_id = null) {
        global $wpdb;
        // Guard: ensure table exists
        $this->maybe_create_table();
        return (bool) $wpdb->insert($this->table, [
            'product_id' => intval($product_id),
            'profile_id' => intval($profile_id),
            'action' => sanitize_text_field($action),
            'status' => self::STATUS_QUEUED,
            'priority' => intval($priority),
            'batch_id' => $batch_id ? sanitize_text_field($batch_id) : null,
        ]);
    }

    private function claim_next_chunk($limit = 50) {
        global $wpdb;
        $wpdb->query('SET TRANSACTION ISOLATION LEVEL READ COMMITTED');
        $wpdb->query('START TRANSACTION');
        try {
            $stale = max(30, intval(apply_filters('delcampe_sync_stuck_threshold', 120)));
            $sql = $wpdb->prepare(
                "SELECT id FROM {$this->table}
                 WHERE (
                    status = %s
                    OR (status = %s AND (next_attempt_at IS NULL OR next_attempt_at <= NOW()))
                    OR (status = %s AND updated_at < DATE_SUB(NOW(), INTERVAL %d SECOND))
                 )
                 ORDER BY priority DESC, id ASC
                 LIMIT %d
                 FOR UPDATE SKIP LOCKED",
                self::STATUS_QUEUED, self::STATUS_RETRY, self::STATUS_PROCESSING, $stale, $limit
            );
            $rows = $wpdb->get_col($sql);
            if (empty($rows)) {
                $wpdb->query('COMMIT');
                return [];
            }
            $ids_in = implode(',', array_map('intval', $rows));
            $wpdb->query("UPDATE {$this->table} SET status='" . self::STATUS_PROCESSING . "', updated_at=NOW() WHERE id IN ({$ids_in})");
            $wpdb->query('COMMIT');

            // Load claimed items fully
            $items = $wpdb->get_results("SELECT * FROM {$this->table} WHERE id IN ({$ids_in}) ORDER BY priority DESC, id ASC");
            return $items;
        } catch (Exception $e) {
            $wpdb->query('ROLLBACK');
            return [];
        }
    }

    public function process_queue($force_reclaim = false) {
        // allow pausing via transient
        if (get_transient('delcampe_sync_queue_paused')) {
            return;
        }
        // Recovery: requeue processing
        if ($force_reclaim) {
            // Aggressively reclaim all processing items for manual runs
            $this->recover_stuck_threshold(0);
        } else {
            // Reclaim only stale ones (2 minutes)
            $this->recover_stuck_threshold(120);
        }
        $limit = apply_filters('delcampe_sync_queue_chunk', 50);
        $chunk = $this->claim_next_chunk($limit);
        if (function_exists('delcampe_log')) {
            delcampe_log('[SyncV2] Claimed ' . count($chunk) . ' item(s) for processing (limit ' . $limit . ')', 'info');
        }
        if (empty($chunk)) {
            return;
        }
        $handler = Delcampe_Sync_Handler::get_instance();
        foreach ($chunk as $row) {
            try {
                $item = [
                    'product_id' => intval($row->product_id),
                    'profile_id' => intval($row->profile_id),
                    'action' => $row->action,
                ];
                $result = $handler->process_item_public($item);

                if ($result === true) {
                    $this->mark_done($row->id);
                    if (function_exists('delcampe_log')) {
                        delcampe_log('[SyncV2] Success product ' . $row->product_id . ' action ' . $row->action, 'info');
                    }
                } else {
                    $this->mark_error_or_retry($row, $result);
                }
            } catch (\Throwable $e) {
                $this->mark_error_or_retry($row, new WP_Error('exception', $e->getMessage()));
            }
        }
        // If items remain, schedule a quick follow-up
        if ($this->has_pending()) {
            if (function_exists('as_enqueue_async_action')) {
                as_enqueue_async_action('delcampe_process_sync_queue', [], 'delcampe');
            } else {
                wp_schedule_single_event(time() + 5, 'delcampe_process_sync_queue');
            }
        }
    }

    private function mark_done($id) {
        global $wpdb;
        $wpdb->update($this->table, [
            'status' => self::STATUS_DONE,
            'updated_at' => current_time('mysql'),
        ], ['id' => intval($id)]);
    }

    private function mark_error_or_retry($row, $result) {
        global $wpdb;
        $attempts = intval($row->attempts) + 1;
        $is_wp_error = is_wp_error($result);
        $message = $is_wp_error ? $result->get_error_message() : (is_string($result) ? $result : 'Unknown error');

        // Retry policy: permanent vs transient
        $permanent = $is_wp_error && $result->get_error_code() && !in_array($result->get_error_code(), ['http_429','http_5xx','timeout'], true);
        if ($permanent || $attempts >= 8) {
            $wpdb->update($this->table, [
                'status' => self::STATUS_FAILED,
                'attempts' => $attempts,
                'last_error' => $message,
                'updated_at' => current_time('mysql'),
            ], ['id' => intval($row->id)]);
            if (function_exists('delcampe_log')) {
                delcampe_log('[SyncV2] FAILED product ' . $row->product_id . ' action ' . $row->action . ' attempts ' . $attempts . ' error: ' . $message, 'error');
            }
        } else {
            $delays = [60, 300, 900, 3600, 7200, 14400, 28800, 86400];
            $idx = min($attempts - 1, count($delays) - 1);
            $next = date('Y-m-d H:i:s', time() + $delays[$idx] + rand(0, 30));
            $wpdb->update($this->table, [
                'status' => self::STATUS_RETRY,
                'attempts' => $attempts,
                'last_error' => $message,
                'next_attempt_at' => $next,
                'updated_at' => current_time('mysql'),
            ], ['id' => intval($row->id)]);
            if (function_exists('delcampe_log')) {
                delcampe_log('[SyncV2] RETRY product ' . $row->product_id . ' action ' . $row->action . ' attempts ' . $attempts . ' next at ' . $next . ' reason: ' . $message, 'warning');
            }
        }
    }

    public function has_pending() {
        global $wpdb;
        return (int) $wpdb->get_var("SELECT COUNT(*) FROM {$this->table} WHERE status IN ('queued','retry')") > 0;
    }

    public function recover_stuck() {
        global $wpdb;
        // Any processing older than 10 minutes goes back to retry
        $wpdb->query("UPDATE {$this->table} SET status='retry', next_attempt_at=NOW(), updated_at=NOW()
                      WHERE status='processing' AND updated_at < DATE_SUB(NOW(), INTERVAL 10 MINUTE)");
    }

    /**
     * Recover processing rows older than a threshold (seconds)
     */
    private function recover_stuck_threshold($seconds) {
        global $wpdb;
        $seconds = intval($seconds);
        if ($seconds <= 0) {
            // Reclaim all processing rows
            $wpdb->query(
                "UPDATE {$this->table}
                 SET status='retry', next_attempt_at=NOW(), updated_at=NOW()
                 WHERE status='processing'"
            );
        } else {
            $wpdb->query($wpdb->prepare(
                "UPDATE {$this->table} 
                 SET status='retry', next_attempt_at=NOW(), updated_at=NOW()
                 WHERE status='processing' AND updated_at < DATE_SUB(NOW(), INTERVAL %d SECOND)",
                $seconds
            ));
        }
    }
}
