<?php
/**
 * Delcampe Webhook Guardian
 * 
 * Production-safe webhook management that prevents duplicates
 * - Ensures only one active registration per (channel,type,destination)
 * - Prevents auto re-adding by using normalized equality + shared token
 * - Schedules a daily ensure job (no CLI required)
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.10.14.0
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Webhook_Guardian {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Essential webhook types (without Curl_ prefix)
     */
    const ESSENTIAL_TYPES = array(
        'Seller_Item_Close_Sold',
        'Seller_Payment_Received'
    );
    
    /**
     * Recommended webhook types for catalog sync
     */
    const RECOMMENDED_TYPES = array(
        'Seller_Item_Update',
        'Seller_Item_Close_Unsold', 
        'Seller_Item_Close_Manually',
        'Seller_Item_Restart',
        'Seller_Item_Move'
    );
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Initialize shared token if not exists
        if (!get_option('delcampe_webhook_token')) {
            add_option('delcampe_webhook_token', wp_generate_password(32, false, false));
        }
        
        // Initialize auto-register setting - DISABLED by default to prevent duplicates
        if (!get_option('delcampe_auto_register_webhooks')) {
            add_option('delcampe_auto_register_webhooks', 'no');
        }
        
        // DISABLED: Automatic webhook checks to prevent duplicates
        // Webhooks should only be created once, manually by the user
        // add_action('init', array($this, 'schedule_daily_check'));
        // add_action('delcampe_daily_webhook_check', array($this, 'ensure_webhooks_and_cleanup'));
        
        // Hook for plugin activation (using main plugin file)
        $plugin_file = dirname(dirname(__FILE__)) . '/main.php';
        register_activation_hook($plugin_file, array($this, 'on_activation'));
        
        // Manual fix endpoint
        add_action('admin_init', array($this, 'handle_manual_fix'));
    }
    
    /**
     * Schedule daily webhook check - DISABLED to prevent automatic webhook creation
     * Webhooks should only be created manually by the user
     */
    public function schedule_daily_check() {
        // DISABLED: Remove any existing scheduled events
        $timestamp = wp_next_scheduled('delcampe_daily_webhook_check');
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'delcampe_daily_webhook_check');
            delcampe_log('[Webhook Guardian] Removed automatic daily webhook check');
        }
    }
    
    /**
     * Run on plugin activation
     */
    public function on_activation() {
        // DISABLED: No automatic webhook checks on activation
        // Webhooks should only be created manually by the user
        // Clear any existing scheduled events
        $timestamp = wp_next_scheduled('delcampe_daily_webhook_check');
        if ($timestamp) {
            wp_unschedule_event($timestamp, 'delcampe_daily_webhook_check');
        }
        
        delcampe_log('[Webhook Guardian] Plugin activated - automatic webhook registration disabled');
    }
    
    /**
     * Normalize URL for comparison
     * Lowercase scheme & host, strip default port, remove trailing slash,
     * sort query params for stable equality
     */
    private function normalize_url($url) {
        $parts = wp_parse_url($url);
        if (!$parts || empty($parts['host'])) {
            return $url;
        }
        
        $scheme = isset($parts['scheme']) ? strtolower($parts['scheme']) : 'https';
        $host = strtolower($parts['host']);
        $port = isset($parts['port']) ? (int)$parts['port'] : null;
        $path = isset($parts['path']) ? rtrim($parts['path'], '/') : '';
        $query_params = array();
        
        if (!empty($parts['query'])) {
            parse_str($parts['query'], $query_params);
            ksort($query_params); // Sort for stable equality
        }
        
        // Strip default port
        $port_str = '';
        if ($port && !(($scheme === 'https' && $port === 443) || ($scheme === 'http' && $port === 80))) {
            $port_str = ':' . $port;
        }
        
        $query = http_build_query($query_params, '', '&', PHP_QUERY_RFC3986);
        return $scheme . '://' . $host . $port_str . $path . ($query ? '?' . $query : '');
    }
    
    /**
     * Build canonical webhook URL with stable token
     */
    private function build_webhook_url() {
        // Get stable token
        $token = get_option('delcampe_webhook_token');
        
        // Build webhook URL
        $url = home_url('/wp-json/delcampe/v1/webhook');
        
        // Force HTTPS as Delcampe requires it
        $url = str_replace('http://', 'https://', $url);
        
        // Add token
        $url = add_query_arg('token', $token, $url);
        
        return $this->normalize_url($url);
    }
    
    /**
     * List current webhook notifications
     */
    private function list_notifications() {
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return $token;
        }
        
        $url = DELCAMPE_API_URL . '/notification/settings?token=' . urlencode($token);
        
        $response = wp_remote_get($url, array(
            'timeout' => 20,
            'headers' => array('Accept' => 'application/xml')
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $xml = simplexml_load_string(wp_remote_retrieve_body($response));
        if (!$xml) {
            return new WP_Error('bad_xml', 'Cannot parse notification settings response');
        }
        
        $notifications = array();
        
        // Parse all notification_settings elements
        if (isset($xml->Notification_Data->body->notification_settings)) {
            foreach ($xml->Notification_Data->body->notification_settings as $setting) {
                $notifications[] = array(
                    'id' => (string)$setting->id_notification,
                    'type' => (string)$setting->type,
                    'channel' => (string)$setting->channel,
                    'destination' => (string)$setting->destination,
                    'active' => isset($setting->active) ? ((string)$setting->active === '1') : true
                );
            }
        }
        
        return $notifications;
    }
    
    /**
     * Register a webhook
     */
    private function register_webhook($type, $destination) {
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return $token;
        }
        
        $url = DELCAMPE_API_URL . '/notification/settings?token=' . urlencode($token);
        
        $payload = array(
            'notificationType' => 'Curl_' . $type,
            'destination' => $destination
        );
        
        $response = wp_remote_post($url, array(
            'timeout' => 20,
            'body' => $payload,
            'headers' => array('Accept' => 'application/xml')
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $code = wp_remote_retrieve_response_code($response);
        return ($code >= 200 && $code < 300);
    }
    
    /**
     * Delete a webhook notification
     */
    private function delete_notification($id) {
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return $token;
        }
        
        $url = DELCAMPE_API_URL . '/notification/' . urlencode($id) . '?token=' . urlencode($token);
        
        $response = wp_remote_request($url, array(
            'method' => 'DELETE',
            'timeout' => 20,
            'headers' => array('Accept' => 'application/xml')
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $code = wp_remote_retrieve_response_code($response);
        return ($code >= 200 && $code < 300);
    }
    
    /**
     * Toggle webhook notification (enable/disable)
     */
    private function toggle_notification($id) {
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return $token;
        }
        
        $url = DELCAMPE_API_URL . '/notification/' . urlencode($id) . '?token=' . urlencode($token);
        
        $response = wp_remote_request($url, array(
            'method' => 'PUT',
            'timeout' => 20,
            'headers' => array('Accept' => 'application/xml')
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $code = wp_remote_retrieve_response_code($response);
        return ($code >= 200 && $code < 300);
    }
    
    /**
     * Main function to ensure webhooks and cleanup duplicates
     */
    public function ensure_webhooks_and_cleanup() {
        // Check if auto-register is enabled (defaults to 'no' to prevent duplicates)
        if (get_option('delcampe_auto_register_webhooks', 'no') !== 'yes') {
            delcampe_log('[Webhook Guardian] Auto-registration disabled (webhooks should only be created manually)');
            return;
        }
        
        // Use transient lock to prevent concurrent runs
        if (get_transient('delcampe_webhook_guard_lock')) {
            delcampe_log('[Webhook Guardian] Another process is already running');
            return;
        }
        
        // Set lock for 60 seconds
        set_transient('delcampe_webhook_guard_lock', 1, 60);
        
        try {
            // Get all required types
            $required_types = self::ESSENTIAL_TYPES;
            $canonical_url = $this->build_webhook_url();
            $canonical_normalized = $this->normalize_url($canonical_url);
            
            delcampe_log('[Webhook Guardian] Starting check. Canonical URL: ' . $canonical_url);
            
            // 1. List current notifications
            $notifications = $this->list_notifications();
            if (is_wp_error($notifications)) {
                delcampe_log('[Webhook Guardian] Failed to list notifications: ' . $notifications->get_error_message());
                return;
            }
            
            delcampe_log('[Webhook Guardian] Found ' . count($notifications) . ' existing notifications');
            
            // 2. Group by normalized identity (channel|type|destination_normalized)
            $groups = array();
            foreach ($notifications as $notification) {
                $dest_normalized = $this->normalize_url($notification['destination']);
                $key = implode('|', array(
                    $notification['channel'],
                    $notification['type'],
                    $dest_normalized
                ));
                
                if (!isset($groups[$key])) {
                    $groups[$key] = array();
                }
                $groups[$key][] = $notification;
            }
            
            // 3. Ensure each required type exists exactly once
            foreach ($required_types as $type) {
                $key = implode('|', array('Curl', $type, $canonical_normalized));
                $existing = isset($groups[$key]) ? $groups[$key] : array();
                
                delcampe_log('[Webhook Guardian] Type ' . $type . ' has ' . count($existing) . ' registrations');
                
                if (count($existing) === 0) {
                    // None exist - register one
                    delcampe_log('[Webhook Guardian] Registering missing webhook: ' . $type);
                    $result = $this->register_webhook($type, $canonical_url);
                    
                    if ($result === true) {
                        delcampe_log('[Webhook Guardian] Successfully registered ' . $type);
                    } else {
                        delcampe_log('[Webhook Guardian] Failed to register ' . $type);
                    }
                    
                } elseif (count($existing) > 1) {
                    // Multiple exist - keep first active one, remove others
                    delcampe_log('[Webhook Guardian] Found duplicates for ' . $type . ', cleaning up...');
                    
                    // Sort: active first, then by ID
                    usort($existing, function($a, $b) {
                        if ($a['active'] === $b['active']) {
                            return (int)$a['id'] <=> (int)$b['id'];
                        }
                        return $a['active'] ? -1 : 1;
                    });
                    
                    // Keep first one
                    $keeper = array_shift($existing);
                    delcampe_log('[Webhook Guardian] Keeping webhook ID ' . $keeper['id']);
                    
                    // Remove duplicates
                    foreach ($existing as $duplicate) {
                        delcampe_log('[Webhook Guardian] Removing duplicate ID ' . $duplicate['id']);
                        
                        $deleted = $this->delete_notification($duplicate['id']);
                        if ($deleted !== true) {
                            // Fallback: disable it
                            delcampe_log('[Webhook Guardian] DELETE failed, trying to disable ID ' . $duplicate['id']);
                            $this->toggle_notification($duplicate['id']);
                        }
                    }
                }
            }
            
            delcampe_log('[Webhook Guardian] Check complete');
            
        } finally {
            // Always release lock
            delete_transient('delcampe_webhook_guard_lock');
        }
    }
    
    /**
     * Handle manual fix request from admin
     */
    public function handle_manual_fix() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        if (!isset($_GET['delcampe_webhook_fix'], $_GET['_wpnonce'])) {
            return;
        }
        
        if ($_GET['delcampe_webhook_fix'] !== '1') {
            return;
        }
        
        if (!wp_verify_nonce($_GET['_wpnonce'], 'delcampe_webhook_fix')) {
            return;
        }
        
        // Run the cleanup
        $this->ensure_webhooks_and_cleanup();
        
        // Add admin notice
        add_action('admin_notices', function() {
            echo '<div class="notice notice-success is-dismissible">';
            echo '<p>Webhook cleanup completed! Check the logs for details.</p>';
            echo '</div>';
        });
        
        // Redirect to remove query args
        wp_safe_redirect(remove_query_arg(array('delcampe_webhook_fix', '_wpnonce')));
        exit;
    }
    
    /**
     * Get manual fix URL for admin
     */
    public static function get_manual_fix_url() {
        return wp_nonce_url(
            add_query_arg('delcampe_webhook_fix', '1', admin_url()),
            'delcampe_webhook_fix'
        );
    }
}