<?php
/**
 * Delcampe Webhook Manager
 * 
 * Manages webhook registration and configuration via Delcampe API
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.10.5.17
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Webhook_Manager {
    
    /**
     * Singleton instance
     * @var Delcampe_Webhook_Manager|null
     */
    private static $instance = null;
    
    /**
     * Get singleton instance
     * 
     * @return Delcampe_Webhook_Manager
     */
    public static function get_instance() {
        if (is_null(self::$instance)) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Add admin menu item for webhook configuration
        // Webhook menu disabled - functionality moved to Settings
        // add_action('admin_menu', array($this, 'add_webhook_menu'), 100);
        
        // Keep only the AJAX handler that exists in this class
        add_action('wp_ajax_delcampe_register_webhooks', array($this, 'ajax_register_webhooks'));
        // ajax_check_webhooks doesn't exist in this class - handled by new Webhook Registration class
        // add_action('wp_ajax_delcampe_check_webhooks', array($this, 'ajax_check_webhooks'));
        add_action('wp_ajax_delcampe_remove_webhook', array($this, 'ajax_remove_webhook'));
    }
    
    
    /**
     * Add webhook configuration menu
     */
    public function add_webhook_menu() {
        // Only add if user has permissions
        if (!current_user_can('manage_options')) {
            return;
        }
        
        add_submenu_page(
            'delcampe',
            __('Webhook Configuration', 'wc-delcampe-integration'),
            __('Webhooks', 'wc-delcampe-integration'),
            'manage_options',
            'delcampe-webhooks',
            array($this, 'render_webhook_page')
        );
    }
    
    /**
     * Get the webhook URL for this site
     * 
     * @return string
     */
    public function get_webhook_url() {
        $base_url = plugins_url('webhook/delcampe-callback.php', dirname(__FILE__));
        
        // Add a security token parameter
        $token = get_option('delcampe_webhook_token');
        if (!$token) {
            $token = wp_generate_password(32, false);
            update_option('delcampe_webhook_token', $token);
        }
        
        return add_query_arg('token', $token, $base_url);
    }
    
    /**
     * Register webhooks with Delcampe API
     * 
     * @param array $notification_types Array of notification types to register
     * @return array Results array with success/error messages
     */
    public function register_webhooks($notification_types = array()) {
        // Default notification types if none provided
        if (empty($notification_types)) {
            $notification_types = array(
                'Curl_Seller_Item_Add',
                'Curl_Seller_Item_Update',
                'Curl_Seller_Item_Close_Sold',
                'Curl_Seller_Item_Close_Unsold',
                'Curl_Seller_Item_Close_Manually',
                'Curl_Seller_Payment_Received'
            );
        }
        
        // Get authentication token
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return array(
                'success' => false,
                'message' => __('Failed to authenticate with Delcampe API', 'wc-delcampe-integration'),
                'error' => $token->get_error_message()
            );
        }
        
        $webhook_url = $this->get_webhook_url();
        $results = array();
        
        // Register each notification type
        foreach ($notification_types as $type) {
            $result = $this->register_single_webhook($token, $type, $webhook_url);
            $results[$type] = $result;
        }
        
        // Also register failover email
        $failover_email = get_option('delcampe_failover_email', get_option('admin_email'));
        $this->register_failover_email($token, $failover_email);
        
        return array(
            'success' => true,
            'webhook_url' => $webhook_url,
            'results' => $results
        );
    }
    
    /**
     * Register a single webhook notification
     * 
     * @param string $token API token
     * @param string $notification_type Notification type
     * @param string $destination Webhook URL
     * @return array Result
     */
    private function register_single_webhook($token, $notification_type, $destination) {
        $url = DELCAMPE_API_URL . '/notification/settings?token=' . $token;
        
        $postData = array(
            'notificationType' => $notification_type,
            'destination' => $destination
        );
        
        $response = wp_remote_post($url, array(
            'body' => $postData,
            'timeout' => 30,
            'headers' => array(
                'Accept' => 'application/xml'
            )
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error' => $response->get_error_message()
            );
        }
        
        $body = wp_remote_retrieve_body($response);
        $xml = simplexml_load_string($body);
        
        if ($xml && isset($xml->status) && (string)$xml->status === 'OK') {
            return array(
                'success' => true,
                'message' => sprintf(__('Webhook registered for %s', 'wc-delcampe-integration'), $notification_type)
            );
        }
        
        return array(
            'success' => false,
            'error' => isset($xml->e) ? (string)$xml->e : __('Unknown error', 'wc-delcampe-integration')
        );
    }
    
    /**
     * Register failover email address
     * 
     * @param string $token API token
     * @param string $email Email address
     * @return array Result
     */
    private function register_failover_email($token, $email) {
        return $this->register_single_webhook($token, 'Email_Error_LimitReached', $email);
    }
    
    /**
     * Get current webhook settings from Delcampe
     * 
     * @return array|WP_Error
     */
    public function get_webhook_settings() {
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return $token;
        }
        
        $url = DELCAMPE_API_URL . '/notification/settings?token=' . $token;
        
        $response = wp_remote_get($url, array(
            'timeout' => 30,
            'headers' => array(
                'Accept' => 'application/xml'
            )
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $body = wp_remote_retrieve_body($response);
        $xml = simplexml_load_string($body);
        
        if (!$xml) {
            return new WP_Error('parse_error', __('Failed to parse webhook settings', 'wc-delcampe-integration'));
        }
        
        $settings = array();
        
        // Parse notification settings
        if (isset($xml->notification)) {
            foreach ($xml->notification as $notification) {
                $settings[] = array(
                    'id' => (string)$notification->id,
                    'type' => (string)$notification->type,
                    'destination' => (string)$notification->destination,
                    'status' => (string)$notification->status
                );
            }
        }
        
        return $settings;
    }
    
    /**
     * Remove a webhook notification
     * 
     * @param int $notification_id Notification ID to remove
     * @return array Result
     */
    public function remove_webhook($notification_id) {
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return array(
                'success' => false,
                'error' => $token->get_error_message()
            );
        }
        
        $url = DELCAMPE_API_URL . '/notification/' . $notification_id . '?token=' . $token;
        
        $response = wp_remote_request($url, array(
            'method' => 'DELETE',
            'timeout' => 30,
            'headers' => array(
                'Accept' => 'application/xml'
            )
        ));
        
        if (is_wp_error($response)) {
            return array(
                'success' => false,
                'error' => $response->get_error_message()
            );
        }
        
        $body = wp_remote_retrieve_body($response);
        $xml = simplexml_load_string($body);
        
        if ($xml && isset($xml->status) && (string)$xml->status === 'OK') {
            return array(
                'success' => true,
                'message' => __('Webhook removed successfully', 'wc-delcampe-integration')
            );
        }
        
        return array(
            'success' => false,
            'error' => isset($xml->e) ? (string)$xml->e : __('Failed to remove webhook', 'wc-delcampe-integration')
        );
    }
    
    /**
     * Render webhook configuration page
     */
    public function render_webhook_page() {
        // Get current webhook settings
        $current_settings = $this->get_webhook_settings();
        $webhook_url = $this->get_webhook_url();
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('Delcampe Webhook Configuration', 'wc-delcampe-integration'); ?></h1>
            
            <div class="notice notice-info">
                <p><?php esc_html_e('Webhooks allow Delcampe to notify your store instantly when events occur.', 'wc-delcampe-integration'); ?></p>
            </div>
            
            <div class="card">
                <h2><?php esc_html_e('Your Webhook URL', 'wc-delcampe-integration'); ?></h2>
                <p>
                    <code style="background: #f0f0f0; padding: 10px; display: block; word-break: break-all;">
                        <?php echo esc_html($webhook_url); ?>
                    </code>
                </p>
                <p class="description">
                    <?php esc_html_e('This URL will receive notifications from Delcampe when events occur.', 'wc-delcampe-integration'); ?>
                </p>
            </div>
            
            <div class="card">
                <h2><?php esc_html_e('Webhook Registration', 'wc-delcampe-integration'); ?></h2>
                
                <table class="form-table">
                    <tr>
                        <th scope="row"><?php esc_html_e('Notification Types', 'wc-delcampe-integration'); ?></th>
                        <td>
                            <label><input type="checkbox" name="notification_types[]" value="Curl_Seller_Item_Add" checked> <?php esc_html_e('Item Added', 'wc-delcampe-integration'); ?></label><br>
                            <label><input type="checkbox" name="notification_types[]" value="Curl_Seller_Item_Update" checked> <?php esc_html_e('Item Updated', 'wc-delcampe-integration'); ?></label><br>
                            <label><input type="checkbox" name="notification_types[]" value="Curl_Seller_Item_Close_Sold" checked> <?php esc_html_e('Item Sold (Auction)', 'wc-delcampe-integration'); ?></label><br>
                            <label><input type="checkbox" name="notification_types[]" value="Curl_Seller_Item_Close_Unsold"> <?php esc_html_e('Item Closed Unsold', 'wc-delcampe-integration'); ?></label><br>
                            <label><input type="checkbox" name="notification_types[]" value="Curl_Seller_Item_Close_Manually"> <?php esc_html_e('Item Closed Manually', 'wc-delcampe-integration'); ?></label><br>
                            <label><input type="checkbox" name="notification_types[]" value="Curl_Seller_Payment_Received" checked> <?php esc_html_e('Payment Received', 'wc-delcampe-integration'); ?></label><br>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><?php esc_html_e('Failover Email', 'wc-delcampe-integration'); ?></th>
                        <td>
                            <input type="email" id="delcampe_failover_email" name="delcampe_failover_email" 
                                   value="<?php echo esc_attr(get_option('delcampe_failover_email', get_option('admin_email'))); ?>" 
                                   class="regular-text" />
                            <p class="description">
                                <?php esc_html_e('Email address to receive notifications if webhook fails 5 times.', 'wc-delcampe-integration'); ?>
                            </p>
                        </td>
                    </tr>
                </table>
                
                <p>
                    <button type="button" class="button button-primary" id="register-webhooks">
                        <?php esc_html_e('Register Webhooks', 'wc-delcampe-integration'); ?>
                    </button>
                    <span class="spinner" style="float: none;"></span>
                </p>
                
                <div id="webhook-result" style="margin-top: 20px;"></div>
            </div>
            
            <?php if (!is_wp_error($current_settings) && !empty($current_settings)): ?>
            <div class="card">
                <h2><?php esc_html_e('Current Webhook Registrations', 'wc-delcampe-integration'); ?></h2>
                <table class="wp-list-table widefat fixed striped">
                    <thead>
                        <tr>
                            <th><?php esc_html_e('Type', 'wc-delcampe-integration'); ?></th>
                            <th><?php esc_html_e('Destination', 'wc-delcampe-integration'); ?></th>
                            <th><?php esc_html_e('Status', 'wc-delcampe-integration'); ?></th>
                            <th><?php esc_html_e('Action', 'wc-delcampe-integration'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($current_settings as $setting): ?>
                        <tr>
                            <td><?php echo esc_html($setting['type']); ?></td>
                            <td><code><?php echo esc_html($setting['destination']); ?></code></td>
                            <td><?php echo esc_html($setting['status']); ?></td>
                            <td>
                                <button type="button" class="button button-small remove-webhook" data-id="<?php echo esc_attr($setting['id']); ?>">
                                    <?php esc_html_e('Remove', 'wc-delcampe-integration'); ?>
                                </button>
                            </td>
                        </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            </div>
            <?php endif; ?>
            
            <div class="card">
                <h2><?php esc_html_e('Testing Webhooks', 'wc-delcampe-integration'); ?></h2>
                <p><?php esc_html_e('You can test your webhook endpoint with this curl command:', 'wc-delcampe-integration'); ?></p>
                <pre style="background: #f0f0f0; padding: 10px; overflow-x: auto;">
curl -X POST <?php echo esc_html($webhook_url); ?> \
  -H "Content-Type: application/x-www-form-urlencoded" \
  -d 'delcampeNotification=<?php echo htmlspecialchars('<?xml version="1.0"?><notification><Notification_Type>Test</Notification_Type></notification>'); ?>'
                </pre>
            </div>
        </div>
        
        <script type="text/javascript">
        jQuery(document).ready(function($) {
            // Register webhooks
            $('#register-webhooks').on('click', function() {
                var $button = $(this);
                var $spinner = $button.next('.spinner');
                var $result = $('#webhook-result');
                
                // Get selected notification types
                var types = [];
                $('input[name="notification_types[]"]:checked').each(function() {
                    types.push($(this).val());
                });
                
                // Save failover email
                var failoverEmail = $('#delcampe_failover_email').val();
                
                $button.prop('disabled', true);
                $spinner.addClass('is-active');
                $result.html('');
                
                $.post(ajaxurl, {
                    action: 'delcampe_register_webhooks',
                    notification_types: types,
                    failover_email: failoverEmail,
                    nonce: '<?php echo wp_create_nonce('delcampe_webhook_nonce'); ?>'
                }, function(response) {
                    $button.prop('disabled', false);
                    $spinner.removeClass('is-active');
                    
                    if (response.success) {
                        $result.html('<div class="notice notice-success"><p>' + response.data.message + '</p></div>');
                        
                        // Reload page after 2 seconds to show updated webhooks
                        setTimeout(function() {
                            location.reload();
                        }, 2000);
                    } else {
                        $result.html('<div class="notice notice-error"><p>' + response.data.message + '</p></div>');
                    }
                });
            });
            
            // Remove webhook
            $('.remove-webhook').on('click', function() {
                var $button = $(this);
                var notificationId = $button.data('id');
                
                if (!confirm('<?php esc_html_e('Are you sure you want to remove this webhook?', 'wc-delcampe-integration'); ?>')) {
                    return;
                }
                
                $button.prop('disabled', true);
                
                $.post(ajaxurl, {
                    action: 'delcampe_remove_webhook',
                    notification_id: notificationId,
                    nonce: '<?php echo wp_create_nonce('delcampe_webhook_nonce'); ?>'
                }, function(response) {
                    if (response.success) {
                        $button.closest('tr').fadeOut();
                    } else {
                        alert(response.data.message);
                        $button.prop('disabled', false);
                    }
                });
            });
        });
        </script>
        <?php
    }
    
    /**
     * AJAX handler for webhook registration
     */
    public function ajax_register_webhooks() {
        check_ajax_referer('delcampe_webhook_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Unauthorized', 'wc-delcampe-integration'));
        }
        
        $notification_types = isset($_POST['notification_types']) ? (array)$_POST['notification_types'] : array();
        $failover_email = isset($_POST['failover_email']) ? sanitize_email($_POST['failover_email']) : '';
        
        // Save failover email
        if ($failover_email) {
            update_option('delcampe_failover_email', $failover_email);
        }
        
        $result = $this->register_webhooks($notification_types);
        
        if ($result['success']) {
            wp_send_json_success(array(
                'message' => __('Webhooks registered successfully!', 'wc-delcampe-integration'),
                'webhook_url' => $result['webhook_url'],
                'results' => $result['results']
            ));
        } else {
            wp_send_json_error(array(
                'message' => $result['message']
            ));
        }
    }
    
    /**
     * AJAX handler for webhook removal
     */
    public function ajax_remove_webhook() {
        check_ajax_referer('delcampe_webhook_nonce', 'nonce');
        
        if (!current_user_can('manage_options')) {
            wp_die(__('Unauthorized', 'wc-delcampe-integration'));
        }
        
        $notification_id = isset($_POST['notification_id']) ? intval($_POST['notification_id']) : 0;
        
        if (!$notification_id) {
            wp_send_json_error(array(
                'message' => __('Invalid notification ID', 'wc-delcampe-integration')
            ));
        }
        
        $result = $this->remove_webhook($notification_id);
        
        if ($result['success']) {
            wp_send_json_success(array(
                'message' => $result['message']
            ));
        } else {
            wp_send_json_error(array(
                'message' => $result['error']
            ));
        }
    }
}

// Initialize the webhook manager
Delcampe_Webhook_Manager::get_instance();