<?php
/**
 * Delcampe Webhook Registration Manager
 * 
 * Handles registration of webhook callbacks with Delcampe API
 * Based on /notification/settings endpoints
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.10.13.21
 */

if (!defined('ABSPATH')) {
    exit;
}

class Delcampe_Webhook_Registration {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Webhook types we care about
     * Priority: Essential > Recommended > Nice-to-have
     */
    const WEBHOOK_TYPES = array(
        // Essential for orders (MUST HAVE)
        'Curl_Seller_Item_Close_Sold' => 'Item Sold (Creates Order)',
        'Curl_Seller_Payment_Received' => 'Payment Received (Updates Order)',
        
        // Strongly recommended for catalog sync
        'Curl_Seller_Item_Update' => 'Item Updated (Sync Changes)',
        'Curl_Seller_Item_Close_Unsold' => 'Item Closed Unsold',
        'Curl_Seller_Item_Close_Manually' => 'Item Closed Manually',
        'Curl_Seller_Item_Restart' => 'Item Restarted',
        'Curl_Seller_Item_Move' => 'Item Moved Category',
        
        // Nice to have for operations
        'Email_Error_LimitReached' => 'Error Notification (Email)'
    );
    
    /**
     * Essential webhooks that must be registered
     */
    const ESSENTIAL_WEBHOOKS = array(
        'Curl_Seller_Item_Close_Sold',
        'Curl_Seller_Payment_Received'
    );
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // REMOVED: Separate menu item - now integrated as Settings tab (v1.10.14.2)
        // add_action('admin_menu', array($this, 'add_webhook_settings_page'), 99);
        
        // Register AJAX handlers
        add_action('wp_ajax_delcampe_check_webhooks', array($this, 'ajax_check_webhooks'));
        add_action('wp_ajax_delcampe_register_webhook', array($this, 'ajax_register_webhook'));
        add_action('wp_ajax_delcampe_unregister_webhook', array($this, 'ajax_unregister_webhook'));
        add_action('wp_ajax_delcampe_preview_cleanup', array($this, 'ajax_preview_cleanup'));
        add_action('wp_ajax_delcampe_cleanup_duplicates', array($this, 'ajax_cleanup_duplicates'));
        
        // Register REST API endpoint for webhook
        add_action('rest_api_init', array($this, 'register_webhook_endpoint'));
    }
    
    /**
     * Add webhook settings page to admin menu
     * DEPRECATED: Now integrated as a tab in Settings (v1.10.14.2)
     * Keeping method for backward compatibility but not registering menu
     */
    public function add_webhook_settings_page() {
        // This method is kept for backward compatibility
        // Webhook settings are now accessed via Settings → Webhooks tab
        return;
    }
    
    /**
     * Get current webhook registrations from Delcampe
     */
    public function get_current_webhooks() {
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return $token;
        }
        
        $url = DELCAMPE_API_URL . '/notification/settings?token=' . $token;
        
        $response = wp_remote_get($url, array(
            'timeout' => 30,
            'headers' => array('Accept' => 'application/xml')
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $body = wp_remote_retrieve_body($response);
        $xml = simplexml_load_string($body);
        
        if (!$xml) {
            return new WP_Error('parse_error', 'Failed to parse XML response');
        }
        
        $webhooks = array();
        
        // Parse notification settings - correct XML structure
        if (isset($xml->Notification_Data->body->notification_settings)) {
            foreach ($xml->Notification_Data->body->notification_settings as $notification) {
                $type = (string)$notification->type;
                $destination = (string)$notification->destination;
                $id = (string)$notification->id_notification;
                $active = (string)$notification->active == '1';
                
                $webhooks[] = array(
                    'id' => $id,
                    'type' => $type,
                    'destination' => $destination,
                    'active' => $active
                );
            }
        }
        
        return $webhooks;
    }
    
    /**
     * Register a webhook with Delcampe
     */
    public function register_webhook($type, $destination) {
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return $token;
        }
        
        $url = DELCAMPE_API_URL . '/notification/settings?token=' . $token;
        
        $body = array(
            'notificationType' => $type,
            'destination' => $destination
        );
        
        $response = wp_remote_post($url, array(
            'timeout' => 30,
            'headers' => array(
                'Content-Type' => 'application/x-www-form-urlencoded',
                'Accept' => 'application/xml'
            ),
            'body' => http_build_query($body)
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        
        if ($response_code !== 200 && $response_code !== 201) {
            $body = wp_remote_retrieve_body($response);
            return new WP_Error('api_error', 'API returned status ' . $response_code, $body);
        }
        
        return true;
    }
    
    /**
     * Unregister a webhook
     */
    public function unregister_webhook($webhook_id) {
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return $token;
        }
        
        // Use the correct endpoint format for deleting notifications
        $url = DELCAMPE_API_URL . '/notification/' . urlencode($webhook_id) . '?token=' . urlencode($token);
        
        $response = wp_remote_request($url, array(
            'method' => 'DELETE',
            'timeout' => 30,
            'headers' => array('Accept' => 'application/xml')
        ));
        
        if (is_wp_error($response)) {
            return $response;
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        // Check for success (200 or 204 for DELETE)
        if ($response_code == 200 || $response_code == 204) {
            return true;
        }
        
        // Check if response contains <ok/> in body
        if (strpos($body, '<ok/>') !== false) {
            return true;
        }
        
        return new WP_Error('delete_failed', 'Failed to delete webhook. Response code: ' . $response_code);
    }
    
    /**
     * Ensure essential webhooks are registered
     * Called on plugin activation and periodically
     */
    public function ensure_essential_webhooks() {
        // FIRST: Clean up any duplicate webhooks
        $cleanup_result = $this->cleanup_duplicate_notifications();
        if (!is_wp_error($cleanup_result)) {
            if ($cleanup_result['duplicates'] > 0) {
                delcampe_log('[Webhook] Cleaned up ' . count($cleanup_result['deleted']) . ' duplicate webhooks');
                
                // Log details about what was cleaned
                $logger = Delcampe_Sync_Logger::get_instance();
                $logger->write_log(array(
                    'timestamp' => current_time('Y-m-d H:i:s'),
                    'event' => 'WEBHOOK_DUPLICATES_CLEANED',
                    'total_before' => $cleanup_result['total'],
                    'duplicates_found' => $cleanup_result['duplicates'],
                    'deleted_count' => count($cleanup_result['deleted']),
                    'failed_count' => count($cleanup_result['failed'])
                ));
            }
        }
        
        // Get current webhooks after cleanup
        $current_webhooks = $this->get_current_webhooks();
        
        if (is_wp_error($current_webhooks)) {
            delcampe_log('[Webhook] Failed to check existing webhooks: ' . $current_webhooks->get_error_message());
            return $current_webhooks;
        }
        
        // Get webhook URL
        $webhook_url = $this->get_webhook_url();
        
        // Check which essential webhooks are missing
        $registered_types = array();
        foreach ($current_webhooks as $webhook) {
            if ($webhook['destination'] === $webhook_url) {
                $registered_types[] = $webhook['type'];
            }
        }
        
        $missing_webhooks = array();
        foreach (self::ESSENTIAL_WEBHOOKS as $type) {
            if (!in_array($type, $registered_types)) {
                $missing_webhooks[] = $type;
            }
        }
        
        // Register missing essential webhooks
        $results = array(
            'checked' => count(self::ESSENTIAL_WEBHOOKS),
            'existing' => count(self::ESSENTIAL_WEBHOOKS) - count($missing_webhooks),
            'registered' => 0,
            'failed' => 0,
            'details' => array()
        );
        
        foreach ($missing_webhooks as $type) {
            $result = $this->register_webhook($type, $webhook_url);
            
            if (is_wp_error($result)) {
                $results['failed']++;
                $results['details'][] = "Failed to register $type: " . $result->get_error_message();
                delcampe_log("[Webhook] Failed to register $type: " . $result->get_error_message());
            } else {
                $results['registered']++;
                $results['details'][] = "Successfully registered $type";
                delcampe_log("[Webhook] Successfully registered essential webhook: $type");
            }
        }
        
        if (count($missing_webhooks) === 0) {
            delcampe_log('[Webhook] All essential webhooks already registered');
        }
        
        return $results;
    }
    
    /**
     * Clean up duplicate webhook notifications
     * Keeps only 1 registration per type+destination and removes extras
     */
    public function cleanup_duplicate_notifications() {
        $auth = Delcampe_Auth::get_instance();
        $token = $auth->get_auth_token();
        
        if (is_wp_error($token)) {
            return $token;
        }
        
        // 1) List current registrations
        $url = DELCAMPE_API_URL . '/notification/settings?token=' . urlencode($token);
        $res = wp_remote_get($url, ['timeout' => 20, 'headers' => ['Accept' => 'application/xml']]);
        
        if (is_wp_error($res)) {
            return $res;
        }
        
        $xml = simplexml_load_string(wp_remote_retrieve_body($res));
        if (!$xml) {
            return new WP_Error('bad_xml', 'Cannot parse /notification/settings');
        }
        
        // 2) Group by (type, destination, channel) and keep the first id
        $groups = [];
        $all_registrations = [];
        
        // Parse all notification_settings elements
        if (isset($xml->Notification_Data->body->notification_settings)) {
            foreach ($xml->Notification_Data->body->notification_settings as $n) {
                $id = (string)$n->id_notification;
                $type = (string)$n->type;
                $dest = (string)$n->destination;
                $chan = (string)$n->channel;
                $active = (string)$n->active;
                
                $key = implode('|', [$type, $dest, $chan]);
                
                if (!isset($groups[$key])) {
                    $groups[$key] = [];
                }
                
                $groups[$key][] = [
                    'id' => $id,
                    'type' => $type,
                    'destination' => $dest,
                    'channel' => $chan,
                    'active' => $active
                ];
                
                $all_registrations[] = [
                    'id' => $id,
                    'type' => $type,
                    'destination' => $dest,
                    'channel' => $chan,
                    'active' => $active,
                    'key' => $key
                ];
            }
        }
        
        // 3) Identify duplicates to remove
        $to_delete = [];
        $kept = [];
        
        foreach ($groups as $key => $registrations) {
            if (count($registrations) <= 1) {
                if (count($registrations) == 1) {
                    $kept[] = $registrations[0];
                }
                continue;
            }
            
            // Keep the first one, mark others for deletion
            $kept[] = $registrations[0];
            for ($i = 1; $i < count($registrations); $i++) {
                $to_delete[] = $registrations[$i];
            }
        }
        
        // 4) Delete the duplicates
        $deleted = [];
        $failed = [];
        
        foreach ($to_delete as $webhook) {
            $del_url = DELCAMPE_API_URL . '/notification/' . urlencode($webhook['id']) . '?token=' . urlencode($token);
            $del = wp_remote_request($del_url, [
                'method' => 'DELETE',
                'timeout' => 20,
                'headers' => ['Accept' => 'application/xml']
            ]);
            
            if (is_wp_error($del)) {
                $failed[] = [
                    'webhook' => $webhook,
                    'error' => $del->get_error_message()
                ];
            } else {
                $response_code = wp_remote_retrieve_response_code($del);
                $body = wp_remote_retrieve_body($del);
                
                if ($response_code == 200 || $response_code == 204 || strpos($body, '<ok/>') !== false) {
                    $deleted[] = $webhook;
                } else {
                    $failed[] = [
                        'webhook' => $webhook,
                        'error' => 'HTTP ' . $response_code
                    ];
                }
            }
        }
        
        return [
            'total' => count($all_registrations),
            'duplicates' => count($to_delete),
            'kept' => $kept,
            'deleted' => $deleted,
            'failed' => $failed
        ];
    }
    
    /**
     * Get the webhook URL for this site
     */
    public function get_webhook_url() {
        // Generate secure token for this site
        $token = get_option('delcampe_webhook_token');
        if (!$token) {
            $token = wp_generate_password(32, false);
            update_option('delcampe_webhook_token', $token);
        }
        
        // Build webhook URL using rest_url to ensure proper format
        $url = rest_url('delcampe/v1/webhook');
        
        // Force HTTPS since Delcampe requires it
        $url = str_replace('http://', 'https://', $url);
        
        $url = add_query_arg('token', $token, $url);
        
        return $url;
    }
    
    /**
     * Render settings page
     */
    public function render_settings_page() {
        ?>
        <div class="wrap">
            <h1><?php _e('Delcampe Webhook Settings', 'wc-delcampe-integration'); ?></h1>
            
            <div class="notice notice-info">
                <p><?php _e('Webhooks allow Delcampe to notify your site in real-time when items are sold or payments are received.', 'wc-delcampe-integration'); ?></p>
            </div>
            
            <div class="card">
                <h2><?php _e('Webhook URL', 'wc-delcampe-integration'); ?></h2>
                <p><?php _e('Your webhook URL is:', 'wc-delcampe-integration'); ?></p>
                <code id="webhook-url"><?php echo esc_html($this->get_webhook_url()); ?></code>
                <button type="button" class="button" onclick="copyWebhookUrl()"><?php _e('Copy', 'wc-delcampe-integration'); ?></button>
            </div>
            
            <div class="card">
                <h2 style="display:flex; align-items:center; gap:10px;">
                    <?php _e('Current Webhook Registrations', 'wc-delcampe-integration'); ?>
                    <button type="button" class="button" onclick="checkWebhooks()">
                        <?php _e('Refresh', 'wc-delcampe-integration'); ?>
                    </button>
                </h2>
                <div id="webhook-status">
                    <p><?php _e('Checking webhook status...', 'wc-delcampe-integration'); ?></p>
                </div>
                <div id="duplicate-actions" style="margin-top: 15px; display: none;">
                    <button type="button" class="button button-secondary" onclick="previewDuplicateCleanup()">
                        <?php _e('Check for Duplicates', 'wc-delcampe-integration'); ?>
                    </button>
                    <div id="cleanup-preview" style="margin-top: 15px;"></div>
                </div>
            </div>
            
            <div class="card">
                <h2><?php _e('Quick Setup', 'wc-delcampe-integration'); ?></h2>
                <button type="button" class="button button-primary" onclick="registerEssentialWebhooks()">
                    <?php _e('Register Essential Webhooks (Recommended)', 'wc-delcampe-integration'); ?>
                </button>
                <p class="description"><?php _e('Registers the two essential webhooks for order processing: Item Sold and Payment Received', 'wc-delcampe-integration'); ?></p>
                
                <?php if (class_exists('Delcampe_Webhook_Guardian')) : ?>
                <hr style="margin: 20px 0;">
                <h3><?php _e('🛡️ Webhook Guardian', 'wc-delcampe-integration'); ?></h3>
                <p><?php _e('The Webhook Guardian ensures proper webhook registration and prevents duplicates.', 'wc-delcampe-integration'); ?></p>
                <a href="<?php echo esc_url(Delcampe_Webhook_Guardian::get_manual_fix_url()); ?>" class="button button-secondary">
                    <?php _e('Run Guardian Check Now', 'wc-delcampe-integration'); ?>
                </a>
                <p class="description"><?php _e('Guardian runs automatically daily. Use this button to force an immediate check.', 'wc-delcampe-integration'); ?></p>
                <?php endif; ?>
            </div>
            
            <div class="card">
                <h2><?php _e('Register Individual Webhooks', 'wc-delcampe-integration'); ?></h2>
                
                <h3><?php _e('📌 Essential (Required for Orders)', 'wc-delcampe-integration'); ?></h3>
                <table class="form-table">
                    <tr>
                        <th><?php _e('Item Sold', 'wc-delcampe-integration'); ?></th>
                        <td>
                            <button type="button" class="button button-primary" onclick="registerWebhook('Curl_Seller_Item_Close_Sold')">
                                <?php _e('Register', 'wc-delcampe-integration'); ?>
                            </button>
                            <p class="description"><?php _e('Creates WooCommerce order when item is sold (On-hold status)', 'wc-delcampe-integration'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th><?php _e('Payment Received', 'wc-delcampe-integration'); ?></th>
                        <td>
                            <button type="button" class="button button-primary" onclick="registerWebhook('Curl_Seller_Payment_Received')">
                                <?php _e('Register', 'wc-delcampe-integration'); ?>
                            </button>
                            <p class="description"><?php _e('Updates order to Processing when payment clears', 'wc-delcampe-integration'); ?></p>
                        </td>
                    </tr>
                </table>
                
                <h3><?php _e('🔄 Catalog Sync (Recommended)', 'wc-delcampe-integration'); ?></h3>
                <table class="form-table">
                    <tr>
                        <th><?php _e('Item Updated', 'wc-delcampe-integration'); ?></th>
                        <td>
                            <button type="button" class="button" onclick="registerWebhook('Curl_Seller_Item_Update')">
                                <?php _e('Register', 'wc-delcampe-integration'); ?>
                            </button>
                            <p class="description"><?php _e('Syncs changes when listing is edited on Delcampe', 'wc-delcampe-integration'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th><?php _e('Item Closed Unsold', 'wc-delcampe-integration'); ?></th>
                        <td>
                            <button type="button" class="button" onclick="registerWebhook('Curl_Seller_Item_Close_Unsold')">
                                <?php _e('Register', 'wc-delcampe-integration'); ?>
                            </button>
                            <p class="description"><?php _e('Updates status when listing ends without sale', 'wc-delcampe-integration'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th><?php _e('Item Closed Manually', 'wc-delcampe-integration'); ?></th>
                        <td>
                            <button type="button" class="button" onclick="registerWebhook('Curl_Seller_Item_Close_Manually')">
                                <?php _e('Register', 'wc-delcampe-integration'); ?>
                            </button>
                            <p class="description"><?php _e('Updates status when seller closes listing', 'wc-delcampe-integration'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th><?php _e('Item Restarted', 'wc-delcampe-integration'); ?></th>
                        <td>
                            <button type="button" class="button" onclick="registerWebhook('Curl_Seller_Item_Restart')">
                                <?php _e('Register', 'wc-delcampe-integration'); ?>
                            </button>
                            <p class="description"><?php _e('Updates when listing is relisted', 'wc-delcampe-integration'); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th><?php _e('Item Moved', 'wc-delcampe-integration'); ?></th>
                        <td>
                            <button type="button" class="button" onclick="registerWebhook('Curl_Seller_Item_Move')">
                                <?php _e('Register', 'wc-delcampe-integration'); ?>
                            </button>
                            <p class="description"><?php _e('Updates when listing category changes', 'wc-delcampe-integration'); ?></p>
                        </td>
                    </tr>
                </table>
                
                <h3><?php _e('⚙️ Operations', 'wc-delcampe-integration'); ?></h3>
                <table class="form-table">
                    <tr>
                        <th><?php _e('Error Notifications', 'wc-delcampe-integration'); ?></th>
                        <td>
                            <input type="email" id="error-email" placeholder="admin@example.com" value="<?php echo esc_attr( get_option('delcampe_failover_email', get_option('admin_email')) ); ?>" />
                            <button type="button" class="button" onclick="registerErrorEmail()">
                                <?php _e('Register Email', 'wc-delcampe-integration'); ?>
                            </button>
                            <p class="description"><?php _e('Email alert if webhook fails 5+ times', 'wc-delcampe-integration'); ?></p>
                        </td>
                    </tr>
                </table>
            </div>
            
            <script>
            // Ensure ajaxurl exists (fallback for contexts where WP doesn't define it)
            if (typeof ajaxurl === 'undefined') {
                var ajaxurl = '<?php echo esc_js( admin_url('admin-ajax.php') ); ?>';
            }

            function copyWebhookUrl() {
                const url = document.getElementById('webhook-url').textContent;
                navigator.clipboard.writeText(url).then(() => {
                    alert('Webhook URL copied to clipboard');
                });
            }
            
            function checkWebhooks() {
                console.log('Checking webhooks...');
                jQuery.post(ajaxurl, {
                    action: 'delcampe_check_webhooks'
                }, function(response) {
                    console.log('Response:', response);
                    if (response.success) {
                        displayWebhooks(response.data);
                    } else {
                        document.getElementById('webhook-status').innerHTML = 
                            '<p class="error">Error: ' + response.data + '</p>';
                    }
                }).fail(function(xhr, status, error) {
                    console.error('AJAX Error:', error);
                    document.getElementById('webhook-status').innerHTML = 
                        '<p class="error">AJAX Error: ' + error + '</p>';
                });
            }
            
            function displayWebhooks(webhooks) {
                let html = '<table class="wp-list-table widefat fixed striped">';
                html += '<thead><tr><th>Type</th><th>Destination</th><th>Status</th><th>Action</th></tr></thead>';
                html += '<tbody>';
                
                // Check for duplicates
                let duplicateFound = false;
                let typeCount = {};
                
                if (webhooks.length === 0) {
                    html += '<tr><td colspan="4">No webhooks registered</td></tr>';
                } else {
                    webhooks.forEach(function(webhook) {
                        // Count by type+destination
                        let key = webhook.type + '|' + webhook.destination;
                        typeCount[key] = (typeCount[key] || 0) + 1;
                        
                        html += '<tr>';
                        html += '<td>' + webhook.type + '</td>';
                        html += '<td>' + webhook.destination + '</td>';
                        html += '<td>' + (webhook.active ? '✅ Active' : '❌ Inactive') + '</td>';
                        html += '<td><button onclick="unregisterWebhook(\'' + webhook.id + '\')" class="button">Remove</button></td>';
                        html += '</tr>';
                    });
                    
                    // Check if any type has duplicates
                    for (let key in typeCount) {
                        if (typeCount[key] > 1) {
                            duplicateFound = true;
                            break;
                        }
                    }
                }
                
                html += '</tbody></table>';
                document.getElementById('webhook-status').innerHTML = html;
                
                // Show/hide duplicate cleanup button
                if (duplicateFound) {
                    document.getElementById('duplicate-actions').style.display = 'block';
                    // Add warning message
                    html += '<div class="notice notice-warning" style="margin-top: 10px;"><p>⚠️ Duplicate webhooks detected! Click "Check for Duplicates" to review and clean them up.</p></div>';
                    document.getElementById('webhook-status').innerHTML = html;
                } else {
                    document.getElementById('duplicate-actions').style.display = 'none';
                }
            }
            
            function registerWebhook(type) {
                const url = document.getElementById('webhook-url').textContent;
                jQuery.post(ajaxurl, {
                    action: 'delcampe_register_webhook',
                    type: type,
                    destination: url
                }, function(response) {
                    if (response.success) {
                        alert('Webhook registered successfully!');
                        checkWebhooks();
                    } else {
                        alert('Error: ' + response.data);
                    }
                });
            }
            
            function registerEssentialWebhooks() {
                if (!confirm('This will register the two essential webhooks (Item Sold and Payment Received). Continue?')) return;
                
                const url = document.getElementById('webhook-url').textContent;
                const essentialTypes = ['Curl_Seller_Item_Close_Sold', 'Curl_Seller_Payment_Received'];
                let completed = 0;
                let errors = [];
                
                essentialTypes.forEach(function(type) {
                    jQuery.post(ajaxurl, {
                        action: 'delcampe_register_webhook',
                        type: type,
                        destination: url
                    }, function(response) {
                        completed++;
                        if (!response.success) {
                            errors.push(type + ': ' + response.data);
                        }
                        
                        if (completed === essentialTypes.length) {
                            if (errors.length === 0) {
                                alert('✅ Essential webhooks registered successfully!');
                            } else {
                                alert('Some webhooks failed:\n' + errors.join('\n'));
                            }
                            checkWebhooks();
                        }
                    });
                });
            }
            
            function registerErrorEmail() {
                const email = document.getElementById('error-email').value;
                if (!email) {
                    alert('Please enter an email address');
                    return;
                }
                
                jQuery.post(ajaxurl, {
                    action: 'delcampe_register_webhook',
                    type: 'Email_Error_LimitReached',
                    destination: email
                }, function(response) {
                    if (response.success) {
                        alert('Error notification email registered!');
                        checkWebhooks();
                    } else {
                        alert('Error: ' + response.data);
                    }
                });
            }
            
            function unregisterWebhook(id) {
                if (!confirm('Remove this webhook?')) return;
                
                jQuery.post(ajaxurl, {
                    action: 'delcampe_unregister_webhook',
                    webhook_id: id
                }, function(response) {
                    if (response.success) {
                        alert('Webhook removed');
                        checkWebhooks();
                    } else {
                        alert('Error: ' + response.data);
                    }
                });
            }
            
            function previewDuplicateCleanup() {
                document.getElementById('cleanup-preview').innerHTML = '<p>Analyzing webhooks for duplicates...</p>';
                
                jQuery.post(ajaxurl, {
                    action: 'delcampe_preview_cleanup',
                    dry_run: true
                }, function(response) {
                    if (response.success && response.data) {
                        let html = '';
                        
                        if (response.data.duplicates_found === 0) {
                            html = '<div class="notice notice-success"><p>✅ No duplicates found!</p></div>';
                        } else {
                            html = '<div class="notice notice-warning">';
                            html += '<h3>Found ' + response.data.duplicates_found + ' duplicate webhook(s)</h3>';
                            html += '<p>The following duplicates will be removed (keeping the first of each type):</p>';
                            html += '<ul>';
                            
                            if (response.data.to_remove) {
                                response.data.to_remove.forEach(function(webhook) {
                                    html += '<li>ID: ' + webhook.id + ' - Type: ' + webhook.type + ' - URL: ' + webhook.destination + '</li>';
                                });
                            }
                            
                            html += '</ul>';
                            html += '<button type="button" class="button button-primary" onclick="executeDuplicateCleanup()">Remove Duplicates</button> ';
                            html += '<button type="button" class="button" onclick="cancelCleanup()">Cancel</button>';
                            html += '</div>';
                        }
                        
                        document.getElementById('cleanup-preview').innerHTML = html;
                    } else {
                        document.getElementById('cleanup-preview').innerHTML = 
                            '<div class="notice notice-error"><p>Error checking for duplicates: ' + (response.data || 'Unknown error') + '</p></div>';
                    }
                });
            }
            
            function executeDuplicateCleanup() {
                if (!confirm('This will remove all duplicate webhooks. Continue?')) return;
                
                document.getElementById('cleanup-preview').innerHTML = '<p>Removing duplicates...</p>';
                
                jQuery.post(ajaxurl, {
                    action: 'delcampe_cleanup_duplicates'
                }, function(response) {
                    if (response.success && response.data) {
                        let html = '<div class="notice notice-success">';
                        html += '<h3>Cleanup Complete!</h3>';
                        html += '<p>✅ Removed ' + response.data.removed + ' duplicate webhook(s)</p>';
                        
                        if (response.data.failed > 0) {
                            html += '<p>⚠️ Failed to remove ' + response.data.failed + ' webhook(s)</p>';
                        }
                        
                        html += '</div>';
                        document.getElementById('cleanup-preview').innerHTML = html;
                        
                        // Refresh webhook list after 2 seconds
                        setTimeout(checkWebhooks, 2000);
                    } else {
                        document.getElementById('cleanup-preview').innerHTML = 
                            '<div class="notice notice-error"><p>Error during cleanup: ' + (response.data || 'Unknown error') + '</p></div>';
                    }
                });
            }
            
            function cancelCleanup() {
                document.getElementById('cleanup-preview').innerHTML = '';
            }
            
            // Check webhooks on page load and when Webhooks tab is activated
            jQuery(document).ready(function($) {
                checkWebhooks();
                // If using the tabbed UI, also refresh when user switches to Webhooks tab
                $(document).on('click', '.delcampe-tabs .nav-tab', function() {
                    var tab = $(this).data('tab');
                    if (tab === 'webhooks') {
                        setTimeout(checkWebhooks, 50);
                    }
                });
            });
            
            // Also check after short delay in case jQuery isn't ready
            setTimeout(function() {
                if (document.getElementById('webhook-status').innerHTML.includes('Checking webhook status...')) {
                    console.log('Fallback check after 2 seconds...');
                    checkWebhooks();
                }
            }, 2000);
            </script>
        </div>
        <?php
    }
    
    /**
     * AJAX handler: Check webhooks
     */
    public function ajax_check_webhooks() {
        // Check nonce if present (though not currently using nonce in JS)
        if (isset($_POST['nonce']) && !wp_verify_nonce($_POST['nonce'], 'delcampe_ajax')) {
            wp_send_json_error('Invalid nonce');
            return;
        }
        
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized - you need admin privileges');
            return;
        }
        
        $webhooks = $this->get_current_webhooks();
        
        if (is_wp_error($webhooks)) {
            delcampe_log('[Webhook] AJAX check error: ' . $webhooks->get_error_message());
            wp_send_json_error($webhooks->get_error_message());
            return;
        }
        
        delcampe_log('[Webhook] AJAX check success: ' . count($webhooks) . ' webhooks found');
        wp_send_json_success($webhooks);
    }
    
    /**
     * AJAX handler: Register webhook
     */
    public function ajax_register_webhook() {
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $type = sanitize_text_field($_POST['type']);
        $destination = sanitize_text_field($_POST['destination']);
        
        // If registering error email, validate and persist option
        if ($type === 'Email_Error_LimitReached') {
            if (!is_email($destination)) {
                wp_send_json_error(__('Please provide a valid email address.', 'wc-delcampe-integration'));
                return;
            }
            update_option('delcampe_failover_email', $destination);
        }

        $result = $this->register_webhook($type, $destination);
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }
        
        wp_send_json_success('Webhook registered');
    }
    
    /**
     * AJAX handler: Unregister webhook
     */
    public function ajax_unregister_webhook() {
        if (!current_user_can('manage_options')) {
            wp_die('Unauthorized');
        }
        
        $webhook_id = sanitize_text_field($_POST['webhook_id']);
        
        $result = $this->unregister_webhook($webhook_id);
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
        }
        
        wp_send_json_success('Webhook removed');
    }
    
    /**
     * AJAX handler: Preview cleanup (dry run)
     */
    public function ajax_preview_cleanup() {
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        // Get current webhooks
        $webhooks = $this->get_current_webhooks();
        
        if (is_wp_error($webhooks)) {
            wp_send_json_error($webhooks->get_error_message());
            return;
        }
        
        // Group by type+destination+channel to find duplicates
        $grouped = array();
        foreach ($webhooks as $webhook) {
            $key = $webhook['type'] . '|' . $webhook['destination'] . '|' . ($webhook['channel'] ?? '');
            if (!isset($grouped[$key])) {
                $grouped[$key] = array();
            }
            $grouped[$key][] = $webhook;
        }
        
        // Identify duplicates to remove
        $to_remove = array();
        $duplicates_found = 0;
        
        foreach ($grouped as $key => $group) {
            if (count($group) > 1) {
                // Keep the first one, mark others for removal
                for ($i = 1; $i < count($group); $i++) {
                    $to_remove[] = $group[$i];
                    $duplicates_found++;
                }
            }
        }
        
        wp_send_json_success(array(
            'duplicates_found' => $duplicates_found,
            'to_remove' => $to_remove
        ));
    }
    
    /**
     * AJAX handler: Execute cleanup
     */
    public function ajax_cleanup_duplicates() {
        if (!current_user_can('manage_options')) {
            wp_send_json_error('Unauthorized');
            return;
        }
        
        // Call the cleanup function
        $result = $this->cleanup_duplicate_notifications();
        
        if (is_wp_error($result)) {
            wp_send_json_error($result->get_error_message());
            return;
        }
        
        wp_send_json_success($result);
    }
    
    /**
     * Register REST API endpoint for webhook callbacks
     */
    public function register_webhook_endpoint() {
        register_rest_route('delcampe/v1', '/webhook', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_webhook'),
            'permission_callback' => array($this, 'verify_webhook_token'),
        ));
    }
    
    /**
     * Verify webhook token in request
     */
    public function verify_webhook_token($request) {
        $token = $request->get_param('token');
        $stored_token = get_option('delcampe_webhook_token');
        
        return $token && $stored_token && $token === $stored_token;
    }
    
    /**
     * Handle incoming webhook from Delcampe
     */
    public function handle_webhook($request) {
        // Get the raw POST body
        $body = $request->get_body();
        
        // Enhanced logging - Log EVERY incoming webhook immediately
        $logger = Delcampe_Sync_Logger::get_instance();
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'WEBHOOK_RAW_RECEIVED',
            'method' => $request->get_method(),
            'headers' => $request->get_headers(),
            'body_length' => strlen($body),
            'body_preview' => substr($body, 0, 1000),
            'source_ip' => $_SERVER['REMOTE_ADDR'] ?? 'unknown',
            'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'unknown'
        ));
        
        // Log the webhook for debugging (keep existing log too)
        delcampe_log('[Webhook] Received notification: ' . substr($body, 0, 500));
        
        // Parse XML
        $xml = simplexml_load_string($body);
        if (!$xml) {
            delcampe_log('[Webhook] Failed to parse XML');
            return new WP_REST_Response(array('error' => 'Invalid XML'), 400);
        }
        
        // Delegate to the centralized webhook handler if available
        if (class_exists('Delcampe_Webhook_Handler')) {
            try {
                $handler = new Delcampe_Webhook_Handler();
                $result = $handler->process_notification($xml, $body);
                $code = (!empty($result['success']) ? 200 : 400);
                
                // Log successful processing
                $logger->write_log(array(
                    'timestamp' => current_time('Y-m-d H:i:s'),
                    'event' => 'WEBHOOK_HANDLER_SUCCESS',
                    'notification_type' => (string)$xml->Notification_Type,
                    'result' => $result
                ));
                
                return new WP_REST_Response($result, $code);
            } catch (Exception $e) {
                // CRITICAL: Catch ALL exceptions from webhook processing
                $error_message = $e->getMessage();
                $error_trace = $e->getTraceAsString();
                
                $logger->write_log(array(
                    'timestamp' => current_time('Y-m-d H:i:s'),
                    'event' => 'WEBHOOK_HANDLER_EXCEPTION',
                    'notification_type' => (string)$xml->Notification_Type,
                    'error' => $error_message,
                    'trace' => $error_trace,
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ));
                
                delcampe_log('[Webhook] EXCEPTION in handler: ' . $error_message);
                delcampe_log('[Webhook] Trace: ' . $error_trace);
                
                return new WP_REST_Response(array(
                    'success' => false,
                    'error' => 'Exception: ' . $error_message
                ), 500);
            } catch (Error $e) {
                // CRITICAL: Catch PHP 7+ fatal errors
                $error_message = $e->getMessage();
                $error_trace = $e->getTraceAsString();
                
                $logger->write_log(array(
                    'timestamp' => current_time('Y-m-d H:i:s'),
                    'event' => 'WEBHOOK_HANDLER_FATAL_ERROR',
                    'notification_type' => (string)$xml->Notification_Type,
                    'error' => $error_message,
                    'trace' => $error_trace,
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ));
                
                delcampe_log('[Webhook] FATAL ERROR in handler: ' . $error_message);
                delcampe_log('[Webhook] Trace: ' . $error_trace);
                
                return new WP_REST_Response(array(
                    'success' => false,
                    'error' => 'Fatal Error: ' . $error_message
                ), 500);
            }
        }

        // Fallback: minimal handling for sold/payment
        $type = (string)$xml->Notification_Type;
        delcampe_log('[Webhook] Notification type: ' . $type);
        if ($type === 'Seller_Item_Close_Sold') {
            $this->handle_item_sold($xml);
        } elseif ($type === 'Seller_Payment_Received') {
            $this->handle_payment_received($xml);
        } else {
            delcampe_log('[Webhook] Unknown notification type (fallback mode): ' . $type);
        }
        return new WP_REST_Response(array('success' => true, 'message' => 'Processed in fallback mode'), 200);
    }
    
    /**
     * Handle item sold notification
     */
    private function handle_item_sold($xml) {
        if (!isset($xml->Notification_Data->body->item)) {
            return;
        }
        
        $item = $xml->Notification_Data->body->item;
        $item_id = (string)$item->id_item;
        
        delcampe_log('[Webhook] Item sold: ' . $item_id);
        
        // Use existing webhook handler if available
        if (class_exists('Delcampe_Webhook_Handler')) {
            $handler = Delcampe_Webhook_Handler::get_instance();
            $handler->handle_item_sold($item);
        } else {
            // Create order directly
            if (class_exists('Delcampe_Order_Manager')) {
                $order_manager = Delcampe_Order_Manager::get_instance();
                $order_manager->create_order_from_webhook($item_id, $item);
            }
        }
    }
    
    /**
     * Handle payment received notification
     */
    private function handle_payment_received($xml) {
        if (!isset($xml->Notification_Data->body->payment)) {
            return;
        }
        
        $payment = $xml->Notification_Data->body->payment;
        
        delcampe_log('[Webhook] Payment received for item: ' . (string)$payment->item_id);
        
        // Update order status to processing
        if (class_exists('Delcampe_Order_Manager')) {
            $order_manager = Delcampe_Order_Manager::get_instance();
            // You might need to implement this method
            // $order_manager->update_order_payment($payment);
        }
    }
    
    /**
     * Ensure default webhooks are registered
     * Automatically registers webhooks if they don't exist
     * THIS IS A WRAPPER - just calls ensure_essential_webhooks which includes cleanup
     */
    public function ensure_default_webhooks() {
        // Just call the better method that includes cleanup
        return $this->ensure_essential_webhooks();
        
        // Define webhooks we want to have registered
        $wanted = array(
            'Curl_Seller_Item_Close_Sold',
            'Curl_Seller_Payment_Received'
        );
        
        // Check what's already registered
        $have = array();
        $our_url = $this->get_webhook_url();
        $webhook_count = array();
        
        if (!empty($current)) {
            // Handle both single and multiple webhook formats
            if (isset($current->Notification_Data->body->notification_settings)) {
                $settings = $current->Notification_Data->body->notification_settings;
                
                // Handle multiple settings (could be array or repeated elements)
                if (is_array($settings) || is_object($settings)) {
                    foreach ($settings as $setting) {
                        if (isset($setting->type) && isset($setting->active) && $setting->active == '1') {
                            $type = (string)$setting->type;
                            $dest = (string)$setting->destination;
                            
                            // Only count webhooks pointing to our URL
                            if ($dest === $our_url) {
                                if (!isset($webhook_count[$type])) {
                                    $webhook_count[$type] = 0;
                                }
                                $webhook_count[$type]++;
                                
                                // Add to have array only once per type
                                if (!in_array($type, $have, true)) {
                                    $have[] = $type;
                                }
                            }
                        }
                    }
                }
            }
        }
        
        // Log if we have duplicates
        foreach ($webhook_count as $type => $count) {
            if ($count > 1) {
                delcampe_log('[Webhook Auto-Register] WARNING: Found ' . $count . ' duplicate webhooks for ' . $type);
            }
        }
        
        // Register any missing webhooks (but don't add more if duplicates exist)
        $registered = false;
        foreach ($wanted as $type) {
            if (!in_array($type, $have, true)) {
                delcampe_log('[Webhook Auto-Register] Registering missing webhook: ' . $type);
                $result = $this->register_webhook($type, $our_url);
                if (!is_wp_error($result)) {
                    $registered = true;
                    delcampe_log('[Webhook Auto-Register] Successfully registered: ' . $type);
                } else {
                    delcampe_log('[Webhook Auto-Register] Failed to register ' . $type . ': ' . $result->get_error_message());
                }
            }
        }
        
        // Also ensure failover email is set (operations alert email)
        $failover_email = get_option('delcampe_failover_email', get_option('admin_email'));
        if ($failover_email && $registered) {
            $auth = Delcampe_Auth::get_instance();
            $token = $auth->get_auth_token();
            if (!is_wp_error($token)) {
                $this->register_failover_email($token, $failover_email);
            }
        }
    }

    /**
     * Register failover email notification (Email_Error_LimitReached)
     *
     * Note: Token parameter is accepted for signature parity but not required
     * because this class retrieves its own token in register_webhook().
     *
     * @param string $token   Ignored (for compatibility)
     * @param string $email   Destination email address
     * @return true|WP_Error
     */
    private function register_failover_email($token, $email) {
        if (!is_email($email)) {
            return new WP_Error('invalid_email', 'Invalid failover email address');
        }
        // Save configured email for future auto-registration
        update_option('delcampe_failover_email', $email);
        return $this->register_webhook('Email_Error_LimitReached', $email);
    }
}
