<?php
/**
 * WooCommerce Backend Integration
 * 
 * Adds Delcampe column to orders list and other WooCommerce integrations
 * Similar to WP-Lister's WooBackendIntegration class
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.10.13.9
 */

// Exit if accessed directly
if (!defined('ABSPATH')) exit;

class Delcampe_WooCommerce_Integration {
    
    /**
     * Singleton instance
     */
    private static $instance = null;
    
    /**
     * Get singleton instance
     */
    public static function get_instance() {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Add custom column for orders table
        add_filter('manage_edit-shop_order_columns', array($this, 'add_order_source_column'), 11);
        add_filter('manage_woocommerce_page_wc-orders_columns', array($this, 'add_order_source_column'), 11);
        
        // Populate custom column
        add_action('manage_shop_order_posts_custom_column', array($this, 'render_order_source_column'), 10);
        add_action('manage_woocommerce_page_wc-orders_custom_column', array($this, 'render_order_source_column'), 10, 2);
        
        // Make orders searchable by Delcampe order ID
        add_filter('woocommerce_shop_order_search_fields', array($this, 'add_delcampe_order_search'));
        
        // Add order attribution tracking
        add_action('woocommerce_checkout_create_order', array($this, 'add_order_attribution'), 20, 2);
        
        // Fix email display for Delcampe orders
        add_filter('woocommerce_email_customer_details', array($this, 'fix_delcampe_email_display'), 10, 3);
        add_filter('woocommerce_order_formatted_billing_address', array($this, 'fix_delcampe_billing_address'), 10, 2);
    }
    
    /**
     * Add order source column to orders list
     */
    public function add_order_source_column($columns) {
        // Add after order_number if it exists, otherwise after cb
        $new_columns = array();
        foreach ($columns as $key => $value) {
            $new_columns[$key] = $value;
            if ($key === 'order_number' || $key === 'cb') {
                $new_columns['order_source'] = __('Source', 'wc-delcampe-integration');
            }
        }
        return $new_columns;
    }
    
    /**
     * Render order source column
     */
    public function render_order_source_column($column, $order = null) {
        if ($column != 'order_source') return;
        
        if ($order) {
            $order_id = $order->get_id();
        } else {
            global $post;
            $order_id = ($post) ? $post->ID : false;
            if (!$order_id) return;
            $order = wc_get_order($order_id);
        }
        
        // Check if order exists
        if (!$order) return;
        
        // Check if order was placed on Delcampe
        $delcampe_order_id = $order->get_meta('_delcampe_order_id');
        
        if ($delcampe_order_id) {
            // Show Delcampe logo/text
            $tooltip = sprintf(__('This order was placed on Delcampe (ID: %s)', 'wc-delcampe-integration'), $delcampe_order_id);
            echo '<span title="' . esc_attr($tooltip) . '" style="color: #0073aa; font-weight: bold;">Delcampe</span>';
            return;
        }
        
        // Check if it's an eBay order (WP-Lister)
        $ebay_order_id = $order->get_meta('_ebay_order_id');
        if ($ebay_order_id) {
            echo '<span style="color: #e53935;">eBay</span>';
            return;
        }
        
        // Check created_via for other sources
        $created_via = $order->get_created_via();
        if ($created_via === 'checkout') {
            echo '<span>Direct</span>';
        } elseif ($created_via === 'admin') {
            echo '<span style="color: #666;">Manual</span>';
        } elseif ($created_via) {
            echo '<span>' . esc_html($created_via) . '</span>';
        }
    }
    
    /**
     * Make orders searchable by Delcampe order ID
     */
    public function add_delcampe_order_search($search_fields) {
        $search_fields[] = '_delcampe_order_id';
        $search_fields[] = '_delcampe_buyer_nickname';
        return $search_fields;
    }
    
    /**
     * Add order attribution tracking for orders created through checkout
     */
    public function add_order_attribution($order, $data) {
        // Only for direct checkout orders, not for imported ones
        if ($order->get_created_via() === 'checkout') {
            // Check if this is a Delcampe-synced product
            $items = $order->get_items();
            foreach ($items as $item) {
                $product_id = $item->get_product_id();
                $delcampe_id = get_post_meta($product_id, '_delcampe_id', true);
                if ($delcampe_id) {
                    $order->update_meta_data('_delcampe_product_source', 'yes');
                    break;
                }
            }
        }
    }
    
    /**
     * Fix email display for Delcampe orders
     * Note: @delcampe.net emails are legitimate forwarding addresses
     */
    public function fix_delcampe_email_display($order, $sent_to_admin, $plain_text) {
        if (!$order) return;
        
        $delcampe_order_id = $order->get_meta('_delcampe_order_id');
        if (!$delcampe_order_id) return;
        
        // No changes needed - Delcampe forwarding emails work properly
    }
    
    /**
     * Fix billing address display for Delcampe orders
     */
    public function fix_delcampe_billing_address($address, $order) {
        if (!$order) return $address;
        
        $delcampe_order_id = $order->get_meta('_delcampe_order_id');
        if (!$delcampe_order_id) return $address;
        
        // Make sure names are set (if empty, use nickname)
        if (empty($address['first_name']) && empty($address['last_name'])) {
            $nickname = $order->get_meta('_delcampe_buyer_nickname');
            if ($nickname) {
                $address['first_name'] = $nickname;
            }
        }
        
        return $address;
    }
}

// Initialize
Delcampe_WooCommerce_Integration::get_instance();