<?php
/**
 * Delcampe Logging Functions
 * 
 * Provides centralized logging functionality for the plugin
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.10.2.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Write a message to the Delcampe log file
 * 
 * @param string $message The message to log
 * @param string $level Log level (debug, info, warning, error)
 * @return void
 */
function delcampe_log($message, $level = 'info') {
    // Check if logging is enabled
    $logging_enabled = get_option(DELCAMPE_LOGGING_OPTION, false);
    
    // For debug messages, only log if WP_DEBUG is also true
    if ($level === 'debug' && (!defined('WP_DEBUG') || !WP_DEBUG)) {
        return;
    }
    
    // Always log errors and warnings
    if (!$logging_enabled && !in_array($level, array('error', 'warning'))) {
        return;
    }
    
    // Ensure log directory exists
    if (defined('DELCAMPE_LOG_DIR') && !file_exists(DELCAMPE_LOG_DIR)) {
        wp_mkdir_p(DELCAMPE_LOG_DIR);
    }
    
    // Determine which log file to use
    $log_file = ($level === 'debug') ? DELCAMPE_DEBUG_LOG_FILE : DELCAMPE_LOG_FILE;
    
    if (!defined('DELCAMPE_LOG_FILE') || !$log_file) {
        return;
    }
    
    // Format the log entry
    $timestamp = current_time('Y-m-d H:i:s');
    $level_upper = strtoupper($level);
    $log_entry = "[{$timestamp}] [{$level_upper}] {$message}" . PHP_EOL;
    
    // Write to log file with explicit error handling
    $written = file_put_contents($log_file, $log_entry, FILE_APPEND | LOCK_EX);
    if ($written === false) {
        // Fallback to PHP error_log if custom log fails
        error_log('[Delcampe Log Fallback] ' . $log_entry);
    }
    
    // Rotate log if it's too large (10MB)
    if (file_exists($log_file) && filesize($log_file) > 10485760) {
        $backup_file = $log_file . '.' . date('Y-m-d-His');
        if (!@rename($log_file, $backup_file)) {
            // If rename fails, log a warning
            error_log('[Delcampe Log Rotation] Failed to rotate log file: ' . $log_file);
        }
    }
}

/**
 * Write a debug message to the Delcampe debug log
 * 
 * @param string $message The debug message
 * @return void
 */
function delcampe_debug($message) {
    delcampe_log($message, 'debug');
}

/**
 * Write an error message to the Delcampe log
 * 
 * @param string $message The error message
 * @return void
 */
function delcampe_error($message) {
    delcampe_log($message, 'error');
}

/**
 * Write a warning message to the Delcampe log
 * 
 * @param string $message The warning message
 * @return void
 */
function delcampe_warning($message) {
    delcampe_log($message, 'warning');
}

/**
 * Write an info message to the Delcampe log
 * 
 * @param string $message The info message
 * @return void
 */
function delcampe_info($message) {
    delcampe_log($message, 'info');
}

/**
 * Whether Delcampe HTTP call logging is enabled.
 */
function delcampe_is_call_log_enabled() {
    $enabled = defined('DELCAMPE_ENABLE_CALL_LOG') && DELCAMPE_ENABLE_CALL_LOG;
    if (!$enabled) {
        $opt = get_option('delcampe_enable_call_log', 'no');
        $enabled = ($opt === 'yes');
    }
    $enabled = (bool) apply_filters('delcampe_enable_call_log', $enabled);
    return $enabled;
}

/**
 * Write a single-line HTTP call entry to the Delcampe call log.
 * Redacts token query param.
 */
function delcampe_call_log($method, $url, $status = null, $elapsed_ms = null) {
    if (!delcampe_is_call_log_enabled()) {
        return;
    }
    if (defined('DELCAMPE_LOG_DIR') && !file_exists(DELCAMPE_LOG_DIR)) {
        wp_mkdir_p(DELCAMPE_LOG_DIR);
    }
    $file = defined('DELCAMPE_CALLS_LOG_FILE') ? DELCAMPE_CALLS_LOG_FILE : (WP_CONTENT_DIR . '/delcampe-logs/delcalls.log');

    $p = wp_parse_url($url);
    $host = isset($p['host']) ? $p['host'] : '';
    $path = isset($p['path']) ? $p['path'] : '';
    $query = isset($p['query']) ? $p['query'] : '';
    if ($query !== '') {
        // Redact token parameter
        parse_str($query, $q);
        if (isset($q['token'])) { $q['token'] = '***'; }
        $query = http_build_query($q);
    }
    $when = gmdate('c');
    $status_s = ($status === null) ? '-' : (string)$status;
    $elapsed_s = ($elapsed_ms === null) ? '-' : number_format((float)$elapsed_ms, 2, '.', '');
    $line = sprintf('[%s] %s %s %s%s%s status=%s ms=%s',
        $when,
        strtoupper($method ?: 'GET'),
        $host,
        $path,
        ($query !== '' ? '?' : ''),
        $query,
        $status_s,
        $elapsed_s
    );
    @file_put_contents($file, $line . "\n", FILE_APPEND | LOCK_EX);
}

/**
 * Hook into WordPress HTTP API to log all calls to rest.delcampe.net
 */
function delcampe_http_api_debug_logger($response, $type, $class, $args, $url) {
    // Only log final responses
    if ($type !== 'response') { return; }
    if (stripos($url, 'rest.delcampe.net') === false) { return; }
    $method = isset($args['method']) ? $args['method'] : 'GET';
    $status = null;
    if (is_wp_error($response)) {
        $status = $response->get_error_code();
    } elseif (is_array($response) && isset($response['response']['code'])) {
        $status = (int)$response['response']['code'];
    }
    delcampe_call_log($method, $url, $status, null);
}

if (function_exists('add_action')) {
    add_action('http_api_debug', 'delcampe_http_api_debug_logger', 10, 5);
}

/**
 * Write a structured entry to the publish audit log (JSON lines).
 * Includes request/response details for create/update calls.
 *
 * @param array $entry Arbitrary data to log (will be JSON-encoded)
 */
function delcampe_is_publish_audit_enabled() {
    // Enable if constant true, or option enabled, or filter forces it; default to false
    $enabled = defined('DELCAMPE_ENABLE_PUBLISH_AUDIT') && DELCAMPE_ENABLE_PUBLISH_AUDIT;
    if (!$enabled) {
        $opt = get_option('delcampe_enable_publish_audit', 'no');
        $enabled = ($opt === 'yes');
    }
    /**
     * Filter to enable/disable publish audit logging dynamically.
     */
    $enabled = (bool) apply_filters('delcampe_enable_publish_audit', $enabled);
    return $enabled;
}

function delcampe_publish_audit(array $entry) {
    if (!delcampe_is_publish_audit_enabled()) {
        return; // Skip heavy audit logging when disabled
    }
    // Ensure directory exists
    if (defined('DELCAMPE_LOG_DIR') && !file_exists(DELCAMPE_LOG_DIR)) {
        wp_mkdir_p(DELCAMPE_LOG_DIR);
    }
    $file = defined('DELCAMPE_PUBLISH_LOG_FILE') ? DELCAMPE_PUBLISH_LOG_FILE : (WP_CONTENT_DIR . '/delcampe-logs/delcampe-publish-audit.log');
    $payload = $entry;
    $payload['ts'] = date('c');
    // Bound body size if present
    if (isset($payload['response_body']) && is_string($payload['response_body'])) {
        $payload['response_body'] = substr($payload['response_body'], 0, 1000);
    }
    if (isset($payload['request_body']) && is_string($payload['request_body'])) {
        $payload['request_body'] = substr($payload['request_body'], 0, 1000);
    }
    $json = wp_json_encode($payload) . "\n";
    @file_put_contents($file, $json, FILE_APPEND | LOCK_EX);
}
