<?php
/**
 * Migration to fix profile assignments in delcampe_listings table
 * 
 * This migration ensures all listings have a valid profile_id
 * and profile_name to prevent "No profile" display issues.
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.10.34.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Fix profile assignments for all listings
 * 
 * @return bool True on success, false on failure
 */
function delcampe_fix_profile_assignments() {
    global $wpdb;
    
    $table_name = $wpdb->prefix . 'delcampe_listings';
    $profiles_table = $wpdb->prefix . 'delcampe_profiles';
    
    // Step 1: Get default profile (prefer profile_id = 2, or first available)
    $default_profile = $wpdb->get_row("
        SELECT profile_id, profile_name 
        FROM $profiles_table 
        WHERE profile_id = 2
    ");
    
    if (!$default_profile) {
        // Fallback to first available profile
        $default_profile = $wpdb->get_row("
            SELECT profile_id, profile_name 
            FROM $profiles_table 
            ORDER BY profile_id ASC 
            LIMIT 1
        ");
    }
    
    if (!$default_profile) {
        // Create a default profile if none exists
        $wpdb->insert(
            $profiles_table,
            array(
                'profile_id' => 2,
                'profile_name' => 'plate blocks',
                'profile_description' => 'Default profile for stamp listings',
                'delcampe_category_id' => '13889',
                'delcampe_category_name' => 'Plate Number & Inscriptions',
                'is_default' => 1,
                'date_created' => current_time('mysql'),
                'date_modified' => current_time('mysql')
            )
        );
        
        $default_profile = (object) array(
            'profile_id' => 2,
            'profile_name' => 'plate blocks'
        );
    }
    
    // Step 2: Update all listings with NULL or 0 profile_id
    $updated = $wpdb->query($wpdb->prepare("
        UPDATE $table_name
        SET 
            profile_id = %d,
            profile_name = %s
        WHERE 
            profile_id IS NULL 
            OR profile_id = 0 
            OR profile_id = ''
    ", $default_profile->profile_id, $default_profile->profile_name));
    
    // Step 3: Sync profile names for listings with valid profile_id but missing name
    $wpdb->query("
        UPDATE $table_name l
        INNER JOIN $profiles_table p ON l.profile_id = p.profile_id
        SET l.profile_name = p.profile_name
        WHERE l.profile_name IS NULL 
        OR l.profile_name = '' 
        OR l.profile_name = 'No profile'
    ");
    
    // Log the migration
    error_log("Delcampe Profile Migration: Updated $updated listings with default profile");
    
    return true;
}

// Run the migration
add_action('init', function() {
    // Check if migration has been run
    $migration_version = get_option('delcampe_profile_migration_version', '0');
    
    if (version_compare($migration_version, '1.10.34.0', '<')) {
        delcampe_fix_profile_assignments();
        update_option('delcampe_profile_migration_version', '1.10.34.0');
    }
});

// Also provide a function that can be called manually
if (!function_exists('delcampe_force_fix_profiles')) {
    function delcampe_force_fix_profiles() {
        return delcampe_fix_profile_assignments();
    }
}