<?php
/**
 * Delcampe Profiles Admin Page
 * Version: 1.1.4.3
 *
 * Handles the admin interface for managing Delcampe listing profiles.
 * Provides UI for creating, editing, and managing category mappings and listing settings.
 * 
 * CHANGELOG:
 * Version 1.1.4.3 - Removed "Select Delcampe Category" button, keeping only saved categories
 * Version 1.1.4.2 - Fixed asset path for CSS and JS files
 * Version 1.1.4.1 - Fixed text domain from 'wc-delcampe-integration' to 'wc-delcampe-integration'
 *                 - Added stamp-specific fields integration
 *                 - Fixed AJAX handler registration
 *
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Admin
 * @since 1.1.3.0
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class Delcampe_Profiles_Admin
 * 
 * Manages the profiles admin page including list table and edit forms
 * 
 * @version 1.9.0.4
 */
class Delcampe_Profiles_Admin {

    /**
     * Profiles model instance
     * @var Delcampe_Profiles_Model
     */
    private $profiles_model;

    /**
     * Constructor
     */
    public function __construct() {
        $this->profiles_model = Delcampe_Profiles_Model::get_instance();
        
        // Add hooks for AJAX handlers
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );
        add_action( 'wp_ajax_delcampe_search_woo_categories', array( $this, 'ajax_search_woo_categories' ) );
        add_action( 'wp_ajax_delcampe_load_category_path', array( $this, 'ajax_load_category_path' ) );
        add_action( 'wp_ajax_delcampe_get_saved_categories', array( $this, 'ajax_get_saved_categories' ) );
        add_action( 'wp_ajax_delcampe_check_category_leaf', array( $this, 'ajax_check_category_leaf' ) );
    }

    /**
     * Render the main profiles page
     * 
     * @return void
     */
    public function render_page() {
        // Check if we're editing a profile
        $action = isset( $_GET['action'] ) ? sanitize_text_field( $_GET['action'] ) : '';
        $profile_id = isset( $_GET['profile_id'] ) ? intval( $_GET['profile_id'] ) : 0;

        // Handle form submissions
        if ( isset( $_POST['submit'] ) && check_admin_referer( 'delcampe_save_profile' ) ) {
            $this->handle_save_profile();
            return;
        }

        // Handle bulk actions
        if ( isset( $_POST['action'] ) && $_POST['action'] != -1 ) {
            $this->handle_bulk_action();
            return;
        }

        // Render appropriate view
        switch ( $action ) {
            case 'edit':
                $this->render_edit_page( $profile_id );
                break;
            case 'new':
                $this->render_edit_page( 0 );
                break;
            case 'delete':
                $this->handle_delete_profile( $profile_id );
                break;
            case 'duplicate':
                $this->handle_duplicate_profile( $profile_id );
                break;
            default:
                $this->render_list_page();
                break;
        }
    }

    /**
     * Render the profiles list page
     * 
     * @return void
     */
    private function render_list_page() {
        // Create list table instance
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'tables/class-delcampe-profiles-list-table.php';
        $list_table = new Delcampe_Profiles_List_Table();
        $list_table->prepare_items();
        ?>
        <div class="wrap">
            <h1 class="wp-heading-inline"><?php _e( 'Delcampe Profiles', 'wc-delcampe-integration' ); ?></h1>
            <a href="<?php echo admin_url( 'admin.php?page=delcampe-profiles&action=new' ); ?>" class="page-title-action">
                <?php _e( 'Add New', 'wc-delcampe-integration' ); ?>
            </a>
            
            <?php if ( isset( $_GET['message'] ) ) : ?>
                <?php $this->display_admin_notice(); ?>
            <?php endif; ?>

            <form method="post">
                <?php $list_table->search_box( __( 'Search Profiles', 'wc-delcampe-integration' ), 'profile' ); ?>
                <?php $list_table->display(); ?>
            </form>
        </div>
        <?php
    }

    /**
     * Render the profile edit page with enhanced stamp-specific fields
     * 
     * @param int $profile_id Profile ID to edit (0 for new)
     * @return void
     * @version 1.1.4.3
     */
    private function render_edit_page( $profile_id = 0 ) {
        // Get profile data
        if ( $profile_id > 0 ) {
            $profile = $this->profiles_model->get_profile( $profile_id );
            if ( ! $profile ) {
                wp_die( __( 'Profile not found.', 'wc-delcampe-integration' ) );
            }
        } else {
            $profile = $this->profiles_model->get_new_profile_template();
        }

        // Get WooCommerce categories for dropdown
        $woo_categories = $this->get_woo_categories_hierarchy();
        
        // Load stamp profile fields if available
        $stamp_fields = null;
        $stamp_fields_file = plugin_dir_path( __FILE__ ) . 'class-delcampe-stamp-profile-fields.php';
        if ( file_exists( $stamp_fields_file ) ) {
            require_once $stamp_fields_file;
            if ( class_exists( 'Delcampe_Stamp_Profile_Fields' ) ) {
                $stamp_fields = new Delcampe_Stamp_Profile_Fields();
            }
        }
        
        ?>
        <div class="wrap">
            <h1>
                <?php echo $profile_id > 0 ? 
                    __( 'Edit Profile', 'wc-delcampe-integration' ) : 
                    __( 'New Profile', 'wc-delcampe-integration' ); 
                ?>
            </h1>

            <form method="post" id="delcampe-profile-form">
                <?php wp_nonce_field( 'delcampe_save_profile' ); ?>
                <input type="hidden" name="profile_id" value="<?php echo esc_attr( $profile['profile_id'] ); ?>">

                <div id="poststuff">
                    <div id="post-body" class="metabox-holder columns-2">
                        
                        <!-- Main Column -->
                        <div id="post-body-content">
                            
                            <!-- General Settings -->
                            <div class="postbox">
                                <h2 class="hndle"><?php _e( 'General Settings', 'wc-delcampe-integration' ); ?></h2>
                                <div class="inside">
                                    <table class="form-table">
                                        <tr>
                                            <th scope="row">
                                                <label for="profile_name">
                                                    <?php _e( 'Profile Name', 'wc-delcampe-integration' ); ?> 
                                                    <span class="required">*</span>
                                                </label>
                                            </th>
                                            <td>
                                                <input type="text" 
                                                       id="profile_name" 
                                                       name="profile_name" 
                                                       value="<?php echo esc_attr( $profile['profile_name'] ); ?>" 
                                                       class="regular-text" 
                                                       required>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th scope="row">
                                                <label for="profile_description">
                                                    <?php _e( 'Description', 'wc-delcampe-integration' ); ?>
                                                </label>
                                            </th>
                                            <td>
                                                <textarea id="profile_description" 
                                                          name="profile_description" 
                                                          rows="3" 
                                                          class="large-text"><?php echo esc_textarea( $profile['profile_description'] ); ?></textarea>
                                                <p class="description">
                                                    <?php _e( 'Optional description for internal use only.', 'wc-delcampe-integration' ); ?>
                                                </p>
                                            </td>
                                        </tr>
                                    </table>
                                </div>
                            </div>

                            <!-- Category Mapping -->
                            <div class="postbox">
                                <h2 class="hndle"><?php _e( 'Category Mapping', 'wc-delcampe-integration' ); ?></h2>
                                <div class="inside">
                                    <table class="form-table">
                                        <tr>
                                            <th scope="row">
                                                <label for="woo_category_id">
                                                    <?php _e( 'WooCommerce Category', 'wc-delcampe-integration' ); ?>
                                                </label>
                                            </th>
                                            <td>
                                                <select id="woo_category_id" 
                                                        name="woo_category_id" 
                                                        class="delcampe-category-select">
                                                    <option value=""><?php _e( '— Select Category —', 'wc-delcampe-integration' ); ?></option>
                                                    <?php echo $this->render_category_options( $woo_categories, $profile['woo_category_id'] ); ?>
                                                </select>
                                                <p class="description">
                                                    <?php _e( 'Products in this WooCommerce category will use this profile by default.', 'wc-delcampe-integration' ); ?>
                                                </p>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th scope="row">
                                                <label>
                                                    <?php _e( 'Delcampe Category', 'wc-delcampe-integration' ); ?>
                                                    <span class="required">*</span>
                                                </label>
                                            </th>
                                            <td>
                                                <input type="hidden" 
                                                       id="delcampe_category_id" 
                                                       name="delcampe_category_id" 
                                                       value="<?php echo esc_attr( $profile['delcampe_category_id'] ); ?>">
                                                <input type="hidden" 
                                                       id="delcampe_category_name" 
                                                       name="delcampe_category_name" 
                                                       value="<?php echo esc_attr( $profile['delcampe_category_name'] ); ?>">
                                                <input type="hidden" 
                                                       id="delcampe_category_path" 
                                                       name="delcampe_category_path" 
                                                       value="<?php echo esc_attr( $profile['delcampe_category_path'] ); ?>">
                                                
                                                <div id="delcampe-category-display">
                                                    <?php if ( ! empty( $profile['delcampe_category_id'] ) ) : ?>
                                                        <div class="delcampe-category-selected" style="padding: 10px; background: #f0f8ff; border: 1px solid #0073aa; border-radius: 4px;">
                                                            <strong style="color: #0073aa;"><?php echo esc_html( $profile['delcampe_category_name'] ); ?></strong>
                                                            <?php if ( ! empty( $profile['delcampe_category_path'] ) ) : ?>
                                                                <br><small style="color: #666;">📁 <?php echo esc_html( $profile['delcampe_category_path'] ); ?></small>
                                                            <?php endif; ?>
                                                            <br><small style="color: #999;">ID: <?php echo esc_html( $profile['delcampe_category_id'] ); ?></small>
                                                        </div>
                                                    <?php else : ?>
                                                        <div style="padding: 10px; background: #fff3cd; border: 1px solid #ffc107; border-radius: 4px;">
                                                            <em style="color: #856404;">⚠️ <?php _e( 'No category selected', 'wc-delcampe-integration' ); ?></em>
                                                        </div>
                                                    <?php endif; ?>
                                                </div>
                                                
                                                <p>
                                                    <button type="button" 
                                                            class="button button-primary" 
                                                            id="select-from-saved">
                                                        <?php _e( 'Select from Saved Categories', 'wc-delcampe-integration' ); ?>
                                                    </button>
                                                </p>
                                                <p class="description">
                                                    <?php 
                                                    printf(
                                                        __( 'You must first save categories in %sSettings > Categories%s before you can select them here.', 'wc-delcampe-integration' ),
                                                        '<a href="' . admin_url( 'admin.php?page=delcampe-settings&tab=categories' ) . '">',
                                                        '</a>'
                                                    ); 
                                                    ?>
                                                </p>
                                            </td>
                                        </tr>
                                    </table>
                                </div>
                            </div>

                            <!-- Listing Details -->
                            <div class="postbox">
                                <h2 class="hndle"><?php _e( 'Listing Details', 'wc-delcampe-integration' ); ?></h2>
                                <div class="inside">
                                    <table class="form-table">
                                        <tr>
                                            <th scope="row">
                                                <label for="listing_duration">
                                                    <?php _e( 'Listing Duration', 'wc-delcampe-integration' ); ?>
                                                </label>
                                            </th>
                                            <td>
                                                <select id="listing_duration" 
                                                        name="listing_details[duration]">
                                                    <option value="7" <?php selected( $profile['listing_details']['duration'], '7' ); ?>>
                                                        <?php _e( '7 days', 'wc-delcampe-integration' ); ?>
                                                    </option>
                                                    <option value="14" <?php selected( $profile['listing_details']['duration'], '14' ); ?>>
                                                        <?php _e( '14 days', 'wc-delcampe-integration' ); ?>
                                                    </option>
                                                    <option value="30" <?php selected( $profile['listing_details']['duration'], '30' ); ?>>
                                                        <?php _e( '30 days', 'wc-delcampe-integration' ); ?>
                                                    </option>
                                                    <option value="60" <?php selected( $profile['listing_details']['duration'], '60' ); ?>>
                                                        <?php _e( '60 days', 'wc-delcampe-integration' ); ?>
                                                    </option>
                                                    <option value="90" <?php selected( $profile['listing_details']['duration'], '90' ); ?>>
                                                        <?php _e( '90 days', 'wc-delcampe-integration' ); ?>
                                                    </option>
                                                </select>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th scope="row">
                                                <label for="auto_renew">
                                                    <?php _e( 'Auto-renew Listings', 'wc-delcampe-integration' ); ?>
                                                </label>
                                            </th>
                                            <td>
                                                <label>
                                                    <input type="checkbox" 
                                                           id="auto_renew" 
                                                           name="listing_details[auto_renew]" 
                                                           value="1" 
                                                           <?php checked( $profile['listing_details']['auto_renew'], '1' ); ?>>
                                                    <?php _e( 'Automatically renew listings when they expire', 'wc-delcampe-integration' ); ?>
                                                </label>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th scope="row">
                                                <label for="price_adjustment">
                                                    <?php _e( 'Price Adjustment', 'wc-delcampe-integration' ); ?>
                                                </label>
                                            </th>
                                            <td>
                                                <input type="text" 
                                                       id="price_adjustment" 
                                                       name="listing_details[price_adjustment]" 
                                                       value="<?php echo esc_attr( $profile['listing_details']['price_adjustment'] ); ?>" 
                                                       class="small-text">
                                                <p class="description">
                                                    <?php _e( 'Examples: +10% (increase by 10%), -5 (decrease by 5), 199 (fixed price)', 'wc-delcampe-integration' ); ?>
                                                </p>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th scope="row">
                                                <label for="quantity_mode">
                                                    <?php _e( 'Quantity Mode', 'wc-delcampe-integration' ); ?>
                                                </label>
                                            </th>
                                            <td>
                                                <select id="quantity_mode" 
                                                        name="listing_details[quantity_mode]">
                                                    <option value="stock" <?php selected( $profile['listing_details']['quantity_mode'], 'stock' ); ?>>
                                                        <?php _e( 'Use WooCommerce stock quantity', 'wc-delcampe-integration' ); ?>
                                                    </option>
                                                    <option value="fixed" <?php selected( $profile['listing_details']['quantity_mode'], 'fixed' ); ?>>
                                                        <?php _e( 'Use fixed quantity', 'wc-delcampe-integration' ); ?>
                                                    </option>
                                                </select>
                                                <div id="fixed-quantity-wrapper" style="margin-top: 10px; <?php echo $profile['listing_details']['quantity_mode'] !== 'fixed' ? 'display:none;' : ''; ?>">
                                                    <label>
                                                        <?php _e( 'Fixed Quantity:', 'wc-delcampe-integration' ); ?>
                                                        <input type="number" 
                                                               name="listing_details[fixed_quantity]" 
                                                               value="<?php echo esc_attr( $profile['listing_details']['fixed_quantity'] ); ?>" 
                                                               min="1" 
                                                               class="small-text">
                                                    </label>
                                                </div>
                                            </td>
                                        </tr>
                                    </table>
                                </div>
                            </div>

                            <?php 
                            // Render stamp-specific fields if available
                            if ( $stamp_fields ) {
                                echo $stamp_fields->render_stamp_fields( $profile );
                                echo $stamp_fields->render_shipping_fields( $profile );
                                echo $stamp_fields->render_payment_fields( $profile );
                                echo $stamp_fields->render_description_fields( $profile );
                            }
                            ?>

                            <!-- Sync Settings -->
                            <div class="postbox">
                                <h2 class="hndle"><?php _e( 'Sync Settings', 'wc-delcampe-integration' ); ?></h2>
                                <div class="inside">
                                    <table class="form-table">
                                        <tr>
                                            <th scope="row">
                                                <?php _e( 'Sync Options', 'wc-delcampe-integration' ); ?>
                                            </th>
                                            <td>
                                                <fieldset>
                                                    <label>
                                                        <input type="checkbox" 
                                                               name="sync_settings[sync_title]" 
                                                               value="1" 
                                                               <?php checked( $profile['sync_settings']['sync_title'], '1' ); ?>>
                                                        <?php _e( 'Sync product title', 'wc-delcampe-integration' ); ?>
                                                    </label><br>
                                                    <label>
                                                        <input type="checkbox" 
                                                               name="sync_settings[sync_description]" 
                                                               value="1" 
                                                               <?php checked( $profile['sync_settings']['sync_description'], '1' ); ?>>
                                                        <?php _e( 'Sync product description', 'wc-delcampe-integration' ); ?>
                                                    </label><br>
                                                    <label>
                                                        <input type="checkbox" 
                                                               name="sync_settings[sync_price]" 
                                                               value="1" 
                                                               <?php checked( $profile['sync_settings']['sync_price'], '1' ); ?>>
                                                        <?php _e( 'Sync product price', 'wc-delcampe-integration' ); ?>
                                                    </label><br>
                                                    <label>
                                                        <input type="checkbox" 
                                                               name="sync_settings[sync_quantity]" 
                                                               value="1" 
                                                               <?php checked( $profile['sync_settings']['sync_quantity'], '1' ); ?>>
                                                        <?php _e( 'Sync product quantity/stock', 'wc-delcampe-integration' ); ?>
                                                    </label><br>
                                                    <label>
                                                        <input type="checkbox" 
                                                               name="sync_settings[sync_images]" 
                                                               value="1" 
                                                               <?php checked( $profile['sync_settings']['sync_images'], '1' ); ?>>
                                                        <?php _e( 'Sync product images', 'wc-delcampe-integration' ); ?>
                                                    </label><br>
                                                    <label>
                                                        <input type="checkbox" 
                                                               name="sync_settings[sync_attributes]" 
                                                               value="1" 
                                                               <?php checked( $profile['sync_settings']['sync_attributes'], '1' ); ?>>
                                                        <?php _e( 'Sync product attributes', 'wc-delcampe-integration' ); ?>
                                                    </label>
                                                </fieldset>
                                            </td>
                                        </tr>
                                        <tr>
                                            <th scope="row">
                                                <label for="auto_sync">
                                                    <?php _e( 'Automatic Sync', 'wc-delcampe-integration' ); ?>
                                                </label>
                                            </th>
                                            <td>
                                                <label>
                                                    <input type="checkbox" 
                                                           id="auto_sync" 
                                                           name="sync_settings[auto_sync]" 
                                                           value="1" 
                                                           <?php checked( isset($profile['sync_settings']['auto_sync']) ? $profile['sync_settings']['auto_sync'] : '', '1' ); ?>>
                                                    <?php _e( 'Enable automatic synchronization', 'wc-delcampe-integration' ); ?>
                                                </label>
                                                <div id="sync-interval-wrapper" style="margin-top: 10px; <?php echo (!isset($profile['sync_settings']['auto_sync']) || $profile['sync_settings']['auto_sync'] !== '1') ? 'display:none;' : ''; ?>">
                                                    <label>
                                                        <?php _e( 'Sync Interval:', 'wc-delcampe-integration' ); ?>
                                                        <select name="sync_settings[sync_interval]">
                                                            <option value="1" <?php selected( $profile['sync_settings']['sync_interval'], '1' ); ?>>
                                                                <?php _e( 'Every hour', 'wc-delcampe-integration' ); ?>
                                                            </option>
                                                            <option value="6" <?php selected( $profile['sync_settings']['sync_interval'], '6' ); ?>>
                                                                <?php _e( 'Every 6 hours', 'wc-delcampe-integration' ); ?>
                                                            </option>
                                                            <option value="12" <?php selected( $profile['sync_settings']['sync_interval'], '12' ); ?>>
                                                                <?php _e( 'Every 12 hours', 'wc-delcampe-integration' ); ?>
                                                            </option>
                                                            <option value="24" <?php selected( $profile['sync_settings']['sync_interval'], '24' ); ?>>
                                                                <?php _e( 'Daily', 'wc-delcampe-integration' ); ?>
                                                            </option>
                                                        </select>
                                                    </label>
                                                </div>
                                            </td>
                                        </tr>
                                    </table>
                                </div>
                            </div>

                        </div><!-- #post-body-content -->

                        <!-- Sidebar -->
                        <div id="postbox-container-1" class="postbox-container">
                            
                            <!-- Save Box -->
                            <div class="postbox">
                                <h2 class="hndle"><?php _e( 'Save Profile', 'wc-delcampe-integration' ); ?></h2>
                                <div class="inside">
                                    <div class="submitbox">
                                        <div id="major-publishing-actions">
                                            <?php if ( $profile_id > 0 ) : ?>
                                                <div id="delete-action">
                                                    <a class="submitdelete deletion" 
                                                       href="<?php echo wp_nonce_url( admin_url( 'admin.php?page=delcampe-profiles&action=delete&profile_id=' . $profile_id ), 'delete-profile_' . $profile_id ); ?>"
                                                       onclick="return confirm('<?php esc_attr_e( 'Are you sure you want to delete this profile?', 'wc-delcampe-integration' ); ?>');">
                                                        <?php _e( 'Delete', 'wc-delcampe-integration' ); ?>
                                                    </a>
                                                </div>
                                            <?php endif; ?>
                                            
                                            <div id="publishing-action">
                                                <input type="submit" 
                                                       name="submit" 
                                                       id="submit" 
                                                       class="button button-primary button-large" 
                                                       value="<?php echo $profile_id > 0 ? 
                                                           esc_attr__( 'Update Profile', 'wc-delcampe-integration' ) : 
                                                           esc_attr__( 'Create Profile', 'wc-delcampe-integration' ); ?>">
                                            </div>
                                            <div class="clear"></div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Profile Options -->
                            <div class="postbox">
                                <h2 class="hndle"><?php _e( 'Profile Options', 'wc-delcampe-integration' ); ?></h2>
                                <div class="inside">
                                    <p>
                                        <label>
                                            <input type="checkbox" 
                                                   name="is_default" 
                                                   value="1" 
                                                   <?php checked( $profile['is_default'], 1 ); ?>>
                                            <?php _e( 'Set as default profile', 'wc-delcampe-integration' ); ?>
                                        </label>
                                    </p>
                                    <p>
                                        <label>
                                            <?php _e( 'Sort Order:', 'wc-delcampe-integration' ); ?>
                                            <input type="number" 
                                                   name="sort_order" 
                                                   value="<?php echo esc_attr( $profile['sort_order'] ); ?>" 
                                                   class="small-text">
                                        </label>
                                    </p>
                                </div>
                            </div>

                            <?php if ( $profile_id > 0 ) : ?>
                                <!-- Profile Info -->
                                <div class="postbox">
                                    <h2 class="hndle"><?php _e( 'Profile Information', 'wc-delcampe-integration' ); ?></h2>
                                    <div class="inside">
                                        <p>
                                            <strong><?php _e( 'Created:', 'wc-delcampe-integration' ); ?></strong><br>
                                            <?php echo date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( $profile['date_created'] ) ); ?>
                                        </p>
                                        <p>
                                            <strong><?php _e( 'Last Modified:', 'wc-delcampe-integration' ); ?></strong><br>
                                            <?php echo date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), strtotime( $profile['date_modified'] ) ); ?>
                                        </p>
                                        <?php
                                        $products_using = $this->profiles_model->get_products_using_profile( $profile_id );
                                        if ( ! empty( $products_using ) ) :
                                        ?>
                                            <p>
                                                <strong><?php _e( 'Products Using This Profile:', 'wc-delcampe-integration' ); ?></strong><br>
                                                <?php echo count( $products_using ); ?>
                                            </p>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            <?php endif; ?>

                        </div><!-- #postbox-container-1 -->

                    </div><!-- #post-body -->
                </div><!-- #poststuff -->

            </form>

        </div><!-- .wrap -->

        <!-- Saved Categories Modal -->
        <div id="saved-categories-modal" class="delcampe-modal" style="display:none;">
            <div class="delcampe-modal-content" style="max-width: 600px;">
                <div class="delcampe-modal-header">
                    <h2><?php _e( 'Select from Saved Categories', 'wc-delcampe-integration' ); ?></h2>
                    <button type="button" class="saved-categories-close delcampe-modal-close">&times;</button>
                </div>
                <div class="delcampe-modal-body">
                    <div id="saved-categories-list">
                        <p class="loading"><?php _e( 'Loading saved categories...', 'wc-delcampe-integration' ); ?></p>
                    </div>
                </div>
                <div class="delcampe-modal-footer">
                    <button type="button" class="button saved-categories-close">
                        <?php _e( 'Close', 'wc-delcampe-integration' ); ?>
                    </button>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Handle saving a profile
     * 
     * @return void
     */
    private function handle_save_profile() {
        $profile_id = isset( $_POST['profile_id'] ) ? intval( $_POST['profile_id'] ) : 0;
        
        // Prepare profile data
        $data = array(
            'profile_name' => sanitize_text_field( $_POST['profile_name'] ),
            'profile_description' => sanitize_textarea_field( $_POST['profile_description'] ),
            'delcampe_category_id' => sanitize_text_field( $_POST['delcampe_category_id'] ),
            'delcampe_category_name' => sanitize_text_field( $_POST['delcampe_category_name'] ),
            'delcampe_category_path' => sanitize_text_field( $_POST['delcampe_category_path'] ),
            'woo_category_id' => intval( $_POST['woo_category_id'] ),
            'listing_details' => isset( $_POST['listing_details'] ) ? $_POST['listing_details'] : array(),
            'sync_settings' => isset( $_POST['sync_settings'] ) ? $_POST['sync_settings'] : array(),
            'is_default' => isset( $_POST['is_default'] ) ? 1 : 0,
            'sort_order' => intval( $_POST['sort_order'] ),
        );
        
        // Validate category is a leaf category (v1.10.18.14)
        // Check both delcampe_category_id and listing_details[category_id]
        $category_to_validate = null;
        $category_name = '';
        
        if ( ! empty( $data['delcampe_category_id'] ) ) {
            $category_to_validate = $data['delcampe_category_id'];
            $category_name = $data['delcampe_category_name'];
        } elseif ( ! empty( $data['listing_details']['category_id'] ) ) {
            $category_to_validate = $data['listing_details']['category_id'];
            $category_name = ! empty( $data['listing_details']['category_name'] ) ? $data['listing_details']['category_name'] : 'Category ' . $category_to_validate;
        }
        
        if ( $category_to_validate ) {
            require_once plugin_dir_path( dirname( __FILE__ ) ) . 'class-delcampe-category-manager.php';
            $category_manager = Delcampe_Category_Manager::get_instance();
            
            $has_children = $category_manager->check_category_has_children( $category_to_validate );
            
            if ( is_wp_error( $has_children ) ) {
                // Log the error but allow save to continue
                delcampe_log( '[Profile Save] Could not validate category ' . $category_to_validate . ': ' . $has_children->get_error_message() );
            } elseif ( $has_children === true ) {
                // Category has children - prevent save
                wp_die( 
                    '<h1>' . __( 'Category Selection Error', 'wc-delcampe-integration' ) . '</h1>' .
                    '<p>' . sprintf( 
                        __( 'The selected category "%s" (ID: %s) has subcategories and cannot be used for listings.', 'wc-delcampe-integration' ),
                        esc_html( $category_name ),
                        esc_html( $category_to_validate )
                    ) . '</p>' .
                    '<p>' . __( 'Please go back and select a leaf category (one without subcategories).', 'wc-delcampe-integration' ) . '</p>' .
                    '<p><a href="javascript:history.back()">' . __( '← Go Back', 'wc-delcampe-integration' ) . '</a></p>',
                    __( 'Invalid Category', 'wc-delcampe-integration' ),
                    array( 'response' => 400, 'back_link' => true )
                );
            }
        }

        // Get WooCommerce category name
        if ( $data['woo_category_id'] ) {
            $term = get_term( $data['woo_category_id'], 'product_cat' );
            if ( $term && ! is_wp_error( $term ) ) {
                $data['woo_category_name'] = $term->name;
            }
        }

        // SECURITY FIX v1.10.21.2: Properly sanitize complex POST data
        // Handle stamp-specific fields if present
        if ( isset( $_POST['stamp_settings'] ) && is_array( $_POST['stamp_settings'] ) ) {
            $data['stamp_settings'] = map_deep( $_POST['stamp_settings'], 'sanitize_text_field' );
        }
        
        // Handle shipping model ID separately (v1.9.0.3)
        if ( isset( $_POST['shipping_model_id'] ) ) {
            $data['shipping_model_id'] = intval( $_POST['shipping_model_id'] );
        }
        
        if ( isset( $_POST['shipping_settings'] ) && is_array( $_POST['shipping_settings'] ) ) {
            $data['shipping_settings'] = map_deep( $_POST['shipping_settings'], 'sanitize_text_field' );
        }
        if ( isset( $_POST['payment_settings'] ) && is_array( $_POST['payment_settings'] ) ) {
            $data['payment_settings'] = map_deep( $_POST['payment_settings'], 'sanitize_text_field' );
        }
        if ( isset( $_POST['description_settings'] ) && is_array( $_POST['description_settings'] ) ) {
            // Description might contain HTML, use wp_kses_post for safe HTML
            $data['description_settings'] = map_deep( $_POST['description_settings'], 'wp_kses_post' );
        }

        // Save profile
        if ( $profile_id > 0 ) {
            $result = $this->profiles_model->update_profile( $profile_id, $data );
            $message = 'updated';
        } else {
            $result = $this->profiles_model->create_profile( $data );
            $profile_id = $result;
            $message = 'created';
        }

        // Handle default profile setting
        if ( $data['is_default'] && ! is_wp_error( $result ) ) {
            $this->profiles_model->set_default_profile( $profile_id );
        }

        // Redirect with message
        if ( is_wp_error( $result ) ) {
            wp_die( $result->get_error_message() );
        } else {
            wp_redirect( admin_url( 'admin.php?page=delcampe-profiles&message=' . $message ) );
            exit;
        }
    }

    /**
     * Handle deleting a profile
     * 
     * @param int $profile_id Profile ID to delete
     * @return void
     */
    private function handle_delete_profile( $profile_id ) {
        check_admin_referer( 'delete-profile_' . $profile_id );
        
        $result = $this->profiles_model->delete_profile( $profile_id );
        
        if ( is_wp_error( $result ) ) {
            wp_die( $result->get_error_message() );
        } else {
            wp_redirect( admin_url( 'admin.php?page=delcampe-profiles&message=deleted' ) );
            exit;
        }
    }

    /**
     * Handle duplicating a profile
     * 
     * @param int $profile_id Profile ID to duplicate
     * @return void
     */
    private function handle_duplicate_profile( $profile_id ) {
        $new_id = $this->profiles_model->duplicate_profile( $profile_id );
        
        if ( is_wp_error( $new_id ) ) {
            wp_die( $new_id->get_error_message() );
        } else {
            wp_redirect( admin_url( 'admin.php?page=delcampe-profiles&action=edit&profile_id=' . $new_id ) );
            exit;
        }
    }

    /**
     * Handle bulk actions
     * 
     * @return void
     */
    private function handle_bulk_action() {
        $action = $_POST['action'];
        $profile_ids = isset( $_POST['profile_ids'] ) ? array_map( 'intval', $_POST['profile_ids'] ) : array();
        
        if ( empty( $profile_ids ) ) {
            return;
        }

        switch ( $action ) {
            case 'delete':
                foreach ( $profile_ids as $profile_id ) {
                    $this->profiles_model->delete_profile( $profile_id );
                }
                wp_redirect( admin_url( 'admin.php?page=delcampe-profiles&message=bulk_deleted' ) );
                exit;
                break;
        }
    }

    /**
     * Display admin notices
     * 
     * @return void
     */
    private function display_admin_notice() {
        $message = isset( $_GET['message'] ) ? sanitize_text_field( $_GET['message'] ) : '';
        
        $messages = array(
            'created' => __( 'Profile created successfully.', 'wc-delcampe-integration' ),
            'updated' => __( 'Profile updated successfully.', 'wc-delcampe-integration' ),
            'deleted' => __( 'Profile deleted successfully.', 'wc-delcampe-integration' ),
            'bulk_deleted' => __( 'Selected profiles deleted successfully.', 'wc-delcampe-integration' ),
        );

        if ( isset( $messages[ $message ] ) ) {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html( $messages[ $message ] ) . '</p></div>';
        }
    }

    /**
     * Get WooCommerce categories hierarchy
     * 
     * @return array Categories array
     */
    private function get_woo_categories_hierarchy() {
        $args = array(
            'taxonomy' => 'product_cat',
            'hide_empty' => false,
            'orderby' => 'name',
            'order' => 'ASC',
        );
        
        return get_terms( $args );
    }

    /**
     * Render category options for select
     * 
     * @param array $categories Categories array
     * @param int   $selected   Selected category ID
     * @param int   $level      Indentation level
     * @return string HTML options
     */
    private function render_category_options( $categories, $selected = 0, $level = 0 ) {
        $html = '';
        
        foreach ( $categories as $category ) {
            if ( $category->parent == 0 && $level == 0 ) {
                $html .= sprintf(
                    '<option value="%d" %s>%s%s</option>',
                    $category->term_id,
                    selected( $selected, $category->term_id, false ),
                    str_repeat( '&nbsp;&nbsp;', $level ),
                    esc_html( $category->name )
                );
                
                // Get children
                $html .= $this->render_category_children( $categories, $category->term_id, $selected, $level + 1 );
            }
        }
        
        return $html;
    }

    /**
     * Render child categories
     * 
     * @param array $categories All categories
     * @param int   $parent_id  Parent category ID
     * @param int   $selected   Selected category ID
     * @param int   $level      Indentation level
     * @return string HTML options
     */
    private function render_category_children( $categories, $parent_id, $selected = 0, $level = 1 ) {
        $html = '';
        
        foreach ( $categories as $category ) {
            if ( $category->parent == $parent_id ) {
                $html .= sprintf(
                    '<option value="%d" %s>%s%s</option>',
                    $category->term_id,
                    selected( $selected, $category->term_id, false ),
                    str_repeat( '&nbsp;&nbsp;', $level ),
                    esc_html( $category->name )
                );
                
                // Recursive call for deeper levels
                $html .= $this->render_category_children( $categories, $category->term_id, $selected, $level + 1 );
            }
        }
        
        return $html;
    }

    /**
     * Enqueue scripts and styles
     * 
     * @param string $hook Current admin page hook
     * @return void
     * @version 1.1.4.3
     */
    public function enqueue_scripts( $hook ) {
        // Only load on our admin pages
        if ( strpos( $hook, 'delcampe-profiles' ) === false ) {
            return;
        }

        // Get the correct base URL for assets
        // The profiles admin is in includes/profiles/ but assets are in the root plugin directory
        $plugin_url = plugin_dir_url( dirname( dirname( __FILE__ ) ) ); // Go up two levels from includes/profiles/

        // Compute asset versions using file mtime to avoid stale caches
        $plugin_dir = plugin_dir_path( dirname( dirname( __FILE__ ) ) );
        $css_path   = $plugin_dir . 'assets/css/delcampe-profiles-admin.css';
        $js_path    = $plugin_dir . 'assets/js/delcampe-profiles-admin.js';
        $css_ver    = file_exists( $css_path ) ? filemtime( $css_path ) : '1.1.4.3';
        $js_ver     = file_exists( $js_path ) ? filemtime( $js_path ) : '1.1.4.3';

        // Enqueue styles
        wp_enqueue_style( 
            'delcampe-profiles-admin', 
            $plugin_url . 'assets/css/delcampe-profiles-admin.css',
            array(),
            $css_ver
        );

        // Enqueue scripts
        wp_enqueue_script(
            'delcampe-profiles-admin',
            $plugin_url . 'assets/js/delcampe-profiles-admin.js',
            array( 'jquery' ),
            $js_ver,
            true
        );

        // Localize script with corrected text domain
        wp_localize_script( 'delcampe-profiles-admin', 'delcampe_profiles', array(
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce' => wp_create_nonce( 'delcampe_profiles_nonce' ),
            'strings' => array(
                'loading' => __( 'Loading...', 'wc-delcampe-integration' ),
                'error' => __( 'An error occurred. Please try again.', 'wc-delcampe-integration' ),
                'select_category' => __( 'Please select a category', 'wc-delcampe-integration' ),
                'no_categories' => __( 'No categories found', 'wc-delcampe-integration' ),
                'no_saved_categories' => __( 'No saved categories found. Please go to Settings > Categories to save some categories first.', 'wc-delcampe-integration' ),
                'verifying' => __( 'Verifying category...', 'wc-delcampe-integration' ),
            ),
        ) );
    }

    /**
     * AJAX handler for getting saved categories
     * 
     * @return void
     */
    public function ajax_get_saved_categories() {
        check_ajax_referer( 'delcampe_profiles_nonce', 'nonce' );
        
        global $wpdb;
        $table_name = $wpdb->prefix . DELCAMPE_TABLE_SELECTED_CATEGORIES;
        
        // Determine if has_children column exists (for backward compatibility)
        $has_children_exists = (bool) $wpdb->get_var( $wpdb->prepare( "SHOW COLUMNS FROM {$table_name} LIKE %s", 'has_children' ) );

        // Build column list dynamically
        $columns = 'category_id, category_name, parent_path' . ( $has_children_exists ? ', has_children' : '' );

        // Get all saved categories
        $categories = $wpdb->get_results( "SELECT {$columns} FROM {$table_name} ORDER BY category_name ASC" );

        // If SQL error, surface it instead of showing empty
        if ( ! empty( $wpdb->last_error ) ) {
            wp_send_json_error( sprintf( __( 'Database error: %s', 'wc-delcampe-integration' ), $wpdb->last_error ) );
        }
        
        if ( empty( $categories ) ) {
            wp_send_json_error( __( 'No saved categories found', 'wc-delcampe-integration' ) );
        }
        
        // DB now stores has_children; if NULL, the JS will verify on click
        
        wp_send_json_success( $categories );
    }

    /**
     * AJAX: Verify if a category is a leaf (no children) lazily on selection
     */
    public function ajax_check_category_leaf() {
        check_ajax_referer( 'delcampe_profiles_nonce', 'nonce' );
        $category_id = isset( $_POST['category_id'] ) ? sanitize_text_field( $_POST['category_id'] ) : '';
        if ( empty( $category_id ) ) {
            wp_send_json_error( __( 'Invalid category ID', 'wc-delcampe-integration' ) );
        }

        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'class-delcampe-category-manager.php';
        $manager = Delcampe_Category_Manager::get_instance();
        $has_children = $manager->check_category_has_children( $category_id );

        if ( is_wp_error( $has_children ) ) {
            // Do not block selection on verification errors; let save-time validation enforce
            wp_send_json_success( array(
                'has_children' => null,
                'is_leaf'      => null,
                'warning'      => $has_children->get_error_message(),
            ) );
            return;
        }

        // Cache the result
        $flags_option_key = 'delcampe_category_meta';
        $flags = get_option( $flags_option_key, array() );
        $flags[ $category_id ] = array(
            'has_children' => (bool) $has_children,
            'checked_at'   => time(),
        );
        update_option( $flags_option_key, $flags, false );

        // Persist to DB as well if the category is saved
        global $wpdb;
        $table = $wpdb->prefix . DELCAMPE_TABLE_SELECTED_CATEGORIES;
        $wpdb->update(
            $table,
            array('has_children' => ( $has_children ? 1 : 0 )),
            array('category_id' => $category_id),
            array('%d'),
            array('%s')
        );

        wp_send_json_success( array(
            'has_children' => (bool) $has_children,
            'is_leaf'      => ! $has_children,
        ) );
    }

    /**
     * AJAX handler for searching WooCommerce categories
     * 
     * @return void
     */
    public function ajax_search_woo_categories() {
        check_ajax_referer( 'delcampe_profiles_nonce', 'nonce' );
        
        $search = isset( $_GET['q'] ) ? sanitize_text_field( $_GET['q'] ) : '';
        
        $args = array(
            'taxonomy' => 'product_cat',
            'hide_empty' => false,
            'search' => $search,
            'number' => 20,
        );
        
        $categories = get_terms( $args );
        $results = array();
        
        foreach ( $categories as $category ) {
            $results[] = array(
                'id' => $category->term_id,
                'text' => $category->name,
            );
        }
        
        wp_send_json( array( 'results' => $results ) );
    }

    /**
     * AJAX handler for loading Delcampe category path
     * 
     * @return void
     */
    public function ajax_load_category_path() {
        check_ajax_referer( 'delcampe_profiles_nonce', 'nonce' );
        
        $category_id = isset( $_POST['category_id'] ) ? sanitize_text_field( $_POST['category_id'] ) : '';
        
        if ( empty( $category_id ) ) {
            wp_send_json_error( __( 'Invalid category ID', 'wc-delcampe-integration' ) );
        }
        
        // Get category API instance
        $category_api = Delcampe_Category_API::get_instance();
        
        // Build category path
        $path = $category_api->get_category_path( $category_id );
        
        if ( is_wp_error( $path ) ) {
            wp_send_json_error( $path->get_error_message() );
        }
        
        wp_send_json_success( array( 'path' => $path ) );
    }
}
