<?php
/**
 * Delcampe Profiles Model
 * Version: 1.1.4.0
 *
 * Handles the storage, retrieval, and management of Delcampe listing profiles.
 * Profiles allow users to create predefined settings that can be applied to multiple products,
 * including category mappings, listing details, and sync settings.
 * Enhanced with stamp-specific fields for philatelic listings.
 *
 * @package WooCommerce_Delcampe_Integration
 * @subpackage Profiles
 * @since 1.1.3.0
 * @version 1.1.4.0
 */

// Exit if accessed directly to prevent direct file access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Class Delcampe_Profiles_Model
 * 
 * Manages Delcampe listing profiles with database operations, caching, and utilities.
 * Implements singleton pattern for consistent data access across the plugin.
 * 
 * @version 1.1.4.0
 */
class Delcampe_Profiles_Model {

    /**
     * Singleton instance
     * @var Delcampe_Profiles_Model|null
     */
    private static $instance = null;

    /**
     * WordPress database object
     * @var wpdb
     */
    private $wpdb;

    /**
     * Table name for profiles
     * @var string
     */
    private $table_name;

    /**
     * Cache key for storing profiles data
     * @var string
     */
    const PROFILES_CACHE_KEY = 'delcampe_profiles_cache';

    /**
     * Cache duration (12 hours)
     * @var int
     */
    const CACHE_DURATION = 12 * HOUR_IN_SECONDS;

    /**
     * Total items count for pagination
     * @var int
     */
    public $total_items;

    /**
     * Private constructor to enforce singleton pattern
     */
    private function __construct() {
        global $wpdb;
        $this->wpdb = $wpdb;
        $this->table_name = $this->wpdb->prefix . 'delcampe_profiles';
    }

    /**
     * Get singleton instance
     * 
     * @return Delcampe_Profiles_Model The singleton instance
     */
    public static function get_instance() {
        if ( null === self::$instance ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Create the profiles database table
     * 
     * Creates table structure for storing profile data with all necessary fields
     * Enhanced in v1.1.4.0 to include stamp_settings and description_settings
     * 
     * @return void
     */
    public function create_table() {
        require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );

        $charset_collate = $this->wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$this->table_name} (
            profile_id mediumint(9) NOT NULL AUTO_INCREMENT,
            profile_name varchar(255) NOT NULL,
            profile_description text,
            delcampe_category_id varchar(50),
            delcampe_category_name text,
            delcampe_category_path text,
            woo_category_id bigint(20),
            woo_category_name varchar(255),
            listing_details longtext,
            sync_settings longtext,
            stamp_settings longtext,
            shipping_model_id int(11) DEFAULT NULL,
            shipping_settings longtext,
            payment_settings longtext,
            description_settings longtext,
            is_default tinyint(1) DEFAULT 0,
            sort_order int(11) DEFAULT 0,
            date_created datetime DEFAULT CURRENT_TIMESTAMP,
            date_modified datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (profile_id),
            KEY delcampe_category_id (delcampe_category_id),
            KEY woo_category_id (woo_category_id),
            KEY is_default (is_default)
        ) $charset_collate;";

        dbDelta( $sql );
        
        // Ensure shipping_model_id column exists (added in v1.9.0.3)
        $this->ensure_shipping_model_column();
        
        delcampe_log( '[Delcampe Profiles Model v1.1.4.0] Profiles table created/updated: ' . $this->table_name );
    }
    
    /**
     * Ensure shipping_model_id column exists
     * Added for backward compatibility with existing installations
     * 
     * @since 1.9.0.3
     * @return void
     */
    private function ensure_shipping_model_column() {
        $column_exists = $this->wpdb->get_results(
            "SHOW COLUMNS FROM {$this->table_name} LIKE 'shipping_model_id'"
        );
        
        if (empty($column_exists)) {
            $this->wpdb->query(
                "ALTER TABLE {$this->table_name} 
                ADD COLUMN shipping_model_id int(11) DEFAULT NULL 
                AFTER stamp_settings"
            );
            delcampe_log( '[Delcampe Profiles Model v1.9.0.3] Added shipping_model_id column to profiles table' );
        }
    }

    /**
     * Get all profiles
     * 
     * Retrieves all profiles ordered by sort order and name
     * 
     * @return array Array of profile records
     */
    public function get_all() {
        $profiles = $this->wpdb->get_results( "
            SELECT * 
            FROM {$this->table_name}
            ORDER BY sort_order ASC, profile_name ASC
        ", ARRAY_A );

        // Decode JSON fields
        foreach( $profiles as &$profile ) {
            $profile['listing_details'] = $this->decode_json( $profile['listing_details'] );
            $profile['sync_settings'] = $this->decode_json( $profile['sync_settings'] );
            $profile['stamp_settings'] = $this->decode_json( $profile['stamp_settings'] );
            $profile['shipping_settings'] = $this->decode_json( $profile['shipping_settings'] );
            $profile['payment_settings'] = $this->decode_json( $profile['payment_settings'] );
            $profile['description_settings'] = $this->decode_json( $profile['description_settings'] );
        }

        return $profiles;
    }

    /**
     * Get a specific profile by ID
     * 
     * @param int $profile_id Profile ID to retrieve
     * @return array|null Profile data or null if not found
     */
    public function get_profile( $profile_id ) {
        $profile = $this->wpdb->get_row( 
            $this->wpdb->prepare( "
                SELECT * 
                FROM {$this->table_name}
                WHERE profile_id = %d
            ", $profile_id ),
            ARRAY_A
        );

        if ( $profile ) {
            $profile['listing_details'] = $this->decode_json( $profile['listing_details'] );
            $profile['sync_settings'] = $this->decode_json( $profile['sync_settings'] );
            $profile['stamp_settings'] = $this->decode_json( $profile['stamp_settings'] );
            $profile['shipping_settings'] = $this->decode_json( $profile['shipping_settings'] );
            $profile['payment_settings'] = $this->decode_json( $profile['payment_settings'] );
            $profile['description_settings'] = $this->decode_json( $profile['description_settings'] );
        }

        return $profile;
    }

    /**
     * Create a new profile
     * 
     * @param array $data Profile data
     * @return int|WP_Error Inserted profile ID or error
     */
    public function create_profile( $data ) {
        // Prepare data for insertion
        $insert_data = $this->prepare_profile_data( $data );
        
        // Validate required fields
        if ( empty( $insert_data['profile_name'] ) ) {
            return new WP_Error( 'missing_name', __( 'Profile name is required.', 'wc-delcampe-integration' ) );
        }

        // Insert profile
        $result = $this->wpdb->insert(
            $this->table_name,
            $insert_data,
            array( '%s', '%s', '%s', '%s', '%s', '%d', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%d' )
        );

        if ( false === $result ) {
            delcampe_log( '[Delcampe Profiles Model v1.1.4.0] Database insert error: ' . $this->wpdb->last_error );
            return new WP_Error( 'db_error', __( 'Failed to create profile.', 'wc-delcampe-integration' ) );
        }

        // Clear cache
        delete_transient( self::PROFILES_CACHE_KEY );
        
        return $this->wpdb->insert_id;
    }

    /**
     * Update an existing profile
     * 
     * @param int   $profile_id Profile ID to update
     * @param array $data       New profile data
     * @return bool|WP_Error True on success or error
     */
    public function update_profile( $profile_id, $data ) {
        // Prepare data for update
        $update_data = $this->prepare_profile_data( $data );
        
        // Remove date_created from update data
        unset( $update_data['date_created'] );

        // Update profile
        $result = $this->wpdb->update(
            $this->table_name,
            $update_data,
            array( 'profile_id' => $profile_id ),
            array( '%s', '%s', '%s', '%s', '%s', '%d', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%d', '%d' ),
            array( '%d' )
        );

        if ( false === $result ) {
            delcampe_log( '[Delcampe Profiles Model v1.1.4.0] Database update error: ' . $this->wpdb->last_error );
            return new WP_Error( 'db_error', __( 'Failed to update profile.', 'wc-delcampe-integration' ) );
        }

        // Clear cache
        delete_transient( self::PROFILES_CACHE_KEY );
        
        return true;
    }

    /**
     * Delete a profile
     * 
     * @param int $profile_id Profile ID to delete
     * @return bool|WP_Error True on success or error
     */
    public function delete_profile( $profile_id ) {
        // Check if profile is in use
        $products_using = $this->get_products_using_profile( $profile_id );
        if ( ! empty( $products_using ) ) {
            return new WP_Error( 
                'profile_in_use', 
                sprintf( 
                    __( 'This profile is used by %d products and cannot be deleted.', 'wc-delcampe-integration' ),
                    count( $products_using )
                )
            );
        }

        // Delete profile
        $result = $this->wpdb->delete(
            $this->table_name,
            array( 'profile_id' => $profile_id ),
            array( '%d' )
        );

        if ( false === $result ) {
            delcampe_log( '[Delcampe Profiles Model v1.1.4.0] Database delete error: ' . $this->wpdb->last_error );
            return new WP_Error( 'db_error', __( 'Failed to delete profile.', 'wc-delcampe-integration' ) );
        }

        // Clear cache
        delete_transient( self::PROFILES_CACHE_KEY );
        
        return true;
    }

    /**
     * Duplicate a profile
     * 
     * @param int $profile_id Profile ID to duplicate
     * @return int|WP_Error New profile ID or error
     */
    public function duplicate_profile( $profile_id ) {
        // Get original profile
        $original = $this->get_profile( $profile_id );
        if ( ! $original ) {
            return new WP_Error( 'not_found', __( 'Profile not found.', 'wc-delcampe-integration' ) );
        }

        // Prepare duplicate data
        unset( $original['profile_id'] );
        unset( $original['date_created'] );
        unset( $original['date_modified'] );
        $original['profile_name'] .= ' ' . __( '(copy)', 'wc-delcampe-integration' );
        $original['is_default'] = 0;

        // Create duplicate
        return $this->create_profile( $original );
    }

    /**
     * Get all profile names for dropdown lists
     * 
     * @return array Array of profile_id => profile_name
     */
    public function get_all_names() {
        $results = $this->wpdb->get_results( "
            SELECT profile_id, profile_name 
            FROM {$this->table_name}
            ORDER BY sort_order ASC, profile_name ASC
        " );

        $profiles = array();
        foreach( $results as $result ) {
            $profiles[ $result->profile_id ] = $result->profile_name;
        }

        return $profiles;
    }

    /**
     * Get paginated profiles for admin list table
     * 
     * @param int $current_page Current page number
     * @param int $per_page     Items per page
     * @return array Array of profile records
     */
    public function get_page_items( $current_page, $per_page ) {
        $orderby  = ! empty( $_REQUEST['orderby'] ) ? esc_sql( $_REQUEST['orderby'] ) : 'profile_name';
        $order    = ! empty( $_REQUEST['order'] ) ? esc_sql( $_REQUEST['order'] ) : 'asc';
        $offset   = ( $current_page - 1 ) * $per_page;

        // Handle special ordering for profile name
        if ( $orderby == 'profile_name' ) {
            $orderby = "sort_order {$order}, profile_name";
        }

        $where_sql = 'WHERE 1 = 1 ';

        // Handle search
        $search_query = isset( $_REQUEST['s'] ) ? esc_sql( sanitize_text_field( $_REQUEST['s'] ) ) : false;
        if ( $search_query ) {
            $where_sql .= $this->wpdb->prepare( "
                AND ( profile_name LIKE %s
                   OR profile_description LIKE %s
                   OR delcampe_category_name LIKE %s )
            ", '%' . $search_query . '%', '%' . $search_query . '%', '%' . $search_query . '%' );
        }

        // Get items
        $items = $this->wpdb->get_results( "
            SELECT *
            FROM {$this->table_name}
            {$where_sql}
            ORDER BY {$orderby} {$order}
            LIMIT {$offset}, {$per_page}
        ", ARRAY_A );

        // Get total count
        $this->total_items = $this->wpdb->get_var( "
            SELECT COUNT(*)
            FROM {$this->table_name}
            {$where_sql}
        " );

        // Decode JSON fields
        foreach( $items as &$profile ) {
            $profile['listing_details'] = $this->decode_json( $profile['listing_details'] );
            $profile['sync_settings'] = $this->decode_json( $profile['sync_settings'] );
            $profile['stamp_settings'] = $this->decode_json( $profile['stamp_settings'] );
            $profile['shipping_settings'] = $this->decode_json( $profile['shipping_settings'] );
            $profile['payment_settings'] = $this->decode_json( $profile['payment_settings'] );
            $profile['description_settings'] = $this->decode_json( $profile['description_settings'] );
        }

        return $items;
    }

    /**
     * Get products using a specific profile
     * 
     * @param int $profile_id Profile ID
     * @return array Array of product IDs
     */
    public function get_products_using_profile( $profile_id ) {
        $meta_key = '_delcampe_profile_id';
        
        $product_ids = $this->wpdb->get_col( 
            $this->wpdb->prepare( "
                SELECT post_id 
                FROM {$this->wpdb->postmeta}
                WHERE meta_key = %s AND meta_value = %s
            ", $meta_key, $profile_id )
        );

        return $product_ids;
    }

    /**
     * Get profile by WooCommerce category ID
     * 
     * @param int $woo_category_id WooCommerce category ID
     * @return array|null First matching profile or null
     */
    public function get_profile_by_woo_category( $woo_category_id ) {
        $profile = $this->wpdb->get_row( 
            $this->wpdb->prepare( "
                SELECT * 
                FROM {$this->table_name}
                WHERE woo_category_id = %d
                ORDER BY is_default DESC, sort_order ASC
                LIMIT 1
            ", $woo_category_id ),
            ARRAY_A
        );

        if ( $profile ) {
            $profile['listing_details'] = $this->decode_json( $profile['listing_details'] );
            $profile['sync_settings'] = $this->decode_json( $profile['sync_settings'] );
            $profile['stamp_settings'] = $this->decode_json( $profile['stamp_settings'] );
            $profile['shipping_settings'] = $this->decode_json( $profile['shipping_settings'] );
            $profile['payment_settings'] = $this->decode_json( $profile['payment_settings'] );
            $profile['description_settings'] = $this->decode_json( $profile['description_settings'] );
        }

        return $profile;
    }

    /**
     * Get default profile
     * 
     * @return array|null Default profile or null
     */
    public function get_default_profile() {
        $profile = $this->wpdb->get_row( "
            SELECT * 
            FROM {$this->table_name}
            WHERE is_default = 1
            LIMIT 1
        ", ARRAY_A );

        if ( $profile ) {
            $profile['listing_details'] = $this->decode_json( $profile['listing_details'] );
            $profile['sync_settings'] = $this->decode_json( $profile['sync_settings'] );
            $profile['stamp_settings'] = $this->decode_json( $profile['stamp_settings'] );
            $profile['shipping_settings'] = $this->decode_json( $profile['shipping_settings'] );
            $profile['payment_settings'] = $this->decode_json( $profile['payment_settings'] );
            $profile['description_settings'] = $this->decode_json( $profile['description_settings'] );
        }

        return $profile;
    }

    /**
     * Set a profile as default
     * 
     * @param int $profile_id Profile ID to set as default
     * @return bool|WP_Error True on success or error
     */
    public function set_default_profile( $profile_id ) {
        // Remove default from all profiles
        $this->wpdb->update(
            $this->table_name,
            array( 'is_default' => 0 ),
            array( 'is_default' => 1 ),
            array( '%d' ),
            array( '%d' )
        );

        // Set new default
        $result = $this->wpdb->update(
            $this->table_name,
            array( 'is_default' => 1 ),
            array( 'profile_id' => $profile_id ),
            array( '%d' ),
            array( '%d' )
        );

        if ( false === $result ) {
            return new WP_Error( 'db_error', __( 'Failed to set default profile.', 'wc-delcampe-integration' ) );
        }

        // Clear cache
        delete_transient( self::PROFILES_CACHE_KEY );
        
        return true;
    }

    /**
     * Prepare profile data for database
     * 
     * @param array $data Raw profile data
     * @return array Prepared data for database
     */
    private function prepare_profile_data( $data ) {
        $prepared = array(
            'profile_name' => isset( $data['profile_name'] ) ? 
                sanitize_text_field( $data['profile_name'] ) : '',
            'profile_description' => isset( $data['profile_description'] ) ? 
                sanitize_textarea_field( $data['profile_description'] ) : '',
            'delcampe_category_id' => isset( $data['delcampe_category_id'] ) ? 
                sanitize_text_field( $data['delcampe_category_id'] ) : null,
            'delcampe_category_name' => isset( $data['delcampe_category_name'] ) ? 
                sanitize_text_field( $data['delcampe_category_name'] ) : null,
            'delcampe_category_path' => isset( $data['delcampe_category_path'] ) ? 
                sanitize_text_field( $data['delcampe_category_path'] ) : null,
            'woo_category_id' => isset( $data['woo_category_id'] ) ? 
                intval( $data['woo_category_id'] ) : null,
            'woo_category_name' => isset( $data['woo_category_name'] ) ? 
                sanitize_text_field( $data['woo_category_name'] ) : null,
            'listing_details' => isset( $data['listing_details'] ) ? 
                wp_json_encode( $data['listing_details'] ) : '{}',
            'sync_settings' => isset( $data['sync_settings'] ) ? 
                wp_json_encode( $data['sync_settings'] ) : '{}',
            'stamp_settings' => isset( $data['stamp_settings'] ) ? 
                wp_json_encode( $data['stamp_settings'] ) : '{}',
            'shipping_model_id' => isset( $data['shipping_model_id'] ) ? 
                intval( $data['shipping_model_id'] ) : null,
            'shipping_settings' => isset( $data['shipping_settings'] ) ? 
                wp_json_encode( $data['shipping_settings'] ) : '{}',
            'payment_settings' => isset( $data['payment_settings'] ) ? 
                wp_json_encode( $data['payment_settings'] ) : '{}',
            'description_settings' => isset( $data['description_settings'] ) ? 
                wp_json_encode( $data['description_settings'] ) : '{}',
            'is_default' => isset( $data['is_default'] ) ? 
                intval( $data['is_default'] ) : 0,
            'sort_order' => isset( $data['sort_order'] ) ? 
                intval( $data['sort_order'] ) : 0,
        );

        return $prepared;
    }

    /**
     * Decode JSON data with error handling
     * 
     * @param string $json JSON string to decode
     * @return array Decoded array or empty array on error
     */
    private function decode_json( $json ) {
        if ( empty( $json ) ) {
            return array();
        }

        $decoded = json_decode( $json, true );
        
        if ( json_last_error() !== JSON_ERROR_NONE ) {
            delcampe_log( '[Delcampe Profiles Model v1.1.4.0] JSON decode error: ' . json_last_error_msg() );
            return array();
        }

        return is_array( $decoded ) ? $decoded : array();
    }

    /**
     * Get new profile template
     * Enhanced in v1.1.4.0 with stamp-specific fields
     * 
     * @return array Template for new profile
     */
    public function get_new_profile_template() {
        // Load stamp-specific field helper
        if ( file_exists( plugin_dir_path( __FILE__ ) . 'stamp-profile-fields.php' ) ) {
            require_once plugin_dir_path( __FILE__ ) . 'stamp-profile-fields.php';
            $stamp_fields = delcampe_get_stamp_profile_fields();
        } else {
            $stamp_fields = array();
        }
        
        return array(
            'profile_id' => false,
            'profile_name' => __( 'New Profile', 'wc-delcampe-integration' ),
            'profile_description' => '',
            'delcampe_category_id' => '',
            'delcampe_category_name' => '',
            'delcampe_category_path' => '',
            'woo_category_id' => '',
            'woo_category_name' => '',
            'listing_details' => isset( $stamp_fields['listing_details'] ) ? 
                $stamp_fields['listing_details'] : array(
                    'duration' => '30',
                    'auto_renew' => '1',
                    'listing_type' => 'FixedPrice',
                    'price_adjustment' => '',
                    'quantity_mode' => 'stock',
                    'fixed_quantity' => '1',
                    'currency' => get_woocommerce_currency(),
                    'minimum_price' => '',
                    'image_selection' => 'all',
                    'selected_images' => array(),
                ),
            'sync_settings' => isset( $stamp_fields['sync_settings'] ) ? 
                $stamp_fields['sync_settings'] : array(
                    'sync_title' => '1',
                    'sync_description' => '1',
                    'sync_price' => '1',
                    'sync_quantity' => '1',
                    'sync_images' => '1',
                    'sync_attributes' => '1',
                    'auto_sync' => '0',
                    'sync_interval' => '24',
                ),
            'stamp_settings' => array(
                'default_condition' => 'Used',
                'catalog_prefix' => 'Michel',
                'add_catalog_number' => '1',
                'perforation_details' => '1',
                'watermark_info' => '1',
                'gum_condition' => '',
            ),
            'shipping_settings' => array(
                'shipping_type' => 'Standard',
                'shipping_cost' => '2.50',
                'additional_item_cost' => '0.50',
                'combined_shipping' => '1',
                'ships_to' => 'Worldwide',
                'excluded_countries' => '',
            ),
            'payment_settings' => array(
                'payment_methods' => array('PayPal', 'BankTransfer'),
                'payment_deadline' => '7',
                'payment_instructions' => __( 'Payment expected within 7 days. Combined shipping available.', 'wc-delcampe-integration' ),
            ),
            'description_settings' => isset( $stamp_fields['description_settings'] ) ? 
                $stamp_fields['description_settings'] : array(
                    'template_type' => 'stamp_standard',
                    'include_technical' => '1',
                    'include_catalog_info' => '1',
                    'include_condition_details' => '1',
                    'include_shipping_info' => '1',
                    'include_payment_terms' => '1',
                    'custom_footer' => '',
                ),
            'is_default' => 0,
            'sort_order' => 0,
        );
    }
}
