<?php
/**
 * WooCommerce Delcampe Integration Uninstall
 *
 * WHEN THIS FILE IS CALLED:
 * This file is automatically executed by WordPress when:
 * 1. The plugin is deleted through the WordPress admin interface (Plugins page)
 * 2. User clicks "Delete" after deactivating the plugin
 * 3. NOT called when plugin is just deactivated (data is preserved)
 * 
 * WHAT IT DOES:
 * Completely removes ALL plugin data from the database including:
 * - All custom database tables (19 tables as of v1.10.18.13)
 * - All plugin options and settings
 * - All transients and cached data
 * - All scheduled cron jobs
 * - All product meta data related to Delcampe
 * - All log files and upload directories
 * 
 * WARNING: This is irreversible! All Delcampe sync data will be permanently deleted.
 *
 * @package     WooCommerce_Delcampe_Integration
 * @version     1.10.18.13
 * @since       1.8.0
 */

// Exit if uninstall not called from WordPress
if (!defined('WP_UNINSTALL_PLUGIN')) {
    exit;
}

/**
 * Remove all plugin data
 * 
 * This function removes:
 * - Database tables
 * - Options
 * - Transients
 * - Scheduled cron jobs
 * - User meta
 * - Post meta related to the plugin
 * 
 * @since 1.8.0
 */
function delcampe_uninstall() {
    global $wpdb;
    
    // Define table names - comprehensive list from v1.10.18.13
    $tables = array(
        $wpdb->prefix . 'delcampe_listings',
        $wpdb->prefix . 'delcampe_profiles', 
        $wpdb->prefix . 'delcampe_orders',
        $wpdb->prefix . 'delcampe_order_fees',
        $wpdb->prefix . 'delcampe_inventory_sync',
        $wpdb->prefix . 'delcampe_inventory_alerts',
        $wpdb->prefix . 'delcampe_selected_categories',
        $wpdb->prefix . 'delcampe_category_mapping',
        $wpdb->prefix . 'delcampe_batch_queue',           // v1.10.18.0
        $wpdb->prefix . 'delcampe_bulk_operations',       // Bulk operations
        $wpdb->prefix . 'delcampe_business_events',       // v1.10.12.0 Business logger
        $wpdb->prefix . 'delcampe_image_sync_queue',      // Image sync queue
        $wpdb->prefix . 'delcampe_license',               // License management
        $wpdb->prefix . 'delcampe_price_sync_queue',      // Price sync queue
        $wpdb->prefix . 'delcampe_sync_queue',            // Sync queue table
        $wpdb->prefix . 'delcampe_webhooks',              // Webhook registrations
        $wpdb->prefix . 'delcampe_listing_tracker',       // v1.10.18.13 Listing audit trail
        $wpdb->prefix . 'delcampe_reconciliation_log'     // v1.10.18.13 Reconciliation logs
    );
    
    // Drop all plugin tables
    // SECURITY FIX v1.10.21.2: Properly escape table names to prevent SQL injection
    foreach ($tables as $table) {
        // Table names come from wpdb->prefix which is safe, but escape for defense in depth
        $safe_table = esc_sql($table);
        $wpdb->query("DROP TABLE IF EXISTS `$safe_table`");
    }
    
    // Remove all plugin options - comprehensive list
    $options = array(
        // Core settings
        'delcampe_api_key',
        'delcampe_enable_logging',
        'delcampe_plugin_version',
        
        // Inventory settings
        'delcampe_auto_sync_inventory',
        'delcampe_inventory_check_frequency',
        'delcampe_low_stock_alert',
        'delcampe_prevent_oversell',
        
        // Order settings
        'delcampe_order_import_enabled',
        'delcampe_order_import_frequency',
        'delcampe_last_order_import',
        'delcampe_auto_import_orders',
        'delcampe_emergency_stop',
        
        // Sync settings
        'delcampe_sync_queue',
        'delcampe_sync_status',
        'delcampe_batch_size',
        'delcampe_processed_closed_items',
        
        // Webhook settings
        'delcampe_webhook_token',
        'delcampe_failover_email',
        
        // License/Trial settings
        'delcampe_trial_start_date',
        'delcampe_trial_active',
        'delcampe_license_key',
        'delcampe_license_status',
        
        // Reconciliation
        'delcampe_last_reconciliation',
        
        // Other
        'delcampe_shipping_models',
        'delcampe_seller_id'
    );
    
    foreach ($options as $option) {
        delete_option($option);
    }
    
    // Remove all transients
    $transients = array(
        'delcampe_auth_token',
        'delcampe_categories',
        'delcampe_category_mapping_cache',
        'delcampe_profiles_cache',
        'delcampe_shipping_models_cache'
    );
    
    foreach ($transients as $transient) {
        delete_transient($transient);
    }
    
    // Clear scheduled cron jobs - comprehensive list
    $cron_hooks = array(
        // Core sync
        'delcampe_sync_cron',
        'delcampe_process_sync_queue',
        
        // Orders
        'delcampe_import_orders_cron',
        'delcampe_check_orders',
        
        // Inventory
        'delcampe_inventory_check_cron',
        
        // Status checking
        'delcampe_check_processing_status',
        
        // Batch processing (v1.10.18.0)
        'delcampe_process_batch_queue',
        'delcampe_check_stuck_batches',
        
        // Reconciliation (v1.10.18.13)
        'delcampe_daily_reconciliation',
        
        // Closed listings sync (v1.10.18.8)
        'delcampe_sync_closed_listings'
    );
    
    foreach ($cron_hooks as $hook) {
        wp_clear_scheduled_hook($hook);
    }
    
    // Remove post meta from all products
    $meta_keys = array(
        '_delcampe_profile_id',
        '_delcampe_listing_id',
        '_delcampe_sync_status',
        '_delcampe_last_sync',
        '_delcampe_last_error',
        '_delcampe_personal_reference',
        '_delcampe_pending_batch',
        '_delcampe_batch_queued',
        '_delcampe_id',  // Alternative listing ID key
        '_delcampe_last_reconciled'
    );
    
    foreach ($meta_keys as $meta_key) {
        $wpdb->delete($wpdb->postmeta, array('meta_key' => $meta_key));
    }
    
    // Remove user meta
    $user_meta_keys = array(
        'delcampe_dismissed_notices',
        'delcampe_screen_options'
    );
    
    foreach ($user_meta_keys as $meta_key) {
        $wpdb->delete($wpdb->usermeta, array('meta_key' => $meta_key));
    }
    
    // Clean up any files in uploads directory
    $upload_dir = wp_upload_dir();
    $delcampe_dir = $upload_dir['basedir'] . '/delcampe';
    
    if (is_dir($delcampe_dir)) {
        delcampe_remove_directory($delcampe_dir);
    }
    
    // Remove log files from old location (for backward compatibility)
    $old_log_files = array(
        $upload_dir['basedir'] . '/delcampe-sync.log',
        $upload_dir['basedir'] . '/delcampe-debug.log'
    );
    
    foreach ($old_log_files as $log_file) {
        if (file_exists($log_file)) {
            @unlink($log_file);
        }
    }
    
    // Remove new log directory and files (since v1.10.1.0)
    $log_dir = WP_CONTENT_DIR . '/delcampe-logs';
    if (is_dir($log_dir)) {
        // Remove log files
        $log_files = glob($log_dir . '/*.log');
        foreach ($log_files as $log_file) {
            @unlink($log_file);
        }
        // Remove directory
        @rmdir($log_dir);
    }
    
    // Clear any cached data
    wp_cache_flush();
}

/**
 * Recursively remove a directory and its contents
 * 
 * @param string $dir Directory path to remove
 * @since 1.8.0
 */
function delcampe_remove_directory($dir) {
    if (!is_dir($dir)) {
        return;
    }
    
    $files = array_diff(scandir($dir), array('.', '..'));
    
    foreach ($files as $file) {
        $path = $dir . '/' . $file;
        
        if (is_dir($path)) {
            delcampe_remove_directory($path);
        } else {
            @unlink($path);
        }
    }
    
    @rmdir($dir);
}

// Run the uninstall function
delcampe_uninstall();
