<?php
/**
 * Delcampe Webhook Callback Handler
 * 
 * This file receives POST notifications from Delcampe when events occur.
 * Delcampe sends the XML in the 'delcampeNotification' POST parameter.
 * 
 * @package WooCommerce_Delcampe_Integration
 * @since 1.10.13.0
 */

// Load WordPress
$wp_load_paths = array(
    dirname(__FILE__) . '/../../../../wp-load.php',
    dirname(__FILE__) . '/../../../wp-load.php',
    '/var/www/html/wp-load.php',
);

$wp_loaded = false;
foreach ($wp_load_paths as $path) {
    if (file_exists($path)) {
        require_once($path);
        $wp_loaded = true;
        break;
    }
}

if (!$wp_loaded) {
    http_response_code(500);
    die('WordPress not found');
}

// Initialize response
http_response_code(200);
header('Content-Type: text/plain');

// Get logger instance
$logger = Delcampe_Sync_Logger::get_instance();

// Log the incoming request
$logger->write_log(array(
    'timestamp' => current_time('Y-m-d H:i:s'),
    'event' => 'WEBHOOK_REQUEST_RECEIVED',
    'method' => $_SERVER['REQUEST_METHOD'],
    'content_type' => $_SERVER['CONTENT_TYPE'] ?? 'not set',
    'ip' => $_SERVER['REMOTE_ADDR'],
    'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'not set'
));

// Check request method
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $logger->write_log(array(
        'timestamp' => current_time('Y-m-d H:i:s'),
        'event' => 'WEBHOOK_INVALID_METHOD',
        'method' => $_SERVER['REQUEST_METHOD']
    ));
    http_response_code(405);
    die('Method not allowed');
}

// Get the XML from POST data
// Delcampe sends it in the 'delcampeNotification' parameter
$xml_data = isset($_POST['delcampeNotification']) ? $_POST['delcampeNotification'] : null;

if (empty($xml_data)) {
    // Try raw POST body as fallback
    $xml_data = file_get_contents('php://input');
}

if (empty($xml_data)) {
    $logger->write_log(array(
        'timestamp' => current_time('Y-m-d H:i:s'),
        'event' => 'WEBHOOK_NO_DATA',
        'post_data' => $_POST,
        'raw_input' => file_get_contents('php://input')
    ));
    http_response_code(400);
    die('No notification data received');
}

// Parse XML
libxml_use_internal_errors(true);
$xml = simplexml_load_string($xml_data);

if ($xml === false) {
    $errors = libxml_get_errors();
    $logger->write_log(array(
        'timestamp' => current_time('Y-m-d H:i:s'),
        'event' => 'WEBHOOK_XML_PARSE_ERROR',
        'errors' => array_map(function($error) {
            return $error->message;
        }, $errors),
        'raw_xml' => substr($xml_data, 0, 1000) // First 1000 chars for debugging
    ));
    http_response_code(400);
    die('Invalid XML');
}

// Log the parsed notification
$logger->write_log(array(
    'timestamp' => current_time('Y-m-d H:i:s'),
    'event' => 'WEBHOOK_XML_PARSED',
    'notification_type' => isset($xml->Notification_Type) ? (string)$xml->Notification_Type : 'unknown',
    'notification_datetime' => isset($xml->Notification_Datetime) ? (string)$xml->Notification_Datetime : '',
    'raw_xml' => $xml_data
));

// SECURITY FIX v1.10.21.0: Mandatory token verification with constant-time comparison
$expected_token = get_option('delcampe_webhook_token');
$received_token = isset($xml->Notification_Token) ? (string)$xml->Notification_Token : '';

// Reject if no token is configured (security requirement)
if (empty($expected_token)) {
    $logger->write_log(array(
        'timestamp' => current_time('Y-m-d H:i:s'),
        'event' => 'WEBHOOK_NO_TOKEN_CONFIGURED',
        'error' => 'Webhook token not configured - rejecting request for security'
    ));
    http_response_code(503);
    die('Service Unavailable - Security configuration required');
}

// Use constant-time comparison to prevent timing attacks
if (empty($received_token) || !hash_equals($expected_token, $received_token)) {
    $logger->write_log(array(
        'timestamp' => current_time('Y-m-d H:i:s'),
        'event' => 'WEBHOOK_AUTHENTICATION_FAILED',
        'expected_length' => strlen($expected_token),
        'received_length' => strlen($received_token),
        'ip_address' => $_SERVER['REMOTE_ADDR'] ?? 'unknown'
    ));
    http_response_code(403);
    die('Authentication failed');
}

// Check for duplicate notification (idempotency)
$notification_hash = md5($xml_data);
$processed_key = 'delcampe_webhook_' . $notification_hash;

// Use transient to track processed notifications (24 hour expiry)
if (get_transient($processed_key)) {
    $logger->write_log(array(
        'timestamp' => current_time('Y-m-d H:i:s'),
        'event' => 'WEBHOOK_DUPLICATE_SKIPPED',
        'hash' => $notification_hash,
        'notification_type' => isset($xml->Notification_Type) ? (string)$xml->Notification_Type : 'unknown'
    ));
    echo "OK (duplicate)";
    exit;
}

// Mark as processed
set_transient($processed_key, current_time('mysql'), 86400); // 24 hours

// Process the notification
try {
    // Get webhook handler
    if (!class_exists('Delcampe_Webhook_Handler')) {
        require_once(DWC_PLUGIN_DIR . 'includes/class-delcampe-webhook-handler.php');
    }
    
    $handler = new Delcampe_Webhook_Handler();
    
    // Process based on notification type
    $notification_type = isset($xml->Notification_Type) ? (string)$xml->Notification_Type : null;
    
    if (!$notification_type) {
        throw new Exception('Missing notification type');
    }
    
    // Remove 'Curl_' prefix if present (Delcampe uses both forms)
    $notification_type = str_replace('Curl_', '', $notification_type);
    
    // Update the XML object to use the cleaned type
    $xml->Notification_Type = $notification_type;
    
    // Process the notification
    $result = $handler->process_notification($xml, $xml_data);
    
    if ($result['success']) {
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'WEBHOOK_PROCESSED_SUCCESS',
            'notification_type' => $notification_type,
            'message' => $result['message']
        ));
        echo "OK";
    } else {
        $logger->write_log(array(
            'timestamp' => current_time('Y-m-d H:i:s'),
            'event' => 'WEBHOOK_PROCESSED_ERROR',
            'notification_type' => $notification_type,
            'error' => $result['message']
        ));
        http_response_code(500);
        echo "Error: " . $result['message'];
    }
    
} catch (Exception $e) {
    $logger->write_log(array(
        'timestamp' => current_time('Y-m-d H:i:s'),
        'event' => 'WEBHOOK_EXCEPTION',
        'error' => $e->getMessage(),
        'trace' => $e->getTraceAsString()
    ));
    http_response_code(500);
    echo "Error: " . $e->getMessage();
}